////////////////////////////////////////////////////////////////////////////////
// Cvt.cpp
////////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"

#include "cvt.h"
#include "cvtdib.h"
#include "cvtutil.h"

#if	TARGET == WIN16
// The 16-bit compiler does not generate proper optimized code
#pragma	optimize("", off)
#endif

CConvert::CConvert(HWND hCtl, HWND hMrkCtl)
{
	ASSERT(hCtl != 0);
	m_hCtl = hCtl;
	m_hMrkCtl = hMrkCtl;
	memset(&m_settings, 0, sizeof(m_settings));
	m_settings.inputFile[0] = '\0';
	m_settings.outputFile[0] = '\0';
	m_settings.outputWidth = 128;
	m_settings.outputHeight = 128;
	m_settings.units = CTLUNIT_PIXEL;
	m_settings.stepsPerInch = 1.0;
	m_settings.outputType = PC_RASTER;
	m_settings.outputFormat = 0;
	m_settings.outputSubFormat = 0;
	m_settings.outputColorDepth = 0;
}
CConvert::~CConvert()
{
}

BOOL CConvert::InitDIB(PAN_CtlFileInfo* pFileInfo)
{
	ASSERT(pFileInfo != 0);
	ASSERT(IsWindow(m_hCtl));

	m_pCvtDIB = new CCvtDIB();

	// Get color depth of source image.
	int	in_colorDepth = (int) ::SendMessage(m_hCtl, PM_CTLGETPALETTE, 0, 0L);
	int	out_colorDepth = m_settings.outputColorDepth;

	/*
	** Get output color depth.
	*/
	if (out_colorDepth <= 0) {
		out_colorDepth = in_colorDepth;
	}

	/*
	**  Match available depths in output filter
	*/
	if (m_settings.outputType == PC_RASTER) {
		int 			maxdepth = 0;
		BOOL			bMatchDepth = FALSE;
		
		CCvtUtil	cut(m_hCtl);
		cut.QueryFormat(m_settings.outputFormat);
		cut.QuerySubFormat(m_settings.outputFormat, m_settings.outputSubFormat);

		for (int i = 0; i < cut.m_wNumDepth; i++) {
			if (maxdepth < cut.m_wColorDepth[i]) {
				maxdepth = cut.m_wColorDepth[i];
			}
			if (out_colorDepth == cut.m_wColorDepth[i]) {
				bMatchDepth = TRUE;
			}
		}
		if (! bMatchDepth) {
			out_colorDepth = maxdepth;
		}
	} else {
		out_colorDepth = in_colorDepth;
	}
	if (out_colorDepth == 0) {
		out_colorDepth = 1;
	}

	// Get width and height.
	LONG width		  = ROUND(LONG, m_settings.outputWidth);
	LONG height 	  = ROUND(LONG, m_settings.outputHeight);

	// Create and initialize palette.
	LOGPALETTE FAR	*pPalette = 0;
	
	if (out_colorDepth != 24 && out_colorDepth != 1	&& in_colorDepth >= 1 && in_colorDepth <= 8) {
		// Allocate memory for palette.
		int	in_nColors = (1 << in_colorDepth);

		pPalette = (LOGPALETTE FAR*) malloc(sizeof(LOGPALETTE) + (in_nColors - 1) * sizeof (PALETTEENTRY));

		// Initialize palette.
		pPalette->palVersion	= 0x300;
		pPalette->palNumEntries = in_nColors;
		::SendMessage(m_hCtl, PM_CTLGETPALETTE, 0, (LPARAM)pPalette);
	}

	// Create DIB.
	if (! m_pCvtDIB->CreateDIB(width, height,
			in_colorDepth, out_colorDepth,
			pFileInfo->type, m_settings.outputType, pPalette)) {

		if (pPalette != 0) {
			free(pPalette);
		}
		return FALSE;
	}

	/*
	** Width and height may have been reduced during DIB creation.
	*/
	if (m_settings.outputType == PC_RASTER) {
		m_settings.outputWidth = (Real) m_pCvtDIB->GetWidthDIB();
		m_settings.outputHeight = (Real) m_pCvtDIB->GetHeightDIB();
	}

	if (pPalette != 0) {
		free(pPalette);
	}
	return TRUE;

} // CConvert::InitDIB()

BOOL	CConvert::RenderDIB(PAN_CtlFileInfo *pFileInfo)
{
	ASSERT(pFileInfo != 0 && IsWindow(m_hCtl));

	HDC	hDC = m_pCvtDIB->GetHDC();

	if (hDC == 0) {
		return FALSE;
	}

	// Initialize render options.
	PAN_CtlRenderOptions 	renderOptions;
	_fmemset(&renderOptions, 0, sizeof renderOptions);

	// Set render mode
	if (pFileInfo->type != PAN_VectorFile) {
		/*
		** Should not set PAN_CTLMODEOPAQUE for vector files
		*/
		renderOptions.mode |= PAN_CTLMODEOPAQUE;
	} else {
		renderOptions.mode |= PAN_CTLMODEANISOTROPIC;

		if (m_pCvtDIB->GetDepthDIB() == 1) {
			// Force to black and white
			renderOptions.mode |= PAN_CTLMODEMONOCHROME;
		}
	}

	renderOptions.hdc			 = hDC;
	renderOptions.source		 = m_settings.inputRange;
	renderOptions.devRect.left	 = 0;
	renderOptions.devRect.top	 = 0;
	renderOptions.devRect.right  = (int) m_pCvtDIB->GetWidthDIB()-1;
	renderOptions.devRect.bottom = (int) m_pCvtDIB->GetHeightDIB()-1;

	// Clear the background color.
//	PatBlt(hDC, 0, 0, (int)m_pCvtDIB->GetWidthDIB(), (int)m_pCvtDIB->GetWidthDIB(),
//		m_pCvtDIB->GetDepthDIB() == 1 ? WHITENESS : BLACKNESS);

	// Clear the background color.
	HBRUSH	bkBrush  = CreateSolidBrush(RGB(255, 255, 255));
	RECT	rect = {0, 0, (int)m_pCvtDIB->GetWidthDIB(), (int)m_pCvtDIB->GetHeightDIB()};
	FillRect(hDC, &rect, bkBrush);
	DeleteObject((HGDIOBJ)bkBrush);
	bkBrush = 0;

	// Force white bg
	COLORREF	bgCol = 0;
	::SendMessage(m_hCtl, PM_CTLGETFGBGCOLOR, 1/*BG*/, (LPARAM) &bgCol);
	COLORREF	fgCol = 0;
	::SendMessage(m_hCtl, PM_CTLGETFGBGCOLOR, 0/*FG*/, (LPARAM) &fgCol);

	COLORREF	white = RGB(255, 255, 255);
	::SendMessage(m_hCtl, PM_CTLSETFGBGCOLOR, 3/*BG*/, (LPARAM) &white);

	COLORREF	black = RGB(0, 0, 0);
	::SendMessage(m_hCtl, PM_CTLSETFGBGCOLOR, 2/*FG*/, (LPARAM) &black);

	// Render contents of control onto DIB device context.
	::SendMessage(m_hCtl, PM_CTLRENDERONTODC, 0, (LPARAM) &renderOptions);

	// Account for origin flip for vector input files.
	if (pFileInfo->type == PAN_VectorFile) {
		Real tmp = renderOptions.source.min.y;

		renderOptions.source.min.y = renderOptions.source.max.y;
		renderOptions.source.max.y = tmp;
	}

	// Render mark up contents of control onto DIB device context.
	if (m_hMrkCtl) {
		MRK_RenderOptions	mrkOpts;

		memset(&mrkOpts, 0, sizeof(MRK_RenderOptions));
		
		mrkOpts.hdc = renderOptions.hdc;
		mrkOpts.source = renderOptions.source;
		mrkOpts.devRect = renderOptions.devRect;
		
#if	TARGET == TWUNIX
		if (m_pCvtDIB->GetDepthDIB() == 1) {
			// Force to black and white: TWIN does not properly dither lines and
			//  sometimes maps colors strangely. Ajit and Ye Yang Nov 25 1999
			mrkOpts.mode |= MRK_RENDERMODEMONOCHROME;
		}
#endif

		if (IsWindow(m_hMrkCtl)) {
			COLORREF	oldBkColor = bgCol;
			int			oldTrueColorFlag = FALSE;
			
			// Set bg color to white.
			::SendMessage(m_hMrkCtl, MRK_GETFGBGCOLOR, 1, (LPARAM)&oldBkColor);
			::SendMessage(m_hMrkCtl, MRK_SETFGBGCOLOR, 1, (LPARAM) white);

			// Render using entity's own color (no alternate color).
			::SendMessage(m_hMrkCtl, MRK_GET_MOPROP, MO_TRUECOLOR, (LPARAM) &oldTrueColorFlag);
			::SendMessage(m_hMrkCtl, MRK_SET_MOPROP, MO_TRUECOLOR, TRUE);

			// Render
			::SendMessage(m_hMrkCtl, MRK_RENDERONTODC, 0, (LPARAM)&mrkOpts);

			// Reset old bg color.
			::SendMessage(m_hMrkCtl, MRK_SETFGBGCOLOR, 1, (LPARAM)oldBkColor);

			// Reset old color mode.
			::SendMessage(m_hMrkCtl, MRK_SET_MOPROP, MO_TRUECOLOR, (LPARAM) oldTrueColorFlag);
		}
	}

 	// Restore fg/bg colour
	::SendMessage(m_hCtl, PM_CTLSETFGBGCOLOR, 3/*BG*/, (LPARAM) &bgCol);
	::SendMessage(m_hCtl, PM_CTLSETFGBGCOLOR, 2/*FG*/, (LPARAM) &fgCol);

	return (m_pCvtDIB->FillOutputDIB());
}

BOOL	CConvert::FreeDIB()
{
	if (m_pCvtDIB != 0) {
		m_pCvtDIB->DestroyDIB();
		delete m_pCvtDIB;
		m_pCvtDIB = 0;
	}
	return TRUE;
}

BOOL	CConvert::Convert(
	LPCSTR			inputFile,
	LPCSTR			outputFile,
	PAN_CtlRange	inputRange,
	Real			outputWidth,
	Real			outputHeight,
	int				units,
	Real			stepsPerInch,
	int				outputType,
	int				outputFormat,
	int				outputSubFormat,
	WORD			outputColorDepth
	)
{
	if (! IsWindow(m_hCtl)) {
		return FALSE;
	}

	// Initialize callbacks.
	CCallbackData callbackData;
	_fmemset(&callbackData, 0, sizeof callbackData);

	PAN_CtlFileInfo	fileInfo;
	memset(&fileInfo, 0, sizeof(fileInfo));
	::SendMessage(m_hCtl, PM_CTLGETFILE, 0, (LPARAM)&fileInfo);

	BOOL fVectorToVector = (fileInfo.type == PAN_VectorFile && outputType == PC_VECTOR);
	BOOL fAllCallbacks = !fVectorToVector;

	MakeCallbacks(&callbackData.callbacks, fAllCallbacks);
	callbackData.PANCVT_ABORT = callbackData.callbacks.PANCVT_ABORT;

	strcpy(m_settings.inputFile, inputFile);
	strcpy(m_settings.outputFile, outputFile);
	m_settings.inputRange = inputRange;
	m_settings.outputWidth = outputWidth;
	m_settings.outputHeight = outputHeight;
	m_settings.units = units;
	m_settings.stepsPerInch = stepsPerInch;
	m_settings.outputType = outputType;
	m_settings.outputFormat = outputFormat;
	m_settings.outputSubFormat = outputSubFormat;
	m_settings.outputColorDepth = outputColorDepth;	/* 0 means default */

	// Initialize DIB on which to render area to convert.
	if (! fVectorToVector && !InitDIB(&fileInfo)) {
		FreeCallbacks(&callbackData.callbacks, fAllCallbacks);
		return FALSE;
	}

	// Render area to convert onto DIB.
	if (! fVectorToVector && ! RenderDIB(&fileInfo)) {
		FreeDIB();
		FreeCallbacks(&callbackData.callbacks, fAllCallbacks);
		return FALSE;
	}

	// Get conversion handle.
	BOOL (__stdcall *lpfn)(LPCVTPANX cbs, DWORD fmt, DWORD subFmt, LPCSTR inFile, LPCSTR outFile);

	::SendMessage(m_hCtl, PM_CTLCONVERT,
		GET_PAFSEXPORT_CONVERT_HANDLE, (LPARAM) (FARPROC FAR*) &lpfn);

	// Convert file.
	callbackData.pThis = this;

	lpfn((LPCVTPANX) &callbackData,
		(DWORD) outputFormat, (DWORD) outputSubFormat,
		(LPSTR) (LPCSTR) inputFile, (LPSTR) (LPCSTR) outputFile);

	if (! fVectorToVector) {
		FreeDIB();
	}
	FreeCallbacks(&callbackData.callbacks, fAllCallbacks);

	return TRUE;
}
