////////////////////////////////////////////////////////////////////////////////
// cvtdib.cpp
////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

#include "export.h"
#include "cvtdib.h"

#if	TARGET == WIN16
// The 16-bit compiler does not generate proper optimized code
#pragma	optimize("", off)
#endif

// Minimum width and height applies to PC_RASTER and PC_VECTOR.
const LONG MIN_DIB_WIDTH			 = 10;
const LONG MIN_DIB_HEIGHT			 = 10;

// Maximum width and height applies to PC_VECTOR only.
const LONG MAX_DIB_WIDTH			 = 1000;
const LONG MAX_DIB_HEIGHT			 = 1000;
const Real DIB_SIZE_REDUCTION_FACTOR = 0.75;

////////////////////////////////////////////////////////////////////////////////
// CCvtDIB - PRIVATE SECTION
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
BOOL CCvtDIB::CreateScreenCompatibleDC(LONG width, LONG height,
	int in_colorDepth, int out_colorDepth, WORD out_Type, LOGPALETTE *pPalette)
{
#if	TARGET == WIN16
	if (out_colorDepth != 1 && out_colorDepth != 24) {
		return FALSE;
	}
#endif

	HDC 		hdcDeskTop;
	HWND		hwndDeskTop;

	m_hDC = (HDC) 0;
	m_hBitmap = (HBITMAP) 0;
	hwndDeskTop = GetDesktopWindow();
	hdcDeskTop	= GetDC(hwndDeskTop);

#if	TARGET == WIN16
	if (out_colorDepth == 24) {
		if (GetDeviceCaps(hdcDeskTop, BITSPIXEL) != 24) {
			// Need to have 24 bit display: DIB.DRV does not support 24bit output
			ReleaseDC(hwndDeskTop, hdcDeskTop);
			return FALSE;
		}
	}
#endif

	if (out_Type == PC_VECTOR) {
		// Reduce size of DIB to reasonable values, otherwise conversion
		// takes too much time.
		while (width > MAX_DIB_WIDTH || height > MAX_DIB_HEIGHT) {
			// Reduce size of DIB.
			width  = ROUND(LONG, width	* DIB_SIZE_REDUCTION_FACTOR);
			height = ROUND(LONG, height * DIB_SIZE_REDUCTION_FACTOR);
		}
	}

	/*
	** In this, the DIB header is used to hold the conversion info.
	** The Image will be rendered onto a memory DC and then copied
	** into the DIB.
	*/
	for (;;) {
		// Try allocating memory for the DIB.
		// If the allocation fails, reduce the size of the DIB and
		// try allocating again.
		// Repeat this procedure until the allocation succeeds or
		// the size of the DIB is too small to be useful.

		if (width < MIN_DIB_WIDTH || height < MIN_DIB_HEIGHT) {
			ReleaseDC(hwndDeskTop, hdcDeskTop);
			return FALSE;
		}
		/*
		** Each line is rounded off to a 16-bit boundary.
		*/
		LONG widthInBytes = GetWidthInBytes(width, out_colorDepth);

#if TARGET == WIN32S || defined(_WIN64)
		/*
		** Under Win32, CreateDIBSection will do the bitmap allocation, so
		** bitmap memory does not have to be allocated.
		*/
		widthInBytes = 0;
#endif

		if (out_colorDepth == 24) {
			m_hDIB = GlobalAlloc(GHND,
						(DWORD) sizeof (BITMAPINFOHEADER) +
						(DWORD) widthInBytes * height);
		} else {
			m_hDIB = GlobalAlloc(GHND,
						(DWORD) sizeof (BITMAPINFOHEADER) +
						(DWORD) (1 << out_colorDepth) * sizeof(RGBQUAD) +
						(DWORD) widthInBytes * height);
		}

		if (m_hDIB != 0) {
			// Allocation succeeded!
			break;
		}

		// Allocation failed: reduce size of DIB and try again.
		width  = ROUND(LONG, width	* DIB_SIZE_REDUCTION_FACTOR);
		height = ROUND(LONG, height * DIB_SIZE_REDUCTION_FACTOR);
	}

	if (m_hDIB == 0) {
		// Not enough memory
		ReleaseDC(hwndDeskTop, hdcDeskTop);
		return FALSE;
	}

	if ((m_pDIB = (LPBITMAPINFO) GlobalLock(m_hDIB)) == 0) {
		// Not enough memory
		DestroyDIB();
		ReleaseDC(hwndDeskTop, hdcDeskTop);
		return FALSE;
	}

	/*
	** save true width and height.
	*/
	m_pixelWidth = width;
	m_pixelHeight = height;
	m_pixelDepth  = out_colorDepth;

	// Initialize DIB.
	m_pDIBBits = InitDIB(pPalette, in_colorDepth, out_colorDepth, FALSE);

	/*
	** Create Memory DC.
	*/
	m_hDC = CreateCompatibleDC(hdcDeskTop);
	if (m_hDC == (HDC) 0) {
		// Not enough memory
		DestroyDIB();
		ReleaseDC(hwndDeskTop, hdcDeskTop);
		return FALSE;
	}

#if	TARGET == WIN32S || defined(_WIN64)
	void		*pvBits = 0;
	m_hBitmap = CreateDIBSection(hdcDeskTop, m_pDIB, DIB_RGB_COLORS, (void **)&m_pDIBBits, 0, 0);
#else
	if (out_colorDepth == 24) {
		m_hBitmap = CreateCompatibleBitmap(hdcDeskTop, (int) m_pixelWidth, (int) m_pixelHeight);
	} else {
		m_hBitmap = CreateBitmap((int) m_pixelWidth, (int) m_pixelHeight, 1, 1, NULL);
	}
#endif
	if (m_hBitmap == (HBITMAP) 0) {
		DestroyDIB();
		ReleaseDC(hwndDeskTop, hdcDeskTop);
		return FALSE;
	}

	SelectObject(m_hDC, (HGDIOBJ)m_hBitmap);

	ReleaseDC(hwndDeskTop, hdcDeskTop);
	return TRUE;

} // CConvert::InitScreenCompatibleDC()

////////////////////////////////////////////////////////////////////////////////
// CCvtDIB - PUBLIC SECTION
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
CCvtDIB::CCvtDIB()
{
	m_hDC = 0;
	m_hDIB = 0;
	m_pDIB = 0;
	m_pDIBBits = 0;
	m_hBitmap = 0;
	m_pixelWidth = 0L;
	m_pixelHeight = 0L;
	m_pixelDepth = 0L;
} // CCvtDIB::CCvtDIB()

////////////////////////////////////////////////////////////////////////////////
CCvtDIB::~CCvtDIB()
{

} // CCvtDIB::~CCvtDIB()

////////////////////////////////////////////////////////////////////////////////
BOOL CCvtDIB::CreateDIB(
	LONG width, LONG height,
	int in_colorDepth, int out_colorDepth,
	WORD in_Type, WORD out_Type, LOGPALETTE *pPalette)
{
	if (out_colorDepth < 0) {
		out_colorDepth = in_colorDepth;
	}

	// WIN32 has the wonderful function CreateDIBSection() that allows
	// you to create bitmaps of any depth.  WIN16 needs to rely on
	// DIB.DRV and/or the display driver.
	return CreateScreenCompatibleDC(width, height, in_colorDepth, out_colorDepth, out_Type, pPalette);

#if 0
	/*
	** DIB driver does not handle 24-bit images.
	*/
	if (out_colorDepth == 24) {
		// TrueColor raster image.
		return CreateScreenCompatibleDC(width, height, in_colorDepth, out_colorDepth, out_Type, pPalette);
	}

	/*
	** DIB.DRV seems to have a bug in performing 16-bit to 1-bit
	**  color reduction.
	*/
	if (out_colorDepth == 1) {
		// Monochrome raster image with windows color reduction.
		return CreateScreenCompatibleDC(width, height, in_colorDepth, out_colorDepth, out_Type, pPalette);
	}

	int out_nColors = 1 << out_colorDepth;
	int in_nColors;

	if (in_colorDepth < 1 || in_colorDepth > 8 || pPalette == 0) {
		// Color depth unreasonable: use default palette.
		in_colorDepth = 8;
		in_nColors = 1 << in_colorDepth;
	} else {
		in_nColors = pPalette->palNumEntries;
	}

	// Get size of DIB.
	if (out_Type == PC_VECTOR) {
		// Reduce size of DIB to reasonable values, otherwise conversion
		// takes too much time.

		while (width > MAX_DIB_WIDTH || height > MAX_DIB_HEIGHT) {
			// Reduce size of DIB.
			width  = ROUND(LONG, width	* DIB_SIZE_REDUCTION_FACTOR);
			height = ROUND(LONG, height * DIB_SIZE_REDUCTION_FACTOR);
		}
	}

	for (;;) {
		// Try allocating memory for the DIB(s).
		// If the allocation fails, reduce the size of the DIB(s) and
		// try allocating again.
		// Repeat this procedure until the allocation succeeds or
		// the size of the DIB is too small to be useful.

		if (width < MIN_DIB_WIDTH || height < MIN_DIB_HEIGHT) {
			// Size of DIB is too small to be useful.
			return FALSE;
		}
		LONG	widthInBytes = GetWidthInBytes(width, out_colorDepth);
		m_hDIB = GlobalAlloc(GHND,
					(DWORD) sizeof (BITMAPINFOHEADER) +
					(DWORD) out_nColors * sizeof (RGBQUAD) +
					(DWORD) widthInBytes * (((height+1) / 2) * 2));

		if (m_hDIB != 0) {
			// Allocation succeeded!
			break;
		}

		// Allocation failed: reduce size of DIB and try again.
		DestroyDIB();
		width  = ROUND(LONG, width	* DIB_SIZE_REDUCTION_FACTOR);
		height = ROUND(LONG, height * DIB_SIZE_REDUCTION_FACTOR);
	}

	if (m_hDIB == 0) {
		// Not enough memory
		DestroyDIB();
		return FALSE;
	}

	if ((m_pDIB = (LPBITMAPINFO) GlobalLock(m_hDIB)) == 0) {
		// Not enough memory
		DestroyDIB();
		return FALSE;
	}

	/*
	** save true width and height.
	*/
	m_pixelWidth = width;
	m_pixelHeight = height;
	m_pixelDepth = out_colorDepth;

	/*
	** It seems like the only safe way to have the DC successfully
	** created from the DIB, is to have an even height and a width
	** which corresponds to a multiple of 8 bytes.
	*/
	m_pDIBBits = InitDIB(pPalette, in_colorDepth, out_colorDepth, TRUE);

	// Create device context.
#if	TARGET == WIN16
	m_hDC = CreateDC("DIB", 0, 0, m_pDIB);
#else
	// Should never get here!
	ASSERT(0);
#endif

	if (m_hDC == 0) {
		DestroyDIB();
		return FALSE;
	}

	return TRUE;

#endif

} // CCvtDIB::CreateDIB()

////////////////////////////////////////////////////////////////////////////////
BOOL CCvtDIB::FillOutputDIB()
{
	/*
	** For 24-bit or 1-bit output with auto color reduction: we need to copy
	** the image data from the memory DC into the DIB.
	*/
#if	TARGET == WIN16
	if (m_pDIB->bmiHeader.biBitCount == 24 || m_pDIB->bmiHeader.biBitCount == 1) {
#else
	/*
	** Win32 already stored bits in m_pDIBBits. Don't have to do copy.
	*/
	if (0) {
#endif
		LONG	widthInBytes = GetWidthInBytes(m_pDIB->bmiHeader.biWidth, m_pDIB->bmiHeader.biBitCount);
		DWORD	dwCount = widthInBytes *  m_pDIB->bmiHeader.biHeight;

		if (m_hBitmap && m_pDIBBits) {
			::GetBitmapBits(m_hBitmap, dwCount , m_pDIBBits);
		} else {
			return FALSE;
		}
	}

	return TRUE;
}

////////////////////////////////////////////////////////////////////////////////
BOOL CCvtDIB::DestroyDIB()
{
	if (m_hDC != 0) {
		DeleteDC(m_hDC);
		m_hDC = 0;
	}

	if (m_hBitmap != 0) {
		DeleteObject((HGDIOBJ)m_hBitmap);
		m_hBitmap = 0;
	}

	if (m_pDIB != 0) {
		GlobalUnlock(m_hDIB);
		m_pDIB = 0;
	}

	if (m_hDIB != 0) {
		GlobalFree(m_hDIB);
		m_hDIB = 0;
	}

	m_pDIBBits = 0;

	return TRUE;

} // CCvtDIB::DestroyDIB()

////////////////////////////////////////////////////////////////////////////////
BOOL	CCvtDIB::CreateDefaultPalette(int out_nColors)
{
	switch (out_nColors) {
	  case 256:
		{{{
		/*
		**  Generate 256 (8 * 8 * 4) RGB combinations to fill the
		**  palette entries.
		*/
		BYTE		red = 0, green = 0, blue = 0;
		for (int i = 0; i < out_nColors; ++i) {
			m_pDIB->bmiColors[i].rgbRed   = red;
			m_pDIB->bmiColors[i].rgbGreen = green;
			m_pDIB->bmiColors[i].rgbBlue  = blue;
			m_pDIB->bmiColors[i].rgbReserved = 0;

			if (!(red += 32))
				if (!(green += 32))
					blue += 64;
		}
		m_pDIB->bmiColors[255].rgbRed   =
		m_pDIB->bmiColors[255].rgbGreen =
		m_pDIB->bmiColors[255].rgbBlue  = 255;
		}}}
		break;
	  case 16:
		{{{
		/*
		**  Set a VGA Colour map:
		**  Black, Blue, Green, Cyan, Red, Magenta, Brown, White,
		**  Dark Grey, Light Blue, Light Green, Light Cyan, Light Red,
		**  Light Magenta, Yellow, Intensified White.
		*/
		static	BYTE	r[16] =
			{0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xAA,
			 0x55, 0x55, 0x55, 0x55, 0xFF, 0xFF, 0xFF, 0xFF};
		static	BYTE	g[16] =
			{0x00, 0x00, 0xFF, 0xFF, 0x00, 0x00, 0x55, 0xAA,
			 0x55, 0x55, 0xFF, 0xFF, 0x55, 0x55, 0xFF, 0xFF};
		static	BYTE	b[16] =
			{0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xAA,
			 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF, 0x55, 0xFF};

		for (int i = 0; i < out_nColors; i++) {
			m_pDIB->bmiColors[i].rgbRed   = r[i];
			m_pDIB->bmiColors[i].rgbGreen = g[i];
			m_pDIB->bmiColors[i].rgbBlue  = b[i];
			m_pDIB->bmiColors[i].rgbReserved = 0;
		}
		}}}
		break;
	  case 2:
		/*
		** Black and white
		*/
		m_pDIB->bmiColors[0].rgbRed   = 255;
		m_pDIB->bmiColors[0].rgbGreen = 255;
		m_pDIB->bmiColors[0].rgbBlue  = 255;
		m_pDIB->bmiColors[0].rgbReserved = 0;
		m_pDIB->bmiColors[1].rgbRed   = 0;
		m_pDIB->bmiColors[1].rgbGreen = 0;
		m_pDIB->bmiColors[1].rgbBlue  = 0;
		m_pDIB->bmiColors[1].rgbReserved = 0;
		break;
	  default:
		/*
		** Default to a greyscale
		*/
		for (int i = 0; i < out_nColors; i++) {
			m_pDIB->bmiColors[i].rgbRed   = i;
			m_pDIB->bmiColors[i].rgbGreen = i;
			m_pDIB->bmiColors[i].rgbBlue  = i;
			m_pDIB->bmiColors[i].rgbReserved = 0;
		}
		break;
	}

	return TRUE;
} // CCvtDIB::CreateDefaultPalette()


BYTE huge*	CCvtDIB::InitDIB(LOGPALETTE *pPalette, int in_colorDepth, int out_colorDepth, BOOL forceToByteBoundary /*=0*/)
{
	int		out_nColors = 1 << out_colorDepth;
	int		in_nColors = 1 << in_colorDepth;
	LONG	widthInBytes = GetWidthInBytes(m_pixelWidth, out_colorDepth);

	// Initialize DIB.
	m_pDIB->bmiHeader.biSize		 = sizeof (BITMAPINFOHEADER);
	m_pDIB->bmiHeader.biWidth		 = forceToByteBoundary ?
										widthInBytes * 8 / out_colorDepth :
										m_pixelWidth;
	m_pDIB->bmiHeader.biHeight		 = m_pixelHeight;
	m_pDIB->bmiHeader.biPlanes		 = 1;
	m_pDIB->bmiHeader.biBitCount	 = out_colorDepth;
	m_pDIB->bmiHeader.biSizeImage	 = forceToByteBoundary ?
										widthInBytes * (((m_pixelHeight+1) / 2) * 2) :
										widthInBytes * m_pixelHeight;
	m_pDIB->bmiHeader.biClrUsed 	 = out_nColors;
	m_pDIB->bmiHeader.biClrImportant = out_nColors;
	m_pDIB->bmiHeader.biCompression  = BI_RGB;

	if (out_colorDepth == 24) {
		m_pDIB->bmiHeader.biClrUsed 	 = 0;
		m_pDIB->bmiHeader.biClrImportant = 0;

		// Get pointer to DIB bits.
		return (BYTE huge *) m_pDIB + m_pDIB->bmiHeader.biSize;
	} else {
		// create the default palette
		CreateDefaultPalette(out_nColors);
		// Copy the source Palette
		SetColorTable(pPalette, m_pDIB, (short)m_pixelDepth);

		// Get pointer to DIB bits.
		return (BYTE huge *) m_pDIB + m_pDIB->bmiHeader.biSize +
					m_pDIB->bmiHeader.biClrUsed * sizeof (RGBQUAD);
	}
}

// apply source pallette on target palette, overriding color entries as required
// copied from CPanDIB::SetColorTable
void CCvtDIB::SetColorTable(LOGPALETTE *pSrcPalette, 
							BITMAPINFO* pDestBmpInfo, 
							const short nDestColorDepth) {
	bool bValidInput = (pSrcPalette != NULL) && (pDestBmpInfo != NULL);

	short numEntries = 0;

	if (bValidInput) {
		numEntries = pSrcPalette->palNumEntries;
		bValidInput = ((numEntries == 2) || (numEntries == 16) || (numEntries == 256)) &&
		((nDestColorDepth == 1) || (nDestColorDepth == 4) || (nDestColorDepth == 8));
	}

	if (bValidInput) {
		short nDestColors = (short)(1 << nDestColorDepth);

		// get the smaller color value
		short nColors = (numEntries < nDestColors) ? numEntries : nDestColors;

		// copy the colors

		for (int i = 0; i < nColors; i++) {
			const PALETTEENTRY &entry = pSrcPalette->palPalEntry[i];
			pDestBmpInfo->bmiColors[i].rgbRed		= entry.peRed;
			pDestBmpInfo->bmiColors[i].rgbGreen		= entry.peGreen;
			pDestBmpInfo->bmiColors[i].rgbBlue		= entry.peBlue;
			pDestBmpInfo->bmiColors[i].rgbReserved	= 0;
		}
	}
}



LONG	CCvtDIB::GetWidthInBytes(LONG width, int depth)
{
#if TARGET == WIN32S || defined(_WIN64)
	return ((width * depth + 31) / 32 * 4);
#else
	return (depth == 24 || depth == 1) ?
			((width * depth + 15) / 16 * 2) :
			((width * depth + 63) / 64 * 8);
#endif
}

