//
// markup\mview.cpp : implementation file of CSampMarkupView
//

#include "stdafx.h"

#include "mfcmark.h"
#include "mainfrm.h"
#include "doc.h"
#include "mview.h"
#include "mdlgs.h"

#include "resource.h"

#include <string.h>
#include <math.h>
#include <float.h>
#include <stdio.h>
#include <stdlib.h>
#include <direct.h>

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#if TARGET == TWUNIX
#define Safe_IsMenuEnabled(x)     IsMenuEnabled(&CSampMarkupView::x)
#define Safe_OnEntitiesXXX(x, y)  OnEntitiesXXX(&CSampMarkupView::x, y) 
#define Safe_OnEntitiesXXX1(x, y, z) OnEntitiesXXX(&CSampMarkupView::x, y, z) 
#else
#define Safe_IsMenuEnabled(x)    IsMenuEnabled(x)
#define Safe_OnEntitiesXXX(x, y) OnEntitiesXXX(x, y) 
#define Safe_OnEntitiesXXX1(x, y, z) OnEntitiesXXX(x, y, z) 
#endif

// Definition for standard AutoVue UDEs
struct {
	int 	udeID;
	char*	title;
} AMustUdes[] = {
	ENTTYPE_UDE_CLOUD,			"Cloud",
	ENTTYPE_UDE_LEADER, 		"Leader",
	ENTTYPE_UDE_DISTANCE,		"Distance",
	ENTTYPE_UDE_CUMDISTANCE,	"Cumulative Distance",
	ENTTYPE_UDE_AREA,			"Area",
	ENTTYPE_UDE_OLE,			"OLE",
};

/*
** Helper function to allocate string memory
*/
char FAR*	MrsAlloc(const char FAR* lpString)
{
	if (lpString == NULL) {
		return NULL;
	}
	char FAR*	lpBuffer = (char FAR *) MrkAlloc(lstrlen(lpString) + 1);
	if (lpBuffer != NULL) {
		lstrcpy(lpBuffer, lpString);
	}
	return lpBuffer;
}

/////////////////////////////////////////////////////////////////////////////
// CSampMarkupView
/////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CSampMarkupView, CSampView)

CSampMarkupView::CSampMarkupView()
{
	m_notifyAction = CHANGE_NONE;

	m_nUdes = 0;
	m_pUdes = NULL;
}

CSampMarkupView::~CSampMarkupView()
{
	DestroyMarkupControl();

	if (m_nUdes && m_pUdes != NULL) {
		delete [] m_pUdes;
	}
}


// Selfposted hyperlink firing message for secure markup control termination
//  wParam: currently unused
//  lParam: (LPCSTR)filename
#define AVM_FIREHYPERLINK		WM_USER + 0x300 + 9


#define	theDerivedClass	CSampMarkupView::
BEGIN_MESSAGE_MAP(CSampMarkupView, CSampView)
	//{{AFX_MSG_MAP(CSampMarkupView)
	ON_WM_CREATE()
	ON_WM_CLOSE()
	ON_WM_DESTROY()
	ON_WM_RBUTTONDOWN()
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, OnFilePrintPreview)
	ON_COMMAND(ID_VIEW_TOOLBAR, OnViewToolbar)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TOOLBAR, OnUpdateViewToolbar)
	ON_COMMAND(ID_VIEW_STATUS_BAR, OnViewStatusbar)
	ON_UPDATE_COMMAND_UI(ID_VIEW_STATUS_BAR, OnUpdateViewStatusbar)
	ON_COMMAND(ID_MARKUP_OPEN, OnMarkupOpen)
	ON_UPDATE_COMMAND_UI(ID_MARKUP_OPEN, OnUpdateMarkupOpen)
	ON_COMMAND(ID_MARKUP_EXIT, OnMarkupExit)
	ON_UPDATE_COMMAND_UI(ID_MARKUP_EXIT, OnUpdateMarkupExit)
	ON_COMMAND(ID_MARKUP_SAVE, OnMarkupSave)
	ON_UPDATE_COMMAND_UI(ID_MARKUP_SAVE, OnUpdateMarkupSave)
	ON_COMMAND(ID_MODIFY_ENTITY_FONT, OnModifyEntityFont)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_ENTITY_FONT, OnUpdateModifyEntityFont)
	ON_COMMAND(ID_MODIFY_MARKUP_ONOFF, OnModifyMarkupOnoff)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_MARKUP_ONOFF, OnUpdateModifyMarkupOnoff)
	ON_COMMAND(ID_MODIFY_DELETE, OnModifyDelete)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_DELETE, OnUpdateModifyDelete)
	ON_COMMAND(ID_MODIFY_DELETE_ALL, OnModifyDeleteAll)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_DELETE_ALL, OnUpdateModifyDeleteAll)
	ON_COMMAND(ID_MODIFY_UNDO, OnModifyUndo)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_UNDO, OnUpdateModifyUndo)
	ON_COMMAND(ID_MODIFY_REDO, OnModifyRedo)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_REDO, OnUpdateModifyRedo)
  	ON_COMMAND(ID_MODIFY_LINESTYLE_DASH, OnModifyLineStyleDash)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINESTYLE_DASH, OnUpdateModifyLineStyleDash)
	ON_COMMAND(ID_MODIFY_LINESTYLE_DOT, OnModifyLineStyleDot)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINESTYLE_DOT, OnUpdateModifyLineStyleDot)
	ON_COMMAND(ID_MODIFY_LINESTYLE_DSDD, OnModifyLineStyleDsdd)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINESTYLE_DSDD, OnUpdateModifyLineStyleDsdd)
	ON_COMMAND(ID_MODIFY_LINESTYLE_DSHD, OnModifyLineStyleDshd)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINESTYLE_DSHD, OnUpdateModifyLineStyleDshd)
	ON_COMMAND(ID_MODIFY_LINESTYLE_SOLID, OnModifyLineStyleSolid)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINESTYLE_SOLID, OnUpdateModifyLineStyleSolid)
	ON_COMMAND(ID_MODIFY_LINEWIDTH_W1, OnModifyLineWidth1)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINEWIDTH_W1, OnUpdateModifyLineWidth1)
	ON_COMMAND(ID_MODIFY_LINEWIDTH_W2, OnModifyLineWidth2)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINEWIDTH_W2, OnUpdateModifyLineWidth2)
	ON_COMMAND(ID_MODIFY_LINEWIDTH_W3, OnModifyLineWidth3)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINEWIDTH_W3, OnUpdateModifyLineWidth3)
	ON_COMMAND(ID_MODIFY_LINEWIDTH_W4, OnModifyLineWidth4)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINEWIDTH_W4, OnUpdateModifyLineWidth4)
	ON_COMMAND(ID_MODIFY_LINEWIDTH_W5, OnModifyLineWidth5)
	ON_UPDATE_COMMAND_UI(ID_MODIFY_LINEWIDTH_W5, OnUpdateModifyLineWidth5)
	ON_COMMAND(ID_HYPERLINK_EDIT, OnHyperlinkEdit)
	ON_UPDATE_COMMAND_UI(ID_HYPERLINK_EDIT, OnUpdateHyperlinkEdit)
	ON_COMMAND(ID_HYPERLINK_ESTABLISH, OnHyperlinkEstablish)
	ON_UPDATE_COMMAND_UI(ID_HYPERLINK_ESTABLISH, OnUpdateHyperlinkEstablish)
	ON_COMMAND(ID_HYPERLINK_FIRE, OnHyperlinkFire)
	ON_UPDATE_COMMAND_UI(ID_HYPERLINK_FIRE, OnUpdateHyperlinkFire)
	ON_COMMAND(ID_ENTITIES_ARC, OnEntitiesArc)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_ARC, OnUpdateEntitiesArc)
	ON_COMMAND(ID_ENTITIES_BOX, OnEntitiesBox)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_BOX, OnUpdateEntitiesBox)
	ON_COMMAND(ID_ENTITIES_CIRCLE, OnEntitiesCircle)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_CIRCLE, OnUpdateEntitiesCircle)
	ON_COMMAND(ID_ENTITIES_CLOUD, OnEntitiesCloud)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_CLOUD, OnUpdateEntitiesCloud)
	ON_COMMAND(ID_ENTITIES_FILLED_BOX, OnEntitiesFilledBox)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_FILLED_BOX, OnUpdateEntitiesFilledBox)
	ON_COMMAND(ID_ENTITIES_FREESTYLE, OnEntitiesFreeStyle)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_FREESTYLE, OnUpdateEntitiesFreeStyle)
	ON_COMMAND(ID_ENTITIES_HIGHLIGHT, OnEntitiesHighlight)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_HIGHLIGHT, OnUpdateEntitiesHighlight)
	ON_COMMAND(ID_ENTITIES_LEADER, OnEntitiesLeader)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_LEADER, OnUpdateEntitiesLeader)
	ON_COMMAND(ID_ENTITIES_LINE, OnEntitiesLine)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_LINE, OnUpdateEntitiesLine)
	ON_COMMAND(ID_ENTITIES_POLYLINE, OnEntitiesPolyline)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_POLYLINE, OnUpdateEntitiesPolyline)
	ON_COMMAND(ID_ENTITIES_TEXT, OnEntitiesText)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_TEXT, OnUpdateEntitiesText)
	ON_COMMAND(ID_ENTITIES_NOTE, OnEntitiesNote)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_NOTE, OnUpdateEntitiesNote)
	ON_COMMAND(ID_ENTITIES_OLE, OnEntitiesOle)
	ON_UPDATE_COMMAND_UI(ID_ENTITIES_OLE, OnUpdateEntitiesOle)
	ON_UPDATE_COMMAND_UI(ID_COLOR_BLUE, OnUpdateColorBlue)
	ON_COMMAND(ID_COLOR_BLUE, OnColorBlue)
	ON_UPDATE_COMMAND_UI(ID_COLOR_RED, OnUpdateColorRed)
	ON_COMMAND(ID_COLOR_RED, OnColorRed)
	ON_UPDATE_COMMAND_UI(ID_COLOR_GREEN, OnUpdateColorGreen)
	ON_COMMAND(ID_COLOR_GREEN, OnColorGreen)
	//}}AFX_MSG_MAP
	ON_MESSAGE(MRKN_CANCEL, 	 OnNotifCancel)
	ON_MESSAGE(MRKN_EDIT,		 OnNotifEdit)
	ON_MESSAGE(MRKN_FIRELINK,	 OnNotifFirelink)
	ON_MESSAGE(MRKN_ENTITYADDED, OnNotifEntityadded)
	ON_MESSAGE(MRKN_SELCHANGED,  OnNotifSelchanged)
	ON_MESSAGE(MRKN_MODIFIED,	 OnNotifModified)
	ON_MESSAGE(MRKN_CURSOR, 	 OnNotifCursor)
	ON_MESSAGE(AVM_FIREHYPERLINK,OnSampFireHyperlink)
END_MESSAGE_MAP()
#undef	theDerivedClass


/************************************************************************
** BEGIN SYSTEM MESSAGES
************************************************************************/
void CSampMarkupView::OnDraw(CDC* pDC)
{
	// Nothing to do: Markup Control handles drawing.
}


int CSampMarkupView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CSampView::OnCreate(lpCreateStruct) == -1) {
		return -1;
	}
	return 0;
}

void CSampMarkupView::OnInitialUpdate()
{
	CSampView::OnInitialUpdate();
}

void CSampMarkupView::OnClose()
{
	// if markup control exists, destroy the markup control first
	CMarkupControl* pMrkCtl = GetDocument()->GetMarkupControl();
	if (pMrkCtl) {
		DestroyMarkupControl();
	}
	CSampView::OnClose();
}

void CSampMarkupView::OnDestroy()
{
	// if markup control exists, destroy the markup control first
	CMarkupControl* pMrkCtl = GetDocument()->GetMarkupControl();
	if (pMrkCtl) {
		DestroyMarkupControl();
	}
	CSampView::OnDestroy();
}
void CSampMarkupView::OnRButtonDown(UINT nFlags, CPoint point)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();

	if (pMrkCtl == 0) {
		CSampView::OnRButtonDown(nFlags, point);
		return;
	}

	int	nAction;
	pMrkCtl->GetAction(&nAction);

	if (nAction == MRKP_ACTION_ADD) {
		pMrkCtl->SetAction(MRKP_ACTION_HYBRID);
		pMrkCtl->SetEntity(0);
	} else {
		CSampView::OnRButtonDown(nFlags, point);
	}
}
/************************************************************************
** END SYSTEM MESSAGES
************************************************************************/



/************************************************************************
** BEGIN MARKUP MENU
************************************************************************/
void CSampMarkupView::OnUpdateMarkupOpen(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl != NULL && pMrkCtl->IsWindowVisible());
}
void CSampMarkupView::OnMarkupOpen()
{
	// open an existing markup file or create untitled markup file
	CMarkupControl *pMrkCtl = GetMarkupControl();
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateMarkupOpen)) {
		return;
	}
	AskForMarkupName();
}

void CSampMarkupView::OnUpdateMarkupSave(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl != NULL && pMrkCtl->IsModified());
}
void CSampMarkupView::OnMarkupSave()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateMarkupSave)) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl != 0);

	BOOL	ret = SaveMarkupFile();

	if (! ret) {
		// Error message
		CString strError;
		strError.LoadString(IDS_MRKPERR_SAVE);
		char	szBuffer[_MAX_PATH];
		wsprintf(szBuffer, strError, pMrkCtl->GetShortID());
		strError.LoadString(IDS_MRKPERR_SAVETITLE);
		MessageBox(szBuffer, strError, MB_ICONHAND | MB_OK);
	}
}

void CSampMarkupView::OnUpdateMarkupExit(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkpCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkpCtl != NULL);
}

void	CSampMarkupView::OnMarkupExit()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateMarkupExit)) {
		return;
	}

	ExitMarkup();
}

void    CSampMarkupView::ExitMarkup()
{
	CSampDoc*		pDoc = GetDocument();
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	// Old active markup.
	int nOldActive = pMrkCtl->GetActiveMarkup();

	// only one markup file
	pMrkCtl->SetActiveMarkup(0);

	if (pMrkCtl->IsModified()) {
		CString	pFile = pMrkCtl->GetMarkupFileName();
		CString	szMarkupName = pMrkCtl->GetLongID();

		CMarkupSaveDlg	saveDlg(this, FALSE/*hideCancel*/, (LPCSTR)szMarkupName);

		switch(saveDlg.DoModal()) {
		  case IDOK:
		  	// Save markup.
			OnMarkupSave();

			if (pMrkCtl->IsModified()) {
				// If file was not saved.  Make sure that is what
				// the user wants.
				if (AfxMessageBox( IDP_MRKP_CANNOT_SAVE_FILE_ASK_QUIT_WITHOUT_SAVING,
						MB_YESNO)  !=  IDYES)
				{
					pMrkCtl->SetActiveMarkup(nOldActive);
					return;
				}
			}

			// Mark as UnModified.
			pMrkCtl->SetIsModified(FALSE);
			break;

		  case IDCANCEL:
		  	pMrkCtl->SetActiveMarkup(nOldActive);
			return;

		  default:
		  	// Mark as UnModified.
			pMrkCtl->SetIsModified(FALSE);
		}
	}

	pMrkCtl->SetActiveMarkup(nOldActive);
	DestroyMarkupControl();

	((CSampMainFrame*)AfxGetMainWnd())->UpdateMenu(FALSE);
	((CSampMainFrame*)AfxGetMainWnd())->ShowMarkupToolbar(FALSE);
}

BOOL CSampMarkupView::AskForMarkupName()
{
	CSampApp*		pApp = (CSampApp*)AfxGetApp();
	CSampDoc*		pDoc = GetDocument();
	CMarkupControl*	pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	// 1. Ask for saving if modified
	if (pMrkCtl->IsModified()) {
		// Ask for saving
		CString	pFile = pMrkCtl->GetMarkupFileName();

		CMarkupSaveDlg	savedlg(this, TRUE /*hideCancel*/, pMrkCtl->GetShortID());
		switch (savedlg.DoModal()) {
		  case IDCANCEL:
			return TRUE;
		  case IDNO:
			// don't bother saving
			break;
		  case IDOK:
		  default:
			// save markup
			OnMarkupSave();
			break;
		}
	}

	// Ask for new markup to load
	CString	selectedMarkup;

	// Note: Dialog should be at top scope, since selectedMarkups references its members.
	CSelectMarkupDlg		dlgexist(this);

	if (dlgexist.DoModal() != IDOK) {
		// User cancelled
		return FALSE;
	} else {
		selectedMarkup = dlgexist.GetMarkupFile();
	}

	if (selectedMarkup.IsEmpty()) {
		// Make up a new name.
		selectedMarkup =
			GetFilenameFromID(pDoc->GetDocFileName(),
							pDoc->GetMarkupDirectory(),
							"001");

		CString	msg("Defaulting new markup to: ");
		msg += selectedMarkup;
		AfxMessageBox(msg);
	}

	// Set markup
	SetMarkup(selectedMarkup);

	// Show them
	pMrkCtl->ShowMarkup(TRUE);

	/* Fix for bug 12238. 
	** When pMrkCtl->Read() fails, SetDefaultMarkupLayerName() will
	** set the first layer name to "0".  This causes the modified
	** flag to be set.  As a result, the user will be prompted to
	** save the markup on exit even if no modifications were made.
	** To prevent this, we should mark the markup as unmodified
	** at this point.
	*/
	pMrkCtl->SetIsModified(FALSE);

	return TRUE;
}
void CSampMarkupView::LoadUdesInfo()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	if (m_nUdes == 0) {
		m_nUdes = pMrkCtl->GetNumUdes();
		m_pUdes = m_nUdes <= 0 ? 0 : new UdeInfo [m_nUdes];
		if ( ! m_pUdes ) {
			m_nUdes = 0;
			return;
		}

		// Store important information about UDEs.

		int nMenuEntryNumber = 0;
		for (int i=0; i < pMrkCtl->GetNumUdes(); i++) {
			UDEINFO udeInfo;
			memset(&udeInfo, 0, sizeof(UDEINFO));

			pMrkCtl->GetUdeInfo(ENTTYPE_UDE_BASE + i, &udeInfo);

			lstrcpyn(m_pUdes[i].szMenuDesc, udeInfo.entMenuDesc,
							sizeof(m_pUdes[0].szMenuDesc));
			m_pUdes[i].idNum = udeInfo.entID;


			// Look up standard UDEs information
			m_pUdes[i].fExternalUde = TRUE;
			int k = 0;
			while (k < sizeof(AMustUdes) / sizeof(AMustUdes[0])) {
				CString s1 = AMustUdes[k].title, s2 = udeInfo.entName;
				if (s1 == s2) {
					m_pUdes[i].fExternalUde = FALSE;
					m_pUdes[i].idInternal = AMustUdes[k].udeID;
				}
				k++;
			}

			if (m_pUdes[i].fExternalUde) {
				m_pUdes[i].nMenuNumber = nMenuEntryNumber++;
			}
		}
	}
}
/************************************************************************
** END MARKUP MENU
************************************************************************/


/************************************************************************
** BEGIN MARKUP 'ENTITIES' MENU
************************************************************************/

/*
**  Checks if a given entity type is available.
**  This really applies only to UDE's (user-defined-entities) that
**  are dynamically loaded at runtime.
*/
BOOL CSampMarkupView::IsEntityTypeAvailable(int Type, BOOL fAllMarkups /*= TRUE*/)
{
	BOOL	fResult = FALSE;

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl == NULL) {
		return fResult;
	}

	// Top level hyperlinks.
	if (pMrkCtl->GetNumEntities(Type)) {
		fResult = TRUE;
	}

	if (!fResult && (Type == ENTTYPE_LINK || Type == ENTTYPE_NOTE || Type == ENTTYPE_TEXT)) {
		// Embedded entities.
		int	numChildren = pMrkCtl->GetNumEntities();

		if (numChildren > 0) {
			LPENTHANDLE 	pEntHandles = new ENTHANDLE [numChildren];
			ASSERT(pEntHandles != NULL);

			memset(pEntHandles, 0, numChildren * sizeof(ENTHANDLE));
			pMrkCtl->GetEntities(numChildren, pEntHandles);

			for (int i = 0; !fResult && i < numChildren; i++) {
				if (pEntHandles[i] != 0 && pMrkCtl->GetNumChildren(pEntHandles[i], Type)) {
					fResult = TRUE;
				}
			}

			delete pEntHandles;
		}
	}

	return fResult;
}

/*
** Check if the UDE's in the list AMustUdes[] are available
*/
BOOL CSampMarkupView::IsEntityAvailable(int nEntityID)
{
	BOOL	fAvailability = FALSE;

	if (nEntityID > ENTTYPE_UDE_FIRST && nEntityID < ENTTYPE_UDE_LAST) {
		// Lookup UDE information list for this UDE presence
		for (int i=0; i < m_nUdes; i++) {
			if (m_pUdes[i].idInternal == nEntityID) {
				fAvailability = TRUE;
				break;
			}
		}
	} else if (nEntityID < ENTTYPE_UDE_BASE) {
		// Built-in entities are always available
		fAvailability = TRUE;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl == NULL || !pMrkCtl->IsWindowVisible()) {
		fAvailability = FALSE;
	}

	return fAvailability;
}

/*
**  Disables an entity in the user interface if it is not
**  available. Can be used for built-in UDEs and generic entities.
*/
void CSampMarkupView::OnUpdateEntitiesXXX(CCmdUI* pCmdUI, int entID, int FillType /*-1*/)
{
	CSampDoc*	pDoc = GetDocument();
	CMarkupControl*	 pMrkCtl = GetMarkupControl();
	if (pMrkCtl != NULL) {
		BOOL	bCheck = TRUE;
		pCmdUI->Enable(IsEntityAvailable(entID));
		if (FillType != -1) {
			bCheck = (pMrkCtl->GetCurrentFillType() == FillType) ? TRUE : FALSE;
		}
		pCmdUI->SetCheck(bCheck && pMrkCtl->GetEntity() == GetUdeRuntimeID(entID));
	} else {
		pCmdUI->Enable(FALSE);
		pCmdUI->SetCheck(FALSE);
	}
}

/*
**  Generic function to begin adding an entity.
*/
void CSampMarkupView::OnEntitiesXXX(void (CSampMarkupView::*fnOnUpdate) (CCmdUI *), int entID, int FillType /*= MRK_FILLNONE*/)
{
	CSampDoc*	pDoc = GetDocument();
	CMarkupControl* pMrkCtl = GetMarkupControl();

	if (!IsMenuEnabled(fnOnUpdate)) {
		return;
	}
	if (pMrkCtl->GetEntity() == GetUdeRuntimeID(entID)) {
		EndAddingEntity();
		return;
	}

	// Set FillType:  Needed for highlight and filled box entities
	// (both are done through a rect entity).
	pMrkCtl->SetCurrentFillType(FillType);

	StartAddingEntity(entID);
}

void CSampMarkupView::StartAddingEntity(int nEntityID, BOOL fExternalUDE /*=FALSE*/)
{
	CMarkupControl	*pMrkCtl = GetMarkupControl();
	if (pMrkCtl == NULL) {
		return;
	}

	pMrkCtl->SetAction(MRKP_ACTION_ADD, 0);

	pMrkCtl->SetEntity(GetUdeRuntimeID(nEntityID));

	m_notifyAction = CHANGE_NONE;
}

void CSampMarkupView::EndAddingEntity()
{
	CMarkupControl* pMrkCtl = GetDocument()->GetMarkupControl();
	if (pMrkCtl == NULL) {
		return;
	}

	// Reset filltype to none.
	pMrkCtl->SetCurrentFillType(MRK_FILLNONE);

	//pMrkCtl->SetAction(MRKP_ACTION_NONE, 0);
	pMrkCtl->SetAction(MRKP_ACTION_HYBRID, 0);
	pMrkCtl->SetEntity(0);
}

/*
**  Setup default layers
*/
void CSampMarkupView::SetDefaultMarkupLayerName()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	int nLayers = pMrkCtl->GetLayers(0, NULL);
	if (nLayers <= 0) {
		return;
	}
	LPPAN_LAYER pLayers = new PAN_LAYER[nLayers];
	if (pLayers == NULL) {
		return;
	}
	pMrkCtl->GetLayers(nLayers, pLayers);
	if (lstrlen(pLayers[0].name) == 0) {
		lstrcpy(pLayers[0].name, "0");
		pMrkCtl->SetLayers(nLayers, pLayers);
	}
}

void	CSampMarkupView::OnUpdateEntitiesLeader(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_UDE_LEADER);
}
void	CSampMarkupView::OnEntitiesLeader()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesLeader, ENTTYPE_UDE_LEADER);
}

void	CSampMarkupView::OnUpdateEntitiesFreeStyle(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_FSTYLE);
}
void	CSampMarkupView::OnEntitiesFreeStyle()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesFreeStyle, ENTTYPE_FSTYLE);
}

void	CSampMarkupView::OnUpdateEntitiesCloud(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_UDE_CLOUD);
}
void	CSampMarkupView::OnEntitiesCloud()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesCloud, ENTTYPE_UDE_CLOUD);
}

void	CSampMarkupView::OnUpdateEntitiesText(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_TEXT);
}

void	CSampMarkupView::OnEntitiesText()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateEntitiesText)) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl->GetSelCount() == 1) {
		MRK_EntitySpec	EntitySpec;
		pMrkCtl->GetSels(1, &EntitySpec.Com.Handle);

		pMrkCtl->LockEntity(&EntitySpec);
		pMrkCtl->UnlockEntity(&EntitySpec, FALSE);
		if (EntitySpec.Com.Type != ENTTYPE_TEXT && EntitySpec.Com.Type != ENTTYPE_LINK &&
			EntitySpec.Com.Type != ENTTYPE_NOTE)
		{
			// Check if entity already contain a hyperlink.
			int	numChildren = pMrkCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_TEXT);
			if (numChildren <= 0) {
				ENTHANDLE	Handle = EntitySpec.Com.Handle;
				memset(&EntitySpec, 0, sizeof(EntitySpec));

				EntitySpec.Com.ParentHandle = Handle;
				EntitySpec.Com.Type = ENTTYPE_TEXT;

				pMrkCtl->AddEntity(&EntitySpec);

				return;
			}
		}
	}

	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesText, ENTTYPE_TEXT);
}

void	CSampMarkupView::OnUpdateEntitiesNote(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_NOTE);
}

void	CSampMarkupView::OnEntitiesNote()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateEntitiesNote)) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl->GetSelCount() == 1) {
		MRK_EntitySpec	EntitySpec;
		pMrkCtl->GetSels(1, &EntitySpec.Com.Handle);

		pMrkCtl->LockEntity(&EntitySpec);
		pMrkCtl->UnlockEntity(&EntitySpec, FALSE);
		if (EntitySpec.Com.Type != ENTTYPE_TEXT && EntitySpec.Com.Type != ENTTYPE_LINK &&
			EntitySpec.Com.Type != ENTTYPE_NOTE)
		{
			// Check if entity already contain a hyperlink.
			int	numChildren = pMrkCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_NOTE);
			if (numChildren <= 0) {
				ENTHANDLE	Handle = EntitySpec.Com.Handle;
				memset(&EntitySpec, 0, sizeof(EntitySpec));

				EntitySpec.Com.ParentHandle = Handle;
				EntitySpec.Com.Type = ENTTYPE_NOTE;

				pMrkCtl->AddEntity(&EntitySpec);

				return;
			}
		}
	}

	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesNote, ENTTYPE_NOTE);
}

void CSampMarkupView::OnUpdateEntitiesLine(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_LINE);
}
void CSampMarkupView::OnEntitiesLine()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesLine, ENTTYPE_LINE);
}

void	CSampMarkupView::OnUpdateEntitiesPolyline(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_POLY);
}
void	CSampMarkupView::OnEntitiesPolyline()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesPolyline, ENTTYPE_POLY);
}

void	CSampMarkupView::OnUpdateEntitiesBox(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_RECT, MRK_FILLNONE);
}
void	CSampMarkupView::OnEntitiesBox()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesBox, ENTTYPE_RECT);
}

void	CSampMarkupView::OnUpdateEntitiesFilledBox(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_RECT, MRK_FILLSOLID);
}

void	CSampMarkupView::OnEntitiesFilledBox()
{
	Safe_OnEntitiesXXX1(&CSampMarkupView::OnUpdateEntitiesFilledBox, ENTTYPE_RECT, MRK_FILLSOLID);
}

void	CSampMarkupView::OnUpdateEntitiesHighlight(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_RECT, MRK_FILLTRANSPARENT);
}

void	CSampMarkupView::OnEntitiesHighlight()
{
	Safe_OnEntitiesXXX1(&CSampMarkupView::OnUpdateEntitiesHighlight, ENTTYPE_RECT, MRK_FILLTRANSPARENT);
}

void	CSampMarkupView::OnUpdateEntitiesCircle(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_CIRCLE);
}
void	CSampMarkupView::OnEntitiesCircle()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesCircle, ENTTYPE_CIRCLE);
}

void	CSampMarkupView::OnUpdateEntitiesArc(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_ARC);
}
void	CSampMarkupView::OnEntitiesArc()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesArc, ENTTYPE_ARC);
}

void CSampMarkupView::OnUpdateEntitiesOle(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_UDE_OLE);
}
void CSampMarkupView::OnEntitiesOle()
{
	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateEntitiesOle, ENTTYPE_UDE_OLE);
}
/************************************************************************
** END MARKUP 'ENTITIES' MENU
************************************************************************/


/************************************************************************
** BEGIN MARKUP 'MODIFY' MENU
************************************************************************/
void CSampMarkupView::OnUpdateModifyRedo(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl && pMrkCtl->IsRedoAvailable() &&
				pMrkCtl->IsWindowVisible());
}
void CSampMarkupView::OnModifyRedo()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyRedo)) {
		return;
	}
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);
	pMrkCtl->Redo();
}

void CSampMarkupView::OnUpdateModifyUndo(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl && pMrkCtl->IsUndoAvailable() && pMrkCtl->IsWindowVisible());
}
void CSampMarkupView::OnModifyUndo()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyUndo)) {
		return;
	}
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);
	pMrkCtl->Undo();
}

void	CSampMarkupView::OnUpdateModifyDelete(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	BOOL	fEnable = FALSE;
	if (pMrkCtl && pMrkCtl->IsWindowVisible() && pMrkCtl->GetSelCount()) {
		fEnable = TRUE;
	}
	pCmdUI->Enable(fEnable);
}
void	CSampMarkupView::OnModifyDelete()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyDelete)) {
		return;
	}

	// Fake "Delete" key pressing for markup control
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	pMrkCtl->SendMessage(WM_KEYDOWN, VK_DELETE);
	pMrkCtl->SendMessage(WM_KEYUP, VK_DELETE);
}

void	CSampMarkupView::OnUpdateModifyDeleteAll(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl != NULL && pMrkCtl->HasEntities() && pMrkCtl->IsWindowVisible());
}
void	CSampMarkupView::OnModifyDeleteAll()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyDeleteAll)) {
		return;
	}
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);
	pMrkCtl->DeleteAllEntities();
}

void CSampMarkupView::OnUpdateModifyMarkupOnoff(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl != 0);
	pCmdUI->SetCheck(pMrkCtl != 0 && pMrkCtl->IsWindowVisible() ? 0 : 1);
}
void CSampMarkupView::OnModifyMarkupOnoff()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyMarkupOnoff)) {
		return;
	}
	CMarkupControl* pMrkpCtl = GetMarkupControl();
	pMrkpCtl->ShowWindow(pMrkpCtl->IsWindowVisible() ? SW_HIDE : SW_SHOW);
}
/************************************************************************
** END MARKUP 'MODIFY' MENU
************************************************************************/


/************************************************************************
** BEGIN MARKUP FONT/COLOUR/LAYER/WIDTH etc
************************************************************************/
// --> Font <--
void CSampMarkupView::OnUpdateModifyEntityFont(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl != NULL && pMrkCtl->IsWindowVisible());
}
void CSampMarkupView::OnModifyEntityFont() {
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateModifyEntityFont)) {
		return;
	}

	CMarkupControl *pMrkCtl = GetMarkupControl();
	if (! pMrkCtl) {
		return;
	}

	CMrkFont font;
	pMrkCtl->GetFont(font);
	LOGFONT lf = font.GetLOGFONT();
	// Set font dialog attributes:
	lf.lfHeight *= 20;
	HDC hdc = ::GetDC(NULL);
	int yRes = GetDeviceCaps(hdc, LOGPIXELSY);
	lf.lfHeight = (int)((double)lf.lfHeight * (double)yRes / 1440.0);

	CFontDialog		fontDlg(&lf, CF_EFFECTS | CF_SCREENFONTS, NULL, NULL);

	// Set font dialog attributes:
	fontDlg.m_cf.lStructSize = sizeof(CHOOSEFONT);
	fontDlg.m_cf.Flags = CF_SCREENFONTS | CF_EFFECTS | CF_INITTOLOGFONTSTRUCT | CF_ENABLEHOOK;
	fontDlg.m_cf.lpLogFont = &lf;
	fontDlg.m_cf.nFontType = SCREEN_FONTTYPE;

	if (fontDlg.DoModal() == IDOK) {
		lf.lfHeight = fontDlg.m_cf.iPointSize / 10;
		pMrkCtl->SetSelectedFont(lf);
	}
}

// --> Line style <--

void CSampMarkupView::OnUpdateModifyLineStyleDash(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}

void CSampMarkupView::OnModifyLineStyleDash()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenStyle(PS_DASH); 
}

void CSampMarkupView::OnUpdateModifyLineStyleDot(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineStyleDot()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenStyle(PS_DOT);
}

void CSampMarkupView::OnUpdateModifyLineStyleDsdd(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineStyleDsdd()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenStyle(PS_DASHDOTDOT);
}

void CSampMarkupView::OnUpdateModifyLineStyleDshd(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineStyleDshd()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenStyle(PS_DASHDOT);
}

void CSampMarkupView::OnUpdateModifyLineStyleSolid(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}

void CSampMarkupView::OnModifyLineStyleSolid()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenStyle(PS_SOLID);
}

// --> Line width <--

void CSampMarkupView::OnUpdateModifyLineWidth1(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineWidth1()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenWidth(1);
}

void CSampMarkupView::OnUpdateModifyLineWidth2(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineWidth2()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenWidth(2);
}

void CSampMarkupView::OnUpdateModifyLineWidth3(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineWidth3()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenWidth(3);
}

void CSampMarkupView::OnUpdateModifyLineWidth4(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineWidth4()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenWidth(4);
}

void CSampMarkupView::OnUpdateModifyLineWidth5(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}
void CSampMarkupView::OnModifyLineWidth5()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pMrkCtl->SetSelectedPenWidth(5);
}

void CSampMarkupView::OnUpdateColorRed(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}

// --> Color <--

void CSampMarkupView::OnColorRed()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);

	pMrkCtl->SetSelectedColor(RGB(255, 0, 0));
	pMrkCtl->SetSelectedFillColor(RGB(255, 0, 0));
}
void CSampMarkupView::OnUpdateColorGreen(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}

void CSampMarkupView::OnColorGreen()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);

	pMrkCtl->SetSelectedColor(RGB(0, 255, 0));
	pMrkCtl->SetSelectedFillColor(RGB(0, 255, 0));
}

void CSampMarkupView::OnUpdateColorBlue(CCmdUI* pCmdUI)
{
	pCmdUI->Enable(TRUE);
}

void CSampMarkupView::OnColorBlue()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);

	pMrkCtl->SetSelectedColor(RGB(0, 0, 255));
	pMrkCtl->SetSelectedFillColor(RGB(0, 0, 255));
}
/************************************************************************
** END MARKUP COLOUR/LAYER/WIDTH etc
************************************************************************/


/************************************************************************
** Begin Hyperlink
************************************************************************/
void CSampMarkupView::OnUpdateHyperlinkEdit(CCmdUI* pCmdUI)
{
	BOOL			fEnable = FALSE;
	CMarkupControl* pMrkCtl = GetMarkupControl();

	if (pMrkCtl && pMrkCtl->IsWindowVisible()) {
		int	nTotalSels = pMrkCtl->GetSelCount();

		if (nTotalSels == 0) {
			fEnable = IsEntityTypeAvailable(ENTTYPE_LINK);

		} else if (nTotalSels == 1) {
			MRK_EntitySpec	EntitySpec;
			pMrkCtl->GetSels(1, &EntitySpec.Com.Handle);
			pMrkCtl->LockEntity(&EntitySpec);
			pMrkCtl->UnlockEntity(&EntitySpec, FALSE);

			fEnable = (EntitySpec.Com.Type == ENTTYPE_LINK) || pMrkCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_LINK);
		}
	}

	pCmdUI->Enable(fEnable);
}

void CSampMarkupView::OnHyperlinkEdit()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateHyperlinkEdit)) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	BOOL		fSuccess = FALSE;
	MRK_EntitySpec EntitySpec;

	// Need to set the proper cursor.
	m_notifyAction = HYPERLINK_EDIT;

	fSuccess = GetHyperlinkSelection(&EntitySpec);

	if (fSuccess) {
		pMrkCtl->LockEntity( &EntitySpec );
		ASSERT(EntitySpec.Com.Type == ENTTYPE_LINK);

		if (HyperlinkEstablish( &EntitySpec, TRUE )) {
			pMrkCtl->UnlockEntity(&EntitySpec, TRUE);
		} else {
			pMrkCtl->UnlockEntity(&EntitySpec, FALSE);
		}
	} else {
		::MessageBeep((UINT)-1);
	}

	// Reset action.
	m_notifyAction = CHANGE_NONE;
}

void CSampMarkupView::OnUpdateHyperlinkEstablish(CCmdUI* pCmdUI)
{
	OnUpdateEntitiesXXX(pCmdUI, ENTTYPE_LINK);
}

void CSampMarkupView::OnHyperlinkEstablish()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateHyperlinkEstablish)) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl->GetSelCount() == 1) {
		MRK_EntitySpec	EntitySpec;
		pMrkCtl->GetSels(1, &EntitySpec.Com.Handle);

		pMrkCtl->LockEntity(&EntitySpec);
		pMrkCtl->UnlockEntity(&EntitySpec, FALSE);
		if (EntitySpec.Com.Type != ENTTYPE_LINK && EntitySpec.Com.Type != ENTTYPE_NOTE) {
			// Check if entity already contain a hyperlink.
			int	numChildren = pMrkCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_LINK);
			if (numChildren <= 0) {
				ENTHANDLE	Handle = EntitySpec.Com.Handle;
				memset(&EntitySpec, 0, sizeof(EntitySpec));

				EntitySpec.Com.ParentHandle = Handle;
				EntitySpec.Com.Type = ENTTYPE_LINK;
				EntitySpec.Ent.Link.dwFlags |= MLINK_FLAG_HIDE_ICON;

				m_notifyAction = HYPERLINK_ESTABLISH;

				pMrkCtl->AddEntity(&EntitySpec);

				m_notifyAction = CHANGE_NONE;

				return;
			}
		}
	}

	Safe_OnEntitiesXXX(&CSampMarkupView::OnUpdateHyperlinkEstablish, ENTTYPE_LINK);

	m_notifyAction = HYPERLINK_ESTABLISH;
}


void CSampMarkupView::OnUpdateHyperlinkFire(CCmdUI* pCmdUI)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	pCmdUI->Enable(pMrkCtl && pMrkCtl->IsWindowVisible() && IsEntityTypeAvailable(ENTTYPE_LINK));
}
void CSampMarkupView::OnHyperlinkFire()
{
	if (!Safe_IsMenuEnabled(&CSampMarkupView::OnUpdateHyperlinkFire)) {
		return;
	}

	CMarkupControl* pMrkpCtl = GetMarkupControl();
	ASSERT(pMrkpCtl);

	// Needed to set the proper cursor.
	m_notifyAction = HYPERLINK_FIRE;

	MRK_EntitySpec	EntitySpec;
	BOOL fSuccess = GetHyperlinkSelection(&EntitySpec);

	if (fSuccess) {
		// At this moment hyperlink entity is selected and handle is in EntitySpec
		PostMessage(AVM_FIREHYPERLINK, 0, (LPARAM) EntitySpec.Com.Handle);
	}

	// Reset group action.
	m_notifyAction = CHANGE_NONE;
}
/************************************************************************
** End Hyperlink
************************************************************************/



/************************************************************************
** BEGIN NOTIFICATION MESSAGES
************************************************************************/
LRESULT CSampMarkupView::OnNotifCancel(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	ASSERT (GetMarkupControl() != 0);
	EndAddingEntity();
	return 0;
}

LRESULT CSampMarkupView::OnNotifEdit(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	CMarkupControl* 	pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);

	LPMRK_EntitySpec	lpEntitySpec = (LPMRK_EntitySpec)lParam;

	if (m_notifyAction) {
		switch(m_notifyAction) {
			// we override default markup control establish dialog
		  case HYPERLINK_ESTABLISH:
			if (HyperlinkEstablish(lpEntitySpec)) {
				return 1;
			} else {
				// Abort.
				return -1;
			}

		  case HYPERLINK_EDIT:
			return 0;

		  default:
			return 0;
		} 
	}
	return 0;
}

LRESULT CSampMarkupView::OnNotifFirelink(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	LPMRK_EntitySpec lpEntity = (LPMRK_EntitySpec)lParam;
	ASSERT(lpEntity);

	PostMessage(AVM_FIREHYPERLINK, wParam, (LPARAM) lpEntity->Com.Handle);

	return 0L;
}

LRESULT CSampMarkupView::OnSampFireHyperlink(WPARAM wParam, LPARAM lParam)
{
	CMarkupControl *pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl != NULL);

	MRK_EntitySpec 	EntitySpec;
	memset(&EntitySpec, 0, sizeof(EntitySpec));

	EntitySpec.Com.Handle = (ENTHANDLE) lParam;
	pMrkCtl->LockEntity(&EntitySpec);

	ASSERT(EntitySpec.Com.Type == ENTTYPE_LINK);

	int	hType = EntitySpec.Ent.Link.LinkType;
	CString	strName(EntitySpec.Ent.Link.szName);
	CString	strDesc(EntitySpec.Ent.Link.szDesc);
	CString	S1(EntitySpec.Ent.Link.szDef1);
	CString	S2(EntitySpec.Ent.Link.szDef2);
	CString	S3(EntitySpec.Ent.Link.szDef3);

	pMrkCtl->UnlockEntity(&EntitySpec, FALSE);

	if (!S1.IsEmpty()) {
		HyperlinkFire(hType, strName, strDesc, S1, S2, S3);
	} else {
		AfxMessageBox("Entity is empty!\nPlease add file name to it.", MB_OK | MB_ICONEXCLAMATION);
	}
	return 0L;
}



LRESULT CSampMarkupView::OnNotifEntityadded(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	ASSERT (GetMarkupControl() != 0);

	CMarkupControl* pMrkCtl = GetMarkupControl();

	// Check special drawing entities first
	switch (pMrkCtl->GetEntity()) {
	  case ENTTYPE_FSTYLE:
		// keep freestyle drawing setting set active anyway
		return 0;
	}

	// Since addition have been performed set inactive state here
	if (pMrkCtl) {
		// Reset filltype to none.
		pMrkCtl->SetCurrentFillType(MRK_FILLNONE);

		pMrkCtl->SetAction(MRKP_ACTION_HYBRID);
		pMrkCtl->SetEntity(0);
	}

	return 0;
}

LRESULT CSampMarkupView::OnNotifSelchanged(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	CMarkupControl*  pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);
	return 0;
}

LRESULT CSampMarkupView::OnNotifModified(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	ASSERT (GetMarkupControl() != 0);
	return 0;
}

LRESULT CSampMarkupView::OnNotifCursor(WPARAM wParam, LPARAM lParam)
{
	PAN_MANAGE_MFCSTATE

	CMarkupControl *pMrkCtl = GetMarkupControl();
	ASSERT (pMrkCtl != 0);

	HCURSOR 	hCursor = NULL;
	UINT		nCursorID = 0;

	ENTHANDLE	NewLinkHandle = 0;
	CString		szNewLinkName = "";

	MRK_EntitySpec	EntitySpec;
	memset(&EntitySpec, 0, sizeof(MRK_EntitySpec));
	EntitySpec.Com.Type = ENTTYPE_NULL;

	if (lParam != 0) {
		EntitySpec.Com.Handle = (ENTHANDLE) lParam;

		pMrkCtl->LockEntity(&EntitySpec);
		if (EntitySpec.Com.Type == ENTTYPE_LINK) {
			NewLinkHandle = EntitySpec.Com.Handle;
			szNewLinkName = EntitySpec.Ent.Link.szName;
		}
		pMrkCtl->UnlockEntity(&EntitySpec, FALSE);

		if (EntitySpec.Com.Type != ENTTYPE_LINK &&
			pMrkCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_LINK) > 0)
		{
			pMrkCtl->GetChildren(EntitySpec.Com.Handle, 1, &EntitySpec.Com.Handle, ENTTYPE_LINK);

			pMrkCtl->LockEntity(&EntitySpec);
			ASSERT(EntitySpec.Com.Type == ENTTYPE_LINK);

			NewLinkHandle = EntitySpec.Com.Handle;
			szNewLinkName = EntitySpec.Ent.Link.szName;

			pMrkCtl->UnlockEntity(&EntitySpec, FALSE);
		}
	}

	switch (m_notifyAction) {
	  case HYPERLINK_DELETE:
	  case HYPERLINK_EDIT:
	  case HYPERLINK_FIRE:
	  	if (EntitySpec.Com.Type == ENTTYPE_LINK) {
	  		// On top of a hyperlink cursor.
	  		nCursorID = IDC_MARKUP_F_HYPER;

	  	} else {
			// Select hyperlink cursor.
			nCursorID = IDC_MARKUP_E_HYPER;
		}
		break;

	  case CHANGE_NONE:
		if (EntitySpec.Com.Type == ENTTYPE_LINK) {
			if (wParam == MRK_ENTNOTSELECTED) {
				// On top of hyplerlink cursor.
				nCursorID = IDC_MARKUP_F_HYPER;

			} else if (wParam == MRK_ENTSELECTED) {
				// On top of hyperlink that can be dragged.
				nCursorID = IDC_MARKUP_FT_HYPER;
			}
		}
		break;
	}

	if (nCursorID) {
		hCursor = AfxGetApp()->LoadCursor(nCursorID);
	}

	return (LRESULT)(LPVOID)hCursor;
}
/************************************************************************
** END NOTIFICATION MESSAGES
************************************************************************/




/************************************************************************
** BEGIN PRINTING
************************************************************************/
void CSampMarkupView::OnFilePrintPreview()
{
	((CSampMainFrame*)AfxGetMainWnd())->SetPreviewMode(TRUE);
	((CSampMainFrame*)AfxGetMainWnd())->m_fFirstTime = TRUE;

	CSampView::OnFilePrintPreview();

	if (GetMarkupControl() == 0) {
		return;
	}

	// Reset view extents after printing
	SetMarkupDefaultExtents();
}

void CSampMarkupView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo)
{
	CSampView::OnEndPrinting(pDC, pInfo);
	CSampMainFrame*	  pFrame = (CSampMainFrame*)AfxGetMainWnd();

	if (GetMarkupControl() == 0) {
		return;
	}

	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(GetDocument()->GetControl() && pMrkCtl);

	// Restore everything:
	SetMarkupDefaultExtents();
}

void CSampMarkupView::RenderMarkupFile(CDC* pDC, CPrintInfo* pInfo,
		  const PAN_CtlPrintOptions& options, UINT pageNum, UINT tileNum,
		  BOOL fForcetoblack)
//	Renders the markup file, if any, onto the given printer DC
//
//	pDC 		IN		Printer device context
//	pInfo		IN		Print information
//	options 	IN		Print options
//	pageNum 	IN		Current document page number (1-based)
//	tileNum 	IN		Current printed page (tile) number (1-based)
{
	ASSERT_VALID(this);
	ASSERT_VALID(pDC);
	ASSERT(AfxIsValidAddress(pInfo, sizeof *pInfo));
	ASSERT(tileNum > 0 && tileNum <= options.printPreview.nPages);

	// Get markup control
	CSampDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl == 0) {
		// Nothing to do! We are probably not in markup mode.
		return;
	}

	ASSERT(pMrkCtl);
	ASSERT_VALID(pMrkCtl);

	pDC->SaveDC();

	// Get image and clip rectangles for current page
	PAN_CtlRange	imageRect;
	CRect			clipRect;
	GetImageAndClipRects(options, tileNum, imageRect, clipRect);

	PAN_CtlFileInfo fileInfo;
	memset(&fileInfo, 0, sizeof(fileInfo));
	pDoc->GetControl()->GetFile(&fileInfo);

	// Take care of flipping
	if (fileInfo.type == PAN_VectorFile) {
		Real	t;
		t = imageRect.min.y;
		imageRect.min.y = imageRect.max.y;
		imageRect.max.y = t;
	}
	// Take care of AR/DB/SS files
	if (fileInfo.type == PAN_SpreadsheetFile ||
		fileInfo.type == PAN_DatabaseFile ||
		fileInfo.type == PAN_ArchiveFile) {

		/*
		**	We need to do the same calculations as done in doc.cpp namely:
		**	1 - Offset the World viewextents of the markup control by the
		**		row/column header sizes in twips.
		**	2 - Offet the Client position of the markup control by the
		**		row/column header sizes in pixels.
		*/

		// Step 2: Calculate the row/column header sizes in pixels, taking
		//	into account the scaling used in the render-onto-dc operation.
		CRect	rcHeader(0, 0, 0, 0);
		rcHeader.right	= pDoc->GetControl()->GetColWidth(CTLUNIT_PIXEL, 0);
		rcHeader.bottom = pDoc->GetControl()->GetRowHeight(CTLUNIT_PIXEL, 0);
		PAN_CtlRange	rg = imageRect;
		CRect			devRect = clipRect;

		pDoc->GetControl()->WorldToClient(&rg);

		double	r2, r3;
		if (rg.max.y - rg.min.y && rcHeader.bottom) {
			r2 = (rg.max.x - rg.min.x) / (rg.max.y - rg.min.y);
			r3 = (double) rcHeader.right / (double)rcHeader.bottom;
			double	a = clipRect.Width(), b = clipRect.Height();
			if (r3 - r2) {
				rcHeader.bottom = (int) (b - (b*r3 - a) / (r3 - r2));
				rcHeader.right	= (int) (rcHeader.bottom * r3);
			}
		}

		if (Fabs(rg.max.x - rg.min.x) > Small && Fabs(rg.max.y - rg.min.y) > Small) {

			Real	fx = Fabs( devRect.Width()	/ (rg.max.x - rg.min.x) );
			Real	fy = Fabs( devRect.Height() / (rg.max.y - rg.min.y) );
			rcHeader.right	= (int) (fx * rcHeader.right);
			rcHeader.bottom = (int) (fy * rcHeader.bottom);
		}
		// step 1:
		imageRect.min.x -= pDoc->GetControl()->GetColWidth(CTLUNIT_TWIPS, 0);
		imageRect.min.y -= pDoc->GetControl()->GetRowHeight(CTLUNIT_TWIPS, 0);
	}

	// Convert clip rectangle to physical device units
	// Use ::LPtoDP() instead of CDC::LPtoDP() because the latter acts
	// on CDC::m_hAttribDC instead of CDC::m_hDC
	::LPtoDP(pDC->GetSafeHdc(), (LPPOINT) (LPRECT) clipRect, 2);

	// Render contents of markup control
	PAN_CtlRenderOptions	ropts;
	memset(&ropts, 0, sizeof(PAN_CtlRenderOptions));

	ropts.hdc = pDC->GetSafeHdc();
	ropts.mode = options.mode;
	ropts.source = imageRect;
	ropts.devRect = clipRect;

	// Select clipping region
	CRgn	rgn;
	rgn.CreateRectRgnIndirect(clipRect);
	pDC->SelectClipRgn(&rgn);
	rgn.DeleteObject();

	pMrkCtl->RenderOntoDC(&ropts);

	pDC->RestoreDC(-1);
}

/************************************************************************
** END PRINTING
************************************************************************/


BOOL CSampMarkupView::CreateMarkupControl()
{
	//	Initialize Markup Library
	TRY {
		if (!CMrkpLibrary::Initialize()) {
			return FALSE;
		}
	} CATCH(CMemoryException, memE) {
		AfxMessageBox("Cannot initialize Markup Library");
		return FALSE;
	} END_CATCH

	CSampDoc* pDoc = GetDocument();
	ASSERT(pDoc);

	CMarkupControl* pMrkCtl = new CMarkupControl();
	ASSERT(pMrkCtl);
	if (pMrkCtl == NULL) {
		return FALSE;
	}

	CRect	rectClient;
	GetClientRect(&rectClient);

	if (pMrkCtl->Create(rectClient, this, pDoc->GetControl()->GetSafeHwnd())) {
		pDoc->SetMarkupControl(pMrkCtl);
	}

	// Set size/position and viewextents
	SetMarkupDefaultExtents();

	// Set background color.
	COLORREF	color = pDoc->GetControl()->GetFgBgColor(TRUE);
	pMrkCtl->SetFgBgColor(TRUE, color);

	pMrkCtl->SetCurrentColor(BYLAYERRGB);
	pMrkCtl->SetCurrentFillColor(BYLAYERRGB);

	pMrkCtl->SetLineWidth(1);
	pMrkCtl->SetPenStyle(PS_SOLID);

	// Base file info.
	PAN_CtlFileInfo fileInfo;
	memset(&fileInfo, 0, sizeof(fileInfo));
	pDoc->GetControl()->GetFile(&fileInfo);

	MRK_BaseInfo	BaseInfo;
	memset(&BaseInfo, 0, sizeof(MRK_BaseInfo));

	BaseInfo.Offset.x = BaseInfo.Offset.y = BaseInfo.Offset.z = 0.0;
	BaseInfo.Scale.x = BaseInfo.Scale.y = BaseInfo.Scale.z = 1.0;
	BaseInfo.Dpi.x = BaseInfo.Dpi.y = BaseInfo.Dpi.z = 1.0;
	BaseInfo.Rotation = 0.;
	BaseInfo.nView = -1;

	if (fileInfo.ins.scale.x > ExtremelyVerySmall &&
		fileInfo.ins.scale.y > ExtremelyVerySmall)
	{
		BaseInfo.Offset.x = fileInfo.ins.offset.x;
		BaseInfo.Offset.y = fileInfo.ins.offset.y;
		BaseInfo.Offset.z = 0.0;

		BaseInfo.Scale.x = fileInfo.ins.scale.x;
		BaseInfo.Scale.y = fileInfo.ins.scale.y;
		BaseInfo.Scale.z = 1.0;

		BaseInfo.Dpi = fileInfo.ins.dpi;

		BaseInfo.Rotation = fileInfo.ins.rot;
	}

	pMrkCtl->SetBaseInfo(&BaseInfo);

	// Show window.
	pMrkCtl->SetWindowPos(0,0,0,0,0,
		SWP_SHOWWINDOW | SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_NOREDRAW);

	LoadUdesInfo();

	return TRUE;
}
void CSampMarkupView::DestroyMarkupControl()
{
	CMarkupControl *pMrkCtl = GetMarkupControl();
	if (pMrkCtl == NULL) {
		return;
	}

	// Save markup modifications if modified
	if (pMrkCtl->IsModified()) {
		CMarkupSaveDlg	saveDlg(this, TRUE /* hideCancel */);
		if (saveDlg.DoModal() == IDOK) {
			OnMarkupSave();
		}
	}

	// Delete UDE Information
	if (m_pUdes && m_nUdes) {
		delete [] m_pUdes;
		m_nUdes = 0;
		m_pUdes = NULL;
	}

	// Destroy markup control if window
	if (::IsWindow(pMrkCtl->GetSafeHwnd())) {
		pMrkCtl->DeleteMarkup(0);
		delete pMrkCtl;
	}
	GetDocument()->SetMarkupControl(0);
}


BOOL CSampMarkupView::HyperlinkEstablish(LPMRK_EntitySpec lpEntity, BOOL fLoadData)
{
	int nResult = IDCANCEL;
	CHyperEstablishDlg	dlg(this);

	if (fLoadData) {
		dlg.SetValues(lpEntity->Ent.Link.LinkType, lpEntity->Ent.Link.szName,
				lpEntity->Ent.Link.szDesc, lpEntity->Ent.Link.szDef1,
				lpEntity->Ent.Link.szDef2);
	}

	nResult = dlg.DoModal();

	if (nResult == IDOK) {
		int		hType = 0;
		CString	strName;
		CString	strDesc;
		CString	strS1;
		CString	strS2;

		dlg.GetValues(hType, &strName, &strDesc, &strS1, &strS2);

		lpEntity->Ent.Link.LinkType	= hType;

		// Free old and set new.
		if (lpEntity->Ent.Link.szName != NULL) {
			MrkFree((LPSTR)lpEntity->Ent.Link.szName);
			lpEntity->Ent.Link.szName = NULL;
		}
		lpEntity->Ent.Link.szName = MrsAlloc(strName);

		if (lpEntity->Ent.Link.szDesc != NULL) {
			MrkFree((LPSTR)lpEntity->Ent.Link.szDesc);
			lpEntity->Ent.Link.szDesc = NULL;
		}
		lpEntity->Ent.Link.szDesc = MrsAlloc(strDesc);

		switch (hType) {
			case MLINK_TO_SCRIPT:
			case MLINK_TO_FILE:
		  		if (lpEntity->Ent.Link.szDef1 != NULL) {
					MrkFree((LPSTR)lpEntity->Ent.Link.szDef1);
					lpEntity->Ent.Link.szDef1 = NULL;
				}
				lpEntity->Ent.Link.szDef1 = MrsAlloc(strS1);
				break;
			case MLINK_TO_DDE:
			case MLINK_TO_APP:
			case MLINK_TO_DLL:
				if (lpEntity->Ent.Link.szDef1 != NULL) {
					MrkFree((LPSTR)lpEntity->Ent.Link.szDef1);
					lpEntity->Ent.Link.szDef1 = NULL;			
				}
				lpEntity->Ent.Link.szDef1 = MrsAlloc(strS1);
			
		 		if (lpEntity->Ent.Link.szDef2 != NULL) {
					MrkFree((LPSTR)lpEntity->Ent.Link.szDef2);
					lpEntity->Ent.Link.szDef2 = NULL;			
				}
				lpEntity->Ent.Link.szDef2 = MrsAlloc(strS2);	
				break;
			default:
				break;
		}
	}
	return (nResult == IDOK) ? TRUE : FALSE;
}

BOOL CSampMarkupView::HyperlinkFire(int hType, CString &strTitle, CString &strDesc, CString &S1, CString &S2, CString &S3)
{
	CSampApp	*pApp = (CSampApp*)AfxGetApp();
	CSampDoc* 	pDoc = GetDocument();
	ASSERT(pDoc);

	CMarkupControl*	pMrkCtl = GetMarkupControl();

	ASSERT(pMrkCtl);

	// If hyperlink have been found change mode
	if (1) {
		CVCETControl	*pCtl = pDoc->GetControl();
		ASSERT(pCtl != 0);

		switch (hType) {
			case MLINK_TO_FILE:
				{{{
				CString strNewDocname = S1;
				OnMarkupExit();

				pApp->OpenDocumentFile(strNewDocname);
				}}}
				break;
			default:
				break;
		}
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Saves currently active markup object (so that to save set active first)
/////////////////////////////////////////////////////////////////////////////
BOOL CSampMarkupView::SaveMarkupFile(LPCSTR lpszNewName /* = NULL */)
{
	// Get document
	CSampDoc* 			pDoc = GetDocument();
	ASSERT(pDoc);
	// Get markup control
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);
	// Current markup
	CString		pFile = pMrkCtl->GetMarkupFileName();

	// Get filename for saving
	CString strMarkupSaveName;	// filename to use for saving
	if (lpszNewName == NULL) {
		if (pFile.IsEmpty()) {
			ASSERT(0);
			return FALSE;
		} else {
			strMarkupSaveName = pFile;
		}
	} else {
		// Verify that specified file does not exist
		struct	stat	statbuf;
		if (! stat(lpszNewName, &statbuf)) {
			// file exists: Load message and alert user
			CString 	str;
			str.LoadString(IDS_FILEEXIST_OVERWRT);
			// prepare message with file name
			char		msg[_MAX_PATH];
			wsprintf(msg, str, lpszNewName);
			if (AfxMessageBox(msg, MB_YESNO) != IDYES) {
				return FALSE;
			}
		}

		// Ok, this filename will be used for markup file saving
		strMarkupSaveName = lpszNewName;
	}

	// Verify that markup directory exists
	char	dr[_MAX_DRIVE], d[_MAX_DIR];
	_splitpath(strMarkupSaveName, dr, d, NULL, NULL);
	// Strip trailing backslash
	int	len = lstrlen(d);
	if (len && (d[len-1] == '\\' || d[len-1] == '/')) {
		d[len-1] = '\0';
	}
	CString named = dr;
	named += d;
#if TARGET == TWUNIX
	mkdir(named, 0);
#else
	mkdir(named);
#endif

	// Actual saving through markup control
	BOOL	fResult = pMrkCtl->Save(strMarkupSaveName);
	if (fResult == FALSE) {
		return FALSE;
	}

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Saves currently active markup object (so that to save set active first)
/////////////////////////////////////////////////////////////////////////////
BOOL	CSampMarkupView::GetHyperlinkSelection(LPMRK_EntitySpec lpEntity)
{
	if (lpEntity == NULL) {
		return FALSE;
	}

	CMarkupControl* pMrkpCtl = GetDocument()->GetMarkupControl();
	ASSERT(pMrkpCtl);

	int 	nTotalSels = pMrkpCtl->GetSelCount();
	int 	nNumLinkSels = 0;

	BOOL		fSuccess = FALSE;
	MRK_EntitySpec	EntitySpec;
	LPENTHANDLE	pEntHandles = NULL;
	if (nTotalSels) {
		pEntHandles = new ENTHANDLE [nTotalSels];
		ASSERT(pEntHandles != NULL);

		memset(pEntHandles, 0, nTotalSels * sizeof(ENTHANDLE));

		pMrkpCtl->GetSels(nTotalSels, pEntHandles);

		for (int i = 0; i < nTotalSels; i++) {
			if (pEntHandles[i] != 0) {
				memset(&EntitySpec, 0, sizeof(MRK_EntitySpec));
				EntitySpec.Com.Handle = pEntHandles[i];

				pMrkpCtl->LockEntity(&EntitySpec);
				pMrkpCtl->UnlockEntity(&EntitySpec, FALSE);

				if (EntitySpec.Com.Type == ENTTYPE_LINK) {
					// Top level hyperlink.
					nNumLinkSels++;

				} else {
					if (pMrkpCtl->GetNumChildren(EntitySpec.Com.Handle, ENTTYPE_LINK)) {
						// Embedded hyperlink.
						pMrkpCtl->GetChildren(EntitySpec.Com.Handle, 1, pEntHandles+i, ENTTYPE_LINK);
						nNumLinkSels++;

					} else {
						pEntHandles[i] = 0;
					}
				}
			}
		}
	}

	if (nNumLinkSels == 0) {
		fSuccess = this->GetEntityInteractively( lpEntity, ENTTYPE_LINK );
	} else {
		for ( int i=0; i < nTotalSels; i++ ) {
			if (pEntHandles[i] != 0) {
				lpEntity->Com.Handle = pEntHandles[i];
				fSuccess = TRUE;
				break;
			}
		}
	}

	if (pEntHandles != NULL) {
		delete pEntHandles;
		pEntHandles = NULL;
	}

	return fSuccess;
}

BOOL CSampMarkupView::GetTrueExtents( PAN_CtlRange* pMarkupTrueExtents)
{
	if (pMarkupTrueExtents == NULL) return FALSE;

	// Starting with View mode Extents values and then extending them
	CSampView::GetTrueExtents(pMarkupTrueExtents);

	CMarkupControl* pMrkpCtl = GetMarkupControl();
	if (pMrkpCtl == NULL) {
		return TRUE;
	}

	/*
	** Markup control will return 0 when it can not compute
	** valid extents (empty or all entities are visible).
	** Should not do anything in that case.
	*/
	PAN_CtlRange	markupExtents;
	if (pMrkpCtl->GetPageSize ( &markupExtents )) {
		// extend extents.
		PAN_CtlRange	*pB = pMarkupTrueExtents, *pM = &markupExtents; // shorter

		pB->min.x = (pB->min.x < pM->min.x) ? pB->min.x : pM->min.x;
		pB->min.y = (pB->min.y < pM->min.y) ? pB->min.y : pM->min.y;
		pB->max.x = (pB->max.x > pM->max.x) ? pB->max.x : pM->max.x;
		pB->max.y = (pB->max.y > pM->max.y) ? pB->max.y : pM->max.y;
	}

	return TRUE;
}

void CSampMarkupView::SetMarkupViewExtents()
{
	// Set viewextents of markup control to match base control.
	// One special case is considered:
	//	The markup control's viewextents are always wrt
	//	topleft tobottom-right so we need to flip for vector files.
	CSampDoc*		pDoc = GetDocument();
	CMarkupControl* pMrkCtl = pDoc->GetMarkupControl();
	if (pMrkCtl == 0) {
		// Nothing to do.
		return;
	}
	ASSERT_VALID(pMrkCtl);

	CVCETControl	*pCtl = GetDocument()->GetControl();
	ASSERT(pCtl != 0);

	PAN_CtlFileInfo fileInfo;
	pCtl->GetFile(&fileInfo);

	PAN_CtlRange	rg;
	pCtl->GetViewExtents(&rg);

	if (fileInfo.type == PAN_VectorFile) {
		Real	t;
		t = rg.min.y;
		rg.min.y = rg.max.y;
		rg.max.y = t;
	}

	// Forward onto markup control
	pMrkCtl->SetViewExtents(&rg);
}

void CSampMarkupView::SetMarkupClientRect()
{
	// Size and position the markup control, based on the base control's
	// client area.
	// One special case is considered:
	//	For table-based formats, need to take into account the row and
	//	column headers.
	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl == 0) {
		// Nothing to do.
		return;
	}
	ASSERT_VALID(pMrkCtl);

	CVCETControl	*pCtl = GetDocument()->GetControl();
	ASSERT(pCtl != 0);

	PAN_CtlFileInfo fileInfo;
	pCtl->GetFile(&fileInfo);

	CRect	rcClient;
	pCtl->GetClientRect(&rcClient);

	if (fileInfo.type == PAN_SpreadsheetFile ||
		fileInfo.type == PAN_DatabaseFile ||
		fileInfo.type == PAN_ArchiveFile) {

		int nColWidth	= GetDocument()->GetControl()->GetColWidth(CTLUNIT_PIXEL, 0);
		int nRowHeight	= GetDocument()->GetControl()->GetRowHeight(CTLUNIT_PIXEL, 0);
		rcClient.left += nColWidth;
		rcClient.top  += nRowHeight;
	}

	// Forward onto markup control
	pMrkCtl->MoveWindow(rcClient, FALSE);
}

void CSampMarkupView::SetMarkupDefaultExtents()
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	if (pMrkCtl == 0) {
		// Nothing to do.
		return;
	}
	ASSERT_VALID(pMrkCtl);

	// Set base file info
	CVCETControl	*pCtl = GetDocument()->GetControl();

	PAN_CtlFileInfo fileInfo;
	memset(&fileInfo, 0, sizeof(fileInfo));
	pCtl->GetFile(&fileInfo);

	PAN_CtlRange	PageSize = fileInfo.dimensions;
	if (fileInfo.type == PAN_DocumentFile && fileInfo.nPages > 1) {
		// Set average page size.  Setting whole
		// document size will affect cause markup text
		// to be extreemely small.


		PageSize.max.x = PageSize.min.x + (PageSize.max.x - PageSize.min.x) / fileInfo.nPages;
		PageSize.max.y = PageSize.min.y + (PageSize.max.y - PageSize.min.y) / fileInfo.nPages;
	}

	pMrkCtl->SetBaseExtents(&PageSize);

	// Set current positioning
	SetMarkupClientRect();

	// Set current viewextents
	SetMarkupViewExtents();
}


BOOL CSampMarkupView::SetMarkup(CString selectedMarkup)
{
	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	// delete the old markup object
	pMrkCtl->DeleteMarkup(0);

	// add the new markup object
	pMrkCtl->AddNewMarkup();
	pMrkCtl->SetMarkupFileName(selectedMarkup);

	// Read new markup data into newly created markup object.
	pMrkCtl->Read(selectedMarkup);
	// Start up as red.
	pMrkCtl->SetCurrentColor(RGB(255, 0, 0));
	pMrkCtl->SetCurrentFillColor(RGB(255, 0, 0));

	pMrkCtl->ShowMarkup(TRUE);
	pMrkCtl->SetAction(MRKP_ACTION_HYBRID, 0);

	// Set viewextents on markup control, to match current viewextents
	SetMarkupDefaultExtents();

	// Setup default layers.
	SetDefaultMarkupLayerName();

	return TRUE;
}



BOOL  CSampMarkupView::GetEntityInteractively(LPMRK_EntitySpec lpEntity, int Type)
{
	SetCapture();
	CWnd*	pwndPreviousFocusHolder = GetFocus();
	SetFocus();

	CMarkupControl* pMrkCtl = GetMarkupControl();
	ASSERT(pMrkCtl);

	MSG 	message;
	BOOL	fEnough = FALSE;
	BOOL	fSuccess = FALSE;

	while ( GetMessage( &message, NULL, 0, 0 ) && fEnough == FALSE ) {

		switch (message.message) {
		  case WM_LBUTTONDOWN:
			{{{{{{{{
			CPoint	ptClick = message.pt;
			::MapWindowPoints( NULL, this->GetSafeHwnd(), &ptClick, 1 );

			// Check that we are on top of the parent window
			CRect	rtClient;
			GetClientRect(&rtClient);
			if ( rtClient.PtInRect( ptClick ) ) {

				// Create 0 width selection region
				PAN_CtlRange	rtNewSelRect = {ptClick.x, ptClick.y, 0.0, ptClick.x, ptClick.y, 0.0};

				// Convert the region to world
				pMrkCtl->ClientToWorld(&rtNewSelRect.min);
				pMrkCtl->ClientToWorld(&rtNewSelRect.max);

				int	numSels = pMrkCtl->SetSel(ENTTYPE_NULL, &rtNewSelRect);
				if (numSels != 1) {
					pMrkCtl->ClearSel();

				} else {
					pMrkCtl->GetSels(1, &lpEntity->Com.Handle);
					pMrkCtl->LockEntity(lpEntity);
					pMrkCtl->UnlockEntity(lpEntity, FALSE);

					if (Type == ENTTYPE_NULL || lpEntity->Com.Type == Type) {
						// Top level entity.
						fSuccess = TRUE;

					} else {
						// Nested entity.
						if (pMrkCtl->GetNumChildren(lpEntity->Com.Handle, Type) <= 0) {
							// None.
							pMrkCtl->ClearSel();

						} else {
							pMrkCtl->GetChildren(lpEntity->Com.Handle, 1, &lpEntity->Com.Handle, Type);
							fSuccess = TRUE;
						}
					}
				}
			}
			}}}}}}}}
			fEnough = TRUE;
			break;

		  case	WM_MOUSEMOVE:
		  	// Patch to be able to have the markup control decide
		  	// which entity is closest to the mouse.  The cursor will
		  	// be switched (if needed) when we get a MRKN_SETCURSOR
		  	// notification from the markup control.
		  	::SendMessage(pMrkCtl->GetSafeHwnd(), WM_SETCURSOR, 0, 0L);
		  	break;

		  // Termination happens on deactivation and rightmouse pressing
		  case WM_ACTIVATE:
			if (message.wParam != WA_INACTIVE) {
				break;
			}
		  case WM_RBUTTONDOWN:
			fEnough = TRUE;
			break;
		}
		TranslateMessage(&message);
		DispatchMessage(&message);
	}

	ReleaseCapture();
	if (pwndPreviousFocusHolder) {
		pwndPreviousFocusHolder->SetFocus();
	}

	if (!fSuccess) {
		::MessageBeep(MB_ICONHAND);
	}
	return fSuccess;
}

int CSampMarkupView::GetUdeRuntimeID(int nInternalID)
{
	if (nInternalID >= ENTTYPE_UDE_FIRST && nInternalID <= ENTTYPE_UDE_LAST) {
		for (int i=0; i < m_nUdes; i++) {
			if (m_pUdes[i].idInternal == nInternalID) {
				return m_pUdes[i].idNum;	// found
			}
		}
		return 0;	// Not found
	}
	return nInternalID; // nothing required
}


// Check if a menu item is enabled.
BOOL CSampMarkupView::IsMenuEnabled(void (CSampMarkupView::*fnOnUpdate) (CCmdUI *))
{
	CDummyCmdUI cd;
	(this->*fnOnUpdate)(&cd);
	BOOL	ret = cd.IsEnabled();

	return (ret);
}

void CSampMarkupView::OnViewToolbar()
{
	CSampMainFrame*	   pFrame = (CSampMainFrame*)AfxGetMainWnd();

	// toggle according to visible state.
	if (pFrame->m_wndToolBar.IsWindowVisible()) {  	// visible
		// HIDE THE TOOLBARS
		pFrame->m_wndToolBar.ShowWindow(SW_HIDE);
		pFrame->m_wndMarkupBar.ShowWindow(SW_HIDE);
	} else {										// invisible
		// SHOW THE TOOLBARS
		pFrame->m_wndToolBar.ShowWindow(SW_SHOW);
		// check if markup control exists.  Show the toolbar when markup
		// control exists.
		if (GetMarkupControl()) {
			pFrame->m_wndMarkupBar.ShowWindow(SW_SHOW);
		}
	}
	pFrame->RecalcLayout();		// recalculate the layout of main frame
}

void CSampMarkupView::OnUpdateViewToolbar(CCmdUI* pCmdUI)
{
	// get the flag to indicate whether the toolbar is visible
	BOOL flag = ((CSampMainFrame*)AfxGetMainWnd())->m_wndToolBar.IsWindowVisible();
	pCmdUI->SetCheck(flag);
}

void CSampMarkupView::OnViewStatusbar()
{
	CSampMainFrame*	   pFrame = (CSampMainFrame*)AfxGetMainWnd();

	// toggle according to the visible state
	if (pFrame->m_wndStatusBar.IsWindowVisible()) {
		// HIDE THE STATUS BAR
		pFrame->m_wndStatusBar.ShowWindow(SW_HIDE);
	} else {
		// SHOW THE STATUS BAR
		pFrame->m_wndStatusBar.ShowWindow(SW_SHOW);
	}
	pFrame->RecalcLayout();
}

void CSampMarkupView::OnUpdateViewStatusbar(CCmdUI* pCmdUI)
{
	BOOL flag = ((CSampMainFrame*)AfxGetMainWnd())->m_wndStatusBar.IsWindowVisible();
	pCmdUI->SetCheck(flag);
}



/*
**  Utility function: Generate the fill filename, given an "ID" (ie extension)
*/
/*static*/
CString	CSampMarkupView::GetFilenameFromID(LPCSTR szBaseFilename, LPCSTR szDir, LPCSTR szID)
{
	ASSERT(szBaseFilename != 0 && *szBaseFilename);
	ASSERT(szDir != 0 && *szDir);
	ASSERT(szID != 0 && *szID);

	char		drive[_MAX_DRIVE], dir[_MAX_DIR], name[_MAX_FNAME], ext[_MAX_EXT];
	char		szResult[_MAX_PATH];
	_splitpath(szBaseFilename, drive, dir, name, ext);
	_makepath(szResult, "", szDir, name, szID);

	return CString(szResult);
}
