#include "stdafx.h"

#include "mfcmark.h"
#include "mainfrm.h"
#include "doc.h"
#include "view.h"
#include "cmarkup.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// CSampView printing

BOOL CSampView::OnPreparePrinting(CPrintInfo* pInfo)
{
	CSampMainFrame*		pMainWnd = (CSampMainFrame*)AfxGetMainWnd();

	CMarkupControl* pMrkCtl = NULL;
	CSampDoc* pDoc = (CSampDoc*)GetDocument();		 
	if (pDoc) {
		pMrkCtl = pDoc->GetMarkupControl();
	}

	// 1997.06.05: in preview mode, the markup toolbar should be hidden
	// for the first time and shown for the second time if print is chosen.
	// The first time this function is called is to display the print
	// preview.  The second time this function is called is to print.  --- FZL
	if (pMainWnd->GetPreviewMode()) {
		if (pMainWnd->m_fFirstTime==TRUE) {
			pMainWnd->ShowMarkupToolbar(FALSE);
			pMainWnd->m_fFirstTime = FALSE;
		} else {
			if(pMrkCtl) {
				pMainWnd->ShowMarkupToolbar(TRUE);
			} else {
				pMainWnd->ShowMarkupToolbar(FALSE);
			}
		}
	} else { 	// In case: cancel the printing in print dialog
		if(pMrkCtl) {
			pMainWnd->ShowMarkupToolbar(TRUE);
		} else {
			pMainWnd->ShowMarkupToolbar(FALSE);
		}
	}
	
	// default preparation	
	return DoPreparePrinting(pInfo);
}

void CSampView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{	
	// Initialization before printing
	m_nOptions = 0;
	m_pOptions = 0;

	// Disable drag and drop
	CVCETControl	*pCtl = GetDocument()->GetControl();
	DWORD	dwMode	=  pCtl->GetMode(FALSE);
	dwMode &= ~PAN_CTLMODEDRAGDROP;
	pCtl->SetMode(FALSE, dwMode);
	CFrameWnd* pMainWnd = (CFrameWnd*) AfxGetMainWnd();
	if (pMainWnd) {
		pMainWnd->DragAcceptFiles(FALSE);
	} 

	GetPrintPreview(pInfo, &pInfo->m_pPD->m_pd);
}


void CSampView::OnPrint(CDC* pDC, CPrintInfo* pInfo)
{
	ASSERT_VALID(this);
	ASSERT_VALID(pDC);

	ASSERT(AfxIsValidAddress(pInfo, sizeof *pInfo));

	if (m_nOptions == 0) {
		// Something bad happened
		return;
	}

	UINT	pageNum = 0;
	UINT	tileNum = 0;
	GetCurrentPageAndTile(pInfo, &pageNum, &tileNum);

	// Render the headers and footers if any
	RenderHeadersFooters(pDC, pInfo, m_pOptions[pageNum-1]);

	// Render the base file, if any.
	RenderBaseFile(pDC, pInfo, m_pOptions[pageNum-1], pageNum, tileNum,	0);

	// Render the  markup file, if any
	RenderMarkupFile(pDC, pInfo, m_pOptions[pageNum-1], pageNum, tileNum);
}

void CSampView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{           
	CSampMainFrame*	 pMainWnd = (CSampMainFrame*)AfxGetMainWnd();

	// Cleanup after printing
	if (m_pOptions != 0) {
		delete [] m_pOptions;
	}
	m_nOptions = 0;
	m_pOptions = 0;

	// Re-enable drag and drop
	CVCETControl	*pCtl = GetDocument()->GetControl();
	DWORD	dwMode	=  pCtl->GetMode(FALSE);
	dwMode |= PAN_CTLMODEDRAGDROP;
	pCtl->SetMode(FALSE, dwMode);
	if (pMainWnd) {
		pMainWnd->DragAcceptFiles(FALSE);
	}                            
	if (GetDocument()->GetMarkupControl()) {
		((CSampMainFrame*)pMainWnd)->ShowMarkupToolbar(TRUE);
	} 
}


// Helper function: Calls VCET's print preview function.
BOOL	CSampView::GetPrintPreview(CPrintInfo *pInfo, PRINTDLG *pd)
{
	ASSERT_VALID(this);
	ASSERT(pd != 0);

	CVCETControl	*pCtl = GetDocument()->GetControl();
	ASSERT(pCtl != 0);

	PAN_CtlFileInfo	fi;
	GetDocument()->GetControlFileInfo(&fi);

	// Get print preview for each page
	int	numPages = pCtl->GetNumPages();

	UINT	fromPage, toPage;
	fromPage = 1;
	toPage = abs(numPages);
	if (toPage < fromPage) {
		toPage = fromPage;
	}
	ASSERT(fromPage > 0 && toPage >= fromPage);

	m_nOptions = (toPage - fromPage + 1);
	m_pOptions = new PAN_CtlPrintOptions[m_nOptions];

	for (UINT iPage = fromPage; iPage <= toPage; iPage++) {
		WORD	curPage = pCtl->GetPage();

		if (iPage != curPage) {
			// Set current page
			pCtl->SetPage(iPage);

			// Make sure that any posted messages are consumed
			// (PM_CTLSETPAGE can post another PM_CTLSETPAGE message)
			MSG msg;
			while (PeekMessage(&msg, pCtl->GetSafeHwnd(), 0, 0, PM_REMOVE)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}

			// Force control to load page
			pCtl->Paint();

			curPage = pCtl->GetPage();
   			if (iPage != curPage) {
   				// We have passed the last page
   				break;
   			}
		}

		// Get print preview for current page
		PAN_CtlPrintOptions& options = m_pOptions[iPage - fromPage];
		_fmemset(&options, 0, sizeof options);
		// PRINTDLG structure
		options.printDlg = pd;

		// Units
		options.units = CTLUNIT_INCH;

		// Scaling: In this example, we will scale Graphics and Document
		// formats to fit the page and print other formats at 1:1
		switch (fi.type) {
		  case PAN_RasterFile:
		  case PAN_VectorFile:
		  case PAN_DocumentFile:
			options.nImageUnits = 0.0;		// Fit
			options.nPaperUnits = 0.0;
			break;
		  default:
			options.nImageUnits = 1440.0;	// Twips
			options.nPaperUnits = 1.0;
			break;
		}

		// Source rectangle: Select the pagesize
		pCtl->GetPageSize(&options.source, iPage);

		// Mode
		options.mode = 0;
		// File type-specific options:
		options.mode |= 0;

		// Margins
		options.margins.units = CTLUNIT_INCH;
		options.margins.left   = 0.0;
		options.margins.top    = 0.0;
		options.margins.right  = 0.0;
		options.margins.bottom = 0.0;

		// Headers/footers
		memset(&options.headers.font, 0, sizeof(LOGFONT));
		lstrcpy(options.headers.font.lfFaceName, "Arial");
		options.headers.font.lfHeight = -10;

		options.headers.topLeftText  = "Left Top";
		options.headers.topCtrText   = "Center Top";
		options.headers.topRightText = "Right Top";
		options.headers.botLeftText  = "Left Bottom";
		options.headers.botCtrText   = "Center Bottom";
		options.headers.botRightText = "Right Bottom";

		// Output filename	
		options.outputFileName = "";

		// Get print preview
		pCtl->PrintPreview(&options);

		// Sanity checks
		if (options.printPreview.nPages <= 0 ||
			options.printPreview.nHorzPages <= 0 ||
			options.printPreview.nVertPages <= 0) {
		
			goto DeleteOptionsAndFail;
		}
	}

	// Clean up and succeed
	pInfo->SetMinPage(1);
	int	nPages;
	int	i;
	nPages = 0;
	i = 0;
	for (i=0; i<m_nOptions; i++) {
		nPages += m_pOptions[i].printPreview.nPages;
	}
	pInfo->SetMaxPage(nPages);

	return TRUE;

	// Clean up and fail
DeleteOptionsAndFail:
	delete [] m_pOptions;
	m_pOptions = NULL;
	m_nOptions = 0;

	return FALSE;
}

/*
** Get the current page and tile number:
**	pInfo		IN		Print information
**  pageNum		OUT		current page (1 indexed)
**  tileNum		OUT		current tile (1 indexed)
*/
void	CSampView::GetCurrentPageAndTile(CPrintInfo *pInfo, UINT *pageNum, UINT *tileNum)
{
	*pageNum = 1;
	*tileNum = 1;
	int		curPage = pInfo->m_nCurPage;	// 1-indexed;
	for (int nPreview = 0; nPreview < m_nOptions; nPreview++) {
		if (curPage <= m_pOptions[nPreview].printPreview.nPages) {
			*pageNum = nPreview + 1;
			*tileNum = curPage;
			break;
		}
		curPage -= m_pOptions[nPreview].printPreview.nPages;
	}
}

void CSampView::GetImageAndClipRects(const PAN_CtlPrintOptions& options,
	UINT tileNum, PAN_CtlRange& imageRect, CRect& clipRect)
//	Returns the image and the clip rectangles corresponding
//	to the given tile number
//
//	options 	IN		Print options from which to get rectangles
//	tileNum 	IN		Tile number
//	imageRect	OUT 	Image rectangle in world units
//	clipRect	OUT 	Clip rectangle in device units
{
	ASSERT_VALID(this);
	ASSERT(tileNum > 0 && tileNum <= options.printPreview.nPages);

	const PAN_CtlPrintPreview* pPreview = &options.printPreview;

	// Get image rectangle
	ASSERT(pPreview->imageRects != NULL);
	PAN_CtlRange FAR* pImageRects =
		(PAN_CtlRange FAR*) GlobalLock(pPreview->imageRects);
	ASSERT(AfxIsValidAddress(
		pImageRects, pPreview->nPages * sizeof (PAN_CtlRange)));
	imageRect = pImageRects[tileNum - 1];
	GlobalUnlock(pPreview->imageRects);

	// Get clip rectangle
	ASSERT(pPreview->clipRects != NULL);
	LPRECT pClipRects = (LPRECT) GlobalLock(pPreview->clipRects);
	ASSERT(AfxIsValidAddress(pClipRects, pPreview->nPages * sizeof (RECT)));
	clipRect = pClipRects[tileNum - 1];
	GlobalUnlock(pPreview->clipRects);
}

void CSampView::RenderHeadersFooters(CDC* pDC, CPrintInfo* pInfo,
	const PAN_CtlPrintOptions& options)
//	Renders the headers and/or footers onto the given printer DC
//
//	pDC 		IN		Printer device context
//	pInfo		IN		Print information
//	options 	IN		Print options from which to get header and/or
//						footer information
{
	ASSERT_VALID(this);
	ASSERT_VALID(pDC);
	ASSERT(AfxIsValidAddress(pInfo, sizeof *pInfo));

	const COLORREF	textColor = RGB(0, 0, 255);

	// Convert header and footer heights to printer logical units
	CPoint hdrHeight(0, options.printPreview.headerHeight);
	pDC->DPtoLP(&hdrHeight);

	CPoint ftrHeight(0, options.printPreview.footerHeight);
	pDC->DPtoLP(&ftrHeight);

	// Use copy of printable area
	CRect rectDraw = pInfo->m_rectDraw;

	// Adjust printable area for headers
	CRect rectHeader = pInfo->m_rectDraw;
	if ( options.headers.topLeftText != NULL &&
		*options.headers.topLeftText != 0
		||
		 options.headers.topCtrText != NULL &&
		*options.headers.topCtrText != 0
		||
		 options.headers.topRightText != NULL &&
		*options.headers.topRightText != 0 )
	{
		rectDraw.top += hdrHeight.y;
		rectHeader.bottom = rectDraw.top;
	}

	// Adjust printable area for footers
	CRect rectFooter = pInfo->m_rectDraw;
	if ( options.headers.botLeftText != NULL &&
		*options.headers.botLeftText != 0
		||
		 options.headers.botCtrText != NULL &&
		*options.headers.botCtrText != 0
		||
		 options.headers.botRightText != NULL &&
		*options.headers.botRightText != 0 )
	{
		rectDraw.bottom -= ftrHeight.y;
		rectFooter.top = rectDraw.bottom;
	}

	// Verify that at least one header and/or footer is non-zero
	if (rectDraw == pInfo->m_rectDraw) {
		return;
	}

	// Save new printable area
	pInfo->m_rectDraw = rectDraw;

	// Save current DC settings
	if (! pDC->SaveDC()) {
		AfxThrowResourceException();
	}

	// Create font and select it into DC
	LOGFONT logFont = options.headers.font;
	logFont.lfHeight =
		::MulDiv(logFont.lfHeight, pDC->GetDeviceCaps(LOGPIXELSY), 96);

	CFont font;
	if (! font.CreateFontIndirect(&logFont)) {
		pDC->RestoreDC(-1);
		AfxThrowResourceException();
	}
	CFont* pOldFont = pDC->SelectObject(&font);

	// Set text color
	pDC->SetTextColor(textColor);

	// Format and output headers
	CString str;

	if (options.headers.topLeftText != 0 && *options.headers.topLeftText != 0) {
		str = options.headers.topLeftText;
		pDC->DrawText(str, -1, &rectHeader, DT_LEFT /*| DT_BOTTOM | DT_SINGLELINE*/);
	}

	if (options.headers.topCtrText != 0 && *options.headers.topCtrText != 0) {
		str = options.headers.topCtrText;
		pDC->DrawText(str, -1, &rectHeader, DT_CENTER /*| DT_BOTTOM | DT_SINGLELINE*/);
	}

	if (options.headers.topRightText != 0 && *options.headers.topRightText != 0) {
		str = options.headers.topRightText;
		pDC->DrawText(str, -1, &rectHeader, DT_RIGHT /*| DT_BOTTOM | DT_SINGLELINE*/);
	}

	// Format and output footers
	if (options.headers.botLeftText != 0 && *options.headers.botLeftText != 0) {
		str = options.headers.botLeftText;
		pDC->DrawText(str, -1, &rectFooter, DT_LEFT);
	}

	if (options.headers.botCtrText != 0 && *options.headers.botCtrText != 0) {
		str = options.headers.botCtrText;
		pDC->DrawText(str, -1, &rectFooter, DT_CENTER);
	}

	if (options.headers.botRightText != 0 && *options.headers.botRightText != 0) {
		str = options.headers.botRightText;
		pDC->DrawText(str, -1, &rectFooter, DT_RIGHT);
	}

	// Clean up
	pDC->SelectObject(pOldFont);
	pDC->RestoreDC(-1);
}

void CSampView::RenderBaseFile(CDC* pDC, CPrintInfo* pInfo,
	const PAN_CtlPrintOptions& options,
	UINT pageNum, UINT tileNum,
	BOOL fForcetoblack /*0*/)
//	Renders the base file onto the given printer DC
//
//	pDC 		IN		Printer device context
//	pInfo		IN		Print information
//	options 	IN		Print options
//	pageNum 	IN		Current document page number (1-based)
//	tileNum 	IN		Current printed page (tile) number (1-based)
//	forcetoBlackIN		Force output to black (default is False)
{
	ASSERT_VALID(this);
	ASSERT_VALID(pDC);
	ASSERT(AfxIsValidAddress(pInfo, sizeof *pInfo));
	ASSERT(tileNum > 0 && tileNum <= options.printPreview.nPages);

	pDC->SaveDC();

	// Get base control
	CSampDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	CVCETControl	*pCtl = pDoc->GetControl();
	ASSERT(pCtl != 0);

	// File information
	PAN_CtlFileInfo	fi;
	pDoc->GetControlFileInfo(&fi);

	// Get image and clip rectangles for current page
	PAN_CtlRange imageRect;
	CRect clipRect;
	GetImageAndClipRects(options, tileNum, imageRect, clipRect);

	// Convert clip rectangle to physical device units
	// Use ::LPtoDP() instead of CDC::LPtoDP() because the latter acts
	// on CDC::m_hAttribDC instead of CDC::m_hDC
	::LPtoDP(pDC->GetSafeHdc(), (LPPOINT) (LPRECT) clipRect, 2);

	// Enable NoRedraw mode in base control
	pCtl->UpdateWindow();
	pCtl->EnableModes(PAN_CTLMODENOREDRAW, TRUE);

	// Go to the current document page
	WORD	curPage = pCtl->GetPage();
	if (pageNum != curPage) {
		pCtl->SetPage(pageNum);
		// Forces the control to load the page
		pCtl->Paint();
	}

	// Render contents of base control
	PAN_CtlRenderOptions roptions;
	_fmemset(&roptions, 0, sizeof roptions);

	// Common options
	roptions.hdc = pDC->GetSafeHdc();
	roptions.mode = PAN_CTLMODEOPAQUE | PAN_CTLMODEEXCESSSCROLL | PAN_CTLMODEANISOTROPIC;
	if (fForcetoblack) {
		roptions.mode |= PAN_CTLMODEMONOCHROME;
	}
	roptions.source = imageRect;
	roptions.devRect = clipRect;

	if (fi.type == PAN_SpreadsheetFile ||
		fi.type == PAN_DatabaseFile ||
		fi.type == PAN_ArchiveFile)
	{
		// PAN_CtlRenderOptions::devRect is relative to (0, 0).
		// Since MFC sets the viewport origin to be the top-left corner
		// of the page rectangle, we need to reset the viewport origin
		// to (0, 0) so that the control renders at the right place.
		// The VCET controls set the viewport origin correctly except
		// for the ss/db/ar controls.
		SetViewportOrgEx(pDC->GetSafeHdc(), 0, 0, NULL);
		roptions.mode |= 0;
	}

	pCtl->RenderOntoDC(&roptions);

	// Restore NoRedraw mode in base control
	pCtl->EnableModes(PAN_CTLMODENOREDRAW, FALSE);

	pDC->RestoreDC(-1);
}

