/*===========================================================================*/
/*                                                                           */
/* SMPLVIEW: A Sample Application for AutoVue 19.3                           */
/*                                                                           */
/* Copyright Oracle, 1995-2008, all rights reserved.                         */
/*                                                                           */
/*===========================================================================*/

/****************************************************************************
	FILE:	Generic.c

	PURPOSE: Generic template for Windows applications

	FUNCTIONS:

	VCET_DisplayFile(HWND hWnd, LPCSTR szFileName)
	VCET_PrintFile(LPSTR FileName)
	VCET_SaveAsBMP(HWND hCtl, LPCSTR szFile)
	VCET_ShutDown(LPSTR FileName)

	COMMENTS:	AutoVue Sample Application
****************************************************************************/

#include <windows.h>			/* required for all Windows applications */
#include <commdlg.h>
#include <memory.h>
#include <malloc.h>
#include <stdarg.h>

#include "target.h"

#if TARGET == UNIX
#include <dlfcn.h>
#endif

#include "pctl.h"

#include "generic.h"			/* specific to this program		*/

#define PM_INITFILE	WM_USER+3682
#define PM_SHOWFILE	WM_USER+3683
#define PM_DUMPFILE	WM_USER+3684
/*
**	Global variables
*/
HANDLE	hInst;				/* Current instance					*/
HWND	hMainWnd;			/* Main Window Handle				*/
HWND	hCtl;				/* CSI VCET control.				*/
WNDPROC	lpfnNotifyProc;		/* CSI VCET Notification procedure	*/

/*
**  Defines
*/
// Define to 0 if you want to handle mouse-zooming yourself.
// Define to 1 if you want to let VCET handle mouse-zooming.
#  define	LET_VCET_HANDLE_MOUSE_ZOOM		1


/****************************************************************************
	FUNCTION: _MyInfo(const char FAR *fmt, ...)
	PURPOSE:  Displays the given formatted text (a la printf()) in a message box.
	COMMENTS:
****************************************************************************/
Public	 void 	_MyInfo(const char FAR *fmt, ...)
{
	va_list	arg_ptr;
	char	buf[240];

	va_start(arg_ptr, fmt);
	wvsprintf((LPSTR) buf, (LPCSTR) fmt, arg_ptr);
	va_end(arg_ptr);

	MessageBox((HWND) 0, (LPCSTR) buf, (LPCSTR) "Info", MB_ICONINFORMATION);
}


/****************************************************************************
	FUNCTION: About(HWND, unsigned, WORD, LONG)
	PURPOSE:  Processes messages for "About" dialog box
	COMMENTS:
****************************************************************************/
BOOL __export CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_INITDIALOG:		/* message: initialize dialog box */
			return (TRUE);

		case WM_COMMAND:			/* message: received a command */
			if (wParam == IDOK		/* "OK" box selected?		*/
				|| wParam == IDCANCEL) /* System menu close command?  */
			{
				EndDialog(hDlg, TRUE); /* Exits the dialog box		*/
				return (TRUE);
			}
			break;
	}
	return (FALSE);			/* Didn't process a message	*/
}

/****************************************************************************
	FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)
	PURPOSE:  Processes messages
	MESSAGES:
		WM_COMMAND	- application menu (About dialog box)
		WM_DESTROY	- destroy window
	COMMENTS:
****************************************************************************/
long	CALLBACK __export MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	RECT	rcClient;
	DWORD	dwMode;

	switch (message) {
	case WM_CREATE:
		/*
		**	WM_CREATE: Create a control that fills the client area
		**	of the main window
		*/
		lpfnNotifyProc = (WNDPROC) MakeProcInstance((FARPROC) NotifyWndProc, hInst);
		GetClientRect(hWnd, &rcClient);
		hCtl = PAN_CreateControl(hWnd, 1, &rcClient, NULL, lpfnNotifyProc);
		/*
		**  Enable a sensible mode
		*/
		dwMode = PAN_CTLMODEOPAQUE | PAN_CTLMODEEXCESSSCROLL | PAN_CTLMODEDRAGDROP;
		SendMessage(hCtl, PM_CTLSETMODE, 0, (LPARAM)dwMode);
		ShowWindow(hCtl, SW_SHOW);
		break;
#if	!LET_VCET_HANDLE_MOUSE_ZOOM
	case WM_LBUTTONDOWN:
		{{{
		// Initialise the zoom box
		RECT	rcZoom = {LOWORD(lParam), HIWORD(lParam), LOWORD(lParam), HIWORD(lParam)};
		if (MyTrackMouse(hWnd, &rcZoom)) {
			// Zoom in on the selected region.
			PAN_CtlRange	rg;
			WORD			wPage;
#if		CSIVER >= 10200
#else	/* CSIVER */
#			define	MAKE_WORD(lo, hi)	((WORD) (((WORD) hi) << 8 | (WORD) (lo)))
			WORD	mode = MAKE_WORD(PAN_CTLCLIENTCOORD, PAN_CTLWORLDCOORD);

			SendMessage(hCtl, PM_CTLGETPAGE, 0, (LPARAM)&wPage);
			memset(&rg, 0, sizeof(rg));
			rg.type = PAN_ClientCoord;
			rg.page = wPage;
			rg.u.client.x	= rcZoom.left;
			rg.u.client.y	= rcZoom.top;
			rg.u.client.width  = rcZoom.right - rcZoom.left;
			rg.u.client.height = rcZoom.bottom - rcZoom.top;
			SendMessage(hCtl, PM_CTLCONVERT_RANGE, (WPARAM) mode, (LPARAM) &rg);
			SendMessage(hCtl, PM_CTLSETVIEWEXTENTS, 0, (LPARAM) &rg);
#endif	/* CSIVER */
		}
		}}}
		break;
#endif	/* LET_VCET_HANDLE_MOUSE_ZOOM */

	case WM_COMMAND:	/* message: command from application menu */
		if (wParam == IDM_ABOUT) {

			DialogBox(hInst,		/* current instance			*/
					"AboutBox",		/* resource to use			*/
					hWnd,			/* parent handle			*/
					About);			/* About() instance address */

			break;
		} else if (wParam == IDM_OPEN) {
			// Select a file to view
			VCET_DisplayFile(hWnd, NULL);
			break;
		} else if (wParam == IDM_PRINT) {
			// Print the current file
			PAN_CtlFileInfo fi;
			SendMessage(hCtl, PM_CTLGETFILE, 0, (LPARAM)(PAN_CtlFileInfo FAR *)&fi);
			// Print the viewed file
			VCET_PrintFile(fi.name, 1);							// Print 1 copy
			// To print other non-viewed files:
//			VCET_PrintFile("file2.doc", 1);	// Print 1 copy
			break;
		} else if (wParam == IDM_SAVE_BMP) {
			OPENFILENAME	ofn;
			char			szFile[256];
			*szFile = '\0';
			/* Set all structure members to zero. */
			_fmemset(&ofn, 0, sizeof(OPENFILENAME));
			ofn.lStructSize = sizeof(OPENFILENAME);
			ofn.hwndOwner = hWnd;
			ofn.lpstrFile = szFile;
			ofn.nMaxFile = sizeof(szFile);
			ofn.lpstrFilter = "BMP Files\0*.bmp\0";
			ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
			if (GetSaveFileName(&ofn)) {
				VCET_SaveAsBMP(hCtl, szFile);
			}
			break;
		} else if (wParam == IDM_EXIT) {
			// Close application
			SendMessage(hWnd, WM_CLOSE, 0, 0L);
			break;
		} else {
			/* Lets Windows process it	*/
			return (DefWindowProc(hWnd, message, wParam, lParam));
		}

	case WM_SIZE:		/* message: window being destroyed */
		/*
		** WM_SIZE: Reseize the VCET Control.
		*/
		MoveWindow(hCtl, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);

		/* It's very important to pass this message to the default procedure! */
		return DefWindowProc(hWnd, message, wParam, lParam);
		break;

	case WM_DESTROY:		/* message: window being destroyed */
		/*
		** WM_DESTROY: Handle the destruction of the window by
		**	causing the destruction of the control.
		*/
		VCET_ShutDown(hCtl);
		if (lpfnNotifyProc) {
			FreeProcInstance((FARPROC) lpfnNotifyProc);
		}
		PAN_FreeControls();		// Unload the VCET DLLs
		PostQuitMessage(0);
		break;

	case PM_INITFILE:
		VCET_DisplayFile(hWnd, (LPCSTR) lParam);
		break;

	case PM_SHOWFILE:
		VCET_DisplayFile(hWnd, (LPCSTR) lParam);
		break;

	case PM_DUMPFILE:
		VCET_SaveAsBMP(hCtl, (LPCSTR) lParam);
		break;

	default:				/* Passes it on if unproccessed	*/
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}

	return (0);
}



/****************************************************************************
	FUNCTION: InitApplication(HANDLE)
	PURPOSE: Initializes window data and registers window class
	COMMENTS:	Loads the VCET Controls.
****************************************************************************/
BOOL	InitApplication(HANDLE hInstance)
{
	WNDCLASS  wc;
	BOOL	status;
	int	pstatus;

	/* Fill in window class structure with parameters that describe the	*/
	/* main window.														*/

	wc.style = 0;						/* Class style(s).					*/
	wc.lpfnWndProc = MainWndProc;		/* Function to retrieve messages for  */
										/* windows of this class.			*/
	wc.cbClsExtra = 0;					/* No per-class extra data.		*/
	wc.cbWndExtra = 0;					/* No per-window extra data.		*/
	wc.hInstance = hInstance;			/* Application that owns the class.   */
	wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICONAPP));
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName =  "GenericMenu";	/* Name of menu resource in .RC file. */
	wc.lpszClassName = "GenericWClass"; /* Name used in call to CreateWindow. */

	/* Register the window class and return success/failure code. */

	status = RegisterClass(&wc);

       	if (status) {
		pstatus = PAN_LoadControls(INIFILE, 0);
		if (!pstatus) {
			UnregisterClass("GenericWClass", hInstance);
			status = 0;
		}
	}

	return (status);
}


/****************************************************************************
	FUNCTION:  InitInstance(HANDLE, int)
	PURPOSE:  Saves instance handle and creates main window
	COMMENTS:
****************************************************************************/
BOOL	InitInstance(HANDLE hInstance, int nCmdShow)
{
	/* Save the instance handle in static variable, which will be used in  */
	/* many subsequence calls from this application to Windows.		*/

	hInst = hInstance;

	/* Create a main window for this application instance.	*/

	hMainWnd = CreateWindow(
		"GenericWClass",				/* See RegisterClass() call.		*/
		"AutoVue Sample Application",/* Text for window title bar.		*/
		WS_OVERLAPPEDWINDOW,			/* Window style.					*/
		CW_USEDEFAULT,					/* Default horizontal position.		*/
		CW_USEDEFAULT,					/* Default vertical position.		*/
		CW_USEDEFAULT,					/* Default width.					*/
		CW_USEDEFAULT,					/* Default height.					*/
		NULL,							/* Overlapped windows have no parent*/
		NULL,							/* Use the window class menu.		*/
		hInstance,						/* This instance owns this window.	*/
		NULL							/* Pointer not needed.				*/
	);

	/* If window could not be created, return "failure" */
	if (! hMainWnd) {
		return (FALSE);
	}

	/* Make the window visible; update its client area; and return "success" */

	ShowWindow(hMainWnd, nCmdShow);	/* Show the window				*/
	UpdateWindow(hMainWnd);			/* Sends WM_PAINT message		*/

	ShowWindow(hCtl, nCmdShow);		/* Update the VCET Control		*/
	UpdateWindow(hCtl);

	return (TRUE);				/* Returns the value from PostQuitMessage */
}

/****************************************************************************
	FUNCTION: WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
	PURPOSE:  Windows Entry Point
	COMMENTS:
****************************************************************************/
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG msg;					/* message			*/

#if TARGET == UNIX
	fprintf(stderr, "Trying to load libthread.so... ");
	if (dlopen("libthread.so",RTLD_GLOBAL | RTLD_NOW)) {
		fprintf(stderr, "Success!\n");
	} else {
		fprintf(stderr, "Failure!\n");
	}
	if (dlopen("libc.so",RTLD_GLOBAL | RTLD_NOW)) {
		fprintf(stderr, "Success!\n");
	} else {
		fprintf(stderr, "Failure!\n");
	}
#endif

	if (! hPrevInstance) {		/* Other instances of app running? */
		if (!InitApplication(hInstance)) { /* Initialize shared things */
			return (FALSE);	/* Exits if unable to initialize	*/
		}
	}
#if TARGET == UNIX
	char *file = getenv("CSI_DISPLAY_FILE");
#endif

	/* Perform initializations that apply to a specific instance */
	if (! InitInstance(hInstance, nCmdShow)) {
		return (FALSE);
	}

	if (lstrlen(lpCmdLine) != 0) {
		PostMessage(hMainWnd, PM_INITFILE, -1, (LPARAM) lpCmdLine);
	}

#if TARGET == UNIX
	/* display a file */
	if (file) {
		printf ("Trying to open %s\n",file);
		PostMessage(hMainWnd, PM_SHOWFILE, -1, file);
		PostMessage(hMainWnd, WM_RBUTTONDOWN, 0, 0);
		PostMessage(hMainWnd, WM_RBUTTONUP,	0, 0);
		PostMessage(hMainWnd, PM_DUMPFILE, -1, "/tmp/smplview_out.bmp");
		PostMessage(hMainWnd, WM_DESTROY, -1, NULL);
	}
#endif

	/* Acquire and dispatch messages until a WM_QUIT message is received. */
	while (GetMessage(&msg,	/* message structure			*/
		0,					/* handle of window receiving the message */
		0,					/* lowest message to examine		*/
		0))					/* highest message to examine		*/
	{
		TranslateMessage(&msg);    /* Translates virtual key codes		*/
		DispatchMessage(&msg);	/* Dispatches message to window		*/
	}
	return (msg.wParam);	/* Returns the value from PostQuitMessage */
}


/****************************************************************************
	FUNCTION: VCET_ShutDown(HWND hWnd)
	PURPOSE:  Shuts down and destroys a VCET controls
	COMMENTS:
****************************************************************************/
Public	void	VCET_ShutDown(HWND hWnd)
{
	SendMessage(hWnd, PM_CTLDESTROY, 0, 0);
	DestroyWindow(hWnd);
}

/****************************************************************************
	FUNCTION: VCET_DisplayFile(HWND hWnd, LPCSTR szFileName)
	PURPOSE:  Display a file in the VCET control
	COMMENTS:
****************************************************************************/
Public	void	VCET_DisplayFile(HWND hWnd, LPCSTR szFileName)
{
	char			szFile[256];

	DWORD			dwCaps;
	PAN_CtlFileInfo fileInfo;

	if (szFileName) {
		lstrcpy(szFile, szFileName);
	} else {
		OPENFILENAME	ofn;
		szFile[0] = '\0';
		/* Set all structure members to zero. */
		_fmemset(&ofn, 0, sizeof(OPENFILENAME));
		ofn.lStructSize = sizeof(OPENFILENAME);
		ofn.hwndOwner = hWnd;
		ofn.lpstrFile = szFile;
		ofn.nMaxFile = sizeof(szFile);
		ofn.lpstrFilter = "All Files\0*.*\0";
		ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
		if (! GetOpenFileName(&ofn)) {
			// User cancelled
			return;
		}
	}

	/*
	**  Clean up previous stuff...
	*/

	/*
	**	Set the file.
	*/
	SendMessage(hCtl, PM_CTLSETFILE, (WPARAM) -1, (LPARAM) szFile);

	/*
	**	Enable all caps on the control
	*/
	SendMessage(hCtl, PM_CTLGETCAPS, (WPARAM)1, (LPARAM)(LPDWORD)&dwCaps);
	SendMessage(hCtl, PM_CTLSETCAPS, (WPARAM)0, (LPARAM) dwCaps);

	/*
	**  Set the Left mouse action
	*/
	memset(&fileInfo, 0, sizeof(fileInfo));
	SendMessage(hCtl, PM_CTLGETFILE, (WPARAM)0, (LPARAM)(PAN_CtlFileInfo FAR *)&fileInfo);
	if (fileInfo.type == PAN_RasterFile || fileInfo.type == PAN_VectorFile) {
#if	LET_VCET_HANDLE_MOUSE_ZOOM
		SendMessage(hCtl, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBZOOM, (LPARAM)0);
		SendMessage(hCtl, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBZOOM, (LPARAM)1);
#else
		SendMessage(hCtl, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBNONE, (LPARAM)0);
		SendMessage(hCtl, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBNONE, (LPARAM)1);
#endif
	} else {
		SendMessage(hCtl, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBSELECT, (LPARAM)0);
	}
}

/****************************************************************************
	FUNCTION: AbortProc(HDC hdc, int error)
	PURPOSE:  Abort procedure for printer.
	COMMENTS:
****************************************************************************/
BOOL	CALLBACK	AbortProc(HDC hdc, int error)
{
	if (error != 0 && error != SP_OUTOFDISK) {
		// Unrecoverable error.
		return FALSE;
	} else {
		MSG	msg;
		while (PeekMessage(&msg, 0, 0, 0, PM_REMOVE)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}
	return TRUE;
}

/****************************************************************************
	FUNCTION:	GetWidthInBytes(LONG width, int depth)
	PURPOSE:	Calculates the width of a raster line in bytes, padded
				to a DWORD boundary.
	COMMENTS:	Used in the Save-as-BMP function.
****************************************************************************/
static	LONG	GetWidthInBytes(LONG width, int depth)
{
	return (((width * depth + 31) / 32 * 4));
}


/****************************************************************************
	FUNCTION:	VCET_SaveAsBMP(HWND hCtl, LPCSTR szFile)
	PURPOSE:	Save the file in the control as a BMP file
	COMMENTS:
****************************************************************************/
Public	void	VCET_SaveAsBMP(HWND hCtl, LPCSTR szFile)
{
	int					outWidth  = 256;
	int					outHeight = 256;
	int					inDepth;
	int					inColors;

	PAN_CtlRenderOptions	renderOptions;
	PAN_CtlRange			rg;

	HDC				hDCCtl = GetDC(hCtl);
	HDC				hDC = CreateCompatibleDC(hDCCtl);
	HBITMAP			hBitmap = CreateCompatibleBitmap(hDCCtl, outWidth, outHeight);
	HGDIOBJ			hOldBitmap = SelectObject(hDC, hBitmap);
	LPLOGPALETTE	pPal = 0;
	HPALETTE		hPal;
#	define			HDIB	HANDLE
	HDIB			hDIB;

	typedef	int		(* CPROC)();
#if	TARGET == WIN16
	typedef HDIB (_far _pascal *BITMAPTODIBPROC)(HBITMAP, HPALETTE);
	typedef WORD (_far _pascal *SAVEDIBPROC)(HDIB, LPCSTR);
#else
	typedef	HDIB (__stdcall *BITMAPTODIBPROC)(HBITMAP, HPALETTE);
	typedef WORD (__stdcall *SAVEDIBPROC)(HDIB, LPCSTR);
#endif
	BITMAPTODIBPROC	fnBitmapToDIB;
	SAVEDIBPROC		fnSaveDIB;
	HMODULE			hDibapi;

	ReleaseDC(hCtl, hDCCtl);
	PatBlt(hDC, 0, 0, outWidth, outHeight, WHITENESS);

	inDepth = (int) SendMessage(hCtl, PM_CTLGETPALETTE, 0, 0);
	if (inDepth > 0 && inDepth <= 8) {
		inColors = 1 << inDepth;
		pPal = _fmalloc(sizeof(LOGPALETTE) + inColors*sizeof(PALETTEENTRY));
		// Initialize palette.
		pPal->palVersion	= 0x300;
		pPal->palNumEntries = inColors;
		SendMessage(hCtl, PM_CTLGETPALETTE, 0, (LPARAM) pPal);
	} else {
		// Set up a default colormap
		inColors = 2;
		pPal = _fmalloc(sizeof(LOGPALETTE) + inColors*sizeof(PALETTEENTRY));
		// Initialize palette.
		pPal->palVersion	= 0x300;
		pPal->palNumEntries = inColors;
	}

	hPal = CreatePalette(pPal);

	// Get curren viewextents.
	SendMessage(hCtl, PM_CTLGETVIEWEXTENTS, 0, (LPARAM)(PAN_CtlRange FAR*) &rg);

	// Initialize render options.
	_fmemset(&renderOptions, 0, sizeof renderOptions);
	renderOptions.hdc			 = hDC;
	renderOptions.source		 = rg;
	renderOptions.mode			 = PAN_CTLMODEOPAQUE;
	renderOptions.devRect.left	 = 0;
	renderOptions.devRect.top	 = 0;
	renderOptions.devRect.right  = outWidth-1;
	renderOptions.devRect.bottom = outHeight-1;

	// Render onto the DC.
	SendMessage(hCtl, PM_CTLRENDERONTODC, 0, (LPARAM) (PAN_CtlRenderOptions FAR *)&renderOptions);

	hDibapi = LoadLibrary("dibapi.dll");
	if (hDibapi) {
		fnBitmapToDIB = (BITMAPTODIBPROC) GetProcAddress(hDibapi, "BitmapToDIB");
		fnSaveDIB = (SAVEDIBPROC) GetProcAddress(hDibapi, "SaveDIB");

		// Convert the DDB to a DIB.
		hDIB = (HDIB)fnBitmapToDIB(hBitmap, hPal);

		// Save the DIB.
		fnSaveDIB(hDIB, (LPSTR)szFile);

		FreeLibrary(hDibapi);
	}

	DeleteObject(hPal);
	_ffree(pPal);
	SelectObject(hDC, hOldBitmap);
	DeleteDC(hDC);
}

/****************************************************************************
	FUNCTION:	VCET_PrintFile(LPSTR FileName, int nCopies)
	PURPOSE:	Print a file, using the VCET controls.
	COMMENTS:
****************************************************************************/
Public	void	VCET_PrintFile(LPSTR FileName, int nCopies)
{
	PRINTDLG	pd;
	RECT		rec = {2000, 2000, 200, 200};	// Create the control offscreen
	int			err;
	char		buf[100];
	HWND		handle = 0;
	PAN_CtlFileInfo fileInfo;
	PAN_CtlPrintOptions PrintOptions;
	char		c_footertxt[256];
	ABORTPROC	lpAbortProc;		// Printer abort procedure
	DWORD		status;
	int			i;

	/*
	**  Get system printer information
	*/
	memset(&pd, 0, sizeof(PRINTDLG));
	pd.lStructSize = sizeof(PRINTDLG);
	// OR with also PD_RETURNDEFAULT if you want to print to the default printer.
	pd.Flags = PD_RETURNDC;
	pd.hwndOwner = hMainWnd;
	if (PrintDlg(&pd) == 0) {
		MessageBox(NULL, "Error...unable to initialize printer", "DEBUG", MB_APPLMODAL);
		return;
	}

	pd.Flags = PD_RETURNDC | PD_RETURNIC;
	pd.nCopies	 = 1;
	pd.nFromPage = 1;
	pd.nToPage	= 0x7FFF;
	pd.nMinPage = 0x0001;
	pd.nMaxPage = 0x7FFF;

	/*
	**  Size control to match the aspect ratio of the page
	*/
	rec.right  = GetDeviceCaps(pd.hDC, HORZRES);
	rec.bottom = GetDeviceCaps(pd.hDC, VERTRES);

	/*
	**  Load the image.
	*/
	if ((handle = PAN_CreateControl(hMainWnd, 3, &rec, NULL, NULL)) == NULL) {
		MessageBox(NULL, "PAN_CreateControl...Error", "DEBUG", MB_APPLMODAL);
		return;
	}

	err = (int) SendMessage(handle, PM_CTLSETFILE, (WPARAM) -1, (LPARAM) (LPSTR) FileName);

	if (err != PAN_CTLERRNONE) {
		wsprintf(buf, "Error on PM_CTLSETFILE Code = %d", err);
		MessageBox(NULL, buf, "DEBUG", MB_APPLMODAL);
		return;
	}

	/*
	**	Force a paint
	*/
	SendMessage(handle, PM_CTLPAINT, 0, 0);

	/*
	**  Zoom to extents
	*/
	memset(&fileInfo, 0, sizeof(fileInfo));
	SendMessage(handle, PM_CTLGETFILE, (WPARAM)0, (LPARAM)(PAN_CtlFileInfo FAR *)&fileInfo);
	if (fileInfo.type == PAN_RasterFile || fileInfo.type == PAN_VectorFile) {
		SendMessage(handle, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBZOOM, (LPARAM)0);
		SendMessage(handle, PM_CTLSETLMBACTION, (WPARAM)PAN_CTLLMBZOOM, (LPARAM)1);
		SendMessage(handle, WM_RBUTTONDOWN, 0, 0);
		SendMessage(handle, WM_RBUTTONUP,	0, 0);
	}

	/*
	**  Set the CSI VCET print options
	*/
	memset(&PrintOptions, 0, sizeof(PAN_CtlPrintOptions));
	// Printer DC structure
	PrintOptions.printDlg = &pd;
	// Print to fit
	if (fileInfo.type == PAN_RasterFile || fileInfo.type == PAN_VectorFile) {
		PrintOptions.units = CTLUNIT_PIXEL;
		// a page fit require these values at 0
		PrintOptions.nImageUnits = 0;
		PrintOptions.nPaperUnits = 0;
	} else {
		PrintOptions.units = CTLUNIT_INCH;
		PrintOptions.nImageUnits = 1440.0;
		PrintOptions.nPaperUnits = 1.0;
	}
	// Margins
	PrintOptions.margins.units = CTLUNIT_INCH;
	PrintOptions.margins.top = 0.75;
	PrintOptions.margins.left = 0.75;
	PrintOptions.margins.bottom = 0.75;
	PrintOptions.margins.right = 0.75;
	// Headers + Footers
	lstrcpy(PrintOptions.headers.font.lfFaceName, "Arial");
	PrintOptions.headers.font.lfHeight   = -10;
	wsprintf(c_footertxt, "Filename: %s", FileName);
	PrintOptions.headers.botCtrText  = c_footertxt;
	// Region.
	PrintOptions.source = fileInfo.dimensions;

	/*
	**  Set AbortProc
	*/
	lpAbortProc = MakeProcInstance(AbortProc, hInst);
	SetAbortProc(PrintOptions.printDlg->hDC, lpAbortProc);

	/*
	**  Important: Disable the main window to avoid reentrancy problems
	*/
	EnableWindow(hMainWnd, FALSE);

	while (nCopies-- > 0) {
		err = (int) SendMessage(handle, PM_CTLPRINT, (WORD) -1,
					(LPARAM)(PAN_CtlPrintOptions FAR *) &PrintOptions);
	}

	if (err != PAN_CTLERRNONE) {
		wsprintf(buf, "Error on PM_CTLPRINT. Code = %d", err);
		MessageBox(NULL, buf, "DEBUG", MB_APPLMODAL);
	}

	// Clean up.
	status = PAN_CTLSTATUSIDLE;
	SendMessage(handle, PM_CTLGETSTATUS, 0, (LPARAM) (LPVOID) &status);
	SendMessage(handle, PM_CTLDESTROY, 0, 0);
	for (i = 0; (i < 10) && status != PAN_CTLSTATUSIDLE; i++) {
		SendMessage(handle, PM_CTLGETSTATUS, 0, (LPARAM) (LPVOID) &status);
	}
	if (status == PAN_CTLSTATUSIDLE) {
		DestroyWindow(handle);
	}
	FreeProcInstance(lpAbortProc);
	EnableWindow(hMainWnd, TRUE);

	DeleteDC(pd.hDC);
}



