#include <windows.h>
#include <commdlg.h>
#include <string.h>
#include <memory.h>
#include <math.h>

#include "pan.h"        // General CSI include file
#include "pctl.h"       // CSI VCET include
#include "mrkupctl.h"   // CSI Markup Control include

#include "ude.h"        // Prototype UDE exports.

#include "resource.h"

// Structure used in Main Memory
typedef struct {    // Color, State, etc.. are managed in MARKUP.DLL
    PAN_Point   x[2];   // Corners of box.
    Int         Locked; // Is the entity locked.
    // Non-retained info
    int         Index;
} RSampleUDE;

/*=====================================================================
== Private Section.
======================================================================*/

// Needed to fetch a Resource
Private HINSTANCE   hLibInst;

// Needed to access resources later on.
Private BOOL    InitApplication(HANDLE hInstance)
{
    hLibInst = hInstance;
    return TRUE;
}

/*********************************************************************/
/**  BEGIN IO FUNCTIONS                                             **/
/**  Note: To be portable across platforms, byte ordering should    **/
/**        be taken into account.                                   */
/*********************************************************************/
Private long    WriteInt(void huge *pData, long offset, Int II)
{
    if (pData != NULL) {
        memcpy((LPSTR)pData+offset, &II, sizeof(II));
    }
    return sizeof(Int);
}

Private long    WriteLong(void huge *pData, long offset, Long LL)
{
    if (pData != NULL) {
        memcpy((LPSTR)pData+offset, &LL, sizeof(LL));
    }
    return sizeof(Long);
}

Private long    WriteDouble(void huge *pData, long offset, Double DD)
{
    if (pData != NULL) {
        memcpy((LPSTR)pData+offset, &DD, sizeof(DD));
    }
    return sizeof(Double);
}

Private long    WritePoint(void huge *pData, long offset, PAN_Point *pPoint)
{
    long nBytes = 0;

    if (pPoint != NULL) {
        nBytes += WriteDouble(pData, offset+nBytes, pPoint->x);
        nBytes += WriteDouble(pData, offset+nBytes, pPoint->y);
        nBytes += WriteDouble(pData, offset+nBytes, pPoint->z);
    }   

    return nBytes;
}

Private long    ReadInt(void huge *pData, long offset, Int *II)
{
    if (pData != NULL) {
        memcpy(II, (LPSTR)pData+offset, sizeof(Int));
        return sizeof(Int);
    }
    return 0L;
}

Private long    ReadLong(void huge *pData, long offset, Long *LL)
{
    if (pData != NULL) {
        memcpy(LL, (LPSTR)pData+offset, sizeof(Long));
        return sizeof(Long);
    }
    return 0L;
}
Private long    ReadDouble(void huge *pData, long offset, Double *DD)
{
    if (pData != NULL) {
        memcpy(DD, (LPSTR)pData+offset, sizeof(Double));
        return sizeof(Double);
    }
    return 0L;
}

Private long    ReadPoint(void huge *pData, long offset, PAN_Point *pPoint)
{
    Double  DD;
    long    nBytes = 0;

    if (pPoint != NULL) {
        nBytes += ReadDouble(pData, offset+nBytes, &DD);
        pPoint->x = DD;
        
        nBytes += ReadDouble(pData, offset+nBytes, &DD);
        pPoint->y = DD;
        
        nBytes += ReadDouble(pData, offset+nBytes, &DD);
        pPoint->z = DD;
    }

    return nBytes;
}
/*********************************************************************/
/**  END IO FUNCTIONS                                               **/
/*********************************************************************/

/*********************************************************************/
/**  BEGIN MOUSE HANDLING FUNCTIONS                                 **/
/*********************************************************************/
Private int
MouseDown(PAN_Point *MousePt, LPMRK_EntitySpec Entity)
{
    HWND        hWndCtl = Entity->Com.hWndCtl; 
    RSampleUDE  *ptr = Entity->Ent.Ude.pData;

    if ( ptr->Index == 0 ) {
        SetCapture(hWndCtl);    // Guarantees a Strict Execution of the State Transition Diagram to create this UDE

        ptr->x[0] = *MousePt;
        ptr->x[1] = *MousePt;

    } else if ( ptr->Index == 1 ) {
        ptr->x[1] = *MousePt;

        ReleaseCapture();

        // Done
        return -1;
    }

    ptr->Index++;
    return 1;   // This means that Everything is Alright
}

Private int
MouseMove(PAN_Point *MousePt, LPMRK_EntitySpec Entity)
{
    MRK_RenderOptions   Render;
    RSampleUDE          *ptr = Entity->Ent.Ude.pData;
    HWND                hWndCtl = Entity->Com.hWndCtl;
    HDC                 hdc = GetDC(hWndCtl);

    memset(&Render, 0, sizeof(MRK_RenderOptions));

    // Let's Draw an Outline
    Render.hdc = hdc;
    Render.mode = MRK_RENDERMODEXOR;    
    SendMessage( hWndCtl, MRK_GETVIEWEXTENTS, 0, (LPARAM) &Render.source );
    GetClientRect( hWndCtl, &Render.devRect );

    Draw( &Render, Entity);

    if ( ptr->Index == 1 ) {
        ptr->x[1] = *MousePt;
    }

    Draw( &Render, Entity);

    // Release DC.
    ReleaseDC(hWndCtl, hdc);

    return 1;
}

Private int
RightMouseUp(PAN_Point *MousePt, LPMRK_EntitySpec Entity)
{
    return 0;   // This Simply means to Destroy the Entity
}
/*********************************************************************/
/**  END MOUSE HANDLING FUNCTIONS                                   **/
/*********************************************************************/

/*=====================================================================
== Public Section.
======================================================================*/

/*********************************************************************/
/**  BEGIN UDE EXPORTS                                              **/
/*********************************************************************/
Public void    PCALLBACK
WhoIAm(LPUDEINFO Info)
{
    strcpy( Info->entName, "Sample UDE" );
    strcpy( Info->entMenuDesc, "Sample UDE" );
    strcpy( Info->entDescription, "Sample UDE Entity." );
    strcpy( Info->entShortDescription, "Sample UDE" );

	Info->BitmapResID	= IDB_SAMPLEUDE;
    Info->hCursor		= LoadCursor(hLibInst, MAKEINTRESOURCE(IDC_SAMPLE));
	Info->dwHints		= UDE_SUPPORTS2D | UDE_SUPPORTSNOFILL;
}

Public  void PCALLBACK
InitEntity(LPMRK_EntitySpec Entity) // Constructor
{
    RSampleUDE *ptr;

    Entity->Ent.Ude.DataSize = sizeof( RSampleUDE );
    Entity->Ent.Ude.pData = MrkAlloc( sizeof( RSampleUDE ) );

    ptr = Entity->Ent.Ude.pData;

    ptr->Index = 0;
    ptr->Locked = 0;

    ptr->x[0].x = 0.;
    ptr->x[0].y = 0.;
    ptr->x[1].x = 0.;
    ptr->x[1].y = 0.;
}

Public  void PCALLBACK
ReleaseEntity(LPMRK_EntitySpec Entity)  // Destructor
{
    if (Entity->Ent.Ude.pData) {
        MrkFree( Entity->Ent.Ude.pData );
        Entity->Ent.Ude.pData = NULL;
    }
}

Public  int PCALLBACK
MouseProc(int NotifMsg, WPARAM wParam, LPARAM lParam, LPMRK_EntitySpec Entity)
{
    switch (NotifMsg) {
      case MRK_LBUTTONDOWN:
        if (lParam != 0) {
            return MouseDown((PAN_Point*) lParam, Entity);
        }
        break;
        
      case MRK_LBUTTONUP:
        if (lParam != 0) {
            // add the point.
            return MouseDown((PAN_Point*) lParam, Entity);
        }
        break;
        
      case MRK_MOUSEMOVE:
        if (lParam != 0) {
            return MouseMove((PAN_Point*) lParam, Entity);
        }
        break;
        
      case MRK_RBUTTONUP:
        if (lParam != 0) {
            return RightMouseUp((PAN_Point*) lParam, Entity);
        }
        break;
    }

    // Continue.
    return 1;
}

Public  BOOL    PCALLBACK
Translate(PAN_Point *Vector, LPMRK_EntitySpec Entity)
{
    RSampleUDE *ptr = Entity->Ent.Ude.pData;

    if (! ptr->Locked) {
        ptr->x[0].x += Vector->x;
        ptr->x[0].y += Vector->y;
        ptr->x[1].x += Vector->x;
        ptr->x[1].y += Vector->y;
    }

    return TRUE;
}

Public  long PCALLBACK
WriteEntity(LPMRK_EntitySpec Entity, BOOL fSizeOnly)
{
    long            nBytes = 0L;
    MRKENTHANDLE    Handle = Entity->Com.Handle;
    RSampleUDE      *ptr = (RSampleUDE*) Entity->Ent.Ude.pData;
    LPSTR           pData = NULL;
    
    if (! fSizeOnly) {
        // Actually writing:  Get data size and allocate.
        long DataSize = WriteEntity(Entity, TRUE);

        if ( ( pData = (LPSTR) MrkAlloc( DataSize ) ) == NULL ) {
            return 0L;
        }
    }

    // Long flag (for future use).
    nBytes += WriteLong(pData, nBytes, 0L); 
    
    // Points.
    nBytes += WritePoint(pData, nBytes, ptr->x);
    nBytes += WritePoint(pData, nBytes, ptr->x+1);
    nBytes += WriteInt(pData, nBytes, ptr->Locked);

    if (pData != NULL) {
        Entity->Ent.Ude.pData = pData;
        Entity->Ent.Ude.DataSize = nBytes;
    }

    return nBytes;
}

Public  long PCALLBACK
ReadEntity(LPMRK_EntitySpec Entity)
{
    Long            LL;
    long            nBytes = 0L;
    LPSTR           ptr = (LPSTR) Entity->Ent.Ude.pData;
    RSampleUDE      *pNewData = NULL;

    Entity->Ent.Ude.pData = NULL;
    Entity->Ent.Ude.DataSize = 0;
    
    if ( ptr == NULL || ( pNewData = (RSampleUDE *) MrkAlloc( sizeof(RSampleUDE) ) ) == NULL ) {
        return 0L;
    }

    // Long flag (for future use).
    nBytes += ReadLong(ptr, nBytes, &LL);

    // Points
    nBytes += ReadPoint(ptr, nBytes, pNewData->x);
    nBytes += ReadPoint(ptr, nBytes, pNewData->x+1);

	MrkAdjustPoints(Entity->Com.hWndCtl, pNewData->x, 2);

    nBytes += ReadInt(ptr, nBytes, &pNewData->Locked);

    // Obviously we have a Finished Product
    pNewData->Index = 1;

    Entity->Ent.Ude.pData = pNewData;
    Entity->Ent.Ude.DataSize = sizeof(RSampleUDE);

    return nBytes;
}

// Warps the Entity on a Global Scale!
Public  BOOL PCALLBACK
DoEdit(PAN_CtlRange *R1, PAN_CtlRange *R2, LPMRK_EntitySpec Entity)
{
    RSampleUDE *ptr = Entity->Ent.Ude.pData;

    if (R1 == NULL && R2 == NULL) {
        // EDIT CONTENTS

        // FOR TESTING ONLY!
        char    msg[256];
        int     nRet;

        // Get base file information from VCET
        HWND    hwndVCET = GetWindow(Entity->Com.hWndCtl, GW_HWNDNEXT);
        PAN_CtlFileInfo fi;
        memset(&fi, 0, sizeof(fi));
        SendMessage(hwndVCET, PM_CTLGETFILE, 0, (LPARAM)&fi);
        wsprintf(msg, "%s\n", fi.name);
        MessageBox(0, msg, "DEBUG", MB_TASKMODAL | MB_ICONSTOP | MB_OK);

        // Ask the user if the entity should be "locked in place"
        wsprintf(msg, "%s", "Do you want to lock the entity?");
        nRet = MessageBox(0, msg, "DEBUG", MB_TASKMODAL | MB_ICONSTOP | MB_YESNOCANCEL);

        if (nRet == IDYES) {
            ptr->Locked = TRUE;
        } else if (nRet == IDNO) {
            ptr->Locked = FALSE;
        }

        return TRUE;
        
    } else if (R2 == NULL) {
        // POINT EDITING
        return FALSE;
    } else if ( R1->min.x == R1->max.x ||
                R1->min.y == R1->max.y ||
                R2->min.x == R2->max.x ||
                R2->min.y == R2->max.y ) {

        // STRANGE VALUES
        return FALSE;
    }

    // BOX EDITING
    if (! ptr->Locked) {
        MrkDoEdit(&ptr->x[0], R1, R2);
        MrkDoEdit(&ptr->x[1], R1, R2);
    }
    return TRUE;
}

// Note that Size if not modified...
Public void PCALLBACK
DoCopy(LPMRK_EntitySpec Entity)
{
    RSampleUDE *P;

    RSampleUDE *ptr = (RSampleUDE *) Entity->Ent.Ude.pData;

    P = (RSampleUDE *) MrkAlloc( sizeof( RSampleUDE ) );
    MrkMemCopy( P, ptr, sizeof( RSampleUDE ) );

    Entity->Ent.Ude.pData = P;
}

Public  void PCALLBACK
Draw(MRK_RenderOptions *RenderSpec, LPMRK_EntitySpec Entity)
{
    MRKENTHANDLE Handle = Entity->Com.Handle;
    RSampleUDE *ptr = Entity->Ent.Ude.pData;

    // Prepare DrawInfo struct.
    MRK_DrawInfo    DrawInfo;
    memset(&DrawInfo, 0, sizeof(DrawInfo));

    DrawInfo.LineColor	= RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->LineColor : -1;
	DrawInfo.FillColor	= RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->FillColor : -1;
	DrawInfo.FillType	= RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->FillType : -1;
	DrawInfo.PenStyle	= RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->PenStyle : -1;
	DrawInfo.PenWidth	= RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->PenWidth : -1;
    DrawInfo.StartArrow = 0;
    DrawInfo.EndArrow	= 0;

    DrawInfo.xDevRes    = RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->xDevRes  : 0;
    DrawInfo.yDevRes    = RenderSpec->lpDrawInfo ? RenderSpec->lpDrawInfo->yDevRes  : 0;
    
    // Use built-in rendering functions.
    MrkDrawRect(Handle, RenderSpec->hdc,
            &ptr->x[0], &ptr->x[1],
            RenderSpec->mode, &DrawInfo );
    // Alteratively, you can draw to the HDC your self.
    // eg to get the client coordinated of the entity you would call:
    // PAN_Point p0 = ptr->x[0];
    // PAN_Point p1 = ptr->x[1];
    // SendMessage( Entity->Com.hWndCtl, MRK_WORLDTOCLIENT, 0, (LPARAM) &p0 );
    // SendMessage( Entity->Com.hWndCtl, MRK_WORLDTOCLIENT, 0, (LPARAM) &p1 );
    // And draw whatever you want.
    // You should respect the settings in DrawInfo to keep the user
    // interface consistent (e.g. colors, filling etc.)
}

Public  BOOL PCALLBACK
SelectionTest(PAN_CtlRange *selrange, LPMRK_EntitySpec Entity)
{
    RSampleUDE *ptr = Entity->Ent.Ude.pData;

    return MrkBoxVisibleInBox(
                &ptr->x[0], &ptr->x[1],
                &selrange->min, &selrange->max);
}

Public BOOL    PCALLBACK
GetControlPoints(LPINT pNumPts, LPPANPOINT *pPts, LPMRK_EntitySpec Entity)
{
    // Advanced function: To add custom-edit handles
    return False;
}

Public  BOOL PCALLBACK
BoundingBox(LPMRK_EntitySpec Entity)
{
    MRKENTHANDLE Handle = Entity->Com.Handle;
    RSampleUDE *ptr = Entity->Ent.Ude.pData;

    // Allocate List of Area(s)
    Entity->Ent.Ude.DataSize = sizeof( PAN_Point ) * 2;
    Entity->Ent.Ude.pData = MrkAlloc( Entity->Ent.Ude.DataSize );

    ((PAN_CtlRange *) Entity->Ent.Ude.pData)->min = ptr->x[0];
    ((PAN_CtlRange *) Entity->Ent.Ude.pData)->max = ptr->x[1];

    return TRUE;
}
/*********************************************************************/
/**  END UDE EXPORTS                                                **/
/*********************************************************************/

/*********************************************************************/
/**  BEGIN Basic Windows DLL Managment                              **/
/*********************************************************************/
#include "pan.h"
#include "pctl.h"
#include "mrkupctl.h"

// DLL Entry Function
BOOL WINAPI DllMain(HINSTANCE hinst, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason) {
    case DLL_PROCESS_ATTACH:

        InitApplication( hinst );
        return ( hinst ? 1 : 0 );

    case DLL_THREAD_ATTACH:
        break;

    case DLL_THREAD_DETACH:
        break;

    case DLL_PROCESS_DETACH:
        break;
    }
    return TRUE;
}

/*********************************************************************/
/**  END Basic Windows DLL Managment                                **/
/*********************************************************************/

