#ifndef	AVTYPES_H
#define	AVTYPES_H

#include "target.h"

#include "avwchar.h"

/*
**  The different types that are used throughout the C.S.I. code
*/

/*
**	Machine-independent types
*/
typedef	double			Double;		// Guaranteed to be 8 bytes, IEEE - signed
typedef	float			Float;		// Guaranteed to be 4 bytes, IEEE - signed
typedef	long long		Long64;		// Guaranteed to be 8 bytes - signed
#if FIX_LONG
typedef	int   			Long;		// Guaranteed to be 4 bytes - signed
#else
typedef	long			Long;		// Guaranteed to be 4 bytes - signed
#endif
typedef	short			Int;		// Guaranteed to be 2 bytes - signed
typedef	unsigned char	Char;		// Guaranteed to be 1 8bit byte - unsigned


// Signed/unsigned type for int32/64
typedef short              int16;
typedef int                int32;
typedef long long          int64;

typedef unsigned short     uint16;
typedef unsigned int       uint32;
typedef unsigned long long uint64;

typedef uint32   id_type;
typedef COLORREF color_type;  // uint32 so there is no dep on WIN?

#define	DoubleSize		8
#define	FloatSize		4
#define	LongSize		4
#define	IntSize			2
#define	CharSize		1

/*
**	Simple types
*/
typedef	signed char		Schar;
typedef	unsigned char	Uchar;

#ifndef	Bool
typedef	int				Bool;
#endif	/* Bool */

typedef	double			Real;
typedef	unsigned int	ULong;

typedef int				PANFILE;		// Type for the handle to a PANIO file
typedef	int				STREAMID;		// OLE stream identifier

/*
**  Function pointers
*/
typedef	void	(* VoidFn)();
typedef	int		(* IntFn)();
typedef	BOOL	(* BoolFn)();

typedef	void	(* FVoidFn)();
typedef	int	(* FIntFn)();
typedef	BOOL	(* FBoolFn)();

/*
**  A point
*/
typedef struct tagPAN_Point {
	double	x;
	double	y;
	double	z;
} PAN_Point, PANPOINT, *PPANPOINT, *LPPANPOINT;

typedef struct tagPHYSPOINT {
	double	x;
	double	y;
} PHYSPOINT, *LPPHYSPOINT;

/*
**  A rectangle
*/
typedef struct tagPAN_Rect {
	PAN_Point		   min;
	PAN_Point		   max;
} PAN_Rect, PANRECT, *PPANRECT, *LPPANRECT,
  PAN_RectangleRange, PAN_Range;

typedef struct tagPHYSRECT {
	double	left;
	double	top;
	double	right;
	double	bottom;
} PHYSRECT, FAR *LPPHYSRECT;

/*
** Thick Poly vertex.
*/
typedef struct {
	double	sw;
	double	ew;
	double	bulge;
} PAN_TVertex, *PPANTVERTEX, *LPPANTVERTEX;

/*
**  Integer coordinate point
*/
typedef	struct lpoint {
	int32  x, y;
} LPOINT,    *LPLPOINT,
  LONGPOINT, FAR *LPLONGPOINT;
typedef	const LPOINT *LPCLPOINT;

/*
**  Integer coordinate size
*/
typedef	struct lsize {
	int32	cx, cy;
} LSIZE,  *LPLSIZE;
typedef	const LSIZE		 *LPCLSIZE;

/*
**  Integer coordinate rectangle
*/
typedef	struct lrect {
	int32	left, top;
	int32	right, bottom;
} LRECT,    *LPLRECT,
  LONGRECT, FAR *LPLONGRECT;
typedef	const LRECT		 *LPCLRECT;

/*
**  Homegenous matrix transformation
*/
#define		HMATRIX_UNIT				0x00000001
#define		HMATRIX_2D					0x00000002
#define		HMATRIX_SCALEONLY			0x00000004
#define		HMATRIX_HASTRANSLATION		0x00000008
#define		HMATRIX_HASROTATION			0x00000010
#define     HMATRIX_UNIFORMSCALEONLY	0x00000020	// upper left 2x2 or 3x3 is a uniform scale
typedef	struct hmatrix {
	double	m[4][4];
	uint32  flags;
} Hmatrix, *PHmatrix,  *LPHmatrix;

/*
** Camera
*/
typedef struct tagPAN_CtlCamera {
	PAN_Point	ViewPoint;
	PAN_Point	Target;
	PAN_Point	UpDir;
} PAN_CtlCamera, * LPPAN_CtlCamera;

/*
**  An entity-handle
*/
typedef struct {
	DWORD	low;
	DWORD	high;
} PAN_Handle, PANHANDLE, *PPANHANDLE,  *LPPANHANDLE;

/**
 *	Structure used to describe Resource Info
 */
typedef struct {
	/** Text font, Linestyle, shape, or external reference file */
	UINT resourceType;

	/** Name of resource ID (Logical Name) */
	char szResourceID[_MAX_PATH];

	/** Number of characters in the name of resource ID */
	UINT cbResourceID;

	/** Name of resource (File Name) */
	char szResource[_MAX_PATH];

	/** Number of characters in the name of resource */
	UINT cbResource;

	/** Full path to the resource or substituted res */
	char szResolvedResource[_MAX_PATH];

	/** Number of characters in the full path */
	UINT cbResolvedResource;

    /** Paths to search in, in order to locate resources */
	char szSearchResources[_MAX_SEARCHPATHS];

    /** Number of characters in the c-string szSearchResources */
	UINT cbSearchResources;

    /** Hints used in search to locate resources */
	char szSearchHints[_MAX_SEARCHHINTS];

	/** Number of characters in the c-string szSearchHints */
	UINT cbSearchHints;

	/** Located / missing flag */
	BOOL fOK;

} PAN_RESOURCEINFO, *PPAN_RESOURCEINFO, *LPPAN_RESOURCEINFO, *HPPAN_RESOURCEINFO;

/*
**	Structure used to describe an External Reference Setting
*/
typedef struct {
	id_type		id; 				// Unique identifier
	char		name[_MAX_PATH];	// Logical Name of xref
	char		fname[_MAX_PATH];	// Full filename of xref
	BOOL		bState; 			// ON or OFF
} PAN_XREF, *PPAN_XREF, FAR *LPPAN_XREF;

/*
**	Structure used to describe a layer.
*/
typedef struct {
	id_type		id; 			// Unique identifier
	char		name[PAN_MAX_PATH]; // Name of layer
	color_type	color;			// RGB value of the layer
	BOOL		bState; 		// ON or OFF
	BOOL		fThawed;		// Frozen or Thawed
	BOOL		bReadOnly;		// Read only or editable
	BOOL		fNotPrintable;
} PAN_LAYER, *PPAN_LAYER, FAR *LPPAN_LAYER;

typedef struct {
	id_type		id; 			// Unique identifier
	BOOL		fNonRect;		// Viewport type: Rectangular, Non_Rectangular
	char		name[STRLEN80]; // View port
	int			pageId;
	PAN_Rect	BBox;

	// TODO: Need to add new viewport properties...

} PAN_VIEWPORT, *PPAN_VIEWPORT, FAR *LPPAN_VIEWPORT;
/*
**	Structure used to describe a block.
*/
typedef struct {
	id_type		id; 			// unique identifier
	char		name[STRLEN80]; // name of block
} PAN_BLOCK, *PPAN_BLOCK, FAR *LPPAN_BLOCK;

/**
 * Structure used to describe a saved view.
 * Note: for 3D views, the transformation matrix (view) is the matrix describing the camera (see CPanCamera(CPanTransform const & viewMatrix))
 */
typedef struct {
	id_type		id; 			// unique identifier
	char		name[STRLEN80]; // name of view
	PAN_Point	vmin, vmax; 	// Window coordinates;
	Hmatrix 	view;			// transformation matrix
	DWORD       dwFlags;        // Additional flags (see PAN_VIEWFLAG_*)
} PAN_VIEW, *PPAN_VIEW, FAR *LPPAN_VIEW;

// Flags for 3D views.
#define PAN_VIEWFLAG_NONE                  0x00 // Default
#define PAN_VIEWFLAG_AUTO_CAMERA_POSITION  0x01 // Allow VCET to move camera outside of model (camera orientation will be respected)
#define PAN_VIEWFLAG_ZOOM_FIT              0x02 // Tell VCET to adjust the zoom factor to do a zoom fit.
#define PAN_VIEWFLAG_IS_PERSPECTIVE        0x04 // If set the view is perspective, otherwsie orthograpgic
#define PAN_VIEWFLAG_AUTO_Z_CLIPPING       0x08 // If set the viewing volume (vmin, vmax).z are computed to encompass model z-extents to avoid clipping, otherwise decoder must pass valid zNear/zFar planes in vmin.z vmax.z
#define PAN_VIEWFLAG_MODELSPACE_CAMERA     0x10 // If set that means the decoder has a camera defined in model space, if not set this means the camera transformation must be applied once the model has been moved to (0,0,0) (standard case)

/*
**	Structure used to describe a vector font.
*/
#define	PAN_MAXFONT_NAME		256
#define	PAN_MAXFONT_CHARWIDTHS	256
typedef	struct {
	DWORD 	dwType;									// Font type:  PAN_FONTTYPE_XXX
	DWORD	dwFlags;								// Font flags:  one or more PAN_FONTFLAG_XXX
	double	Size;									// Font size.
	BOOL	fHmValid;								// Does hm contain a valid matrix.
	Hmatrix hm;										// Transformation matrix.
	double	CharWidths[PAN_MAXFONT_CHARWIDTHS+1];	// Buffer of character widths
	char	szName[PAN_MAXFONT_NAME+1];				// Font name.
	char	szFileName[_MAX_PATH+1];				// File name containing the font definition.
} PAN_FONT, *PPAN_FONT, FAR *LPPAN_FONT;

typedef	struct PAN_BOOKMARK {
	id_type	ID;					// Any app-defined ID
	int		level;				// 0:=root level
	int		fState;				// 0:=closed, 1:=opened, 2:=leaf
	wchar_t	name[_MAX_PATH];	// Descriptive name
	char	target[1024];		// Target destination.
}	PAN_BOOKMARK, *PPAN_BOOKMARK,
	 *LPPAN_BOOKMARK, *HPPAN_BOOKMARK;

/************************************************************************
**	Declare the basic Document types ************************************
*************************************************************************/
typedef enum {
	PAN_UnknownFile,
	PAN_RasterFile,
	PAN_VectorFile,
	PAN_DatabaseFile,
	PAN_SpreadsheetFile,
	PAN_DocumentFile,
	PAN_ArchiveFile
} PAN_FileType;

/************************************************************************
** Begin vector-specific types
************************************************************************/

typedef struct tagPAN_TEXTMETRIC {
	double tmHeight; 
    double tmAscent; 
    double tmDescent; 
    double tmInternalLeading; 
    double tmExternalLeading; 
    double tmAveCharWidth; 
    double tmMaxCharWidth; 
    double tmOverhang; 

} PAN_TEXTMETRIC, FAR * LPPAN_TEXTMETRIC;

typedef struct {
	PAN_Point	start;
	PAN_Point	end;
	double		height;
	double		width;
	WORD		align;
	uint32		attrib;
	uint32		color;
	double		strwidth;
	double		strheight;
	double		oblique;   //not used for now
	PAN_TEXTMETRIC FAR * ptm;
} TTEXTLF, FAR * LPTTEXTLF;

//structure for PANX_SetLineStyle function
typedef struct tagPAN_LineStyle {

	int		nElements;
	double	FAR * lpElement;
	DWORD Reserved;

} PAN_LineStyle, FAR * LPPAN_LineStyle;	


// structures for the PANX_SetFontEx callback
typedef struct tagPANPDFFontMapping {
	int nCount;						// element number of toGIDMapping or type1Encoding
	char** pType1Encoding;			// type1 font encoding for building to-GID mapping 
	unsigned short* pToGIDMapping;	// to-GID mapping
} PAN_PDFFontMapping, *PPAN_PDFFontMapping;

typedef struct {
	unsigned int 	nFirst;			// this record applies to
	unsigned int 	nLast;			// CIDs <first>..<last>
	double 			fWidth;			// char width
} PAN_PDFCIDFontWidthExcep;

typedef struct {
	unsigned int	nFirst;			// this record applies to
	unsigned int	nLast;			// CIDs <first>..<last>
	double 			fHeight;		// char height
	double 			fVx, fVy;		// origin position
} PAN_PDFCIDFontWidthExcepV;

typedef struct {
		double 						fDefWidth;	// default char width
		double 						fDefHeight;	// default char height
		double 						fDefVY;		// default origin position
		PAN_PDFCIDFontWidthExcep* 	exceps;		// exceptions
		int 						nExceps;	// number of valid entries in exceps
		PAN_PDFCIDFontWidthExcepV*	excepsV;	// exceptions for vertical font
		int 						nExcepsV;	// number of valid entries in excepsV
} PAN_PDFCIDFontWidths;

typedef struct tagPANPDFFontWidths {
	int nCount;
	union {
		double*						pNonCIDFontWidths;
		PAN_PDFCIDFontWidths*		pCIDFontWidths;
	} widths;
} PAN_PDFFontWidths, *PPAN_PDFFontWidths;


// structure for PANX_Point user defined point style - currently not implemented (!)
typedef struct tagPAN_UserPointStyle {
	int nID;
} PAN_UserPointStyle;
				

// enum for PANX_Point
enum PanPointStyle{
	PAN_POINTSTYLE_DOT,		    // Simple dot point
	PAN_POINTSTYLE_SQUARE,		// The point is drawn as a small square
	PAN_POINTSTYLE_CIRCLE,		// The point is drawn as a small circle
	PAN_POINTSTYLE_TRIANGLE,	// The point is drawn as a small triangle
	PAN_POINTSTYLE_STAR,		// The point is drawn as a small star (5 vertices)
	PAN_POINTSTYLE_PLUS,        // the point is drawn as a small plus sign (+)
	PAN_POINTSTYLE_CROSS,       // the point is drawn as a small cross x
	PAN_POINTSTYLE_DIAMOND,     // the point is drawn as a small diamond
	PAN_POINTSTYLE_ASTERISKX,   // The point is drawn as a small asterisk
	PAN_POINTSTYLE_ASTERISKY,   // The point is drawn as a small asterisk
	PAN_POINTSTYLE_ASTERISK,    // The point is drawn as a small asterisk
	PAN_POINTSTYLE_CUSTOM		// Custom point style should be defined in an additionnal structure PAN_UserPointStyle
};

/************************************************************************
** End vector-specific types
************************************************************************/

/************************************************************************
** Begin AB/SS/AR-specific types
************************************************************************/

typedef struct {
	unsigned char	iAlign; 				// PAN_ALIGN_{LEFT|RIGHT|CENTER}
	double			Width; 					// in characters
	WORD			wFlags;					// special behaviour flags
	char			szTitle[STRLEN240 + 1];
} PAN_Column;

// Used only with PANX_SetSSProps
// num is either column or row number
typedef struct {
	int 	num;
	double	value;
} PAN_RowColAttrib, *PPAN_RowColAttrib, FAR *LPPAN_RowColAttrib;

typedef struct {
	int				type;

	int				formatFam;			// Main number format
	int				subFormat;			// Sub number format
	int 			formatDigs;			// Number of digits in number format

	int 			formatFlgs;			// Can be a combination of:
										// PAN_SSFRMTFLG_HILITE, PAN_SSFRMTFLG_PAREN, PAN_SSFRMTFLG_REV, PAN_SSFRMTFLG_FILL
										// PAN_ALIGN_RIGHT, PAN_ALIGN_CENTER, PAN_ALIGN_JUSTIFY, PAN_ALIGN_LEFT
										// PAN_ALIGN_TOP, PAN_ALIGN_VCENTER, PAN_ALIGN_BOTTOM, PAN_ALIGN_VJUSTIFY
										// PAN_ALIGN_TOPTOBOTTOM, PAN_ALIGN_WRAP, PAN_SSFRMTFLG_DISTRIBUTEDINDENT
	
	int 			brdrFlgs;			// Specifies what borders are visible

	int 			brdrStyle[8];		// Style for every one of the 8 borders (Left, Right, Top, Bottom, Up Diagonal, Down Diagonal, Vertical, Horizontal)
	color_type		brdrColor[8];		// Color for every one of the 8 borders
	RECT			brdrRect;			// Range of cells that the current cell is part of (needed for Vertical/Horizontal border support)

	double			rotation;			// Angle of text rotation
	double			leftIndent;			// Left indent amount

	color_type		fgColor;			// Text color
	color_type		bkColor;			// Fill background color
	color_type		patternFgColor;		// Pattern fill foreground color
	unsigned int	pattern;			// 4x4 monochrome bit pattern

	WORD			wDefaultFields;		// Indicates which of the above members have not been user set
										// Uses PAN_SSATTRIBSET_XXX flags

} PAN_CellAttrib, *PPAN_CellAttrib, FAR *LPPAN_CellAttrib;


// Represents different text properties
typedef struct {
	LOGFONTW	lfont;				// logfont
	color_type	color;				// font colour
	WORD		flags;				// flags for superscript, subscript, whatever is missing in LOGFONT
	DWORD		dwDefaultFontProps;	// indicates which fields of the logfont were actually set by the decoder, combination of PAN_SSATTRIBDEFAULT_FONTXXX
} PAN_TextProperies;


// Represents a complex text run.
typedef struct {
	WORD				nNumRuns;		// number of formatting runs
	PAN_TextProperies*	pProperties;	// array of properties
	DWORD*				pPositions;		// array of text positions where matching property starts
	WCHAR16*			pwText;			// unicode text string
} PAN_MultiRunText;


typedef struct {
	int 	type;
	union {
		int 	intVal;
		double	dbleVal;
		LPSTR	strVal;
		PAN_MultiRunText	multi;
	} value;
} PAN_CellContents, *PPAN_CellContents, FAR *LPPAN_CellContents;


typedef struct {
	HDC		hDC;				// dc on which to draw the text
	PANRECT	boundingRect;		// bounding rect in which to draw the text
	double	rotation;			// text rotation angle
	WORD	wAlign;				// text alignment
	PAN_MultiRunText*	pRuns;	// text and its corresponding runs	
} PAN_SSTextBox, *PPAN_SSTextBox, *LPPAN_SSTextBox;


typedef struct {

	// Cell of the comment
	WORD	wRow;		// Row of the comment
	WORD	wCol;		// Column of the comment

} PAN_SSComment;


typedef struct {
	// Bit flags indicating which properties have to be applied
	BYTE	cFlags;		// Bits: 0 - Font Color, 1 - Cell Color, 2 - Font Style

	// Cell with conditional format
	WORD	wRow;		// Cell row
	WORD	wCol;		// Cell column

	// Colors to apply for the cell
	color_type rgbFontColor;	// Color for the text (foregroung color)
	color_type rgbCellColor;	// Color for the cell (backgroung color)

	int nFontStyle;			// 0 - Regular, 1 - Italic, 2 - Bold, 3 - Bold Italic

} PAN_SSConditionalFormat;

/************************************************************************
** End DB/SS/AR-specific types
************************************************************************/

/************************************************************************
** Begin document-specific types
************************************************************************/

/*
** Document Table.
*/
typedef struct tagTABLE {
	int 	nRows;			// number of rows, or 0 if not known
	WORD	wBreakMode;		// How to break table if it intersects another
							// rigid element.
} PAN_Table, FAR *LPPAN_Table;

/*
** Document Table Row.
*/
typedef struct tagROW {
	id_type	idRow;			// identifies row for the core to reference in future
	int 	xTable; 		// set to -1 to use default
	int 	yTable; 		// set to -1 to use default
	int 	yaHeight;		// row height at least, exactly, or variable
	BYTE	jc; 			// justification of a row within page column
							// ALSO USES FLAGS TO CONTROL TABLE ROW BEHAVIOUR
							// FLAG_HEADER_ROW  // 0x80
							// FLAG_ROW_NOSPLIT	// 0x40
	int 	nCells; 		// number of cells, or 0 if not known
} PAN_Row, FAR *LPPAN_Row;

/* Universal structure for a border segment around text units */
typedef struct tagBORDER {
	BYTE	brdStyle;		// No border, single-line, double-line, thick, dotted ...
	int 	brdLineWidth;	// Width of a single line of the border.
	int 	dxpSpace;		// Width of space to maintain between border and text within border.
							//		(Must be 0 when used to describe table cell border.)
	int 	brdType;		// One of predefined types of border length
	int 	brdLength;		// Length of border line if brdType = PAN_BL_CUSTOM
	DWORD	rgbColor;		// Border color
} PAN_Border, FAR * LPPAN_Border;

/*
** General shading structure
*/
typedef struct {
	int		iPercentage;	// In hundredths of a percent !
	int		iPattern;
	DWORD	rgbForeColor;
	DWORD	rgbBackColor;
} PAN_Shading, FAR *LPPAN_Shading;

typedef struct {
	char			ObjectID[16];
	char			ShareID[16];
	int 			ObjectType;
	char			filename[PAN_MAX_PATH];
	WORD			flags;				// special flags

	LPSTR			lpszHeader; 		// pointer to file prefix
	int 			cbHeader;			// size of the file prefix

	WORD			wPageNumber;		// Page number of the file to use
	int 			zOrder; 			// -1: underlay, 0: inline, 1: overlay.

	size_t			ObjectOffset;		// starting of data in the file
	size_t			ObjectSize; 		// size of data to extract
	PANRECT			source; 			// for images: current view extents or cropping percentages.
	PAN_Border *	borders;
	PAN_Shading *	shading;
	WORD			wRotate;			// Rotation angle, in degrees (0, 90, 180 or 270)
	WORD			wFlip;				// Flipping (PAN_FLIPNONE, PAN_FLIPX, PAN_FLIPY or PAN_FLIPXY)
	BOOL			fTransform;			// Whether transformation should be applied
	BYTE			TransMethod;		// Transformation method
	Hmatrix			hm;					// Transformation matrix
	PAN_Point		TransformedRegion[4]; 
	color_type      fgColor;			// Foreground color
	BOOL			fUsefgColor;		// Whether foreground color should be set in control
	color_type      bgColor;			// Background color
	BOOL			fUsebgColor;		// Whether background color should be set in control
	color_type      trColor;			// Transparent color
	BOOL			fUsetrColor;		// Whether transparent color should be aplied - (OBJECTFLAG_TRANSPARENT must be set).
	void *			pLoadingAttributes;	// Attributes to be passed to the overlay decoder
} PAN_OBJECT, *PPAN_OBJECT, FAR *LPPAN_OBJECT;

/* A structure for lines and shadows around text units */
typedef struct	{
	PAN_Border	leftBord;	// Border to be used to the left of an object
	PAN_Border	topBord;	// Border to be used above an object
	PAN_Border	rightBord;	// Border to be used to the right of an object
	PAN_Border	bottomBord; // Border to be used below an object
	PAN_Shading intShade;	// Shading to be used in the interior of borders
	int 		iRadius;	// Curvature of border corners in percent
							// 0% - square corners, 100% - all-curved frame
	int 		shadDir;	// Existence and/or direction of a shadow for the border
	int 		shadDepth;	// Depth of border shadow in TWIPS. If 0, no shadow.
	DWORD		rgbShadColor;// Color to be used to paint border shadow
} PAN_Edge, FAR * LPPAN_Edge;

/*
** Specific area on a page template
*/
typedef struct {
	PANRECT 		prBoundary;
	PAN_DocElemType	type;
	FlowID			flowId;
	WORD			wWrap;
	WORD			wRelHorzPos;
	WORD			wRelVertPos;
	BYTE			Transparency;	// Transparency code
	LPPAN_Edge		pBordFrame;		// Borders and shadows associated with element
									// Set to NULL for no borders.
} PAN_Element, *PPAN_Element, FAR *LPPAN_Element;

/*
** Line items in documents
*/
typedef struct {
	PAN_Point		from;
	PAN_Point		to;
	PAN_Border FAR	*lpRuleStyle;
} PAN_Rule, *PPAN_Rule, FAR *LPPAN_Rule;

/*
**  A tab-position definition
*/
typedef struct {
	BYTE	bTabType;
	short	wTabPos;
} PAN_TAB, FAR *LPPAN_TAB;


// To be used with PANX_SetDocProps to send default paragraph information
typedef struct	{
	WORD		wAlign;						// One of PAN_ALIGN_LEFT, PAN_ALIGN_CENTER, PAN_ALIGN_RIGHT, PAN_ALIGN_JUSTIFY
	int			nLeftIndent;				// Left indent
	int			nRightIndent;				// Right indent
	int			nFirstIndent;				// First line indent
	int			nBeforeSpacing;				// Paragraph before spacing
	int			nAfterSpacing;				// Paragraph after spacing
	WORD		wLineSpacing;				// Line spacing
	PAN_Edge 	OuterBorders;				// Outer borders (top, left, right, bottom) and paragraph fill shading
	PAN_Border 	InnerBorder;				// Inner horizontal border
	BOOL		bAutoBeforeSpacing;			// TRUE if before spacing is automatic
	BOOL		bAutoAfterSpacing;			// TRUE if after spacing is automatic
	BOOL		bHasBullets;				// TRUE if the paragraph has bullets
	BOOL		bWidowOrphanControl;		// TRUE if widow/orphan control is turned on
	BOOL		bKeepLinesTogether;			// TRUE if keep lines together is turned on
	BOOL		bKeepWithNext;				// TRUE if keep with next is turned on
	int 		nTabs;						// Count of tabs in the list
	PAN_TAB* 	pTabList;					// Tab list
} PAN_Paragraph, *PPAN_Paragraph;

/*
** Text element information
*/
typedef struct {
	POINT		Page;	// page size
	POINT		Off;	// page offsets
	RECT		Marg;	// page margins
} TxSize;

/*
** Absolutely-positioned object
*/
typedef struct	{
	int 	iPage;			// Page no. or 0 if unknown
	BYTE	pcHorz; 		// 0 - relative to column, 1 - margin, 2 - page
	BYTE	pcVert; 		// 0 - relative to margin, 1 - page, 2 - to text (where non_APO
							//				 would have been placed)
	int 	dxaAbs; 		// Positive - horiz. dist. from reference frame,
							//				 0  - positioned at the left,
							//				-4  - centered horizontally,
							//				-8  - adjusted right,
							//				-12 - placed immediately inside RF,
							//				-16	- placed immediately outside of RF
	int 	dyaAbs; 		// Positive - vert. dist. from reference frame,
							//				 0  - y-position is unconstrained,
							//				-4  - on top of RF,
							//				-8  - centered vertically,
							//				-12 - at bottom of RF,
	RECT	distFromText;	// Distance between APO and the text around it in all four directions
	//int 	dxaFromText;	// Horiz.dist. between APP and any non-AP text
	//int 	dyaFromText;	// Vertical.dist. between APP and any non-AP text
	int 	dxaWidth;		// When !=0, margin & column settings have no effect
	int		wHeightAbs; 	// Sign bit = 0 -> exact, 1 -> at least
	BYTE	Wrap;			// Wrap code
	BYTE	Transparency;	// Transparency code:  PAN_TR_TRANSPARENT, PAN_TR_OPAQUE, PAN_TR_UNDERLIE
	PAN_Edge	bordFrame;	// Borders and shadows associated with an APO
	DWORD	behavior;		// Behavioral flags
							// Could be one of the following:
							//   PAN_APOPOS_FORCE_ON_PAGE - if set we have to force 
							//                      the APO to be on page completely, 
							//                      even if its coordinates are outside of the page.
							//	 PAN_APO_IS_FRAME - APO is a frame
	int		nZOrder;		// z-order. values less than 2000h are below the main text
	int		dxFrameSpace;	// If the APO is a frame, this is the distance required between the 
							// column and the object before text is allowed to wrap in that 
							// area (see bug 27564).  Not used if not a frame.

} PAN_APO, FAR *LPPAN_APO;

typedef struct tagCELL {
	id_type	idCell; 		// Identifies cell for the core to reference in future
	BOOL	fNewCall;		// True by default
	int 	fType;			// type of data which cell contains
	int 	xRow;			// set to -1 to use default
	int 	yRow;			// set to -1 to use default
	int 	xaWidth;		// cell width must be non-zero
	int		yaHeight;		// Default height if cell is empty.
	int 	dxaMarginLeft;	// space between left border & data
	int 	dyaMarginTop;	// space between top border & data
	int 	dxaMarginRight; // space between right border & data
	int 	dyaMarginBottom;// space between bottom border & data
	DWORD   dwFlags;        //now it used for merged cells       
	PAN_Edge	bordCell;	// Cell borders and shading
	PAN_Border	LeftTopToRightBottomDiagonal; // diagonal border of the cell going from the cell left top corner to its right bottom corner
	PAN_Border	RightTopToLeftBottomDiagonal; // diagonal border of the cell going from the cell right top corner to its left bottom corner
} PAN_Cell, FAR *LPPAN_Cell;	

/************************************************************************
** End document-specific types
************************************************************************/

////////////////////////////////////////////////////////////////////////////////////
//	Some type definitions for the drawing objects
////////////////////////////////////////////////////////////////////////////////////

typedef struct {
	POINT		start;
	POINT		end;
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow
	WORD		eppStart;	// Start endpoint properties.
	WORD		eppEnd;		// End endpoint properties.
							// (& 0x0003) : 0 - none; 1 - hollow; 2 - filled
							// (& 0x000c) : weight
							// (& 0x0030) : length		
}	PAN_LINE, FAR *LPPAN_LINE;

typedef struct {
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow
	BYTE		fLeft;		// True when endpoint is to the left of the start point.
	BYTE		fUp;		// True when endpoint is above the start point.
}	PAN_ARC, FAR *LPPAN_ARC;

typedef struct {
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow
}	PAN_ELLIPSE, FAR *LPPAN_ELLIPSE;

typedef struct {
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow
}	PAN_RECTANGLE, FAR *LPPAN_RECTANGLE;

typedef struct {
	int			nPts;	
	POINT FAR	*lpPoints;
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow
	WORD		eppStart;	// Start endpoint properties.
	WORD		eppEnd;		// End endpoint properties.
							// (& 0x0003) : 0 - none; 1 - hollow; 2 - filled
							// (& 0x000c) : weight
							// (& 0x0030) : length		
}	PAN_POLYLINE, FAR *LPPAN_POLYLINE;

typedef struct {
	int			nPts;	
	POINT*		pPoints;
	int			nCurves;
	int*		pCurves;
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow	
}	PAN_POLYPOLYGON, FAR *LPPAN_POLYPOLYGON;

typedef struct {
	color_type	color;
	int			weight;
	int			style;		// 0 - solid; 1 - dashed; 2 - dotted; 3 - dash dot;
							// 4 - dash dot dot; 5 - hollow

	BOOL		fOpen;		// TRUE to open text box for input, FALSE to close it.
	DWORD		dwFlags;	// Flags for specific text box properties, e.g. text flow direction (text rotation - vertical text)
}	PAN_TEXTBOX, FAR *LPPAN_TEXTBOX;

/*
** Drawing Objects
*/
typedef struct	{
	int 	iPage;			// Page no. or 0 if unknown
	BYTE	pcHorz; 		// 0 - relative to column, 1 - margin, 2 - page
	BYTE	pcVert; 		// 0 - relative to margin, 1 - page, 2 - to text (where non_APO
							//				 would have been placed)
	int 	dxaAbs; 		// Positive - horiz. dist. from reference frame,
							//				 0 - positioned at the left,
							//				-4 - centered horizontally,
							//				-8 - adjusted right,
							// -12 - placed immediately inside RF,
							// -16	- placed immediately outside of RF
	int 	dyaAbs; 		// Positive - vert. dist. from reference frame,
							//				 0 - y-position is unconstrained,
							//				-4 - on top of RF,
							//				-8 - centered vertically,
							// -12 - at bottom of RF,
	int 	dxaWidth;
	int		dyaHeight;
	int		dzaHeight;		// z-order or height (0x1000 to 0x3000)
							// Value less than 0x2000 means transparent.

	PAN_Shading	shading;

	PAN_DrawingObjType	type;
	HBITMAP				hBmp;		// background fill bitmap

	union {
		PAN_LINE		line;
		PAN_RECTANGLE	rectangle;
		PAN_ARC			arc;
		PAN_ELLIPSE		ellipse;
		PAN_POLYLINE	polyline;
		PAN_TEXTBOX		textbox;
		PAN_POLYPOLYGON	polypolygon;
	} u;	
} PAN_DrawingObj, FAR *LPPAN_DrawingObj;


//////////////////////////////////////////////////////
//  Font Mapping Structure

typedef struct fontMapping {
	PANOSE		panose;							//  Windows PANOSE structure
	WCHAR16		AltFontName[LF_FACESIZE];		//  Alternative font in Unicode
	DWORD		mode;							//  Mode for font mapping algorithm
	LPLOGFONTW	pLogFontW;						//  Unicode version of the font
	int			nCharSpacing;					//  Spacing between characters
} FontMapping;


//  Font mapping modes
#define		FONTMAP_ALTFONTNAME_ABOVE_PANOSE	0x00	//  Alternative font name is more important than panose
#define		FONTMAP_PANOSE_ABOVE_ALTFONTNAME	0x01	//  Panose information is more important than alternative font name
#define		FONTMAP_ALTFONTNAME_UNICODE			0x02	//  Alternative font face name is passed as unicode
#define		FONTMAP_NO_REPORTING				0x04	//  Do not report font substitution in the prioperties dialog.
#define		FONTMAP_NO_USEFUL_INFO				0x08	//  Font substitution has no useful information. Treat as empty.
#define		FONTMAP_USE_UNICODE_FONT			0x10	//  Use the Unicode logfont, ignore LPLONGFONT parameter
#define		FONTMAP_WORD_REGISTRY_MAPPING		0x20	//  Use the Word registry key font mapping



////////////////////////////////////////////////////////////////////////////////////
// End of the type definitions for the drawing objects
////////////////////////////////////////////////////////////////////////////////////

#endif	/* AVTYPES_H */
