/******************************************************************************
** export.h
**
** PANORAMIC! Conversion DLL Interface.
******************************************************************************/

#ifndef	PAFSEXPORT_H
#define	PAFSEXPORT_H

#include "pan.h"

#if	TARGET == WIN16
#pragma	pack(1)
#else
#pragma	pack(8)
#endif

#ifdef __cplusplus
extern "C" {
#endif 

/*************************************************************************
*******  Shared defines and typedefs for the Conversion Engine     *******
**************************************************************************/


/* 
**  Type of the output format.
*/
#define	PC_RASTER	0x0001
#define	PC_VECTOR	0x0002
#define	PC_3D   	0x0003
#define	PC_PDF   	0x0004


/*
**  Output capabilities
*/
#define PC_CAPS(x)			((x)&0x0000FFFF)

#define	PC_CAPS_SUPPORTS_BITMAP			0x00000001
#define	PC_CAPS_SUPPORTS_LINE			0x00000002
#define	PC_CAPS_SUPPORTS_ELLIPTICARC	0x00000004
#define	PC_CAPS_SUPPORTS_ELLIPSE		0x00000008
#define	PC_CAPS_SUPPORTS_POLY			0x00000010
#define	PC_CAPS_SUPPORTS_VTEXT			0x00000020
#define	PC_CAPS_SUPPORTS_FILLEDPOLY		0x00000040
#define	PC_CAPS_SUPPORTS_MULTIPAGE		0x00000080

/*
**  Hints for format output (x,y or A to E)
**  and Steps per inch
*/
#define PC_HINT(x, hint)		(((x)&(hint))==(hint))
#define PC_HINT_XBYY			0x00000000		// default value
#define PC_HINT_ATOE			0x00010000
#define PC_HINT_STEPSPERINCH	0x00020000
#define PC_HINT_TRIANGULATION	0x00040000

#ifndef WIDTHBYTES
#define	WIDTHBYTES(x)			(((x) + 7) >> 3)
#endif

/*
**  The PANMETAFILE record types
*/
#define	PANMETAREC_NOP			0x0000
#define	PANMETAREC_EOM			0x0001
#define	PANMETAREC_BITMAP		0x0002
#define	PANMETAREC_LINE			0x0003
#define	PANMETAREC_POLY			0x0004
#define	PANMETAREC_ELLIPTICARC	0x0005
#define	PANMETAREC_ELLIPSE		0x0006
#define	PANMETAREC_VTEXT		0x0007
#define	PANMETAREC_ROTFLIP		0x0008


/*
**  Generic "don't care" define
*/
#define	PC_DONTCARE			0

/*
**  Unit definitions
*/
#define PC_UNIT_PIXEL			0x0001
#define PC_UNIT_INCH			0x0002
#define PC_UNIT_MM			0x0004

/*
** inch to mm conversion factor
*/
#define PC_INCHTOMM			25.4

/*
**  Information describing Rotation/Flipping record.
*/
typedef struct  {
	WORD	Rotate;
	WORD	Flip;
} PANMETA_ROTFLIP, FAR * LPPANMETA_ROTFLIP;

/*
**  Information describing a Raster Line entity.
*/
typedef struct  {
	Long		Offsetx;
	Long		Offsety;
	Long		NumBytes;
	LPBYTE		lpPixels;
} PANMETA_BITLINE, FAR * LPPANMETA_BITLINE;

/*
**  Information describing a Vector Line entity.
*/
typedef struct  {
	PAN_Point	p1;
	PAN_Point	p2;
	Long		LineColor;
} PANMETA_LINE,  FAR * LPPANMETA_LINE; 

/*
**  Information describing a Polyline entity.
*/
//Fethi 11/30/98: 128 changed this value to fix bug#1743
//Ajit Jan 25, 2001: Increase from 256 to 4096 for bug 5014
#define	MAX_PCPOLY_POINTS	4096
typedef struct  {
	Int		numPts;
	Long		LineColor;
	Long		FillColor;
	PAN_Point	Points[MAX_PCPOLY_POINTS + 4];
} PANMETA_POLY,	FAR * LPPANMETA_POLY;

/*
**  Information describing an Elliptic Arc entity.
*/
typedef struct  {
	PAN_Point	center;
	Real		radius1;
	Real		radius2;
	Real		stang;
	Real		endang;
	Long		LineColor;
	Long		FillColor;
	Bool		b;
} PANMETA_ELLIPTICARC, FAR * LPPANMETA_ELLIPTICARC;

/*
**  Information describing an Ellipse entity.
*/
typedef struct  {
	PAN_Point	center;
	Real		radius1;
	Real		radius2;
	Long		LineColor;
	Long		FillColor;
	Bool		b;
} PANMETA_ELLIPSE, FAR * LPPANMETA_ELLIPSE;

/*
**  Information describing a Text entity.
*/
#define	MAX_PCVTEXT_CHARS	256
typedef struct  {
	PAN_Point	p;
	Real		width;
	Real		height;
	Real		rotation;
	Real		oblique;
	Long		LineColor;
	Real		rotm[9];
	char		sz[MAX_PCVTEXT_CHARS];
} PANMETA_VTEXT, FAR * LPPANMETA_VTEXT;

/*
**  Structure which holds one entity type.
*/
typedef  struct	tagPANMETARECORD {
	WORD		type;
	union	{
		PANMETA_BITLINE		bitline;
		PANMETA_LINE		line;
		PANMETA_POLY		poly;
		PANMETA_ELLIPTICARC	ea;
		PANMETA_ELLIPSE		el;
		PANMETA_VTEXT		vtext;
	} rec;
} PANMETARECORD, FAR * LPPANMETARECORD;

/*
** FIXME: Export still needs the old PAN_Cmap structure for the 
** PANCVT_GetOutputColormap callback. This should be removed ASAP. 
** Dave Belanger, 2004/09/13.
*/
#define MAXCMAP 256

typedef struct {
    int  ncolours;
    BYTE r[MAXCMAP];
    BYTE g[MAXCMAP];
    BYTE b[MAXCMAP];
} PAN_Cmap, PANCMAP, *PPANCMAP, *LPPANCMAP;

/*
**	Callback structure
*/
typedef	struct	 tagCVTPANX {
	/*
	** Common functions.
	*/
	BOOL (CALLBACK *PANCVT_GetOutputSize)(LONG ID, struct tagCVTPANX FAR *cbs, LPWORD wUnits, Real FAR *stepsperinch, Real FAR *width, Real FAR *height);
	BOOL (CALLBACK *PANCVT_GetOutputNumPages)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPINT /*numPages*/);
	BOOL (CALLBACK *PANCVT_GetOutputColormap)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPINT /*bitsperpixel*/, LPPANCMAP /*colormap*/);
	BOOL (CALLBACK *PANCVT_GetOptionInt)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR /*module*/, LPSTR /*section*/, LPSTR /*name*/, int /*default*/, LPINT /*result*/);
	BOOL (CALLBACK *PANCVT_GetOptionReal)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR module, LPSTR /*section*/, LPSTR /*name*/, Real /*default*/, Real FAR * /*result*/);
	BOOL (CALLBACK *PANCVT_GetOptionString)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR /*module*/, LPSTR /*section*/, LPSTR /*name*/, LPSTR /*default*/, LPSTR /*result*/, int max);
	BOOL (CALLBACK *PANCVT_SetOptionInt)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR /*module*/, LPSTR /*section*/, LPSTR /*name*/, int /*value*/);
	BOOL (CALLBACK *PANCVT_SetOptionReal)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR module, LPSTR /*section*/, LPSTR /*name*/, Real /*value*/);
	BOOL (CALLBACK *PANCVT_SetOptionString)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPSTR /*module*/, LPSTR /*section*/, LPSTR /*name*/, LPSTR /*value*/);

	BOOL (CALLBACK *PANCVT_OutputBeginPage)(LONG /*ID*/,  struct tagCVTPANX FAR * /*cbs*/, int /*pageIndex*/);
	
	BOOL (CALLBACK *PANCVT_ABORT)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/);

	/*
	** Raster.
	*/
	BOOL (CALLBACK *PANCVT_GetPixel)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LONG /*xpos*/, LONG /*ypos*/, LPDWORD /*pixelvalue*/);
	BOOL (CALLBACK *PANCVT_GetBitLine)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LONG /*xoffset*/, LONG /*yoffset*/, LONG /*nbBytes*/, LPBYTE /*line*/);		  
	
	/*
	** Vector
	*/
	BOOL (CALLBACK *PANCVT_GetPenMapping)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, WORD /*wPen*/, LPWORD /*wToPen*/, Real FAR * /*lprWidth*/, LPWORD /*wSpeed*/);  
	BOOL (CALLBACK *PANCVT_GetNextCommand)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, LPPANMETARECORD /*entity to encode*/);	

	/*
	** PDF
	*/
	BOOL (CALLBACK *PANCVT_GetPDFData)(LONG /*ID*/, struct tagCVTPANX FAR * /*cbs*/, HWND* /*hwnd*/,Real* MrkRotation/*markup rotation*/, int* MrkFlip/*markup flip*/);  

} CVTPANX, FAR *LPCVTPANX;




/*************************************************************************
*******  PAFS Export Filter Entry Points                           *******
**************************************************************************/


#if	TARGET == DOS86 || TARGET == DOS386 || TARGET == UNIX
#define	DECLAREDRIVER(ext)						\
Public	Bool PCALLBACK	PANCVT_Identify_##ext(				\
	LPSTR  szModule,  LPSTR szDesc, LPWORD  wOutputType,		\
	LPWORD  wSubFormats);						\
Public	Bool PCALLBACK	PANCVT_Query_##ext(LONG ID, LPCVTPANX cbs, 	\
	WORD wSubFormat, LPSTR szDesc, 					\
	LPSTR szFilter, LPSTR szExt,					\
	LPWORD  wNumDepths, LPWORD _far *wColorDepths,			\
	LPDWORD  dwOutputCaps,						\
	LPWORD wUnits,    Real _far *Width, Real _far *Height);		\
Public	Bool PCALLBACK PANCVT_ConvertFile_##ext(			\
	LONG ID, LPCVTPANX cvcbs,					\
	WORD wSubFormat, LPCSTR szInFileName, LPCSTR szOutFileName);

DECLAREDRIVER(RADI);
DECLAREDRIVER(BMP);
DECLAREDRIVER(EPSON);
DECLAREDRIVER(EPS);
DECLAREDRIVER(GIF);
DECLAREDRIVER(IBM);
DECLAREDRIVER(DCX);
DECLAREDRIVER(LTX);
DECLAREDRIVER(HPLJ);
DECLAREDRIVER(PCX);
DECLAREDRIVER(PDF);
DECLAREDRIVER(FAX);
DECLAREDRIVER(RLC);
DECLAREDRIVER(TIF);
DECLAREDRIVER(VTC);
DECLAREDRIVER(ADI);
DECLAREDRIVER(DXF);
DECLAREDRIVER(PCI);
DECLAREDRIVER(HPGL);
DECLAREDRIVER(PS);
DECLAREDRIVER(SLD);

#else	/* TARGET == DOS86 || TARGET == DOS386 || TARGET == UNIX */

Public	BOOL PCALLBACK
PANCVT_Identify_DLL(LPSTR szModule, LPSTR szDesc, LPWORD wOutputType, LPWORD wSubFormats); 

Public	BOOL PCALLBACK
PANCVT_Query_DLL(LONG ID, LPCVTPANX cbs,
	WORD wSubFormat, LPSTR szDesc,
	LPSTR szFilter,  LPSTR szExt,
	LPWORD wNumDepths, LPWORD _far *wColorDepths,
	LPDWORD dwOutputCaps, 
	LPWORD wUnits, Real FAR *Width, Real FAR *Height); 

Public	BOOL PCALLBACK
PANCVT_ConvertFile_DLL(LONG ID, LPCVTPANX cvcbs, WORD wSubFormat, LPCSTR szInFileName, LPCSTR szOutFileName); 

#endif	/* TARGET == DOS86 || TARGET == DOS386 || TARGET == UNIX */

/*************************************************************************
******* Conversion Engine Module Private Entryry Points            *******
**************************************************************************/
Public	LPCSTR	GetExportProfile(void);

/*************************************************************************
******* Conversion Engine Entry Points                             *******
**************************************************************************/
Public	Bool	CCALLBACK
LoadPAFSImageDLLS(LPCSTR szProfile, LPCSTR szSection);

Public	Bool	CCALLBACK
UnLoadPAFSImageDLLS(void);

Public	Bool	CCALLBACK
PAFSExport_IdentifyImageDLL(LPWORD lpNumDLLs);

Public	Bool	CCALLBACK
PAFSExport_QueryImageDLL(WORD index, LPWORD lpType, LPSTR lpDesc);

Public	Bool	CCALLBACK
LoadPAFSExportDLLS(LPCSTR szProfile, LPCSTR szSection);

Public	Bool	CCALLBACK
UnLoadPAFSExportDLLS(void);

Public	Bool	CCALLBACK
PAFSExport_Identify(LPWORD  wFormat, LPWORD  wSubFormat, LPWORD  wOutputType);

Public	Bool	CCALLBACK
PAFSExport_QueryFormat(LONG ID,
	LPCVTPANX cvcbs, DWORD wFormat, LPSTR szModule, LPSTR szDesc);

Public	Bool	CCALLBACK
PAFSExport_QuerySubFormat(LONG ID,
	LPCVTPANX cvcbs, DWORD wFormat, DWORD wSubFormat,
	LPSTR szDesc, LPSTR szFilter, LPSTR szExt,
	LPWORD  wNumDepths, LPWORD _far *wColorDepths, LPDWORD  dwOutputCaps,
	LPWORD wUnits, Real _far *Width, Real _far *Height);

Public	BOOL	CCALLBACK
PAFSExport_ConvertFile(LONG ID, LPCVTPANX cvcbs,
	DWORD wFormat, DWORD wSubFormat,
	LPCSTR  szInFileName, LPCSTR szOutFileName);

Public	BOOL	CCALLBACK
PAFSExport_Convert(LPCVTPANX cvcbs,
	DWORD wFormat, DWORD wSubFormat,
	LPCSTR  szInFileName, LPCSTR szOutFileName);
 
Public BOOL CCALLBACK 
PAFSExport_OptimizePalette(LONG index, HANDLE hDIB, RGBQUAD _far *lpColors, WORD nColors);

Public BOOL CCALLBACK 
PAFSExport_ReduceColors(LONG index, HANDLE hDIBIn, HANDLE hDIBOut);

//Public	Bool	CCALLBACK
//PAFSExport_MakeCallbacks(LPCVTPANX cvcbs, LPCSTR szTmpIniFile);

//Public	Bool	CCALLBACK
//PAFSExport_Shell(LPCSTR szTmpIniFile);

//Public	Bool	CCALLBACK
//PAFSExport_FreeCallbacks(LPCVTPANX cvcbs);

#ifdef __cplusplus
}
#endif 
 

//******************************************************************************


#pragma	pack()

#endif	/* PAFSEXPORT_H */
