// -*- C++ -*-
#ifndef _CSI_MARKUP_FONT_
#define _CSI_MARKUP_FONT_

// How small can a font size be allowed to be, in order to avoid
// having zero size fonts
#define MRK_FONT_DOUBLE_MIN_SIZE 1e-4
// This is our reference size, normally this should be otmEMSquare
// (ntmSizeEM in NEWTEXTMETRIC) value extracted from GetOutLineMetrics,
// but this only works for TrueType fonts.
// Most of the TTF are based on a 2048 grid size and are a powers of two.
// This value should get as much resolution as possible from
// the font metrics after rasterization.
// MSDN suggest 65536= 2^16, but that is too big and definitively far more
// than the 16384 TTF max grid size
// 1024 seems a reasonable size.
#define MRK_FONT_REFERENCE_SIZE  1024

// This is required, otherwise std::min/max will not work
#undef max
#undef min

// WARNING: HACK ALERT
// We still have some code that is C and at least one UDE (m_note) that
// requires access to this header file.
// Since such code should migrate at some point in time to C++ this hack
// was implemented.  It should be removed as soon as all code is c++.
// WARNING: DO NOT DECLARE ANYTHING VIRTUAL HERE.
#ifdef __cplusplus
#include <string>
#include <algorithm>
#include <math.h>

class CMrkFont {
private:
#else 
typedef struct _MRK_Font {
#endif
	LOGFONT   m_lf;
	PAN_Point m_fs;
#ifdef __cplusplus
public:
	/**
	 * Create a empty font
	 * font size is set to zero, and logfont is memset to zero
	 */
	CMrkFont() {
		Clear();
	}
	/**
	 * Create a font from a LOGFONT struct
	 * font size in doubles is the same as lfWidth, lfHeight
	 */
	CMrkFont(const LOGFONT& lf) {
		m_lf   = lf;
		m_fs.x = lf.lfWidth;
		m_fs.y = lf.lfHeight;
		m_fs.z = 0;
	}
	/**
	 * Copy constructor
	 */
	CMrkFont(const CMrkFont& mf) {
		m_lf = mf.m_lf;
		m_fs = mf.m_fs;
	}
	/**
	 * font size is set to zero, and logfont is memset to zero
	 */
	void Clear() {
		memset(&m_lf, 0, sizeof(LOGFONT));
		m_fs.x = m_fs.y = m_fs.z = 0;
	}

	/**
	 * Get the LOGFONT associated with this font
	 */
	const LOGFONT& GetLOGFONT() const {
		return m_lf;
	}
	/**
	 * Get the font size
	 * @return the <code>PAN_Point</code> with the size info. 
	 */
	const PAN_Point& GetFontSize() const {
		return m_fs;
	}
	/**
	 * Get the font height.
	 * Usually height and lfHeight are at most 0.5 units apart
	 * from each other.
	 * If using the default height, it will return 0
	 */
	const double GetFontHeight() const {
		return m_fs.y;
	}
	/**
	 * Get the font width.
	 * Usually height and lfHeight are at most 0.5 units apart
	 * from each other.
	 * If using the default width, it will return 0
	 */
	const double GetFontWidth() const {
		return m_fs.x;
	}
	/**
	 * This is a utility call. It will set the integer value of the font
	 * size, without any checks for validity.  It will also propagate
	 * the changes to the double font size.
	 * It should only be used when either h or w has to be zero (values from
	 * the UI)
	 * Otherwise use  SetFontWidth, SetFontHeight, SetFontSize
	 * It will avoid setting any value to zero.
	 * @see MRK_FONT_DOUBLE_MIN_SIZE
	 */
	void SetLogFontSize(int h, int w) {
		m_lf.lfWidth  = w;
		m_lf.lfHeight = h;
		m_fs.x = w;
		m_fs.y = h;
	}

	void SetFontWidth(const double w) {
		m_fs.x = std::max(w, MRK_FONT_DOUBLE_MIN_SIZE);
		m_lf.lfWidth  = std::max(1, (int) (m_fs.x+0.5));
	}

	void SetFontHeight(const double h) {
		m_fs.y = std::max(h, MRK_FONT_DOUBLE_MIN_SIZE);
		m_lf.lfHeight = std::max(1, (int) (m_fs.y+0.5));
	}

	void SetFontSize(const PAN_Point& fs) {
		SetFontSize(fs.x, fs.y);
	}

	void SetFontSize(double x, double y) {
		SetFontWidth(x);
		SetFontHeight(y);
	}
	void ScaleFont(double sx, double sy) {
		SetFontWidth(GetFontWidth() *fabs(sx));
		SetFontHeight(GetFontHeight()*fabs(sy));
	}

	void SetEscapement(int e) {
		m_lf.lfEscapement = e;
	}

	int GetEscapement() {
		return m_lf.lfEscapement;
	}
	void SetOrientation(int e) {
		m_lf.lfOrientation = e;
	}

	int GetOrientation() {
		return m_lf.lfOrientation;
	}

	void SetBold(int bold) {
		m_lf.lfWeight = bold;
	}
	int GetBold() {
		return m_lf.lfWeight;
	}

	void SetItalic(BYTE it) {
		m_lf.lfItalic = it;
	}
	BYTE GetItalic() {
		return m_lf.lfItalic;
	}

	void SetUnderline(BYTE un) {
		m_lf.lfUnderline = un;
	}
	BYTE GetUnderline() {
		return m_lf.lfUnderline;
	}

	void SetStrikeOut(BYTE un) {
		m_lf.lfStrikeOut = un;
	}
	BYTE GetStrikeOut() {
		return m_lf.lfStrikeOut;
	}

	void SetCharSet(int charset) {
		m_lf.lfCharSet = charset;
	}
	int GetCharSet() {
		return m_lf.lfCharSet;
	}

	void SetPitchAndFamily(int paf) {
		m_lf.lfPitchAndFamily = paf;
	}
	int GetPitchAndFamily() {
		return m_lf.lfPitchAndFamily;
	}

	void SetFaceName(const std::string& name) {
		size_t sz = std::max((size_t)LF_FACESIZE, name.size());
		size_t i = 0;
		for ( i = 0; i < sz; i++) {
			m_lf.lfFaceName[i] = name[i];
		}
		if ( i < LF_FACESIZE ) {
			m_lf.lfFaceName[i] = '\0';
		}
	}

	std::string GetFaceName() {
		std::string name;
		GetFaceName(name);
		return name;
	}
	void GetFaceName(std::string& name
					 ) {
		name.clear();
		size_t i = 0;
		for ( i = 0 ; i < LF_FACESIZE; i++) {
			if ( m_lf.lfFaceName[i] == '\0' ) {
				break;
			}
			name += m_lf.lfFaceName[i];
		}
	}

	/**
	 * For two font to be considered identical
	 * The equivalent logfont must match and the font size
	 * must not be more than MRK_FONT_DOUBLE_MIN_SIZE units apart
	 */
	bool operator==(const CMrkFont& f) const {
		return memcmp(&m_lf, &f.m_lf, sizeof(LOGFONT)) == 0 &&
			(fabs(m_fs.x-f.m_fs.x) <= MRK_FONT_DOUBLE_MIN_SIZE) &&
			(fabs(m_fs.y-f.m_fs.y) <= MRK_FONT_DOUBLE_MIN_SIZE);
	}

	/**
	 * For two font to be considered different
	 * The equivalent logfont must not match or the font size
	 * must be more than MRK_FONT_DOUBLE_MIN_SIZE units apart
	 */
	bool operator!=(const CMrkFont& f) const {
		return memcmp(&m_lf, &f.m_lf, sizeof(LOGFONT)) != 0 ||
			(fabs(m_fs.x-f.m_fs.x) > MRK_FONT_DOUBLE_MIN_SIZE) ||
			(fabs(m_fs.y-f.m_fs.y) > MRK_FONT_DOUBLE_MIN_SIZE);
	}
};
#else 
} CMrkFont;
#endif

#endif
