/*
**		Functions external to the Panoramic library
*/
#ifndef PANX_H
#define PANX_H

#include "pan.h"

/*
**	Callback interface as defined in "DESIGN DOCUMENT Version 0.1".
**	When new callbacks need to be added, they should replace
**  the DUMMY placeholders.
*/
#ifdef __cplusplus

#include <string>
#include <boost/any.hpp>
#include "boost/shared_ptr.hpp"
#include "panattr.fwd.h"
#include "geometry/point.fwd.hpp"

namespace csi {
	class ColorMap;
	class IPanAccumulate;
    class IPanMultipleInserter;
	class CPanConfigurationSet;
	class CPanOverlay;
}
class IBuildModel;
class CPanTransform;
class ICommonBody;	
class IPAFSBody;
class CPanTableEntry;
class CPanAttribute;
class CPanPrintSettings;

namespace csi {
class IVCETFontFactory;
class IPanResourceFactory;
class IVCETText;
}

/** Special state value for PANX_State() */
/**
 * State by layer
 * @note A variable value provided by layer
 */
#define STATE_BY_LAYER				(long)-5	/* == 'COLOR_BY_LAYER' */
/**
 * State by block
 * @note A variable value provided by block
 */
#define STATE_BY_BLOCK				(long)-6	/* == 'COLOR_BY_BLOCK' */
/**
 * State by insertion layer
 * @note A variable value provided by block insertion's layer
 */
#define STATE_BY_INSERTION_LAYER	(long)-7
/**
 * State by entity
 * @note A variable value provided by entity
 */
#define STATE_BY_ENTITY				(long)-8

/** State types for PANX_State() */
enum PanStateType {
	StateTypeLineColor,
	StateTypeLineColorAlpha,
	StateTypeFillColorAlpha,
	StateTypeThickness,
	StateTypeLineStyleSetting, // TODO: Available soon
	StateTypeLayer,
	StateTypeClip,
	StateTypeTransform,
	StateTypeFont,
	StateTypeBitmapGlobalAlpha,
	StateTypeBitmapTransparentColor,
	StateTypeFillSettings,
	// Note: StateTypeCount must be the last element
	StateTypeCount
};

/** State actions for PANX_State() */
enum PanStateAction {
	StateActionReset,
	StateActionPush,
	StateActionPop,
	StateActionGet,
	StateActionSet,
	StateActionSetID,
};

/** Clipping operations: typically used with PANX_State() */
enum PanClipOperation {
	ClipOpSet,
	ClipOpAnd,
	ClipOpDiff,
	ClipOpOr,
	ClipOpXor
};

/** Transformation operations: typically used with PANX_State() */
enum PanTransformOperation {
	TransformOpSet,
	TransformOpCombine  /* Left matrix multiplication */
};

enum PanProcessType
{
	PROCESS_LOAD,
	PROCESS_DISPLAY,
	PROCESS_MISC,
	PROCESS_TYPE_SIZE
};

enum PanOptionType
{
	IDENTICAL_PART_DETECTION,
	ANTIALIASING_SUPPORT,
	OPTION_TYPE_SIZE	
};

static const int ANTIALIASING_SUPPORT_NONE		= 0;
static const int ANTIALIASING_SUPPORT_IMAGE		= 1;
static const int ANTIALIASING_SUPPORT_TEXT		= 2;
static const int ANTIALIASING_SUPPORT_GEOMETRY	= 6;
static const int ANTIALIASING_SUPPORT_FULL_IMAGE= 9;

#endif /* __cplusplus */

typedef struct panx {

	/* Common Functions */
	Bool	(CALLBACK *PANX_NOP)(LONG);
	Bool	(CALLBACK *PANX_SetFileType)(LONG, PAN_FileType, DWORD);
	Bool	(CALLBACK *PANX_SetSubFileType)(LONG, DWORD);
	Bool	(CALLBACK *PANX_SetColAttr)(LONG, PAN_Column  *, unsigned int);
	Bool	(CALLBACK *PANX_SetColormap)(LONG, WORD, WORD, WORD, BYTE  *, BYTE  *, BYTE  *);
	Bool	(CALLBACK *PANX_GetRange)(LONG, PAN_FileType, LONG  *, LONG  *, LONG  *, LONG  *);
	Bool	(CALLBACK *PANX_SetExtentsLong)(LONG, LONG, LONG, LONG, LONG);
	Bool	(CALLBACK *PANX_SetExtentsReal)(LONG, Real, Real, Real, Real, Real, Real);
	Bool	(CALLBACK *PANX_GetExtents)(LONG, LPPOINT, LPPOINT, PAN_Point  *, PAN_Point  *);
	Bool	(CALLBACK *PANX_SetNumPages)(LONG, int);
	Bool	(CALLBACK *PANX_SetCachedData) (LONG, LPVOID);
	LPVOID	(CALLBACK *PANX_GetCachedData) (LONG);
	int		(CALLBACK *PANX_GetFormatClassPrivilege)(LONG ID, int eClassName);
	Bool	(CALLBACK *PANX_SetFileProperty)(LONG ID,
                                             const WCHAR16* pKey,
                                             const WCHAR16* pProperty);
	/* Support for setting bookmarks */
	int		(CALLBACK *PANX_SetBookMarks)(LONG ID, int numBookmarks, HPPAN_BOOKMARK pBookMarks);
	/* Support for setting links */
	DWORD	(CALLBACK *PANX_SetLink)(LONG ID, DWORD link, DWORD dwFlags);


	Bool	(CALLBACK *PANX_DUMMY_X05) (void);
	Bool	(CALLBACK *PANX_DUMMY_X04) (void);
	Bool	(CALLBACK *PANX_DUMMY_X03) (void);
	Bool	(CALLBACK *PANX_DUMMY_X02) (void);
	Bool	(CALLBACK *PANX_DUMMY_X01) (void);

	/* Raster Functions */
	Bool	(CALLBACK *PANX_Bitmap)(LONG, LONG, LONG, LONG, BYTE  *, LONG);
	Bool	(CALLBACK *PANX_DUMMY_RS5)(void);
	Bool	(CALLBACK *PANX_DUMMY_RS4)(void);
	Bool	(CALLBACK *PANX_DUMMY_RS3)(void);
	Bool	(CALLBACK *PANX_DUMMY_RS2)(void);
	Bool	(CALLBACK *PANX_DUMMY_RS1)(void);

	/* Vector Functions */
	Bool	(CALLBACK *PANX_Line)(LONG, PAN_Point*, PAN_Point*, LONG);
	Bool	(CALLBACK *PANX_Rect)(LONG, PAN_Point*, PAN_Point*, Bool, LONG, LONG);
	Bool	(CALLBACK *PANX_Poly)(LONG, PAN_Point*, int, LONG, LONG);
	Bool	(CALLBACK *PANX_PolyPolygon)(LONG, PAN_Point*, LPINT, int, LONG, LONG, LONG);
	Bool	(CALLBACK *PANX_Arc)(LONG, PAN_Point*, Real, Real, Real, LONG, LONG, Real);
	Bool	(CALLBACK *PANX_3PtArc)(LONG, PAN_Point*, PAN_Point*, PAN_Point*, LONG, LONG, Real);
	Bool	(CALLBACK *PANX_EllipticArc)(LONG, const PAN_Point*, Real, Real, Real, Real, LONG, LONG, Bool, Real, const PAN_Point*);
	Bool	(CALLBACK *PANX_Ellipse)(LONG, const PAN_Point*, Real, Real, LONG, LONG, Real, const PAN_Point*);
	Bool	(CALLBACK *PANX_VText)(LONG, struct panx*, PAN_Point*, LPCSTR, Real, Real, Real, Real, void*, int, LONG, LPPANRECT);
	Bool	(CALLBACK *PANX_MonoBitmap)(LONG, BYTE*, Real, Real, int, int, int, LONG, LONG, Real);
	Bool	(CALLBACK *PANX_TText)(LONG, LPCSTR, PAN_Point*, PAN_Point*, LPSTR, Real, Real, Real, Real, WORD, LONG, LONG, BOOL, BOOL);
	Bool	(CALLBACK *PANX_TTextSize)(LONG, LPCSTR, LPCSTR, Real, Real, LONG, Real*, Real*);
	Bool	(CALLBACK *PANX_SetMetafileBBox)(LONG, LPRECT);
	Bool	(CALLBACK *PANX_SetMetafileObjects)(LONG, UINT);
	Bool	(CALLBACK *PANX_Points)(LONG ID, 
					const PAN_Point* pPoints, 
					int nPoints,
					enum PanPointStyle eStyle,
					int nSize,
					LONG nLineColor,
					LONG nFillColor,
					PAN_UserPointStyle* pUserStyle);
	Bool	(CALLBACK *PANX_Symbols)(LONG ID, 
					const PAN_Point* pPoints, 
					int nPoints,
					enum PanPointStyle eStyle,
					PAN_Point ptSize,
					LONG nLineColor,
					LONG nFillColor);

#if CSIVER >= 10200
	Bool	(CALLBACK *PANX_ThickPoly) (LONG, PAN_Point*, int, Real, Real, PAN_TVertex*, WORD, LONG, LONG);
#else
	Bool	(CALLBACK *PANX_DUMMY_VC_DUMMY) (void);
#endif

	Bool	(CALLBACK *PANX_PlayMetafileRecord)(LONG, METARECORD  *);
	Bool	(CALLBACK *PANX_PlayMetafile)(LONG, LPCSTR, LPPANRECT, DWORD);

	Bool	(CALLBACK *PANX_SetHandle) (LONG, DWORD, LPPANHANDLE);
	Bool	(CALLBACK *PANX_SetClipRegion) (LONG ID, PAN_Point  *pPoints, int PolyCnt, LPINT pPtCnt, DWORD Flags);
	Bool	(CALLBACK *PANX_DefineLineStyle)(LONG , LPWORD , LPPAN_LineStyle , DWORD );
	Bool	(CALLBACK *PANX_SetLineStyle)(LONG , LPWORD, DWORD, Real );

	BOOL	(CALLBACK *PANX_SetFontEx)(LONG	ID, Int	nFontType, char* szFontName,		
										double	height, double	width, char* szFontFileName,	
										char*	szParameters, PAN_PDFFontMapping* pFontMappingInfo,
										PAN_PDFFontWidths* pCharWidths);
	BOOL	(CALLBACK *PANX_TextEx) (LONG ID, PAN_Point* pPoints, int nPoints, unsigned int* pText, 
									unsigned int *pTextUnicode, DWORD dwMode, LONG color, LONG outlineColor, 
									double charSpacing, double wordSpacing, Hmatrix* pMatrix);
	Bool	(CALLBACK *PANX_SetLayer) (LONG, LPINT, LPINT);

	/* XRef callbacks */
	Bool	(CALLBACK *PANX_SetXRefs)(LONG, WORD, LPPAN_XREF);
	Bool	(CALLBACK *PANX_GetXRefs)(LONG, WORD, LPPAN_XREF);

	BOOL	(CALLBACK *PANX_LocateResource)(LONG ID, UINT type, LPCSTR szResource,
											LPCSTR szPath, LPCSTR szSearchPaths, LPCSTR szHints,
											LPSTR szResolvedPath, UINT cbResolvedPath);

	Bool	(CALLBACK *PANX_DUMMY_VC5) (void);
	Bool	(CALLBACK *PANX_DUMMY_VC4) (void);
	Bool	(CALLBACK *PANX_DUMMY_VC3) (void);
	Bool	(CALLBACK *PANX_DUMMY_VC2) (void);
	Bool	(CALLBACK *PANX_DUMMY_VC1) (void);

	/* Document Functions */
	Bool	(CALLBACK *PANX_Text)(LONG ID, LPSTR lpStr);
	Bool	(CALLBACK *PANX_SetFont)(LONG ID, LPLOGFONT lpFont, const FontMapping* pFontMapping);
	Bool	(CALLBACK *PANX_SetAlignment)(LONG ID, WORD wAlign);
	Bool	(CALLBACK *PANX_SetIndents)(LONG ID, int iLeft, int iRight, int iFirstLine);
	Bool	(CALLBACK *PANX_SetTabs)(LONG ID, LPPAN_TAB lpTabList, int nTabs);
	Bool	(CALLBACK *PANX_SetAttribute)(LONG ID, DWORD dwAttr, DWORD dwColor, int iBlAlign);
	Bool	(CALLBACK *PANX_SetSpacing)(LONG ID, WORD wLine, WORD wParAbove, WORD wParBelow);
	Bool	(CALLBACK *PANX_SetMargins)(LONG ID, WORD wLeft, WORD wTop, WORD wRight, WORD wBottom);
	Bool	(CALLBACK *PANX_SetTable) (LONG ID, PAN_Table *lpTable);
	Bool	(CALLBACK *PANX_SetTableRow) (LONG ID, PAN_Row *lpRow);
	Bool	(CALLBACK *PANX_SetTableCell) (LONG ID, PAN_Cell *lpCell);
	Bool	(CALLBACK *PANX_SetImage)(LONG ID, LPCSTR lpPictureFileName, DWORD fcImage, DWORD cbImage,
				POINT ptCurrDimensions, POINT ptTargetDimensions,
				LPPAN_Edge lpEdge, RECT rcCrop, POINT ptScale, LPSTR lpExt, DWORD dwFlags);
	Bool	(CALLBACK *PANX_SetColumns)(LONG ID, BYTE bkc, WORD ccolM1, WORD dxaColumns, LPPAN_Border lpBorder);
	Bool	(CALLBACK *PANX_SetParagraphBorder)(LONG ID, LPPAN_Edge lpEdge,
				 LPPAN_Border lpbrdBetween, LPPAN_Border lpbrdBar);
	LONG	(CALLBACK *PANX_DefineAPO) (LONG ID, LPPAN_APO lpAPO);
	Bool	(CALLBACK *PANX_DefineDocument) (LONG ID, int nParam, LPSTR *rgTxtPrm);
	Bool	(CALLBACK *PANX_DefineParagraph) (LONG ID, int nParam, LPSTR *rgTxtPrm);
	Bool	(CALLBACK *PANX_DefineSection) (LONG ID, int nParam, LPSTR *rgTxtPrm);
	Bool	(CALLBACK *PANX_AddElement) (LONG ID, int nFromPage, int nToPage, PAN_Element *lpNewElement);
	Bool	(CALLBACK *PANX_InitLayout) (LONG ID, int nPage, int xSize, int ySize);
	Bool	(CALLBACK *PANX_SetTextFlow) (LONG ID, int nPage, FlowID newFlow);
	Bool	(CALLBACK *PANX_SetPageLayout) (LONG ID, int nPage, int nLayoutPage);
	Bool	(CALLBACK *PANX_SetRule) (LONG ID, int nPage, PAN_Rule *lpNewRule);

	/* Database Functions */
	Bool	(CALLBACK *PANX_DBaseRecord)(LONG, BYTE  *, int);
	Bool	(CALLBACK *PANX_DUMMY_DB5) (void);
	Bool	(CALLBACK *PANX_DUMMY_DB4) (void);
	Bool	(CALLBACK *PANX_DUMMY_DB3) (void);
	Bool	(CALLBACK *PANX_DUMMY_DB2) (void);
	Bool	(CALLBACK *PANX_DUMMY_DB1) (void);

	/* Cont' of Document Fucntions */
	Bool	(CALLBACK *PANX_SetColumns1)(LONG ID, BYTE bkc, WORD ccolM1, WORD dxaColumns,
			 LPWORD lpColWidths, LPWORD lpColSpaces, LPPAN_Border lpBorder);
	Bool	(CALLBACK *PANX_SpecialCharacters) (LONG ID, LPSTR lpStr);
	Bool	(CALLBACK *PANX_DrawingObject) (LONG ID, LPPAN_DrawingObj lpDrawObj);
	Bool	(CALLBACK *PANX_SetDocProps) (LONG ID, WORD nProp, void *pProp);

	/* Spreadsheet Functions */
	Bool	(CALLBACK *PANX_SpreadsheetRow)(LONG, BYTE  *, int);
	Bool	(CALLBACK *PANX_SetCell)(LONG, LONG, LONG, LONG, PAN_CellAttrib*);
	Bool	(CALLBACK *PANX_SetCellContents)(LONG, LONG, LONG, LONG, PAN_CellContents*);
	Bool	(CALLBACK *PANX_SetSSProps)(LONG, LONG, WORD, void*);
	Bool	(CALLBACK *PANX_SetFlow)(LONG, LONG);
	Bool	(CALLBACK *PANX_GetSSProps)(LONG, LONG /* sheet*/, LONG /*col*/, LONG /*row*/, WORD, void*);

	/* Archive */
	Bool	(CALLBACK *PANX_ArchiveFile)(LONG, LPSTR, LPSTR, Real, BOOL);
	Bool	(CALLBACK *PANX_SetExtractFunc)(LONG, LONG);
	Bool	(CALLBACK *PANX_DUMMY_AR5) (void);
	Bool	(CALLBACK *PANX_DUMMY_AR4) (void);
	Bool	(CALLBACK *PANX_DUMMY_AR3) (void);
	Bool	(CALLBACK *PANX_DUMMY_AR2) (void);
	Bool	(CALLBACK *PANX_DUMMY_AR1) (void);

	/* Layer callbacks */
	Bool	(CALLBACK *PANX_SetLayers)(LONG, WORD, LPPAN_LAYER, int, DWORD);
	int		(CALLBACK *PANX_GetLayers)(LONG, WORD, LPPAN_LAYER);

	/* Block callbacks */
	Bool	(CALLBACK *PANX_SetBlocks)(LONG, WORD, LPPAN_BLOCK);
	Bool	(CALLBACK *PANX_GetBlock)(LONG,  LPLONG);

	/* View callbacks */
	Bool	(CALLBACK *PANX_SetViews)(LONG, WORD, LPPAN_VIEW);
	Bool	(CALLBACK *PANX_GetView)(LONG,	LPLONG, LPHmatrix);

	/* Option flags */
	int 	(CALLBACK *PANX_GetOptionString)(LONG, LPCSTR /*Section*/, LPCSTR /*optionname*/, LPCSTR /*defaultvalue*/,int /* Max Char to be read */, LPSTR /*result*/, LPCSTR /* Ini FileName or Null */);
	int 	(CALLBACK *PANX_GetOptionInt)(LONG, LPCSTR /*Section*/, LPCSTR /*optionname*/, int /*defaultvalue*/,LPCSTR /*Ini File Name or Null */);

	/* Support for Overlayed files / OLE objects  */
	Bool	(CALLBACK *PANX_SetInsertionPoint)(LONG, PAN_Point  * /*offset*/, PAN_Point  * /*scale*/, PAN_Point  * /*dpi*/, Real /*rotation rad*/, WORD /*flip*/, WORD /*units PAN_UNITXXX*/, LONG /*rgbcolour*/, int /* Current page */);
#if CSIVER >= 10200
	Bool	(CALLBACK *PANX_OverlayImage)(LONG, PANRECT /*placement*/, PAN_OBJECT  * /*olSpec*/);
#else
	Bool	(CALLBACK *PANX_OverlayFile)(LONG, struct panx  * /*cbs*/, DWORD dwObjectID, LPSTR /*filename*/, PAN_Point  * /*point1*/, PAN_Point  * /*point2*/);
#endif
	Bool	(CALLBACK *PANX_OleObject)(LONG, LPCSTR /*ObjectNum*/, int /*ObjType*/, PANRECT /*Placement*/, LONG /*ObjOffset*/, LONG /*cbObj*/);

	/* Support for tiled tiff images */
	Bool	(CALLBACK *PANX_SetTileSize)(LONG, LONG, LONG);

	Bool	(CALLBACK *PANX_SetOptionString)(LONG, LPCSTR /*Section*/, LPCSTR /*optionname*/, LPCSTR /*value*/, LPCSTR /* Ini FileName or Null */);
	Bool	(CALLBACK *PANX_SetOptionInt)(LONG, LPCSTR /*Section*/, LPCSTR /*optionname*/, int /*value*/, LPCSTR /*Ini File Name or Null */);

	Bool	(CALLBACK *PANX_TTextLF)(LONG, LPSTR, LPLOGFONT, LPTTEXTLF);
	Bool	(CALLBACK *PANX_SS_TTextSize)(LONG , LPLOGFONT , Real * , Real  *, LPSTR );

	// Dummy functions
	Bool	(CALLBACK *PANX_DUMMY_VCC_1) (void);
	Bool	(CALLBACK *PANX_DUMMY_VCC_2) (void);
	Bool	(CALLBACK *PANX_DUMMY_VCC_3) (void);

	// C++ interface
#ifdef __cplusplus

	bool	(CALLBACK *PANX_SetDocumentAttribute)(LONG ID,
												  const std::string & attribute_name,
												  const boost::any & attrib_value,
												  csi::IPanAccumulate & attrib_accum);
												  
	// Used to obtain overlay loading attributes
	const CPanAttrValueVector &	(CALLBACK *PANX_GetLoadingAttributes)(LONG ID);

	Bool	(CALLBACK *PANX_BeginModel)(LONG ID,
					 					const std::string & model_name,
										const CPanAttrValueVector & attribs);
 	Bool	(CALLBACK *PANX_EndModel)(LONG ID);

	Bool	(CALLBACK *PANX_SetPrintOptions)(LONG, bool, const CPanPrintSettings*);
    Bool	(CALLBACK *PANX_SetColorMapEx)(LONG, unsigned char, const csi::ColorMap*);

	// Block support
	Bool	(CALLBACK *PANX_BeginBlock)(
					LONG ID,
                    const std::string & table_name,	// For futur use: Just set it to "BLOCK" for now
					const std::string & block_name,
					const CPanAttrValueVector & attribs
					);
	Bool	(CALLBACK *PANX_EndBlock)(LONG ID);
	Bool	(CALLBACK *PANX_InsertBlock)(
					LONG ID,
                    const std::string & table_name,	// For futur use: Just set it to "BLOCK" for now
					const std::string & block_name,
					const CPanAttrValueVector & attribs
					);

	Bool	(CALLBACK *PANX_MultipleInsertBlock)(
					LONG ID,
                    const std::string & table_name,	// For futur use: Just set it to "BLOCK" for now
					const std::string & block_name,
					const CPanAttrValueVector & attribs,
                    const csi::IPanMultipleInserter & inserter
					);


	// High-level entity support (Used for now by ECAD decoders)
	Bool	(CALLBACK *PANX_DefineEntity)(
					LONG ID,
                    const std::string & table_name,
                    const std::string & type,
					const std::string & definition
					);
	Bool	(CALLBACK *PANX_BeginEntity)(
					LONG ID,
                    const std::string & table_name,
					const std::string & entity_name,
					const CPanAttrValueVector & attribs
					);
	Bool	(CALLBACK *PANX_EndEntity)(LONG ID);
	Bool	(CALLBACK *PANX_SetEntityAttribute)(
                    LONG ID,
					const std::string & table_name,
					const std::string & entity_name,
					const std::string & attrib_name,
					const boost::any & attrib_value
					);
	Bool	(CALLBACK *PANX_SetEntityAttributes)(
                    LONG ID,
					const std::string & table_name,
					const std::string & entity_name,
					const CPanAttrValueVector & attribs
					);
	Bool	(CALLBACK *PANX_AddTableEntryEx)(
                    LONG ID,
					const std::string & table_name,
					const std::string & entity_name,
					const CPanAttrValueVector & attribs
					);

	long	(CALLBACK *PANX_State) (LONG ID, PanStateType state, PanStateAction action, int operation, const boost::any & value);

	const boost::shared_ptr<csi::IVCETFontFactory> & (CALLBACK *PANX_CreateFont)(LONG ID);
	const boost::shared_ptr<csi::IVCETText> & (CALLBACK *PANX_MeasureText)(LONG ID);
	const boost::shared_ptr<csi::IVCETText> & (CALLBACK *PANX_SetText)(LONG ID);

	bool (CALLBACK *PANX_SetOption)(LONG ID, PanProcessType process, PanOptionType option, const boost::any & value);

	// 3D Control functions
	Bool	(CALLBACK *PANX_DoesGroupExist)(LONG, DWORD, PanGroupType);
	LONG	(CALLBACK *PANX_BeginGroup)(LONG, DWORD, PanGroupType);
	LONG	(CALLBACK *PANX_EndGroup)(LONG);
	LONG	(CALLBACK *PANX_InsertGroup)(LONG, DWORD, const CPanTransform*);
	Bool	(CALLBACK *PANX_StartSMSession)(LONG, int);
	LONG	(CALLBACK *PANX_AddBody)(LONG, DWORD, ICommonBody*, const CPanTransform*);
	LONG	(CALLBACK *PANX_AddTableEntry)(LONG, DWORD, const CPanTableEntry *, DWORD);
	LONG	(CALLBACK *PANX_SetVecAttribute)(LONG, DWORD, const CPanAttribute *, DWORD);

	/**
	 * Set an attribute for a given entity
	 * @param groupID identifier of the entity
	 * @param attrib_name name of the attribute to set
	 * @param attrib_value value of the attribute
	 */
	Bool	(CALLBACK *PANX_Set3DEntityAttribute)(
                    LONG ID,
					DWORD groupID,
					const std::string & attrib_name,
					const boost::any & attrib_value
					);
	/**
	 * Set a configuration set on VCET control
	 * @param pConfigSet Pointer to configuration set added by pafs
	 */
	Bool  (CALLBACK *PANX_SetConfigurations)(LONG ID, const csi::CPanConfigurationSet *pConfigSet);


	Bool	(CALLBACK *PANX_CreateBody)(LONG /*ID*/,
										LONG /*sm_type*/,
										LONG /*subtype*/,
										LPCSTR, 
										void*,
										void** &,
									    int& /* nNumBodies */,
										IBuildModel*);
	// PMI entity support
    LONG	(CALLBACK *PANX_AddPMIEntity)(LONG /*ID*/,
										  LONG /*type*/,
										  const CPanAttrValueVector& /*attributes*/);

	// PMI entity support
    LONG	(CALLBACK *PANX_InsertPMIEntity)(LONG /*ID*/,
											 DWORD /*parentID*/,
											 DWORD /*groupID*/,
											 const CPanTransform* /*pTransf*/);
	// Return the text size of the given string (for 3D PMI text).
	Bool	(CALLBACK *PANX_TextSize)(LONG /*ID*/,
									  const WCHAR16* /*szText*/,
									  unsigned long /*fontID*/,
									  DWORD /*dwFlags*/,
									  double /*height*/,
									  double /*width (> 0: Absolute width, < 0: Relative width (factor), 0: Default width)*/,
									  const std::vector<csi::Point2DDouble> & insertionPoints,
									  double& /*height*/,
									  double& /*width*/,
									  double& /*descent*/);

	const boost::shared_ptr<csi::IPanResourceFactory> & (CALLBACK *PANX_CreateResource)(LONG /*ID*/);

	Bool	(CALLBACK *PANX_Overlay)(LONG /*ID*/, const csi::CPanOverlay & /*olSpec*/);

#else 
	// Dummy function entries to keep the same panx structure size in case of "C" code 

	Bool	(CALLBACK *PANX_SetDocumentAttribute)(void);
	Bool	(CALLBACK *PANX_GetLoadingAttributes)(void);

	Bool	(CALLBACK *PANX_BeginModel)(void);
 	Bool	(CALLBACK *PANX_EndModel)(void);
	Bool	(CALLBACK *PANX_SetPrintOptions)(void);
    Bool	(CALLBACK *PANX_SetColorMapEx)(void);
	Bool	(CALLBACK *PANX_BeginBlock)(void);
	Bool	(CALLBACK *PANX_EndBlock)(void);
	Bool	(CALLBACK *PANX_InsertBlock)(void);
	Bool	(CALLBACK *PANX_MultipleInsertBlock)(void);

	Bool	(CALLBACK *PANX_DefineEntity)(void);
	Bool	(CALLBACK *PANX_BeginEntity)(void);
	Bool	(CALLBACK *PANX_EndEntity)(void);
	Bool	(CALLBACK *PANX_SetEntityAttribute)(void);
	Bool	(CALLBACK *PANX_SetEntityAttributes)(void);
	Bool	(CALLBACK *PANX_AddTableEntryEx)(void);
	Bool	(CALLBACK *PANX_State) (void);
	Bool	(CALLBACK *PANX_CreateFont)(void);
	Bool	(CALLBACK *PANX_MeasureText)(void);
	Bool	(CALLBACK *PANX_SetText)(void);
	Bool	(CALLBACK *PANX_SetOption)(void);

	Bool	(CALLBACK *PANX_DoesGroupExist)(void);
	LONG	(CALLBACK *PANX_BeginGroup)(void);
	LONG	(CALLBACK *PANX_EndGroup)(void);
	LONG	(CALLBACK *PANX_InsertGroup)(void);
	Bool	(CALLBACK *PANX_StartSMSession)(void);
	LONG	(CALLBACK *PANX_AddBody)(void);
	LONG	(CALLBACK *PANX_AddTableEntry)(void);
	LONG	(CALLBACK *PANX_SetVecAttribute)(void);
	Bool	(CALLBACK *PANX_Set3DEntityAttribute)(void);
	Bool	(CALLBACK *PANX_SetConfigurations)(void);
	Bool	(CALLBACK *PANX_CreateBody)(void);
    Bool	(CALLBACK *PANX_AddPMIEntity)(void);
	Bool	(CALLBACK *PANX_InsertPMIEntity)(void);
	Bool	(CALLBACK *PANX_TextSize)(void);
	Bool    (CALLBACK * PANX_CreateResource)(void);
	Bool	(CALLBACK *PANX_Overlay)(void);

#endif

} PANX, *PPANX,  *LPPANX;


#endif	/* PANX_H */
