/*****************************************************************************

   Copyright (C) 1992, 1993, 1994, 1995, 1996 by Autodesk, Inc.

   Permission to use, copy, modify, and distribute this software in
   object code form for any purpose and without fee is hereby granted,
   provided that the above copyright notice appears in all copies and
   that both that copyright notice and the limited warranty and
   restricted rights notice below appear in all supporting
   documentation.

   AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
   AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
   MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
   DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
   UNINTERRUPTED OR ERROR FREE.

   Use, duplication, or disclosure by the U.S. Government is subject to
   restrictions set forth in FAR 52.227-19 (Commercial Computer
   Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
   (Rights in Technical Data and Computer Software), as applicable.

   DESCRIPTION: Header file for AcadMap "C" call interface

*****************************************************************************/

#if defined(__cplusplus)
extern "C" {
#endif

#ifndef STDIO_H
# define STDIO_H
# include <stdio.h>
#endif

#ifndef ADSLIB_H
# define ADSLIB_H
# include <adslib.h>
#endif

#ifndef ADSERRORS_H
# define ADSERRORS_H
# include <ol_errno.h>
#endif

#include "topoads.h"

/*
 *  Break linear objects that cross boundary edge(s).
 *
 *  Parameters:
 *  sscut       - selection set of objects to cut;
 *  boundary    - boundary entity name of list of points.
 *  skiptopo    - 1 = Skip objects referenced by a topology
 *                0 = Clip objects referenced by a topology
 *  keepod      - 1 = Retains all object data on the resulting objects from
 *                    any cut object
 *                0 = Drops object data on any cut clipped object
 *  ssreturn    - selection set of cut objects or Nil in case of error
 *
 *  Returns RTNORM or RTERROR
 */
    int map_dwgBreakObj (ads_name sscut,
                         ads_name boundary,
                         short skiptopo, 
                         short keepod,
                         ads_name ssreturn);


/*
 *  Trims linear objects which cross boundary edge(s)
 *
 *  Parameters:
 *  ssclip      - selection set of objects to clip;
 *  boundary    - boundary entity name of list of points.
 *  inorout     - 1 = cut inside boundary
 *                0 = cut outside boundary
 *  skiptopo    - 1 = Skip objects referenced by a topology
 *                0 = Clip objects referenced by a topology
 *  keepod      - 1 = Retains all object data on any clipped object
 *                0 = Drops object data on any clipped object
 *  bitflag     - 0 = Delete any non edge objects within or on clip boundary
 *                1 = Ignore any non edge objects within or on clip boundary
 *                2 = Reference the insertion point of any non edge objects
 *                    within or on clip boundary
 *  ssreturn    - selection set of trimmed objects or Nil in case of error
 *
 *  Returns RTNORM or RTERROR
 */
    int map_dwgTrimObj (ads_name ssclip,
                        ads_name boundary,
                        short inorout,
                        short skiptopo,
                        short keepod,
                        short bitflag,
                        ads_name ssreturn);


/*
 *  Get statistics for topology
 *
 *  Parameters:
 *  tpm_id      - Topology identifier, topology should be opened for read
 *
 *  Returns a list of dotted pairs, (variable name . variable value),
 *  or NULL in case of error. The list is defined as follows:
 *  (
 *     (node_count . #nodes)
 *     (link_count . #links)
 *     (polygon_count . #polygons)
 *     ((min_x . #n) (min_y . #n))
 *     ((max_x . #n) (max_y . #n))
 *  ;; The following pairs are attached only for Polygon topologies:
 *     (area_total . #n)
 *     (area_average . #n)
 *     (area_min . #n)
 *     (area_max . #n)
 *     (area_variance . #n)
 *     (area_deviation . #n)
 *     (perimeter_total . #n)
 *     (perimeter_average . #n)
 *     (perimeter_min . #n)
 *     (perimeter_max . #n)
 *     (perimeter_variance . #n)
 *     (perimeter_deviation . #n)
 *  ;; The following pairs are attached only for Network topologies:
 *     (length_total . #n)
 *     (length_average . #n)
 *     (length_min . #n)
 *     (length_max . #n)
 *     (length_variance . #n)
 *     (length_deviation . #n)
 *  )
 *  You have to release this resbuf.
 */
    struct resbuf *map_topoStat(ade_id tpm_id);


/*
 *  Checks if a topology is correct
 *
 *  Parameters:
 *  tpm_id      - Topology identifier, topology should be opened for read
 *
 *  The function audits the geometry of a topology.
 *  This determines if the geometrical relationships defined by the
 *  topology object data are correct.
 *
 *  Returns RTNORM or RTERROR
 */
    int map_topoAudit(ade_id tpm_id);


/*
 *  Completes all objects in a loaded partial topology
 *
 *  Parameters:
 *  tpm_name    - Topology name, topology should be closed
 *  ss          - Entity selection set of all new objects queried
 *
 *  Returns RTNORM or RTERROR
 */
    int map_topoComplete(char *tpm_name, ads_name ss);


/************************************************************************/
/*         Map Closed polylines                                         */
/************************************************************************/

/*
 *  Create closed polylines
 *
 *  Parameters:
 *
 *  topoName - topology name
 *  layer    - layer for closed polylines
 *  group    - group complex polygons toggle
 *  odata    - copy object data toggle
 *  aselink  - copy ase links toggle
 *  ssreturn - selection set containing closed polylines
 *
 *  Returns RTNORM or RTERROR
 */

int map_topoClose( char* topoName, 
                   char* layer, 
                   int group, 
                   int odata, 
                   int aselink,
                   ads_name ssreturn) ;

/************************************************************************/
/*         Map Plotting functions                                       */
/************************************************************************/
/*
 *  called before any other plot routines
 *
 *  Parameters:
 *  none
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltInit (void);


/*
 *  restore original settings (pre map_pltinit)
 *
 *  Parameters:
 *  none
 *
 *  Returns RTNORM or RTERROR
 *
 */
    int map_pltCleanup (void);


/*
 *   plot an entire plot set
 *
 *   Parameters:
 *   name   - Plot set name
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltExecute (char * name);
 

/*
 *  generate plot display for a boundary
 *
 *  Parameters:
 *  bnd      - Boundary object name
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltDisplay (char *bnd);


/*
 *  plot the display
 *
 * Parameters
 * none
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltPlot (void);


/*
 * restore the display (pre map_pltdisplay)
 *
 * Parameters
 * none
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltRestore (void);


/*
 *  read in plot set definitions from dictionary
 *
 * Parameters
 * none
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltDefRead (void);


/*
 *  save plot set definitions to dictionary
 *
 * Parameters
 * none
 *
 *  Returns RTNORM or RTERROR
 */

    int map_pltDefSave (void);


/*
 * return list of available plot set defs
 *
 * Parameters
 * res      pointer to resbuf, return list
 *
 *  Returns result buffer of plot set definitions. List 
 *  should be released.
 */
    int map_pltDefList (struct resbuf ** res);
 

/*
 *  get value of a plot attribute
 *
 * Parameters
 * pn    plot set name
 * attr  plot attribute
 * res   resulting value. Should be released.
 *
 *  The following plot set attributes can be defined:
 *
 *  name    - query name (RTSTR)
 *  desc    - description (RTSTR)
 *  pnam    - plotter name (RTSTR)
 *  pscr    - plot script (RTSTR)
 *  
 *  block   - name of plot layout (RTSTR)
 *  atts    - layout attributes (list of RTSTR)
 *
 *  vlayer  - main viewport layer (RTSTR)
 *  vdispl  - main viewport disable layers (list of RTSTR)
 *                                                  
 *  kflg    - enable key viewport (RTSHORT (flag))
 *  klayer  - key viewport layer (RTSTR (flag))
 *  kdispl  - key display layers (list of RTSTR)
 *  kscl    - key viewport scale factor (double)
 *
 *  dwgs    - list of source drawings (list of RTSTR)
 *  dact    - force active at plot time (RTSHORT (flag))
 *
 *  qcat    - list of query catalogs paired with qnam (list of RTSTR)
 *  qnam    - list of query names (list of RTSTR)
 *
 *  bdwg    - boundary drawing (RTSTR)
 *  bnds    - list of boundaries (list of RTSTR)
 *  blyr    - boundary layer (RTSTR)
 *  bnamt   - boundary name table (RTSTR)
 *  bnamf   - boundary name field (RTSTR)
 *  bodfs   - boundary fields (mapped to layout atts) (list of RTSTR)
 *
 *  sflg    - plot to specifiec scale (RTSHORT (flag))
 *  scale   - plot scale (RTSTR) (eg, 1:2000)
 *  clip    - clip objects against boundary (RTSHORT (flag))
 *  pbnd    - plot boundary (RTSHORT (flag)) (regardless of vdispl setting)
 *  bbuf    - buffer the boundary (RTSHORT (flag))
 *  btyp    - buffer type (RTSTR, "true" or "rect")
 *  bdist   - buffer distance (RTREAL)
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltDefGet (char * pn, char * attr, struct resbuf ** res);


/*
 *  test of plot set is valid. 
 *
 *  Parameters
 *  pn    plot set name
 *
 *  Returns RTNORM if plot set is valid or RTERROR if not. All 
 *  diagnostics can be retrieved from the ADE diagnostics area
 *  
 */
    int map_pltDefValid (char * pn);


/*
 *   extensive verification of a plot set
 *
 *  Parameters
 *  pn    plot set name
 *
 *  Returns RTNORM if plot set is valid or RTERROR if not. All 
 *  diagnostics can be retrieved from the ADE diagnostics area
 *  
 */
    int map_pltDefVerify (char *pn);


/*  delete a plot set
 *
 *  Parameters
 *  pn    plot set name
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltDefDelete (char *pn);


/* 
 *  set the currenty plot set
 *
 *  Parameters
 *  pn    plot set name
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltCurrDef (char *pn);


/*
 *  save the current plot set
 *
 *  Parameters
 *  none
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltCurrSave (void);



/*
 *  get attribute of current plot set
 *
 *  Parameters
 *  attr  plot attribute (the same as map_pltDefGet)
 *  res   resulting value. Should be released.
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltCurrGet (char * attr, struct resbuf ** res);


/*
 *  set attribute of current plot set
 *
 *  Parameters
 *  attr  plot attribute (the same as map_pltDefGet)
 *  val   attribute value
 *
 *  Returns RTNORM or RTERROR
 */
    int map_pltCurrSet (char * attr, struct resbuf * val);


/* 
 *  unset attribute of current plot set (reset attribute to default
 *  value)
 *
 *  Parameters
 *  attr  plot attribute
 *
 *  Returns RTNORM or RTERROR
 */

    int map_pltCurrDel (char * attr);


/*
 *  return list of valid plot layouts (blocks)
 *
 *  Parameters
 *  none
 *
 *  Returns result buffer of blocks
 */
    struct resbuf * map_pltBlkList (void);


/*
 *  return list of valid viewport layers in layout blocks
 *
 *  Parameters
 *  name   layout block name
 *  res    result buffer of viewport layers. Return result buffer must 
 *         released.
 *
 *  Return RTERROR if error occured (invalid block name, invalid
 *         layout block), RTNORM if list initialized
 */
    int map_pltBlkVps (char * name, struct resbuf ** res);


/*
 *  return list of block attributes.
 *
 *  Parameters
 *  name   layout block name
 *  res    result buffer of block attributes.
 *
 *  Return RTERROR if error occured, RTNORM if list initialized
 */
    int map_pltBlkAtts (char * name, struct resbuf ** res);

#if defined(__cplusplus)
}
#endif
