/*    
   adeapp.c

   (C) Copyright 1995-1996 by Autodesk, Inc.
 
   Permission to use, copy, modify, and distribute this software 
   for any purpose and without fee is hereby granted, provided 
   that the above copyright notice appears in all copies and 
   that both that copyright notice and the limited warranty and 
   restricted rights notice below appear in all supporting 
   documentation.
 
   AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.  
   AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
   MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
   DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
   UNINTERRUPTED OR ERROR FREE.
 
   Use, duplication, or disclosure by the U.S. Government is subject to 
   restrictions set forth in FAR 52.227-19 (Commercial Computer 
   Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
   (Rights in Technical Data and Computer Software), as applicable.
*/

/**************************************************************************** 

	Sample ADS RX Application Demonstrating ADE 2.0 ADS RX API 

	Four application interface functions are defined.
	
	1. RUN_SAMPLE_QUERY - defines a drive alias called TUTOR which 
	points to the directory where the tutorial drawings are stored.
	Then it attaches and activates the sample drawings, defines a
	query and then executes the query.

	2. CREATE_TABLE - defines an object data table getting information
	about the fields from the user.

	3. ATTACH_DATA - attaches records of the created table to a selection
	set of objects.
	
	4. SHOW_DATA - allows the user to select an object.  The data associated
	with the selected object is displayed.

****************************************************************************/
#ifndef _MBCS
#error Define _MBCS to compile.
#endif

#include <accmd.h>
#include <adslib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <adeads.h>
#include <rxregsvc.h>
#include <tchar.h>

#define ELEMENTS(array) (sizeof(array)/sizeof((array)[0]))

#define USR_STRG_LENGTH	133
#define DWGNAME_LENGTH	15
#define PNT_STRG_LENGTH	133
#define COORD_LENGTH	30

static char *Tutor = "TUTOR";

/* All Function Prototypes */

extern "C" {
   AcRx::AppRetCode acrxEntryPoint(AcRx::AppMsgCode msg, void* ptr);
}

static int loadfuncs(void);
static int dofun(void);

void create_alias();
void attach_drawings();
void define_query();
void run_sample_query();
void create_table();
void attach_data();
void show_data();

static char *func_table[] = {
	 "C:run_sample_query"
	,"C:create_table"
	,"C:attach_data"
	,"C:show_data"
};


/* The main entry point */
AcRx::AppRetCode acrxEntryPoint(AcRx::AppMsgCode msg, void *appId)
{
   switch (msg) {

   case AcRx::kInitAppMsg:
       acrxRegisterAppMDIAware(appId);
       acrxDynamicLinker->unlockApplication(appId);
       break;

   case AcRx::kLoadDwgMsg:
       loadfuncs();
       break;
       
   case AcRx::kInvkSubrMsg:
       dofun();
   }

   return AcRx::kRetOK;
}


/* LOADFUNCS - Define this application's external functions with AutoLISP */
static int
loadfuncs()
{
	int i;
	int func_num = ELEMENTS(func_table);

   	for (i = 0; i < func_num; i++) 
   	{
		if (!acedDefun(func_table[i], (short)i))
	    	return RTERROR;
   	}
   
   	acutPrintf("Functions loaded.\n");
   	acutPrintf("The following commands are defined:\n\
		you can \"run_sample_query\", \"create_table\", \"attach_data\"\
		,\"show_data\".\n");
   
   	return RTNORM;
}


/* Execute a defined function. */ 
int
dofun()
{
   int id;

   /* Get the function code */
   if ((id = acedGetFunCode()) < 0)
		return RTERROR;

   switch (id) 
   {
   	case 0:
		run_sample_query();
		break;
   	case 1:
   		create_table();
		break;
	case 2:
   		attach_data();
		break;
	case 3:
		show_data();
		break;
   	default:
		break;
   }

   return 1;
}


void 
check_error()
{
 	if ( ade_errqty() )	{
		/* acedAlert( ade_errmsg(0) );	*/
		char *tempStr = ade_errmsg(0);
		if(NULL != tempStr)
		{
			acutPrintf("\n%s\n", tempStr);
		}
		ade_errclear();
	}
}

void 
print_errors()
{
 	int errn;
	int error_count = ade_errqty();

	for (errn=0; errn<error_count; errn++) 
	{
		char *tempStr = ade_errmsg(errn);
		if(NULL != tempStr)
		{
			acutPrintf("\n%s\n", tempStr);
		}
	}

	/* Clear the error message stack... */
	ade_errclear();
}


/* Function to create Alias specified in "Tutor" for Tutorial Directory */
void
create_alias(char *dir_loc)
{
	int retCode = 0;


	/* Ask the user for the location of the Tutorial Directory */

	acedGetString(TRUE, "\nTutorial Directory Location: ", dir_loc);

	/* now create an alias called TUTOR for this directory... */

	retCode = ade_aliasadd (Tutor, dir_loc);

	if (retCode != RTNORM) {
		acutPrintf("\nError creating alias called %s.\n", Tutor);
		char *tempStr = ade_errmsg(0);
		if(NULL != tempStr)
		{
			acutPrintf("\n%s\n", tempStr);
		}
		ade_errclear();
		acedRetNil();
		return;
	}
	else {
		acutPrintf("\nAlias %s was successfully created.\n", Tutor);
		acutPrintf("%s = %s\n", Tutor, dir_loc);
		acedRetT();	
	}
}


void
attach_drawings(char *tutor_loc)
{
	struct resbuf *dwg_list, *prev_val, *set_val, *cur_buf = NULL;
	char dwg_path[USR_STRG_LENGTH + DWGNAME_LENGTH];
	char dwg_alias[USR_STRG_LENGTH + DWGNAME_LENGTH];
	ade_id dId = 0;

	/* get the list of all files in the tutorial directory */

	dwg_list = ade_osfexpand (tutor_loc, "dwg", "~tblock*");
	
	if (dwg_list == NULL) {
	 	acedAlert("No drawings found in the tutorial directory.");
	}

	/* set the preference to activate on attach....	*/

	prev_val = ade_prefgetval("ActivateDwgsOnAttach");
	set_val = acutNewRb(RTT);
	ade_prefsetval("ActivateDwgsOnAttach", set_val);
	
	for (cur_buf = dwg_list; cur_buf != NULL; cur_buf = cur_buf->rbnext) 
	{
		strcpy(dwg_path, tutor_loc);
		strcat(dwg_path, "\\");
		strcpy(dwg_alias, Tutor);
		strcat(dwg_alias, ":\\");

		/* Check if each drawing has already been attached. */
 
		strcat(dwg_path, cur_buf->resval.rstring);
		strcat(dwg_alias, cur_buf->resval.rstring);
		dId = ade_dwggetid(dwg_alias);
		if (dId == 0) {
			acutPrintf("\nAttaching drawing: %s\n", dwg_alias);
			ade_errclear();
			ade_dsattach(dwg_alias);	
			check_error();
		}
		else { /* activate drawing */
		 	acutPrintf("\nActivating drawing: %s\n", dwg_path);
			ade_dwgactivate(dId);
		}
	}  
	ade_prefsetval ("ActivateDwgsOnAttach", prev_val);
    acedRetT();	
}


/* 
   Checks the given alias name: 
   if it's found returns the correspondent path, otherwise returns NULL 
*/
int 
alias_find(char *alias_name, char *alias_loc)
{
 	struct resbuf *alias_list, *cur_def = NULL;
	int retCode = FALSE;

	alias_list = ade_aliasgetlist();

	if (alias_list == NULL) 
		return retCode;

	for (cur_def = alias_list; cur_def != NULL; cur_def = cur_def->rbnext) 
	{
		if (cur_def->restype == RTLB) {
		 	cur_def = cur_def->rbnext;
			if ( !strcmp(cur_def->resval.rstring, alias_name) )	{
				strcpy (alias_loc, cur_def->rbnext->resval.rstring);
				retCode = TRUE;
			}
		} 	
	}
	acutRelRb(alias_list);	
	return retCode;
}


/* 
 Define a query to retrieve all objects inside a window
 which are on layer STREETS.
*/
void
define_query()
{
	int retCode = 0;
	ads_point pt1, pt2;
	struct resbuf *qrycond1, *qrycond2;

	/* Clear out the current query */

	ade_qryclear();

	/* Have the user specify the Window first... */

	ade_dwgzoomextents();
    acutPrintf("\nDefine a window to locate queried objects...");
	acedInitGet(RSG_NONULL, NULL);
    acedGetPoint(NULL, "\nFirst corner: ", pt1);
	acedInitGet(RSG_NONULL + RSG_DASH, NULL);
	acedGetCorner(pt1, "\nOther corner: ", pt2);

	/* now define a condition using these two points... */

	qrycond1 = acutBuildList(
	  			RTLB,
	  			RTSTR,"window", RTSTR,"crossing", RTPOINT,pt1, RTPOINT,pt2,
	  			RTLE, 0);

	ade_qrydefine ("", "", "", "Location", qrycond1, "");

	/* now add the layer condition... */
	qrycond2 = acutBuildList(
	  			RTLB,
	  			RTSTR,"layer", RTSTR,"=", RTSTR,"STREETS",
	  			RTLE, 0);

	ade_qrydefine ("AND", "", "", "Property", qrycond2, "");
	
}


/* Application interface functions */

void
run_sample_query()
{
	char tutor_loc[USR_STRG_LENGTH];
	int queried_stat;

	/* Create alias if not already created... */

	if (alias_find(Tutor, tutor_loc) == FALSE) {
		create_alias(tutor_loc);
	}
	else {
	 	acutPrintf("\nAlias %s is already defined as %s\n", Tutor, tutor_loc);
	}
		
	/* Attach and activate all drawings in the tutorial directory */

	attach_drawings(tutor_loc);

	/* define and execute a query */

	define_query();

	/* execute the query */
	
	acutPrintf("\nExecuting query...\n");

	queried_stat = (int)ade_qryexecute();

	if (queried_stat == 0)
		acutPrintf("\nNo objects were found which cross "
            "the specified window.\n");
	else
		acutPrintf("\n%d objects were found which cross "
            "the specified window.\n", queried_stat);

    acedRetVoid();
}


/*
 * Gets struct resbuf *(*defn).
 * Builds the definition ("name" . Value).
 * Upon completion: (*defn) points to the RTLE resbuf.
 */
void 
create_defn(int Code, char *name, 
		struct resbuf **defn, struct resbuf **value, char *strval)
{
	struct resbuf *curdefn = *defn;

	curdefn->rbnext = acutNewRb(RTLB);
	curdefn = curdefn->rbnext;

	curdefn->rbnext = acutNewRb(RTSTR);
	curdefn = curdefn->rbnext;
	curdefn->resval.rstring = (char *)malloc( sizeof(char)*
										(strlen(name)+1) );
	strcpy(curdefn->resval.rstring, name);

	curdefn->rbnext = acutNewRb(RTDOTE);
	curdefn = curdefn->rbnext;

	curdefn->rbnext = acutNewRb(Code);
	curdefn = curdefn->rbnext;
	*value = curdefn;
	if (Code == RTSTR) {
		curdefn->resval.rstring = (char *)malloc( sizeof(char)*
										(strlen(strval)+1) );
		strcpy(curdefn->resval.rstring, strval);
	}	
 	curdefn->rbnext = acutNewRb(RTLE);		
	*defn = curdefn->rbnext;	
}


int 
create_col_defn(struct resbuf **curcoldefn)
{
  	char colname[USR_STRG_LENGTH] = ""; 
  	char coldesc[USR_STRG_LENGTH] = "";
  	char coltype[USR_STRG_LENGTH] = "" ;
	char usr_input[USR_STRG_LENGTH] = "";
	char point_str[PNT_STRG_LENGTH];
	char coord_str[COORD_LENGTH];	
	struct resbuf *curdefn, *value;
	int val;
	ads_point point_value;
	
	curdefn = *curcoldefn;

	/* open column defn parenth (...  */
	curdefn->rbnext = acutNewRb(RTLB);
	curdefn = curdefn->rbnext;

	/* get column name */
	acedGetString(FALSE, "\nColumn Name: ", colname);
	if (!strcmp(colname, ""))
		return FALSE;
	
	create_defn(RTSTR, "colname", &curdefn, &value, colname);
	/* curdefn now points to RTLE resbuf */ 	
		
	/* get column description */
	acedGetString(TRUE, "\nDescription: ", coldesc);
	create_defn(RTSTR, "coldesc", &curdefn, &value, coldesc);	
	
	/* get column type */
	acedInitGet(0, "Real Integer Point Character");
	acedGetKword("\nType:Real/Integer/Point/<Character>: ",coltype);
	if (!strcmp(coltype, ""))
		strcpy(coltype, "Character");

	create_defn(RTSTR, "coltype", &curdefn, &value, coltype);
	
	/* get default value */
	if ( !strcmp(coltype, "Real") )
	{
		create_defn(RTREAL, "defaultval", &curdefn, &value, NULL);
	 	acedInitGet(1, NULL);
	 	acedGetReal("\nDefault value: ", &(value->resval.rreal));	
	}
	else if ( !strcmp(coltype, "Integer") ) 
	{
		create_defn(RTSHORT, "defaultval", &curdefn, &value, NULL);
		acedInitGet(1, NULL);
	 	acedGetInt("\nDefault value: ", &val);
	 	value->resval.rint = val;	
	}
	else if ( !strcmp(coltype, "Point") ) 
	{
		acedInitGet(1, NULL); 
	 	acedGetPoint(NULL, "\nDefault point: ", point_value);
		// convert point to the coordinates string: point_str
		acdbRToS(point_value[0],2,2,coord_str);
		strcpy(point_str, coord_str);
		strcat(point_str, ",");
		acdbRToS(point_value[1],2,2,coord_str);
		strcat(point_str, coord_str);
		strcat(point_str, ",");
		acdbRToS(point_value[2],2,2,coord_str);
		strcat(point_str, coord_str);
		
	 	create_defn(RTSTR, "defaultval", &curdefn, &value, point_str);	
	}
	else if ( !strcmp(coltype, "Character") ) 
	{
	 	acedGetString(TRUE, "\nDefault value: ", usr_input);
		create_defn(RTSTR, "defaultval", &curdefn, &value, usr_input);	
	}
	/* close column def parenth ...)  */
	curdefn->rbnext = acutNewRb(RTLE);
	curdefn = curdefn->rbnext;

	*curcoldefn = curdefn;

	return TRUE;		
}


void
create_table()
{
	char tabname [USR_STRG_LENGTH] = "";
	char tabdesc [USR_STRG_LENGTH] = "";
	char usrinput [USR_STRG_LENGTH] = "";
	int retCode = FALSE;
	struct resbuf *callist = NULL, *curcoldefn = NULL, *columns = NULL;
	struct resbuf *curpos, *value = NULL;
	struct resbuf *table_def;

	/* get the name of the table to create ... */

	acedGetString(FALSE, "\nEnter name of table <SAMPLE>: ", tabname); 
	if ( !strcmp(tabname, "") )
		strcpy(tabname, "SAMPLE");

	/* get the table description */

	acedGetString(TRUE, "\nDescription: ", tabdesc);
	if ( !strcmp(tabdesc, "") )
		strcpy(tabdesc, tabname);

	/* start the column list: ( "columsn" ... */
	curcoldefn = callist = acutNewRb(RTLB);
	curcoldefn->rbnext = acutNewRb(RTSTR);
	curcoldefn = curcoldefn->rbnext;
	curcoldefn->resval.rstring = (char *)malloc( sizeof(char)*
										(strlen("columns")+1) );
	strcpy(curcoldefn->resval.rstring, "columns");
	columns = curcoldefn; 

	/* get columns... */
	while ( (retCode = create_col_defn(&curcoldefn)) == TRUE );

	if (columns->rbnext	== NULL) {
		acutPrintf("\nNo columns were entered.");
		acutRelRb(callist);
		acedRetNil();
		return;
	}
  
	curcoldefn->rbnext = acutNewRb(RTLE);

	curpos = table_def = acutNewRb(RTLB);
	create_defn(RTSTR, "tablename", &curpos, &value, tabname);	
	create_defn(RTSTR, "tabledesc", &curpos, &value, tabdesc);
	
	curpos->rbnext = callist;

	for(curpos = table_def; curpos->rbnext != NULL; curpos = curpos->rbnext);
	curpos->rbnext = acutNewRb(RTLE);;

//	acedRetList(table_def);
	
	retCode = ade_oddefinetab(table_def);

	acutRelRb(table_def); 

	if (retCode == RTNORM) { 
		acutPrintf("\nTable successfully created.\n");
		acedRetT();
	}
	else { 
	/* else errors occured so print them out... */ 
		print_errors();	
		acedRetNil();	
	} 	
} 


void
attach_data()
{
	char tabname [USR_STRG_LENGTH] = "", tabprompt [USR_STRG_LENGTH];
	int retCode = FALSE;
	struct resbuf *tab_list = NULL, *curpos;
	ads_name ss, ename;
	long sslen, sscur;

	tab_list = ade_odtablelist();
	if (tab_list == NULL) {
		acedAlert("No Object Tables are defined yet.\n");
		return;
	}

	acutPrintf("\nAvailable tables: \n");
	for(curpos = tab_list; curpos != NULL; curpos = curpos->rbnext)
	{
		acutPrintf("\t%s\n",curpos->resval.rstring);
	}

	/* Specify table to attach... */
	
	sprintf(tabprompt, "\nSelect table < %s >: ",tab_list->resval.rstring);
	acedGetString(FALSE, tabprompt, tabname);
	if ( !strcmp(tabname, "") )
		 strcpy(tabname, tab_list->resval.rstring);

	acutPrintf("\nYou have specified table \"%s\"\n", tabname);
		 
	/* have the user select objects */
	
	acutPrintf("\nSelect object to attach data to...");
	retCode = acedSSGet(NULL, NULL, NULL, NULL, ss); 
	if (retCode != RTNORM) {
		acutPrintf("\nNo objects selected.\n");
		acutRelRb(tab_list);
		acedSSFree(ss);
		return;
	}
	
	/* attach records to the table */

	acedSSLength(ss, &sslen);
	for (sscur = 0; sscur < sslen; sscur++)
	{
		/* get entity name of next object */

		acedSSName(ss, sscur, ename);

		/* attach a new record to the object... */

		if (ade_odaddrecord (ename, tabname) != RTNORM)
			acutPrintf("\nCannot attach the table \"%s\"\n", tabname);
	}
	
	if ( ade_errqty() > 0 )
		print_errors();
	else
	 	acutPrintf("\nSuccessfully attached data to %d object(s)\n", sslen);
	 
	acutRelRb(tab_list); 
	acedSSFree(ss);
	acedRetT();
}


/* Returns pointer to RTSTR resbuf node, containing given name */ 
struct resbuf*  
find_resbuf(char *name, struct resbuf *list)
{
 	struct resbuf *curbuf = NULL;

	if (list == NULL) 
		return NULL;

	for (curbuf = list; curbuf != NULL; curbuf = curbuf->rbnext) 
	{
		if (curbuf->restype != RTSTR)
			continue;
			  
		if ( !_tcsicmp(curbuf->resval.rstring,name) )
		{
		 	return curbuf;
		} 	
	}					 
	return NULL;				 
}

void
show_columns(struct resbuf *collist, char* tabname, int recnum, ads_name ename)
{
 	struct resbuf *column, *field, *fldvalue;
	char *fldname;
	int code;

	while (collist != NULL)
	{
 		column = find_resbuf("colname", collist);
		if (column == NULL)
				break;

		
		field = column->rbnext;
		if (field != NULL && field->restype != RTSTR)
			  field = field->rbnext;
		if (field == NULL || field->restype != RTSTR) {
			check_error();
			continue;
		}
		fldname = field->resval.rstring;

		fldvalue = ade_odgetfield(ename, tabname, fldname, recnum);
		if (fldvalue == NULL) {
			check_error();
			break;
		}
			
		acutPrintf("\n\tColumn \"%s\": ", fldname);
		code = fldvalue->restype;
		switch (code) {
		 case RTREAL:
			acutPrintf("%f", fldvalue->resval.rreal);
		 	break;
		 case RTSHORT:
		 	acutPrintf("%d", fldvalue->resval.rint);
		 	break;
		 case RTSTR:
		 	acutPrintf("%-s", fldvalue->resval.rstring);
		 	break;
		 case RTPOINT:	
		 	acutPrintf("%f, %f, %f", fldvalue->resval.rpoint[1],
		 					fldvalue->resval.rpoint[2],
		 					fldvalue->resval.rpoint[3]);
		 	break;
		}
		acutRelRb(fldvalue);
		collist = field;
	}
}

void
show_data()
{
	//char tabname [USR_STRG_LENGTH] = "";
	char *tabname;
	int retCode = FALSE;
	struct resbuf *tab_list = NULL, *curpos, *tab_defn;
	ads_name  ename;
	ads_point spt;
	int rec_count, rec;

	/* have the user select one object */

	retCode = acedEntSel(NULL, ename, spt); 
	if (retCode != RTNORM) {
		acutPrintf("\nNo object selected.\n");
		return;
	}

	/* get the names of the tables which have records attached
	   to the object... */

	tab_list = ade_odgettables (ename);
	if (tab_list == NULL) {
		acedAlert("No Object Tables are attached to the object selected.\n");
		return;
	}

	for(curpos = tab_list; curpos != NULL; curpos = curpos->rbnext)
	{
		tabname = curpos->resval.rstring;
		acutPrintf("\nData for table: \"%s\"\n",tabname);

		/* show the number of records attached to the object for that table */

		rec_count = ade_odrecordqty(ename, tabname);
		if (rec_count == RTERROR)
			check_error();
			
		acutPrintf("\n%d  record(s) are attached.\n", rec_count);

		/* get the definition of the table */

		tab_defn = ade_odtabledefn(tabname);

		/* show the value of each column for all records from this table... */

		for (rec = 0; rec < rec_count; rec++)
		{
			acutPrintf("\n\nRecord %d:", rec + 1);
			show_columns(tab_defn, tabname, rec, ename);	
		}
		acutPrintf("\n");
	}	
	acutRelRb(tab_list); 
	acedRetT();
}
