/*****************************************************************************
      
    PLOT.C

    (C) Copyright 1996 by Autodesk, Inc.

    This program is copyrighted by Autodesk, Inc. and is  licensed
    to you under the following conditions.  You may not distribute
    or  publish the source code of this program in any form.   You
    may  incorporate this code in object form in derivative  works
    provided  such  derivative  works  are  (i.) are  designed and 
    intended  to  work  solely  with  Autodesk, Inc. products, and 
    (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright  
    1988-1999 by Autodesk, Inc."

    AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
    AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
    CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
    DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
    UNINTERRUPTED OR ERROR FREE.

    Written by: Bernard Sastourne
                Milton Hagler
    Date: July 1996

*****************************************************************************

    Description: 

    This routine demonstrates running a plot using ADS and the Map API.
    The user is given two options for how to plot. One that displays the 
    potential  plot before being plotted and a second that executes the
    plot immediately.

    Three Map tutorial drawings - parcels7.dwg, houses7.dwg and tblock.dwg -
    are necessary for this example. The title sheet is contained within 
    tblock.dwg and the others are used as map source drawings. A boundary
    drawing is created on the fly and saved to the Map tutorial directory.
    
    The tutorial drawings must be located in the \acadmap\maptut\tutorial VI 
	directory. The code will find the AutoCAD executable and construct a path 
	to this directory.

    Usage: 
    
    After compiling plot.cpp into an executable, load it into AutoCAD.
    Type TPM_PLOT at the Command prompt to execute it. When prompted by
    ADE to export object data, answer yes.
      
*****************************************************************************

   Clarification to Function: map_pltDefGet

      "bdwg"   RTSTR  
               A specific boundary drawing to use.

      "blyr"   RTSTR
               The name of the layer the boundary is placed on. It can be
               any layer from the boundary drawing. 

      "bnamt"  RTSTR
               The name of the OD table attached to a boundary object.

      "bnamf"  RTSTR
               The name of a field to use, from the table assigned in 
               "bnamt", for the attribute "bnds". 

      "bnds"   RTSTR List
               Which boundary (or boudaries) to use by providing the value 
               for the field assigned in "bnamf". 

      "bodfs"  RTSTR List
               The list of field names to be mapped to attributes in the 
               title block. The title block is assigned using the attributes 
               from the block assigned to "block". 

               Mapping of object data fields to block attributes is managed by
               the two plot attributes "bodfs" and "atts". The mapping is best 
               explained by example.

               1. Lets suppose that boundary object data table, defined by the 
                  "bnamf" attribute, has the following fields:
                    Field1
                    Field2
                    Field3 
                    Field4
               
               2. Lets suppose that layout block, defined by  the "block" 
                  attribute, has the following attributes:
                    Attr1
                    Attr2
                    Attr3
                    Attr4
                    Attr5

               3. Let suppose the uer wants value of Field1 to appear in place 
                  of Attr2, value of Field2 in place of Attr3, and value of 
                  Field4 in place of Attr1. To define this mapping we must set
                  "bodfs" and "atts" attributes as follows: 
               
                    bodfs = ("field1"    "field2"    "field4")
                                |           |           |
                                V           V           V
                    atts  = ("attr2"     "attr3"     "attr1")

                  When plotting is done, values of the fields from the object
                  data table attached to the boundary's closed polyline will 
                  be assigned to the specified block attributes. The number of 
                  elements in each list for "bodfs" and "atts" must be the 
                  same.

      "kflg"   RTSHORT
               Enables the reference viewport by setting the attribute value to
               RTT, and disables it by setting the value to RTNIL. There are 
               two dependent attributes "klayer" and "kdispl".

      "klayer" RTSTR
               The layer of the reference viewport, that must be a different 
               layer from the layer used for the main viewport (attribute 
               "vlayer"). If the function map_pltBlkVps returns only one 
               viewport layer, it should be used for the main viewport.

      "kdispl" RTSTR List
               A list of layers to be displayed in the referenced viewport. 
               This should not be void if "kflg" is enabled. Valid layers are
               all the layers returned by ade_dsproplist using the option
               "layer".


***************************************************************************************/
#include <string.h>
#include <stdio.h>
#include "adslib.h"
#include "aced.h"
#include "mapads.h"
#include <dbapserv.h>
#include <acdblmgr.h>			//For the AcDbLayoutManager stuff
#include <tchar.h>

/*
   DEFINES
*/
#define ELEMENTS(array) (sizeof(array)/sizeof(array)[0])

extern "C" {
	
	AcRx::AppRetCode acrxEntryPoint(AcRx::AppMsgCode msg, void*);
}


#ifndef SEPARATOR
#ifdef UNIX
#define SEPARATOR '/'
#else
#define SEPARATOR '\\'
#endif
#endif

#ifndef PROGRAM
#ifdef UNIX
#define PROGRAM "acad"
#else
#define PROGRAM "acad.exe"
#endif
#endif

/*
    FUNCTION DECLARATIONS
*/
static void DefinePlotSet(void);
static void CreateBoundaryDwg(char*);
static void DefineBoundaryTable(ads_name);
static void AttachDrawings(char*);
static void DefineQuery(void);
static void EndPlot(void);
static void GetAcadPath(char*);
static void PrintErrors(char*);
static int  LoadFunctions (void);
static int  ExecuteFunction (void);
Adesk::Boolean GenerateLayoutName(char* pLayoutName, unsigned int nBuflen);
/*
    STATIC VARIABLES                                          
*/
static struct {
  char *name; 
  void (*func)();
} func_table[] = { "C:TPM_PLOT", DefinePlotSet };

static char name[15]      = "Plot_Set";      /* plot set name     */
static char alias[15]     = "My_Disk";       /* alias name        */
static char category[15]  = "category";      /* category name     */
static char query[15]     = "query1";        /* query name        */
static char btable[20]    = "BOUNDARIES";    /* boundary OD table */
static char bfield[20]    = "Name";          /* boundary OD field */
static char bname[20]     = "My City";       /* boundary name     */
static char blayer[33]    = "boundary";      /* boundary layer    */
static char clayer[33]    = "";              /* current layer     */

const char* kpszBoundaryDrawingName = "bndry.dwg";
const char *kpszParcelDrawingName = "parcels7.dwg";
const char *kpszHousesDrawingName = "houses7.dwg";
const char *kpszTitleBlockDrawingName = "tblock.dwg";

/**********************************************************************
 * Function: DefinePlotSet
 *
 * Main routine for controlling the plotting, that creates the 
 * boundary drawing and creates the layout block, attaches the 
 * drawings to plot, defines the query to use to get the objects
 * to plot from the source drawings and sets up the plot attributes.
 *
 */
void DefinePlotSet ()
{
  struct resbuf rbval;
  struct resbuf *rb = NULL;
  struct resbuf *rb1 = NULL;
  char block_name[512], vlayer[512];
  char str1[132], str2[132];
  char answer[33], acad_path[512];
  int result;
  ads_name en;

  /* Get current layer */
  result = acedGetVar("CLAYER", &rbval);
  if( result == RTNORM )
    strcpy(clayer, rbval.resval.rstring);

  /*
     Setup Boundary drawing, attach source
     drawings and define a query.
  */
  GetAcadPath(acad_path);
  CreateBoundaryDwg(acad_path);                       
  AttachDrawings(acad_path);
  DefineQuery();

  /* Insert the title block */
  sprintf(str1, "%s%s", acad_path, kpszTitleBlockDrawingName);
  acedCommand(RTSTR, "._insert", RTSTR, str1, RTSTR, "0,0", RTSTR, "", 
              RTSTR, "", RTSTR, "", RTSTR, "", 0);
  acdbEntLast(en);
  acdbEntDel(en);

  /* /////////////////// Start Map Plot DCL /////////////////////////// */

  /* Define a new plot set */
  if (map_pltCurrDef (name) != RTNORM)
    PrintErrors("ERROR: Unable to define a new plot set.");
                   
  /* Set description */
  rb = acutBuildList (RTSTR, "Map Plot API Example", NULL);
  if (map_pltCurrSet ("desc", rb) != RTNORM)
    PrintErrors("ERROR: Unable to define a plot description.");
  acutRelRb (rb);

  /* /////////////////// Plot Layout Button  ////////////////////////// */

  /*
     Get the list of defined blocks in current drawing.
     And, choose a block to use.
  */
  if ((rb1 = map_pltBlkList ()) == NULL) 
    PrintErrors("ERROR: block layout creation failed.");
  else
    strcpy (block_name, rb1->resval.rstring);              /*  TBLOCK (only one block) */
  acutRelRb (rb1);

  /* Set plot/title block */
  rb = acutBuildList (RTSTR, block_name, NULL);
  if (map_pltCurrSet ("block", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set a plot block.");
  acutRelRb (rb);

  /* Get the block's attributes and assign them to Plot engine */
  rb = NULL;
  if (map_pltBlkAtts (block_name, &rb) != RTNORM) 
    PrintErrors("ERROR: Unable to get block attributes.");
  else {
    if (map_pltCurrSet ("atts", rb) != RTNORM)
      PrintErrors("ERROR: Unable to set block attributes.");
  }
  acutRelRb (rb);

  /*
     Get the block's viewport layers, the first one will be main viewport.
  */
  if (map_pltBlkVps (block_name, &rb1) != RTNORM) 
    PrintErrors("ERROR: Unable to get block viewports.");
  else
    strcpy (vlayer, rb1->resval.rstring);
  acutRelRb (rb1);

  /* Main Viewport layer */
  rb = acutBuildList (RTSTR, vlayer, NULL);
  if (map_pltCurrSet ("vlayer", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set main viewport layer.");
  acutRelRb (rb);

  /* Set "kflg" to ON to have reference viewport */
  rb = acutBuildList (RTT, NULL);
  if (map_pltCurrSet ("kflg", rb) != RTNORM) 
   PrintErrors("ERROR: Unable to enable key viewport.");
  acutRelRb (rb);
  
  /* Reset reference viewport layer */
  rb = acutBuildList(RTSTR, "REF_VPORT", 0);
  if (map_pltCurrSet ("klayer", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to diable key viewport layer.");
  acutRelRb(rb);
  
  /* Set the list of layers to display from the source drawings */
  rb = acutBuildList (RTSTR, "PARCELS", RTSTR, "STREETS", NULL);
  if (map_pltCurrSet ("kdispl", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set key display layers.");
  acutRelRb(rb);
  

 /* ///////////////////// Source Drawings Button //////////////////// */

  /* Source drawings */
  sprintf(str1, "%s:\\%s", alias, kpszParcelDrawingName);
  sprintf(str2, "%s:\\%s", alias, kpszHousesDrawingName);
  rb = acutBuildList (RTSTR, str1, RTSTR, str2, NULL);
  if (map_pltCurrSet ("dwgs", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set list of plot drawings.");
  acutRelRb (rb);

  /* Query category */
  rb = acutBuildList (RTSTR, category, NULL);
  if (map_pltCurrSet ("qcat", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set query catalog.");
  acutRelRb (rb);

  /* Query name */
  rb = acutBuildList (RTSTR, query, NULL);
  if (map_pltCurrSet ("qnam", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set query name.");
  acutRelRb (rb);


  /* //////////////////////// Boundaries Button //////////////////////// */

 
  /* Boundary drawing to use */
  sprintf(str1, "%s:\\%s", alias, kpszBoundaryDrawingName);
  rb = acutBuildList (RTSTR, str1, NULL);
  if (map_pltCurrSet ("bdwg", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set boundary drawing name.");

  /* List of OD values attched to boundaries in BOUNDARY.DWG */
  rb = acutBuildList (RTSTR, bname, NULL);        //My City
  if (map_pltCurrSet ("bnds", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set list of boundaries.");
  acutRelRb (rb);

  /* Layer of the boundary built in function CreateBoundaryDwg */

  rb = acutBuildList (RTSTR, blayer, NULL);
  if (map_pltCurrSet ("blyr", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set boundary layer.");
  acutRelRb (rb);

  /* Boundary OD table */
  rb = acutBuildList (RTSTR, btable, NULL);
  if (map_pltCurrSet ("bnamt", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set boundary name table.");
  acutRelRb (rb);

  /* Boundary OD field from table */
  rb = acutBuildList (RTSTR, bfield, NULL);
  if (map_pltCurrSet ("bnamf", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set boundary field names.");
  acutRelRb (rb);

  /* Map boundary list of fields to use in attribute mapping */
  /* parallel list to 'atts' */
  rb = acutBuildList (RTSTR, bfield, NULL);                      // TITLE????
  if (map_pltCurrSet ("bodfs", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to assign list of boundary fields.");
  acutRelRb (rb);

  /* ////////////////////////// Plot Option Button //////////////////////// */

	
  
  /* Use first layout we find*/
  const int knLayoutNameLength = 256; 
  char cBuffer[knLayoutNameLength];
  if(Adesk::kTrue == GenerateLayoutName(cBuffer, knLayoutNameLength))
  {

    rb = acutBuildList(RTSTR, cBuffer, RTNONE);
	if (map_pltCurrSet ("pnam", rb) != RTNORM)
      PrintErrors("ERROR: Unable to define a plotter name.");
	acutRelRb (rb);
  }
  else
  {
	PrintErrors("\nERROR: Unable to use any of the layouts in this drawing");
  }

  /* Use plot scale */
  rb = acutBuildList (RTT, NULL);
  if (map_pltCurrSet ("sflg", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set plot scale flag.");
  acutRelRb (rb);

  /* Plot Scale */
  rb = acutBuildList (RTSTR, "1:250", NULL);
  if (map_pltCurrSet ("scale", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set plot scale.");
  acutRelRb (rb);

  /* Trim objects inside the boundary */
  rb = acutBuildList (RTT, NULL);
  if (map_pltCurrSet ("clip", rb) != RTNORM) 
    PrintErrors("ERROR: Unable to set clip flag.");
  acutRelRb (rb);

  /* Buffer the boundary */
  rb = acutBuildList (RTT, NULL);
  if (map_pltCurrSet ("bbuf", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set buffer flag.");
  acutRelRb (rb);

  /* Buffer type 'rect' */
  rb = acutBuildList (RTSTR, "rect", NULL);
  if (map_pltCurrSet ("btyp", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set buffer type.");
  acutRelRb (rb);

  /* Buffer distance */
  rb = acutBuildList (RTREAL, 0.25, NULL);
  if (map_pltCurrSet ("bdist", rb) != RTNORM)
    PrintErrors("ERROR: Unable to set buffer offset.");
  acutRelRb (rb);

  /* Save the plot set */
  if (map_pltCurrSave () != RTNORM)
    PrintErrors("ERROR: Unable to save plot set.");

  /* Save plot set dictionary */
  if (map_pltDefSave () != RTNORM) 
    PrintErrors("ERROR: Unable to plot set dictionary.");

  /* Set ADE variable before verifying the plot set */
  rb = acutBuildList(RTT, NULL);
  if (ade_prefsetval("MkSelSetWithQryObj", rb) != RTNORM )
    PrintErrors("ERROR: Unable to set 'MkSelSetWithQryObj'.");
  acutRelRb(rb);

  /* 
     Attributes are all setup, now initialize in
     plot and verify they are correct.
  */

  /* Initialize and display the plot set */
  result = map_pltInit();

  acutPrintf("\n\nVerifying the plot set...");
  result = map_pltDefVerify(name);
  if( result != RTNORM ) {
    PrintErrors("\nERROR: Plot verification failed.\n");
    map_pltCleanup();              /* restore ADE preferences           */
    EndPlot();                     /* detach drawings, delete category, */
                                   /* and clear queries                 */
    return;
  }                                                                                                   

  /*
     Ask the user if the displayed objects 
     are correct. If so, plot them.
  */
  acedInitGet(0, "Plot Execute");
  result = acedGetKword("\nRun plot or execute? (Plot/Execute) <Plot>: ", answer);
  if ( result == RTNONE || (result == RTNORM && ! strcmp(answer, "Plot")) ) {
    result = map_pltDisplay(bname);
    if ( result != RTNORM )
      PrintErrors(NULL);

    /*
       Ask the user if the displayed objects 
       are correct. If so, plot them.
    */
    acedInitGet(0, "Yes No");
    result = acedGetKword("\nPlot display? (Yes/No) <Yes>: ", answer);
    if ( result == RTNONE || (result == RTNORM && ! strcmp(answer, "Yes")) ) {
      result = map_pltPlot();
      if( result != RTNORM )
        PrintErrors("ERROR: Plot execution failed.");
    }
  }
  else {
    result = map_pltExecute(name);
    if( result != RTNORM )
      PrintErrors("ERROR: Plot execution failed.");
  }

  /*
     Restore environment
  */
  map_pltRestore();              /* restore display to previous state */
  map_pltCleanup();              /* restore ADE preferences           */
  EndPlot();                     /* detach drawings, delete category, */
                                 /* and clear queries                 */

  acutPrintf("\n\nProcessing completed.\n");

}/* DefinePlotSet */


/***********************************************************************
 * Function: CreateBoundaryDwg
 *
 * Create a drawing file that contains a polyline with an object
 * data table attached to it that defines the boundary of the 
 * plot. 
 *
 */
void CreateBoundaryDwg(char* path)
{
  char dwgname[512], str[512];
  ads_point pt1, pt2, pt3, pt4;
  ads_name en;

  /*
    Coordinates are from the attached drawings, defining 
    the part of the drawings to query in.
  */
  pt1[X] = 3082000.0;  pt1[Y] = 1274500.0;  pt1[Z] = 0.0;
  pt2[X] = 3087000.0;  pt2[Y] = 1274500.0;  pt2[Z] = 0.0;
  pt3[X] = 3087000.0;  pt3[Y] = 1278500.0;  pt3[Z] = 0.0;
  pt4[X] = 3082000.0;  pt4[Y] = 1278500.0;  pt4[Z] = 0.0;


  acedCommand(RTSTR, "._layer", RTSTR, "_m", RTSTR, blayer, RTSTR, "", 0);

  acedCommand(RTSTR, "._pline", RTPOINT, pt1, RTPOINT, pt2, 
              RTPOINT, pt3, RTPOINT, pt4, RTSTR, "_c", 0);

  acdbEntLast(en);
  DefineBoundaryTable(en);                         /* attach OD table */

  sprintf(dwgname, "%s%s", path, kpszBoundaryDrawingName);
  pt1[X] = pt1[Y] = pt1[Z] = 0.0;

  if (acedFindFile(dwgname, str) == RTNORM )
    acedCommand(RTSTR, "._wblock", RTSTR, dwgname, RTSTR, "_yes", 
                RTSTR, "", RTPOINT, pt1, RTSTR, "_all", RTSTR, "", 0);
  else 
    acedCommand(RTSTR, "._wblock", RTSTR, dwgname, RTSTR, "", 
                RTPOINT, pt1, RTSTR, "_all", RTSTR, "", 0);
 
  if( acedFindFile(dwgname, dwgname) != RTNORM ) {
    sprintf(str, "ERROR: Unable to create %s", kpszBoundaryDrawingName);
    PrintErrors(str);
  }
  else {
    sprintf(str, "\n%s create successfully.", kpszBoundaryDrawingName);
    acutPrintf(str);
  }

}/* CreateBoundaryDwg */


/**********************************************************************
 * Function: DefineBoundaryTable
 * 
 * Define a table with two columns that contain the data used to
 * fill the attributes of the plot layout block created in the 
 * routine CreateBlockLayout. The table is attached to the boundary
 * border drawing created in the routine CreateBoundaryDwg.
 *
 */
void DefineBoundaryTable(ads_name en)
{
  int result = 0;

  struct resbuf *tabldefn = 
     acutBuildList (RTLB,
                      RTLB,
                          RTLB,
                          RTSTR, "tablename",
                          RTSTR, btable,
                        RTDOTE,
                        RTLB,
                          RTSTR, "tabledesc",
                          RTSTR, "Boundaries Table",
                        RTDOTE,
                        RTLB,
                          RTSTR, "columns",
                          RTLB,
                            RTLB,
                              RTSTR, "colname",
                              RTSTR, bfield,
                            RTDOTE,
                            RTLB,
                              RTSTR, "coldesc",
                              RTSTR, "Name of City",
                            RTDOTE,
                              RTLB,
                              RTSTR, "coltype",
                              RTSTR, "character",
                            RTDOTE,
                            RTLB,
                              RTSTR, "defaultval",
                              RTSTR, bname,
                            RTDOTE,
                          RTLE,
                          RTLB,
                            RTLB,
                              RTSTR, "colname",
                              RTSTR, "Sheet",
                            RTDOTE,
                            RTLB,
                              RTSTR, "coldesc",
                              RTSTR, "Sheet description",
                            RTDOTE,
                            RTLB,
                              RTSTR, "coltype",
                              RTSTR, "character",
                            RTDOTE,
                            RTLB,
                              RTSTR, "defaultval",
                              RTSTR, "Sheet 1",
                            RTDOTE,
                          RTLE,
                        RTLE,
                      RTLE,
                    RTLE, 0 );
 
   result = ade_oddefinetab (tabldefn);
   if( result != RTNORM )
     PrintErrors("ERROR: Unable to define OD table.");
   else {
     result = ade_odaddrecord(en, "BOUNDARIES");
     if( result != RTNORM )
       PrintErrors("ERROR: Unable to attach table to object.");
     else
       acutPrintf("\nTable attached to object successfully.");
   }

   acutRelRb(tabldefn);
   
}/* DefineBoundaryTable */


/***********************************************************************
 * Function: AttachDrawings
 *
 * Attach source drawings that contain the objects to plot.
 *
 */
void AttachDrawings (char* path)
{
  char acad_path[132] = "";
  char str[50] = "";

  /* Define a drive alias */
  ade_aliasdelete(alias);
  if (ade_aliasadd (alias, path) != RTNORM)
    PrintErrors("ERROR: Unable to define alias.");

  /* Attach drawings */
  sprintf(str, "%s:\\%s", alias, kpszParcelDrawingName);
  if (ade_dsattach (str) == ADE_NULLID)
    PrintErrors("ERROR: Unable to attach PARCELS.DWG");

  sprintf(str, "%s:\\%s", alias, kpszHousesDrawingName);
  if (ade_dsattach (str) == ADE_NULLID) 
    PrintErrors("ERROR: Unable to attach STREETS.DWG");
  ade_errclear ();

  sprintf(str, "%s:\\%s", alias, kpszBoundaryDrawingName);
  if (ade_dsattach (str) == ADE_NULLID)
    PrintErrors("ERROR: Unable to attach BOUNDARY.DWG");

}/* AttachDrawings */

/***********************************************************************
 * Function: DefineQuery
 *
 * Define a query that will bring in the desired objects 
 * from the attached source drawings.
 *
 */
void DefineQuery ()
{
  struct resbuf *qry_list;
  ads_point point1 = {3083670.0, 1277440.0, 0.0};
  ads_point point2 = {3085420.0, 1275820.0, 0.0};
  ade_id result;

  ade_qryclear ();

	/* First condition : crossing window using point1 and point2 */
  qry_list = acutBuildList (RTLB, 
                              RTSTR, "window",
                              RTSTR, "crossing",
                              RTPOINT, point1,
                              RTPOINT, point2,
                            RTLE, RTNONE);

  if (ade_qrydefine ("", "", "", "location", qry_list, "") == ADE_NULLID) 
    PrintErrors("ERROR: Unable to define location query.");
  acutRelRb (qry_list);

  /* Object in layer parcels or roads */
  qry_list = acutBuildList (RTLB,
                              RTSTR, "layer",
                              RTSTR, "=",
                              RTSTR, "parcels",
                            RTLE, RTNONE);

  if (ade_qrydefine ("and", "(", "", "property", qry_list, "") == ADE_NULLID)
    PrintErrors("ERROR: Unable to define parcels property query.");
  acutRelRb (qry_list);

  qry_list = acutBuildList (RTLB,
                              RTSTR, "layer",
                              RTSTR, "=",
                              RTSTR, "streets",
                            RTLE, RTNONE);

  if (ade_qrydefine ("or", "", "", "property", qry_list, ")") == ADE_NULLID) 
    PrintErrors("ERROR: Unable to define streets property query.");
  acutRelRb (qry_list);

  /* Draw the objects at execution time */
  if (ade_qrysettype ("draw", ADE_FALSE, "", "") != RTNORM)
    PrintErrors("ERROR: Unable to set query to draw.");

  /* Saves the query */
  qry_list = acutBuildList (RTLB,
                              RTLB,
                                RTSTR, bfield, 
                                RTSTR, query,
                              RTLE,
                              RTLB,
                                RTSTR, "description",
                                RTSTR, "Sample Query",
                              RTLE,
                              RTLB,
                                RTSTR, "qtype",
                                RTSHORT, 1,
                              RTLE,
                              RTLB,
                                RTSTR, "saveoption",
                                RTSHORT, 2,
                              RTLE,
                            RTLE, RTNONE);

  /* Save the query under a category */
  result = ade_qrysave (category, qry_list);
  if( result == ADE_NULLID )
    PrintErrors("ERROR: Unable to save query.");
  acutRelRb (qry_list);

}/* DefineQuery */

/****************************************************************************
 * Function: EndPlot
 *
 * Detach source drawings, delete query category and clear all queries.
 *
 */
void EndPlot()
{
  struct resbuf *rb = NULL;
  struct resbuf *rb1 = NULL;
  int result = 0;

  rb = ade_dslist(0.0, ADE_FALSE);
  if ( ! rb )           
    PrintErrors("ERROR: Unable to get attached drawing list.");
  else {
    for( rb1 = rb ; rb1 != NULL ; rb1 = rb1->rbnext ) {
      if( rb1->restype == RTREAL )
        ade_dsdetach(rb1->resval.rreal);
    }
    acutPrintf("\nDrawings detached successfully.");
  }
  acutRelRb(rb);  

  result = ade_aliasdelete(alias);
  if ( result != RTNORM )
    PrintErrors("ERROR: Unable to delete alias.");
  else
    acutPrintf("\nAlias deleted successfully.");

  rb = ade_qllistctgy();
  if ( ! rb )
    PrintErrors("ERROR: Unable to get category list.");
  else {
    for( rb1 = rb ; rb1 != NULL ; rb1 = rb1->rbnext ) {
      if( rb1->restype == RTREAL )
        ade_qldelctgy(rb1->resval.rreal);
    }
    acutPrintf("\nCategory deleted successfully.");
  }
  acutRelRb(rb);  

  result = ade_qryclear();
  if ( result != RTNORM )
    PrintErrors("ERROR: Unable to clear queries.");
  else
   acutPrintf("\nQueries cleared successfully.");

  /* Restore current layer */
  rb = acutBuildList(RTSTR, clayer, 0);
  result = acedSetVar("CLAYER", rb);

}/* EndPlot */

/**************************************************************************
 * Function: GetAcadPath
 *
 * Find AutoCAD executable to get the path for tutorial drawings.
 *
 */
void GetAcadPath(char* path)
{
  char* chr, tpm_path[512];

  acedFindFile (PROGRAM, path);
  chr = strrchr (path, SEPARATOR);
  *chr = 0;
  
  sprintf (tpm_path, "%s%cmaptut%ctutorial VI%c%s"
	  , path, SEPARATOR, SEPARATOR, SEPARATOR, kpszParcelDrawingName);

  if (acedFindFile (tpm_path, tpm_path) != RTNORM) {
    chr = strrchr (path, SEPARATOR);
    *chr = 0;
  }
  sprintf (tpm_path, "%cmaptut%ctutorial VI%c", SEPARATOR, SEPARATOR, SEPARATOR);
  strcat (path, tpm_path);

}/* GetAcadPath */


/***************************************************************************
 * Function: PrintErrors
 *
 * Prints the ADE error stack.
 *
 */
 void PrintErrors(char* msg)
 {
   int nb, i;

   if( msg )
     acutPrintf("\n%s", msg);

   nb = ade_errqty();
   for ( i = 0 ; i < nb ; i++ ) 
     acutPrintf("\n   ADE ERROR %d: %s", i, ade_errmsg(i));

   ade_errclear();

 }

AcRx::AppRetCode
acrxEntryPoint(AcRx::AppMsgCode msg, void* pkt)
{
    switch (msg) {
    case AcRx::kInitAppMsg:
		acrxDynamicLinker->registerAppMDIAware(pkt);
        acrxDynamicLinker->unlockApplication(pkt);
        break;
	case AcRx::kLoadDwgMsg:
		LoadFunctions();
		break;
	case AcRx::kInvkSubrMsg:
		ExecuteFunction();
		break;
    case AcRx::kUnloadAppMsg:
		break;
    }
    return AcRx::kRetOK;
}


/****************************************************************************
 * Function: LoadFunctions
 *
 *  Standard code for loading the ADS functions from 'func_table[]'.
 *
 */
static int LoadFunctions()
{
  short i = 0;
  for ( i = 0 ; i < ELEMENTS(func_table) ; i++ ) {
    if ( ! acedDefun(func_table[i].name, i))
      return RTERROR;
  }
    
  return RTNORM;
    
} /* LoadFunctions */


/****************************************************************************
 * Function: ExecuteFunction
 *
 *   Standard code for executing an ADS function from 'func_table[]'.
 *
 */
static int ExecuteFunction()
{
  int i = 0;

  i = acedGetFunCode();

  if ( i < 0 || i >= ELEMENTS(func_table)) {
    acdbFail("]nRecieved nonexistent function code.");
    return RTERROR;
  }

  (*func_table[i].func)();

  return RTNORM;

} /* ExecuteFunction */

    

/////////////////////////////////////////////////////////////////////////
//This function will fill pLayoutName with the first layout it finds that
//can fit with in the layout buffer.
Adesk::Boolean
GenerateLayoutName(char* pLayoutName, unsigned int nBuflen)
{
   	Acad::ErrorStatus eError;
	Adesk::Boolean bRetCode = Adesk::kFalse;

	AcDbDatabase *pCurDwgDb = acdbHostApplicationServices()->workingDatabase();    
	AcDbDictionary *pDict = NULL;
	
	//Get the layout dictionary
	eError = pCurDwgDb->getLayoutDictionary(pDict, AcDb::kForRead);
	assert(pDict); if(NULL == pDict) return bRetCode;
	
	//Get the iterator for this dictionary
	AcDbDictionaryIterator *pDictIter = pDict->newIterator();
	assert(pDictIter);
	if(NULL == pDictIter)
	{
		pDict->close();
		return bRetCode;
	}
	
	//Itereate through the layouts searching for a layout name that will
	//fit within the buffer.
	for(; !pDictIter->done(); pDictIter->next())
	{
		const char* kpszName = pDictIter->name();

		if(NULL != kpszName)
		{
			if(_tcslen(kpszName) < nBuflen)
			{
				//We have a layout name to use
				_tcscpy(pLayoutName, kpszName);
				bRetCode = Adesk::kTrue;
				break;
			}
		}
	}
	
	pDict->close();
	return bRetCode;
}
