#ifndef MAP_ALTERATION_H
#define MAP_ALTERATION_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the descriptions of Map ARX API classes to manipulate
// the property alteration.
// Class AcMapPropertyAlterationDefinition represents an interface to the set
// of property alteration items. An application gets an instance of this class 
// via AcMapQuery::GetPropertyAlteration() and is responsible for deleting it.
// This class supports the standard set of methods to manipulate the collection.
//
#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

#include <MapConstants.h>

// Forward declarations
class AcMapPropertyAlteration ;
class AcMapProject;

class AcMapPropertyAlterationDefinition
{
public:
	// Gets the associated project.
	virtual AcMapProject*	Project			() const = 0 ;

    // Clears the property alteration set.
    //
    virtual AcMap::EErrCode	Clear           () const = 0 ;

    // Returns the number of items defined in the property alteration set.
    //
    virtual int				CountAlterations() const = 0 ;

    // The type of object allocated by this method depends on the kType argument as follows:
    //    AcMap::kAlterationTextEntity		AcMapTextAlteration object
    //    AcMap::kAlterationHatch			AcMapHatchAlteration object
    //    Other kType arguments				AcMapPropertyAlteration object
    // NOTE! AcMap::kAlterationText does not get you an AcMapTextAlteration object! The 
    // only correct kType for AcMapTextAlteration objects is AcMap::kAlterationTextEntity.
    // When this method creates AcMapTextAlteration or AcMapHatchAlteration objects, it uses 
    // default values for their parameters.
    // Unlike similar methods, this method always allocates a new pObj object, which the    
    // application is responsible for deleting.
    //
	virtual AcMap::EErrCode	AddAlteration   (AcMapPropertyAlteration *&pObj,
											 AcMap::EAlterationType kType) const = 0 ;

    // Gets the i-th property alteration object. Unlike some similar methods, 
    // a new object is always allocated. It is the responsibility of an application to delete 
    // the pObj object.
    //
    virtual AcMap::EErrCode	GetAlteration   (AcMapPropertyAlteration *&pObj, int i) const = 0 ;
 
    // Gets the property alteration object by the Id. Unlike some similar methods, a new
    // object is always allocated. It is the responsibility of an application to delete the pObj object. 
    // An application can get the Id of the AcMapPropertyAlteration object using AcMapPropertyAlteration::ObjectId().
    //
    virtual AcMap::EErrCode	GetAlteration   (AcMapPropertyAlteration *&pObj,
                                             AcMapId Id) const = 0 ;
 
    // Removes the property alteration object.
    //
    virtual AcMap::EErrCode	RemoveAlteration(int i) const = 0 ;
  
    // Removes the property alteration object, specified by the Id. An application 
    // can get the Id of the AcMapPropertyAlteration object using AcMapPropertyAlteration::ObjectId().
    //
    virtual AcMap::EErrCode	RemoveAlteration(AcMapId Id) const = 0 ;

protected:
							AcMapPropertyAlterationDefinition
											() ;

    virtual					~AcMapPropertyAlterationDefinition
											() ;
} ;

 
class AcMapPropertyAlteration
{
public:
    virtual					~AcMapPropertyAlteration
											() ;

    // Returns the Id of the associated internal object. 
    //
    virtual AcMapId			ObjectId        () const = 0 ;

	// Gets the associated project
	//
	virtual AcMapProject*	Project			() const = 0 ;

    // Gets the property type. Originally the object type is set when the object is 
    // created by AcMapQuery::AddPropertyAlteration() and cant be changed. 
    //
    virtual AcMap::EErrCode	GetType         (AcMap::EAlterationType &kType) const = 0 ;

    // Sets the expression to calculate object property. The expression is an AutoLISP statement 
    // with MAP dot variables. For the child AcMapTextAlteration this expression defines the 
    // text value. For the child AcMapHatchAlteration this expression defines the hatch 
    // pattern. If using a range table to calculate values conditionally, the expression has 
    // the following format:
    //     "(Range PropertyName RangeTableName)"
    // where Range is a required keyword that never changes. 
    //
	virtual AcMap::EErrCode	SetExpression   (const char* pcStr) const = 0 ;

    // This method gets the expression for calculating the object property. An application is responsible for releasing 
    // the string buffer using the delete [] operator.
    //
    virtual AcMap::EErrCode	GetExpression   (char *&pcStr) const = 0 ;

protected:
							AcMapPropertyAlteration
								            () ;
} ;

class AcMapTextAlteration : public AcMapPropertyAlteration
{
public:

    // Accesses the expression to calculate text height. 
    //
    virtual AcMap::EErrCode	GetHeight       (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetHeight       (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate text rotation.
    //
    virtual AcMap::EErrCode	GetRotation     (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetRotation     (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate text location. The expression should evaluate to a 
    // 3D point value and can use .LABELPT and .CENTROID dot variables.
    //
    virtual AcMap::EErrCode	GetLocation     (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetLocation     (const char *pcStr) const = 0 ;

    // Accesses an AutoLISP expression to calculate text justification. The expression should
    // evaluate to one of the following values "CENTER", "MIDDLE", "RIGHT", "TL", "TC", "TR", "ML",
    // "MC", "MR" ,"BL" ,"BC" ,"BR".
    //
    virtual AcMap::EErrCode	SetJustification(const char *pcStr) const = 0 ;
    virtual AcMap::EErrCode	GetJustification(char *&pcStr) const = 0 ;

    // Accesses the expression to calculate text style. 
    //
    virtual AcMap::EErrCode	GetStyle        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetStyle        (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate text layer. 
    //
    virtual AcMap::EErrCode	GetLayer        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetLayer        (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate text color. 
    //
    virtual AcMap::EErrCode	GetColor        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetColor        (const char *pcStr) const = 0 ;

    // Sets the default values
    //  Height              - 0.2
    //  Insertion Point     - .LABELPT
    //  Justification       - CENTER
    //  Style               - current text style
    //  Layer               - current layer
    //  Color               - current color
    //  Rotation            - 0.
    //
    virtual AcMap::EErrCode	SetDefaults     () const = 0 ;

protected:
							AcMapTextAlteration
								            () ;

    virtual					~AcMapTextAlteration
								            () ;
} ;

class AcMapHatchAlteration : public AcMapPropertyAlteration
{
public:
    // Accesses the expression to calculate hatch scale. 
    //
    virtual AcMap::EErrCode	GetScale        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetScale        (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate hatch layer.
    //
    virtual AcMap::EErrCode	GetLayer        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetLayer        (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate hatch color.
    virtual AcMap::EErrCode	GetColor        (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetColor        (const char *pcStr) const = 0 ;

    // Accesses the expression to calculate hatch rotation.
    virtual AcMap::EErrCode	GetRotation     (char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode	SetRotation     (const char *pcStr) const = 0 ;

    // Sets the default values
    //  Pattern             - USER
    //  Scale               - 1.
    //  Rotation            - 0.
    //  Layer               - current layer
    //  Color               - current color
    //
    virtual AcMap::EErrCode	SetDefaults     () const = 0 ;

protected:
    // the constructor sets default values
							AcMapHatchAlteration
								            () ;

    virtual					~AcMapHatchAlteration
								            () ;

} ;

// Inlines.
inline AcMapPropertyAlterationDefinition::AcMapPropertyAlterationDefinition()
{}

inline AcMapPropertyAlterationDefinition::~AcMapPropertyAlterationDefinition()
{}

inline AcMapPropertyAlteration::AcMapPropertyAlteration()
{}

inline AcMapPropertyAlteration::~AcMapPropertyAlteration()
{}

inline AcMapTextAlteration::AcMapTextAlteration()
{}

inline AcMapTextAlteration::~AcMapTextAlteration()
{}

inline AcMapHatchAlteration::AcMapHatchAlteration()
{}

inline AcMapHatchAlteration::~AcMapHatchAlteration()
{}

#endif /* MAP_ALTERATION_H */
/*EOF*/
