#ifndef MAP_ARXAPI_H
#define MAP_ARXAPI_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
//  Main Exported MAP Library definitions.

#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

// INCLUDES
#include <wtypes.h>
#include <gevec2d.h>
#include <gepnt3d.h>
#include <gept3dar.h>
#include "MapConstants.h"
#include "MapSession.h"      // For AcMapSession
#include "MapObjectId.h"
#include "MapColl.h"
#include <dbidar.h>          // For AcDbObjectIdArray

typedef AcMapObjArray<AcMapObjectId> AcMapObjectIdArray ;

// FORWARD DECLARATIONS
class AcMapAttachedDrawing;
class AcMapPropertyAlteration;
class AcMapTemplateLine ;
class AcMapQueryCategory ;
class AcMapQueryAttribute ;
class AcMapDriveAlias ;
class AcMapQuery ;
class AcMapRangeTable ;
class AcMapRangeLine ;
class AcMapErrorEntry ;
class AcMapErrorParameter ;
class AcMapDrawingSetReactor ;
class AcMapAliasesReactor ;
class AcMapQueryLibraryReactor ;
class AcMapQueryBranch ;
class AcMapPropertyAlterationDefinition ;
class AcMapReportTemplate ;
class AcMapExpression ;
class AcMapObjectId;
class AcMapODRecordIterator;
class AcMapODTableDefinition;
class AcMapODTable;
class AcMapStringArray ;
class AcMapValue ;

// AutoCAD classes
class AcDbDatabase ;
class AcDbObjectId ;
class AcDbBlockTableRecordIterator ;
class AcDbExtents ;

class CWnd ;


class AcMapExpression
{
public:

    virtual					~AcMapExpression() ;

	// Gets the associated project
	//
	virtual AcMapProject*	Project			() const = 0 ;

    // Gets the result of the Val object depending on the Val type. If the
    // Val type is not initialized, the type of the value is set automatically.
    //
    virtual AcMap::EErrCode Execute			(AcMapValue &Val, 
											 AcDbObjectId &Id = AcDbObjectId(0)) = 0 ;

    // Accesses the expression string.
    //
    virtual AcMap::EErrCode GetExpressionString
											(const char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode GetExpressionString
											(char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode SetExpressionString
											(const char *pcStr) = 0 ;
protected:
							AcMapExpression	() ;
} ;

class AcMapAliases
{
public:
    // Counts all aliases defined in the current session.
    //
    virtual int				CountAliases    () const = 0 ;
    
    // Adds new alias.
    //
    virtual AcMap::EErrCode AddAlias        (const char *pName, 
											 const char *pPath) const = 0 ;
    
    // Gets alias definition by the index or by alias name. An application 
    // is responsible for releasing the output buffer via operator delete[].
    //
    virtual AcMap::EErrCode GetAlias        (AcMapDriveAlias *&pObj,
											 int index) const = 0 ;
    virtual AcMap::EErrCode GetAlias        (AcMapDriveAlias *&pObj,
                                             const char *pName) const = 0 ;
    
    // Looks for the alias, associated with the specified path.An application 
    // is responsible for releasing the output buffer via operator delete[].
    //
    virtual AcMap::EErrCode FindAlias       (AcMapDriveAlias *&pObj,
                                             const char *pPath) const = 0 ;
    
    // Removes the alias(es).
    //
    virtual AcMap::EErrCode Clear           () const = 0 ;
    virtual AcMap::EErrCode RemoveAlias     (const char *pName) const = 0 ;
    virtual AcMap::EErrCode RemoveAlias     (int index) const = 0 ;
    
    // Makes the alias path for the specified actual path. As far as the result 
    // can be unambiguous, there is an option to specify the alias to be used as 
    // the optional parameter. If this parameter is missing, the first alias in 
    // the list, that suits to the specified path, will be used. An application 
    // is responsible for releasing the output buffer via operator delete[].    
    //
    virtual AcMap::EErrCode TranslateToAliasPath
											(char *&pcRes, 
											 const char* pcPath, 
											 char *pcAlias = NULL) const = 0 ;

    // Creates the actual path for the specified alias path. An application is responsible 
    // for releasing the output buffer via operator delete[].
    //
    virtual AcMap::EErrCode TranslateToActualPath
											(char *&pcRes, 
											 const char* pcAliasPath) const = 0 ;

    // Accesses the aliases reactor.
    //
    virtual void			AddReactor		(AcMapAliasesReactor *pReact) const = 0 ;
    virtual void			RemoveReactor	(AcMapAliasesReactor *pReact) const = 0 ;

protected:    
							AcMapAliases    () ;
    virtual					~AcMapAliases   () ;
} ;

class AcMapDriveAlias
{
public:
    virtual					~AcMapDriveAlias() ;
                            
    // Sets the alias parameters.
    //
    virtual AcMap::EErrCode SetName         (const char* name) = 0 ;
    virtual AcMap::EErrCode SetPath         (const char* path) = 0;
    
    // Gets the alias parameters.
    //
    virtual AcMap::EErrCode GetPath         (const char* &path) const = 0;
    virtual AcMap::EErrCode GetPath         (char *&path) const = 0 ;
    virtual AcMap::EErrCode GetName         (const char* &pName) const = 0 ;
    virtual AcMap::EErrCode GetName         (char *&pName) const = 0 ;
    
protected:    
							AcMapDriveAlias () ;
} ;

class AcMapSaveSet  
{
public:
    // Gets the associated project.
	//
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Adds the specified objects to the save set.
    //
    virtual int				AddObjects      (AcDbObjectIdArray &) const = 0 ;
    
    // Removes the specified objects from the save set.
    //
    virtual int				RemoveObjects   (AcDbObjectIdArray &) const = 0 ;
    
    // Counts the objects of the specified type in the save set, where
    // type is the logical sum of the following bit values:
    // AcMap::kQueriedExisted   queried existed object
    // AcMap::kQueriedErased	queried erased object
    // AcMap::kNewlyCreated     newly created object
    //
    virtual int				CountObjects    (int type) const = 0 ;          
    
    // Gets the objects of the specified type in the save set, where
    // type is the logical sum of the following bit values:
    // AcMap::kQueriedExisted	queried existed object
    // AcMap::kQueriedErased	queried erased object
    // AcMap::kNewlyCreated     newly created object
    //
    virtual int				GetObjects      (int type, 
											 AcDbObjectIdArray &) const = 0 ;          
    
    // Gets lock information from the queried object.
    //
    virtual Adesk::Boolean  
							WhoHasIt        (AcDbObjectId   &Id,
											 char           *&pDwgName,
                                             char           *&pUserName ,
										     char           *&pProjectName ,
                                             struct tm      &tTime) const = 0 ;
                                             
    // Checks whether the object is in the save set.
    //
    virtual Adesk::Boolean  
							IsInSaveSet     (AcDbObjectId ) const = 0 ;
    
protected:
							AcMapSaveSet    () ;
    virtual					~AcMapSaveSet   () ;
} ;

class AcMapDrawingSet
{
public:
    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Counts the attached drawings. If the flag is set to True, nested
    // drawings are counted as well.
    //
    virtual long			CountDrawings   (Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;
    
    // Attaches a new drawing. If initially the pointer is NULL, a new interface object is allocated. 
    // An application is responsible for deleting it.
    //
    virtual AcMap::EErrCode	AttachDrawing   (AcMapAttachedDrawing *&pDwg,
											 const char *pcAliasPath) const = 0 ;
    
    // Detaches a drawing. Only 0-level drawings can be detached.
    //
    virtual AcMap::EErrCode DetachDrawing   (int iIndex) const = 0 ;
    virtual AcMap::EErrCode DetachDrawing   (const char *pcAliasPath) const = 0 ;
    virtual AcMap::EErrCode DetachDrawing   (AcMapId Id) const = 0 ;
    
    // Gets an interface object for the specified attached drawing. If initially the pointer is NULL,
    // a new interface object is allocated. An application is responsible for deleting it. Boolean flag controls 
    // whether the nested drawings should be counted.
    //
    virtual AcMap::EErrCode GetDrawing      (AcMapAttachedDrawing *&pDwg,
											 int iIndex,
											 Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;
 
    // Gets an interface object for the specified attached drawing. If initially the pointer is NULL,
    // a new interface object is allocated. An application is responsible for deleting it. Boolean flag controls 
    // whether the nested drawings should be accessed.
    // NOTE: Map Drawing Sets can have multiple entries of the same drawing. This method gets the first
    // entry. 
    //
    virtual AcMap::EErrCode GetDrawing      (AcMapAttachedDrawing *&pDwg,
											 const char *pcAliasPath,
											 Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;
 
    // Gets an interface object for the specified attached drawing. If initially the pointer is NULL,
    // a new interface object is allocated. An application is responsible for deleting it. 
    //
    virtual AcMap::EErrCode GetDrawing      (AcMapAttachedDrawing *&pDwg,
											 AcMapId Id) const = 0 ;
 
    // Fills the array with the names of the entries of the specified symbol table.
    //
    virtual AcMap::EErrCode GetTableList    (AcMapStringArray &tArray,
											 AcMap::ETableType kType) const = 0 ;
                                             
    // Gets resulting drawing extents of all active drawings in WCS.
    //
    virtual AcMap::EErrCode GetExtents      (AcDbExtents &tExtents) const = 0 ;
    
    // Zooms resulting drawing extents of all active drawings. 
    //
    virtual AcMap::EErrCode ZoomExtents     () const = 0 ;
    
    // Adds the Drawing Set Reactor instance.
    //
    virtual void			AddReactor		(AcMapDrawingSetReactor *pObj) const = 0 ;
    
    // Removes the Drawing Set Reactor instance.
    //
    virtual void			RemoveReactor	(AcMapDrawingSetReactor *pObj) const = 0 ;
    
protected:
							AcMapDrawingSet () ;
    virtual					~AcMapDrawingSet() ;
} ;

class AcMapAttachedDrawing
{
public:
    virtual					~AcMapAttachedDrawing  
											() ;
    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns the identification of the drawing's internal object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;

    // Gets either the drawing alias path or WC path.
    //
    virtual AcMap::EErrCode	GetAliasPath    (const char *&pcPath) const = 0 ;
    virtual AcMap::EErrCode GetAliasPath    (char *&pcPath) const = 0 ;

    // Gets the actual path to the drawing file.
    //
    virtual AcMap::EErrCode GetActualPath   (const char *&pcPath) const = 0 ;
    virtual AcMap::EErrCode GetActualPath   (char *&pcPath) const = 0 ;

    // Gets the drawing description. 
    //
    virtual AcMap::EErrCode GetDescription  (const char *&pcDsc) const = 0 ;
    virtual AcMap::EErrCode GetDescription  (char *&pcDsc) const = 0 ;

    // Sets the drawing description. 
    //
    virtual AcMap::EErrCode SetDescription  (const char *pcDsc) = 0 ;

    // Activates the specified drawing. 
    //
    virtual AcMap::EErrCode Activate        () = 0 ;

    // Deactivates the specified drawing. 
    //
    virtual AcMap::EErrCode Deactivate      () = 0 ;

    // Returns the active status of the specified drawing.
    //
    virtual AcMap::EAdeDwgStatus    
							ActiveStatus    () const = 0 ;

    // Returns True if another AutoCAD Map user has modified the drawing so that 
    // the entity and symbol table information were updated.
    //
    virtual Adesk::Boolean  IsUpdatedByAnotherMapUser   
											() const = 0 ;

    // Returns the nesting level of the drawing in the drawing set. Drawings attached to the
    // current one are considered to have a level of 0.
    //
    virtual int				NestedLevel     () const = 0 ;

    // Gets the interface object of the parent drawing for a drawing with 
    // nested level greater than 0.
    //
    virtual AcMap::EErrCode GetParentDrawing(AcMapAttachedDrawing *&) const = 0 ;

    // Gets the interface object for the i-th drawing from the nested drawing set. 
    // The 'bSkipNestedDrawings parameter, if True, means skip drawings from the L+2 level 
	// and greater, where L is level of the object.
    //
    virtual AcMap::EErrCode GetNestedDrawing(AcMapAttachedDrawing *&pObj,
											 int i,
                                             Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;

    // Gets an interface object for the specified nested drawing. If initially the pointer is NULL,
    // a new interface object is allocated. An application is responsible for deleting it. Boolean flag controls 
    // whether the nested drawings should be accessed.
    // NOTE: A nested Drawing Set can have multiple entries of the same drawing. This method gets the first
    // entry. 
    //
    virtual AcMap::EErrCode GetNestedDrawing(AcMapAttachedDrawing *&pObj,
											 const char *pcAliasPath,
											 Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;

    // Counts number of the drawings in the nested drawing set. If Boolean 'bSkipNestedDrawings' is True, 
    // only drawings from level L+1 are counted, where L is nested level of this object.
    //
    virtual long			CountNestedDrawings
											(Adesk::Boolean bSkipNestedDrawings = Adesk::kTrue) const = 0 ;

    // Gets Ids of all objects in the source drawing that match the current query.
    //
    virtual AcMap::EErrCode	ApplyQuery      (AcDbObjectIdArray &Ids) = 0 ;
    virtual AcMap::EErrCode ApplyQuery      (AcMapObjectIdArray &Ids)= 0 ;

    // Creates a report for the specified objects in the source drawing 
    // in accordance with the current template.
    //
    virtual AcMap::EErrCode Report          (AcDbObjectIdArray &Ids) = 0 ;
    virtual AcMap::EErrCode Report          (AcMapObjectIdArray &Ids) = 0 ;

    // Draws objects from the external drawing. If there are blocks with the same names 
    // in the current and the source drawing, the block definition from the external drawing 
    // is taken.
    //
    virtual AcMap::EErrCode Preview         () = 0 ;

    // Draws specified objects from the external drawing. If there are blocks with the 
    // same names in the current and the source drawing, the block definition from the current 
    // drawing is taken.
    //
    virtual AcMap::EErrCode Preview         (AcDbObjectIdArray &Ids) = 0 ;
    virtual AcMap::EErrCode Preview         (AcMapObjectIdArray &Ids) = 0 ;

    // Deep clones the specified objects from the external drawing to the current session. 
    // If applyFeatureAlteration parameter is True, the current feature alteration is applied 
    // to the Cloned objects.
    //
    virtual AcMap::EErrCode QueryIn         (AcDbObjectIdArray &Ids) = 0 ;
    virtual AcMap::EErrCode QueryIn         (AcMapObjectIdArray &Ids) = 0 ;

    // Deep clones back objects queried from the drawing, if they are in the save set.
    // Call LockForWriteHeavy() before calling this method. Call Save() to save the 
    // drawing database to the file.
    //
    virtual AcMap::EErrCode CloneBack       () = 0 ;

    // Deep clones the newly created objects from the save set, if they
    // match the "Area" criteria.
    // Call LockForWriteHeavy() before calling this method. Call Save() to 
    // save the drawing database to the file.
    //
    virtual AcMap::EErrCode CloneBackArea   () = 0 ;

    // Deep clones the newly created objects from the save set, if they
    // match the "Layer" criteria.
    // Call LockForWriteHeavy() before calling this method. Call Save() to save 
    // the drawing database to the file.
    //
    virtual AcMap::EErrCode CloneBackLayer  ()  = 0 ;

    // Deep clones the specified objects from the save set to the 
    // external drawing. 
    // Call LockForWriteHeavy() before calling this method. Call Save() to save 
    // the drawing database to the file.
    //
    virtual AcMap::EErrCode CloneBack       (AcDbObjectIdArray &Ids) = 0 ;

    // Gets the object iterator for the Model Space of the active drawing. Place any
    // kind of heavy lock before calling this method.
    //
    virtual AcDbBlockTableRecordIterator* 
							GetObjectIterator
											() = 0 ;

    // Places a read lock on the drawing file.
    //
    virtual AcMap::EErrCode LockForRead		() = 0 ;

    // Places a write lock on the drawing file. The lock can be exclusive to allow 
    // changing the projection.
    //
    virtual AcMap::EErrCode LockForWrite	(Adesk::Boolean bIsExclusive) = 0 ;

    // Removes a read lock from the drawing file.
    //
    virtual AcMap::EErrCode UnlockForRead	() = 0 ;

    // Removes a write lock from the drawing file.
    //
    virtual AcMap::EErrCode UnlockForWrite	() = 0 ;

    // Sets the drawing update status to control behavior of the save() method.
    //
    virtual AcMap::EErrCode SetUpdatedStatus(AcMap::EAdeDwgUpdateStatus newVal) = 0 ;

    // Returns the drawing update status to control behavior of the save() method. The returned
    // value is a combination of bits, defined in AcMap::EAdeDwgUpdateStatus.
    //
    virtual int				UpdatedStatus	() const = 0 ;

    // Saves the drawing if it has been updated.
    //
    virtual AcMap::EErrCode Save            (char *message = NULL) = 0 ;

    // Gets pointer to the drawing database. If database has been unloaded, 
    // loads it into memory. Drawing should be either active or locked and 
    // be locked with any heavy lock.
    //
    virtual AcDbDatabase*   
							Database        () const = 0 ;

    // Switches On/Off simple transformation for the drawing.
    //
    virtual AcMap::EErrCode EnableTransformation
											(Adesk::Boolean bIsEnabled) = 0 ;

    // Checks if simple transformation for the drawing is switched On/Off.
    //
    virtual Adesk::Boolean  
							IsTransformationEnabled
											() const = 0 ;

    // Accesses the transformation offset.
    //
    virtual AcMap::EErrCode GetOffset       (AcGeVector2d &tOffset) const = 0 ;
    virtual AcMap::EErrCode SetOffset       (AcGeVector2d &tOffset) = 0 ;

    // Accesses the transformation scale.
    //
    virtual AcMap::EErrCode GetScale        (double &dScale) const = 0 ;
    virtual AcMap::EErrCode SetScale        (double dScale) = 0 ;

    // Accesses the transformation rotation angle.
    virtual AcMap::EErrCode GetRotation     (double &dRotation) const = 0 ;
    virtual AcMap::EErrCode SetRotation     (double dRotation) = 0 ;

    // Gets the list of object names from the source drawings. 
    //
    virtual AcMap::EErrCode GetTableList    (AcMapStringArray &tArray,
											 AcMap::ETableType kType) = 0;

    // Gets extents of the drawing. If the bTransformToCurrent is True, the result is transformed to the
    // project coordinate system and is in WCS, otherwise it is in the source coordinate system 
    // in UCS.
    //
    virtual AcMap::EErrCode GetExtents      (AcDbExtents &tExtents,
											 Adesk::Boolean bTransformToCurrent) = 0 ;

    // Accesses the projection in the external database. 
    //
    virtual AcMap::EErrCode SetProjection   (const char* pcNewProjection) = 0 ;
    virtual AcMap::EErrCode GetProjection   (const char* &pcProjection) = 0 ;
    virtual AcMap::EErrCode GetProjection   (char* &pcProjection) = 0 ;

    // Accesses the save back extents. 
    //
    virtual AcMap::EErrCode SetSaveBackExtents
											(AcGePoint3dArray &tPoints) = 0 ;
    virtual AcMap::EErrCode SaveBackExtents (AcGePoint3dArray &tPoints,
											 Adesk::Boolean bTransformToCurrent) const = 0 ;

    // Adds all objects, queried from the drawing, to the save set.
    //
    virtual AcMap::EErrCode AddAllToSaveSet () = 0 ;

    // Removes all objects, queried from the drawing, from the save set.
    //
    virtual AcMap::EErrCode RemoveAllFromSaveSet
											() = 0 ;

    // Gets the number of objects, queried from the drawing, in the save set. 
    // Each bit of the iType has the following value
    // AcMap::kQueriedExisted - queried existed object
    // AcMap::kQueriedErased  - queried erased object
    // if the bCountInNested parameter is true, counts objects in the save set queried from the 
    // nested drawings.
    virtual int				NumInSaveSet    (int iType, 
											 Adesk::Boolean bCountInNested = Adesk::kFalse) const = 0 ;

    // Removes locks, owned by the users that match the pattern, from the drawing. 
    // This method works like the ADEDWGMAINT command. Only a superuser can remove the 
    // foreign locks. The unique lock is required in the case.
    //
    virtual AcMap::EErrCode RemoveLocks     (const char* pcPattern) = 0 ;

    // Accesses the complex transformation.
    //
    virtual AcMap::EErrCode SetComplexTransformation
											(const AcGeMatrix3d &tTransform) = 0 ;
    virtual AcMap::EErrCode GetComplexTransformation
											(AcGeMatrix3d &tTransform)const = 0 ;
protected:
							AcMapAttachedDrawing  
											() ;
} ;

class AcMapQueryLibrary 
{
public:

	// Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

        // Counts the query categories.
    //
    virtual int				CountCategories () const = 0 ;                                            
    
    // Gets the array of the query category names.
    //
    virtual AcMap::EErrCode GetCategoryNames(AcMapStringArray &) const = 0 ;
    
    // Gets an interface object for the specified query category.
    // If initially the pointer is NULL, a new interface object is allocated. An application is 
    // responsible for deleting it.
    //
    virtual AcMap::EErrCode GetCategory     (AcMapQueryCategory *&pCat,
											 int i) const = 0  ;

    virtual AcMap::EErrCode GetCategory     (AcMapQueryCategory *&pCat,
											 const char *pcName) const = 0 ;

    virtual AcMap::EErrCode GetCategory     (AcMapQueryCategory *&pCat,
											 AcMapId Id) const = 0 ;
                                             
    // Removes the specified query category.
    //
    virtual AcMap::EErrCode RemoveCategory  (int i) = 0 ;
    virtual AcMap::EErrCode RemoveCategory  (const char *pcName) = 0 ;
    virtual AcMap::EErrCode RemoveCategory  (AcMapId Id) = 0 ;
    
    // Adds a new query category. If initially the pointer is NULL, a new 
    // interface object is allocated. An application is responsible for deleting 
    // it.
    // 
    virtual AcMap::EErrCode AddCategory     (AcMapQueryCategory *&pCat,
											 const char *pcName) = 0 ;
                              
    // Accesses the reactor object.
    virtual void			AddReactor      (AcMapQueryLibraryReactor *pReact) const = 0 ;
    virtual void			RemoveReactor   (AcMapQueryLibraryReactor *pReact) const = 0 ;

protected:
							AcMapQueryLibrary  
											() ;
    virtual					~AcMapQueryLibrary
											() ;
} ;

class AcMapQueryCategory
{
public:
    virtual					~AcMapQueryCategory
											() ;

    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns the Id of the associated internal Map object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;

    // Counts the queries.
    //
    virtual int				CountQueries    () const = 0 ;                                            
    
    // Sets the category name.
    //
    virtual AcMap::EErrCode SetName         (const char *pcName) = 0 ;

    // Gets the category name.
    virtual AcMap::EErrCode GetName         (const char *&pcName) const = 0 ;

    // Gets the category name. The output buffer should be released using the 
    // delete[] operator.
    //
    virtual AcMap::EErrCode GetName         (char *&pcName) const = 0 ;

    // Gets an interface object for the specified query attributes. If pObj
    // is NULL, a new interface object is allocated. An application is responsible 
    // for deleting this object.
    //
    virtual AcMap::EErrCode GetQuery        (AcMapQueryAttribute *&pObj, 
											 int iIndex) const = 0 ;
    virtual AcMap::EErrCode GetQuery        (AcMapQueryAttribute *&pObj, 
											 const char *pcName) const = 0 ;
    virtual AcMap::EErrCode GetQuery        (AcMapQueryAttribute *&pObj, 
											 AcMapId Id) const = 0 ;

    // Removes the query from the category and destroys it.
    //
    virtual AcMap::EErrCode RemoveQuery     (int iIndex) = 0 ;
    virtual AcMap::EErrCode RemoveQuery     (const char *pcName) = 0 ;
    virtual AcMap::EErrCode RemoveQuery     (AcMapId Id) = 0 ;

    // Gets the names of the nested query objects.
    //
    virtual AcMap::EErrCode GetQueryNames   (AcMapStringArray &tArray) const = 0 ;

    // Adds a link to an externally saved query. 
    //
    virtual AcMap::EErrCode AddQuery        (const char *pcName,
											 const char *pcDsc,
											 const char *pcFileName) = 0 ;

    // Adds a query internally.
    //
    virtual AcMap::EErrCode AddQuery        (const char *pcName,
											 const char *pcDsc,
											 const AcMapQuery &tQuery) = 0 ;
protected:
							AcMapQueryCategory
											() ;

} ;

class AcMapQueryAttribute
{
public:
    virtual					~AcMapQueryAttribute
											() ;
    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns the Id of the associated internal Map object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;

    // Sets the query name. The name should be a valid AutoCAD name unique 
    // within the query library.
    //
    virtual AcMap::EErrCode SetName         (const char *pcName) = 0 ;

    // Accesses the query name. 
    //
    virtual AcMap::EErrCode GetName         (const char *&pcName) const = 0 ;
    virtual AcMap::EErrCode GetName         (char *&pcName) const = 0 ;

    // Accesses the query description.
    //
    virtual AcMap::EErrCode SetDescription  (const char *pcDsc) = 0 ;
    virtual AcMap::EErrCode GetDescription  (const char *&pcDsc) const = 0 ;
    virtual AcMap::EErrCode GetDescription  (char *&pcDsc) const = 0 ;

    // Sets the name of the parent category. This method allows you to move a query 
    // between the query categories.
    //
    virtual AcMap::EErrCode SetCategory     (const char *pcName) = 0 ;
    virtual AcMap::EErrCode SetCategory     (AcMapId    Id) = 0 ;

    // Gets the associated category.
    //
    virtual AcMap::EErrCode GetCategory     (AcMapId    &Id) const = 0 ;
    virtual AcMap::EErrCode GetCategory     (const char *&pcName) const = 0 ;
    virtual AcMap::EErrCode GetCategory     (char *&pcName) const = 0 ;

    // Checks whether the query is saved externally.
    //
    virtual AcMap::EErrCode IsExternal      (Adesk::Boolean &bIsExtern) const = 0 ;

    // Accesses the full query file path for the external query. 
    //
    virtual AcMap::EErrCode SetFile         (const char *pcFileName) = 0 ;
    virtual AcMap::EErrCode GetFile         (const char *&pcFileName) const = 0 ;
    virtual AcMap::EErrCode GetFile         (char *&pcFileName) const = 0 ;

    // Loads the query. External query can be loaded only if the specified
    // query object refers to the current Map query.
    virtual AcMap::EErrCode Load            (AcMapQuery &tQuery) = 0 ;

protected:  
							AcMapQueryAttribute
											() ;
} ;

class AcMapRangeLibrary 
{
public:
    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Counts range tables.
    //
    virtual int				CountRangeTables() const = 0 ;                                            

    // Reinitializes the object.
    //
    virtual AcMap::EErrCode Clear           () = 0 ;
    
    // Gets the array of the range table names.
    //
    virtual AcMap::EErrCode GetRangeTableNames   
											(AcMapStringArray &tArray) const = 0 ;
    
    // Gets an interface object for the specified range table.
    // If initially the pointer is NULL, a new interface object is allocated. An application is 
    // responsible for deleting it.
    //
    virtual AcMap::EErrCode GetRangeTable   (AcMapRangeTable *&pTab,
											 int i) const = 0 ;
    virtual AcMap::EErrCode GetRangeTable   (AcMapRangeTable *&pTab,
											 const char *pcName) const = 0 ;
    virtual AcMap::EErrCode GetRangeTable   (AcMapRangeTable *&pTab,
											 AcMapId Id) const = 0 ;
                                             
    // Removes the specified range table.
    //
    virtual AcMap::EErrCode RemoveRangeTable(int i) = 0 ;
    virtual AcMap::EErrCode RemoveRangeTable(const char *pcName) = 0 ;
    virtual AcMap::EErrCode RemoveRangeTable(AcMapId Id) = 0 ;
    
    // Adds new range table. If initially pointer has NULL value, a new interface object is allocated. 
    // An application is responsible for deleting it.
    //
    virtual AcMap::EErrCode AddRangeTable   (AcMapRangeTable *&pObj,
											 const char *pcName,
											 const char *pcDsc = NULL) = 0 ;
                                                 
protected:
							AcMapRangeLibrary  
											() ;
    virtual					~AcMapRangeLibrary
											() ;
} ;

class AcMapRangeTable
{
public:
    virtual					~AcMapRangeTable() ;

    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns the identifier of the range table's internal object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;                  

    // Counts range lines.
    //
    virtual int				CountRangeLines () const = 0 ;

    // Reinitializes the object.
    //
    virtual AcMap::EErrCode Clear           () = 0 ;

    // Accesses the table name.
    //
    virtual AcMap::EErrCode SetName         (const char *pcName) = 0 ;
    virtual AcMap::EErrCode GetName         (const char *&pcName) const = 0 ;
    virtual AcMap::EErrCode GetName         (char *&pcName) const = 0 ;

    // Accesses the table description.
    //
    virtual AcMap::EErrCode SetDescription  (const char *pcString) = 0 ;
    virtual AcMap::EErrCode GetDescription  (const char *&pcString) const = 0 ;
    virtual AcMap::EErrCode GetDescription  (char *&pcString) const = 0 ;

    // Removes the range line.
    //
    virtual AcMap::EErrCode RemoveRangeLine (int i) = 0 ;
    virtual AcMap::EErrCode RemoveRangeLine (AcMapId Id) = 0 ;

    // Gets the range line.
    //
    virtual AcMap::EErrCode GetRangeLine    (AcMapRangeLine *&pObj,
											 int i) const = 0 ;
    virtual AcMap::EErrCode GetRangeLine    (AcMapRangeLine *&pObj,
											 AcMapId Id) const = 0 ;

    // Adds new range line. Only add a line with AcMap::kRangeOtherwise operation
    // if the table is not empty. pcDefinition must be NULL in this case.
    // You cannot add a new line to the table if the last line contains the 
    // AcMap::kRangeOtherwise operation.
    //
    virtual AcMap::EErrCode AddRangeLine    (AcMap::ERangeOperator kOp,
											 const char *pcDefinition,
											 const char *pcRetValue) = 0 ;
                                            
protected:
							AcMapRangeTable () ;
} ;

class AcMapRangeLine
{
public:
    virtual					~AcMapRangeLine () ;

    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns the identifier of the corresponding internal object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;                  

    // Sets the range definition. If the rangle line's operation is AcMap::kRangeOtherwise,
    // pcStr must be NULL, otherwise must not be NULL.
    //
    virtual AcMap::EErrCode	SetRangeDefinition
											(const char *pcStr) = 0 ;

    // Gets the range definition.
    //
    virtual AcMap::EErrCode GetRangeDefinition
											(const char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode GetRangeDefinition
											(char *&pcStr) const = 0 ;

    // Sets the range operation. AcMap::kRangeOtherwise is allowed only for the last line
    // in the table. If the operation is AcMap::kRangeOtherwise, the range definition is dropped.
    //
    virtual AcMap::EErrCode SetRangeOperation
											(AcMap::ERangeOperator kOp) = 0 ;

    // Gets the range operation.
    //
    virtual AcMap::EErrCode GetRangeOperation
											(AcMap::ERangeOperator &kOp) const = 0 ;

    // Accesses the range value.
    //
    virtual AcMap::EErrCode SetRangeValue   (const char *pcStr) = 0 ;
    virtual AcMap::EErrCode GetRangeValue   (const char *&pcStr) const = 0 ;
    virtual AcMap::EErrCode GetRangeValue   (char *&pcStr) const = 0 ;

protected:
							AcMapRangeLine  () ;
};

class AcMapErrorStack 
{
public:
    // Clears Map error stack.
    //
    virtual void			Clear           () const = 0 ;
    
    // Gets the number of entries in Ade error stack.
    //
    virtual int				CountEntries    () const = 0 ;
    
    // Pushes new entry in Map Error Stack.
    //
    virtual Adesk::Boolean  
							Push            (AcMap::EErrCode kCode, 
											 AcMap::EErrType kType,
											 const char *pcMessage) const = 0 ;
                                             
    // Sets additional parameters to the last pushed error.
    //
    virtual Adesk::Boolean  
							SetErrorParameter
											(int		classId,
											 const char *pcMessage,
											 const char *pcSql,
											 int iPos) const = 0 ; 
                             
                             
    // Displays "ADE warning" dialog in accordance with the current CMDDIA setting 
    // and clears the stack. Caption of the dialog can be specified by the caption parameter. 
    // pParent optional parameter allows you to specify the parent box information.
    //
    virtual void			Print           (const char * pcCaption = NULL,
											 CWnd* pParent = NULL) const = 0 ;
    
    // Gets error entry interface object for the I-th error.
    //
    virtual Adesk::Boolean  
							GetEntry        (AcMapErrorEntry *&pObj,
											 int iIndex) const = 0 ;

    
protected:
							AcMapErrorStack () ;
    virtual					~AcMapErrorStack() ;
};

class AcMapErrorEntry
{
public:

    virtual					~AcMapErrorEntry() ;
                                            
    // Adds additional parameters to the last pushed error.
    //
    virtual Adesk::Boolean  
							AddParameter    (int		Id,
											 const char *pcMessage,
											 const char *pcSql,
											 int iPos) const = 0 ; 

    // Gets the number of error parameters.
    //
    virtual	int				CountParameters () const = 0 ;
    
    // Gets error type.
    //
    virtual AcMap::EErrType	ErrorType       () const = 0 ;
    
    // Gets error code.
    //
    virtual int				ErrorCode       () const = 0 ;
    
    // Gets error message.
    //
    virtual Adesk::Boolean  
							GetErrorMessage (const char* &pMsg) const = 0 ;
    virtual Adesk::Boolean  
							GetErrorMessage (char *&pMsg) const = 0 ;
    
    // Gets and optionally allocates the interface to the i-th error parameter.
    //
    virtual Adesk::Boolean  
							GetParameter    (AcMapErrorParameter *&pObj,
											 int iIndex) const = 0 ; 
                                             
protected:
							AcMapErrorEntry() ;
} ;

class AcMapErrorParameter
{
public:
    virtual					~AcMapErrorParameter
											() ;

    // Gets class Id of the parameter.
    //
    virtual int				GetAdeClassId   () const = 0 ;
    
    // Gets error description for the parameter.
    //
    virtual Adesk::Boolean  
							GetParameterDescription
											(const char* &) const = 0 ;
    virtual Adesk::Boolean  
							GetParameterDescription
											(char *&) const = 0 ;
                                            
    // Gets statement for the parameter.
    //
    virtual Adesk::Boolean  
							GetStatement    (const char*&) const = 0 ;
    virtual Adesk::Boolean  
							GetStatement    (char *&) const = 0 ;
    
    // Gets error position in the statement.
    //
    virtual int				GetErrorPosition() const = 0 ;
    
protected:
							AcMapErrorParameter
											() ;
                            
} ;

class AcMapQuery
{
public:
							~AcMapQuery     () ;

    // Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

    // Returns Id of the associated internal object.
    //
    virtual AcMapId			ObjectId        () const = 0 ;

    // Clears the query definition. Depending on bClearAll, clears either
    // query, property alteration and report template or only the query 
    // definition itself.
    //
    virtual AcMap::EErrCode Clear           (Adesk::Boolean bClearAll = Adesk::kTrue) = 0 ;

    // Accesses the query mode.
    //
    virtual AcMap::EErrCode SetMode         (AcMap::EQueryType kType) = 0 ;
    virtual AcMap::EErrCode GetMode         (AcMap::EQueryType &kType) = 0 ;

    // Defines a query with a AcMapQueryBranch object. 
    //
    virtual AcMap::EErrCode Define          (const AcMapQueryBranch *pObj) = 0 ;

    // Constructs the AcMapQueryBranch object for the query definition.
    // All changes to this branch affect the query definition only after call Define() with
    // this branch specified.
    //
    virtual AcMap::EErrCode GetBranch       (AcMapQueryBranch *&pObj) const = 0 ;

    // Runs the query. 
    //
    virtual AcMap::EErrCode Run             () const = 0 ;

    // Represents the query in the output string. An application is responsible for 
    // releasing the string buffer using delete [].
    //
    virtual AcMap::EErrCode GetStringContext(char *&pcBuff) const = 0 ;

    // Associates the interface object with the current query. If the current query is not empty, 
    // gets query properties, feature alteration and templates from the current query.
    //
    virtual AcMap::EErrCode LoadFromCurrent () = 0 ;

    // Clears the current query and associates the current query with the interface object. 
    //
    virtual AcMap::EErrCode SetCurrent      () = 0 ;

    // Saves the query externally. If file with the specified name already exists, it 
    // will be rewritten.
    // Options parameter can be created by combining any of the following bit values:
    // AcMap::kSaveDrawingSet             - to save active drawings in a query file
    // AcMap::kSaveLocationCoordinates    - to save location condition coordinates in a query file
    // AcMap::kSavePropertyAlteration     - to save property alteration in a query file
    // AcMap::kAutoExecute                - to execute the query while loading  
    //
    virtual AcMap::EErrCode Save            (const char* pcPath, int iOptions) const = 0 ;

    // Restores a query from the external file. This method can be applied only to the current query. 
    //
    virtual AcMap::EErrCode Load            (const char* pcPath) = 0 ;

    // Saves the query internally. If the specified entry already exists in the query library, it will
    // be rewritten. Options parameter can be created by combining any of the following bit values:
    // AcMap::kSaveDrawingSet             - to save active drawings in a query file
    // AcMap::kSaveLocationCoordinates    - to save location condition coordinates in a query file
    // AcMap::kSavePropertyAlteration     - to save property alteration in a query file
    // AcMap::kAutoExecute                - to execute the query while loading
    //  
    // If fileName is specified, query is saved in the file and file name is stored in the query library.
    // Options only make sense in this case.
    //
    virtual AcMap::EErrCode Save            (const char* pcCategory,
											 const char* pcName,
											 const char* pcDesc = NULL,
											 const char* pcFileName = NULL,
											 int         iOption = 0) const = 0 ;

    // Restores a query from the specified category and name. If
    // a query library contains only a link to the query file, the query can't be loaded
    // and code kErrCantLoadExternQuery will be returned.
    //
    virtual AcMap::EErrCode Load            (const char* pcCategory,
											 const char* pcName) = 0 ;

    // Gets the flag value indicating whether the property alteration is applied.
    //
    virtual AcMap::EErrCode IsPropertyAlterationEnabled
											(Adesk::Boolean &bIsEnabled) const = 0 ;

    // Sets the flag value indicating whether the property alteration is applied.
    //
    virtual AcMap::EErrCode EnablePropertyAlteration
											(Adesk::Boolean bIsEnabled) const = 0 ;

    // Checks whether property alteration is defined.
    //
    virtual AcMap::EErrCode IsPropertyAlterationDefined
											(Adesk::Boolean &bIsDefined) const = 0 ;

    // Checks whether a report template is defined.
    //
    virtual AcMap::EErrCode IsReportTemplateDefined
											(Adesk::Boolean &bIsDefined) const = 0 ;

    // Creates AcMapPropertyAlterationDefinition interface object. 
    //
    virtual AcMap::EErrCode GetPropertyAlteration
											(AcMapPropertyAlterationDefinition *&pPAlt) = 0 ;

    // Creates AcMapReportTemplate interface object. 
    //
    virtual AcMap::EErrCode GetReportTemplate   
											(AcMapReportTemplate *&pTemp ) = 0 ;

protected:
							AcMapQuery      () ;
} ;

class   AcMapODContainer
{
public:
	// Gets the associated project.
	virtual AcMapProject*			
							Project			() const = 0 ;

        // Creates a new OD Table with the specified name, description 
    // and definition. The name should be a valid AutoCAD name. A description is 
    // optional; pcDsc can have NULL value if the description is empty. tDef 
    // object is allocated and released by an application. Depending on 
    // bXDataRecs parameter the table records will be stored either in XRecords 
    // (the default setting) or in XData. Table is created in the current drawing.
    //
    virtual AcMap::EErrCode CreateODTable   (const char *kpName, 
											 const AcMapODTableDefinition& kDef,
											 const char *kpDesc = NULL, 
											 Adesk::Boolean bXdataRecs = Adesk::kFalse) const = 0;

    // Returns the number of the OD Data Table in the accessible drawings.
    //
    virtual int				CountODTables   () const = 0;

    // Gets the names of the OD Data Tables in the accessible drawings.
    //
    virtual AcMap::EErrCode	GetODTableNames (AcMapStringArray& tNames) const = 0;

    // Allocates an object OD Record iterator for the specified 
    // drawing to iterate records of any OD table attached to the AutoCAD
    // object. 
    //
    virtual AcMap::EErrCode GetObjectODRecordIterator
											(AcMapODRecordIterator *&pIter,
											 AcMapId lDwgId = 0L) const = 0;

    // Checks whether the OD Table with the specified name is defined in the 
    // accessible drawings.
    //
    virtual Adesk::Boolean  IsODTableDefined(const char *pcTableName) const = 0;

    // Gets the specified OD Table. It allocates AcMapODTable object. The application is 
    // responsible for deleting it.
    //
    virtual AcMap::EErrCode GetODTable      (AcMapODTable *&pODTable, 
											 const char *pcTableName) const = 0; 

	// Removes Object Data table from the project and all active attached drawings. Nobody
	// can be using the source drawings.
	//
    virtual AcMap::EErrCode RemoveODTable   (const char *pcTableName)  = 0;

    // Updates Object Data table in the project and all active attached drawings. Nobody
	// can be using the source drawings.
	//
    virtual AcMap::EErrCode UpdateODTable   (const char *pcName, 
											 const AcMapODTableDefinition &Definition) = 0;

    // Renames Object Data table in the project and all active attached drawings. Nobody
	// can be using the source drawings.
	//
    virtual AcMap::EErrCode RenameODTable   (const char *pcOldName, 
											 const char *pcNewName) = 0;  

protected:
							AcMapODContainer();
    virtual					~AcMapODContainer
											();
};    

// Inlines.
//
inline AcMapQuery::AcMapQuery()
{}

inline AcMapQuery::~AcMapQuery()
{}

////////////////////////////////////////////////////////////////////////////////
//  AcMapODAccess inlines
inline
AcMapODContainer::AcMapODContainer()
{}

inline
AcMapODContainer::~AcMapODContainer()
{}


#endif // MAP_ARXAPI_H

// End of File.
