#ifndef MAP_BOUNDARY_H
#define MAP_BOUNDARY_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the definitions of classes that represent all 
// AutoCAD MAP location query conditions. They are as follows:
//
//   AcMapLocationBoundary
//      AcMapAllBoundary
//      AcMapPointBoundary
//          AcMapCircleBoundary
//      AcMapFenceBoundary
//          AcMapBufferFenceBoundary
//          AcMapPolygonBoundary
//              AcMapWindowBoundary
//      AcMapPolylineBoundary
//          AcMapBufferPolylineBoundary
//          AcMapClosedPolylineBoundary
//          
// AcMapPolylineVertex represents a polyline vertex in 
// AcMapPolylineBoundary classes. 
//
#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

#include <acdb.h>
#include <gepnt3d.h>
#include <gevec3d.h>
#include <gept3dar.h>
#include <adslib.h>
#include "MapConstants.h"
#include "MapColl.h"

class AcDbEntity ;
class AcMapPolylineVertex ;

typedef AcMapObjArray<AcMapPolylineVertex> AcMapVertexArray;

extern "C"
{
    Adesk::Boolean			pointWcsToEcs   (AcGePoint3d &p,
											 const AcGeVector3d &normal) ;

    Adesk::Boolean			pointEcsToWcs   (AcGePoint3d &p,
											 const AcGeVector3d &normal) ;

    Adesk::Boolean			vectorWcsToEcs  (AcGeVector3d &p,
											 const AcGeVector3d &normal) ;

    Adesk::Boolean			vectorEcsToWcs  (AcGeVector3d &p,
											 const AcGeVector3d &normal) ;

    int						WcsToEcs        (ads_point p,
											 const AcGeVector3d &normal,
											 int disp,
											 ads_point res) ;
    int						EcsToWcs        (ads_point p,
											 const AcGeVector3d &normal,
											 int disp,
											 ads_point res) ;
} ;

class AcMapPolylineVertex 
// 
// Class that represents vertex information
//
{
public:
							AcMapPolylineVertex     
											() ;
							AcMapPolylineVertex     
											(const AcGePoint3d &p, double bulge) ;
							AcMapPolylineVertex     
											(const AcMapPolylineVertex &) ;
    virtual					~AcMapPolylineVertex    
											() ;

    // Copy operator.
    //
    AcMapPolylineVertex&	operator =      (const AcMapPolylineVertex &Source) ;
    
    // Sets the vertex parameters.
    //
    void					Set             (const AcGePoint3d &p, double bulge) ;

    // Gets the vertex parameters.
    //
    double					GetBulge        () const ;
    const AcGePoint3d&		GetVertex       () const ;    
    
private:
    AcGePoint3d				mVertex ;       // Vertex coordinates
    double					mBulge ;        // Vertex bulge
} ;

// This is an abstract class that all boundary classes are derived from.
class AcMapLocationBoundary
{
public:

    virtual					~AcMapLocationBoundary  
											() ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const = 0;
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource)  ;

protected:
							AcMapLocationBoundary   
											() ;
} ;

// This class represents the "All" boundary condition.
class AcMapAllBoundary : public AcMapLocationBoundary
{
public:
							AcMapAllBoundary() ;

    virtual					~AcMapAllBoundary   
											() ;

    // Runtime class identification.
    //	
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be only an AcMapAllBoundary object.
    // If tSource is the wrong type, throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

} ;

// This class represents the "Point" boundary condition.
// The condition is defined by the coordinates of the point.
class AcMapPointBoundary : public AcMapLocationBoundary
{
public:
							AcMapPointBoundary      
								            () ;
							AcMapPointBoundary      
								            (const AcGePoint3d &p) ;
    virtual					~AcMapPointBoundary     
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be either an AcMapPointBoundary or AcMapCircle object.
    // If tSource is the wrong type, throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Accesses the coordinates.
    //
    virtual void			SetPoint        (const AcGePoint3d &p) ;
    virtual const AcGePoint3d&  
							GetPoint        () const ;

private:

    AcGePoint3d				mPoint ;        // Point coordinates
} ;

// This class represents the "Circle" boundary condition.
// The condition is defined by the coordinate of the center 
// point and the radius.
class AcMapCircleBoundary : public AcMapPointBoundary
{
public:
							AcMapCircleBoundary     
								            () ;
							AcMapCircleBoundary     
								            (const AcGePoint3d &p,
									         double radius) ;
    virtual					~AcMapCircleBoundary  
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be either an AcMapPointBoundary or AcMapCircle object.
    // If tSource is not AcMapCircle object, width is set to 0.
    // If tSource is the wrong type, throws kAdeWrongType.
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Accesses the radius.
    //
    virtual void			SetRadius       (double rad) ;
    virtual double			GetRadius       () const ;

private:
    double					mRadius ;       // Circle radius
} ;

// This class represents the "Fence" boundary condition.
// The condition is defined by the coordinates of the points.
// There must be at least two different points.
class AcMapFenceBoundary : public AcMapLocationBoundary
{
public:
							AcMapFenceBoundary      
								            () ;

							AcMapFenceBoundary      
								            (const AcGePoint3dArray &pp) ;

    virtual					~AcMapFenceBoundary     
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapFence.
    // If tSource is the wrong type throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Accesses the points.
    //
    virtual void			SetPoints       (const AcGePoint3dArray &pp) ;
    virtual const AcGePoint3dArray& 
							GetPoints       () const ;
    virtual AcGePoint3dArray&   
							GetPoints       () ;


protected:

    AcGePoint3dArray		mPoints ;       // Array of coordinates
} ;


// This class represents the "Buffer Fence" boundary condition.
// The condition is defined by the coordinates of the points and
// the width of the buffer. A Buffer Fence with 0 width behaves just 
// like a Fence with the same points. 
class AcMapBufferFenceBoundary : public AcMapFenceBoundary
{
public:
							AcMapBufferFenceBoundary
								            () ;

							AcMapBufferFenceBoundary
								            (const AcGePoint3dArray &pp,
									         double width) ;

    virtual					~AcMapBufferFenceBoundary
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapFence.
    // If tSource is not an AcMapBufferFence object, width is set 0.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Accesses the buffer width.
    //
    virtual void			SetWidth        (double width) ;
    virtual double			GetWidth        () const ;

private:

    double					mWidth ;        // Buffer width
} ;

class AcMapPolygonBoundary : public AcMapFenceBoundary
{
public:
							AcMapPolygonBoundary    
								            () ;
							AcMapPolygonBoundary    
								            (const AcGePoint3dArray &) ;
    virtual					~AcMapPolygonBoundary   
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapFence.
    // If tSource is the wrong type throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Sets the points.
    //
    virtual void			SetPoints       (const AcGePoint3dArray &pp) ;
} ;


// This class represents the "Window" boundary condition.
// The condition is defined by the coordinates of two different 
// points. 
class AcMapWindowBoundary : public AcMapPolygonBoundary
{
public:
							AcMapWindowBoundary    
								            () ;

							AcMapWindowBoundary     
								            (const AcGePoint3d &p1,
									         const AcGePoint3d &p2) ;

    virtual					~AcMapWindowBoundary    
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapFence.
    // If tSource is the wrong type throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Sets the points.
    //
    virtual void			Init            (const AcGePoint3d &p1,
								             const AcGePoint3d &p2) ;
} ;

// This class represents the "Polyline Fence" boundary condition.
// The condition is defined by the Polyline, Line or Arc object.
class AcMapPolylineBoundary : public AcMapLocationBoundary
{
public:
							AcMapPolylineBoundary   
								            () ;

							AcMapPolylineBoundary   
								            (const AcMapVertexArray &tArray,
									         const AcGeVector3d &normal,
										     Adesk::Boolean isClosed) ;

    virtual					~AcMapPolylineBoundary  
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapPolyline.
    // If tSource is the wrong type throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Initialization by the following AutoCAD objects:
    // AcDb2dPolyline
    // AcDb3dPolyline
    // AcDbPolyline
    // AcDbLine
    // AcDbArc
    //
    virtual AcMap::EErrCode Init            (const AcDbObjectId &Id) ;

    // Gets the parameters.
    //
    virtual const AcMapVertexArray&  
							GetPoints       () const ;
    virtual const AcGeVector3d& 
							GetNormal       () const ;
    virtual Adesk::Boolean  IsClosed        () const ;

private:
    AcMap::EErrCode         Init			(AcDbEntity *pEnt) ;

    AcGeVector3d			mNormal ;       // Object normal vector
    AcMapVertexArray		mVertexArray ;  // Array of vertexes
    Adesk::Boolean			mIsClosed ;     // Flag for the closed contours
} ;

// This class represents the "Polyline Buffer Fence" boundary condition.
// The condition is defined by the Polyline, Line or Arc object and the 
// buffer width. If the width is 0, it behaves just like a Polyline Fence
// initialized by the AutoCAD object.
class AcMapBufferPolylineBoundary : public AcMapPolylineBoundary 
{
public:
							AcMapBufferPolylineBoundary 
											() ;

							AcMapBufferPolylineBoundary 
								            (const AcMapVertexArray &tArray,
									         const AcGeVector3d &normal,
										     Adesk::Boolean isClosed,
											 double width) ;
                                                        
	virtual					~AcMapBufferPolylineBoundary    
								            () ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapPolyline.
    // If tSource is not an AcMapBufferPolyline object, width is set 0.
    // If tSource is the wrong type throws kAdeWrongType.
    //
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Initialization by the following AutoCAD objects:
    // AcDb2dPolyline
    // AcDb3dPolyline
    // AcDbPolyline
    // AcDbLine
    // AcDbArc
    //
    virtual AcMap::EErrCode Init            (const AcDbObjectId &Id,
											 double width) ;

    // Access the buffer width.
    //
    virtual void			SetWidth        (double width) ;
    virtual double			GetWidth        () const ;

private:

    double					mWidth ;		// Buffer width
} ;

class AcMapClosedPolylineBoundary : public AcMapPolylineBoundary 
{
public:
							AcMapClosedPolylineBoundary 
											() ;

							AcMapClosedPolylineBoundary 
											(const AcMapVertexArray &tArray,
											 const AcGeVector3d &normal,
											 Adesk::Boolean isClosed = Adesk::kTrue) ;

    virtual					~AcMapClosedPolylineBoundary    
											() ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapLocationBoundary*
							Clone           () const ;

    // tSource can be an object of any class derived from AcMapPolylineBoundary.
    // If tSource is the wrong type throws kAdeWrongType.
    virtual AcMapLocationBoundary&
							operator =      (const AcMapLocationBoundary &tSource) ;

    // Initialization by the following AutoCAD objects:
    // AcDb2dPolyline
    // AcDb3dPolyline
    // AcDbPolyline
    // AcDbArc
    //
    virtual AcMap::EErrCode Init            (const AcDbObjectId &Id) ;
} ;

#endif /* MAP_BOUNDARY_H */
