#ifndef MAP_COLL_H
#define MAP_COLL_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the definitions of templates for dynamic arrays, called
// AcMapObjPtrArray and AcMapObjArray.
//
// "Dynamic array" means that the array can grow without bounds,
// unlike declaring an array of objects of type "ApiObj" in the
// usual manner.  For example declaring "ApiObj myArray[10]" is 
// limited to holding only ten entries.
//
// In order to use the templates AcMapObjPtrArray and AcMapObjArray, you need to 
// understand a couple of simple, yet key, concepts:
//
//     1) The logical length of the array.
//            - How many entries have been placed into the array,
//              initially always zero.
//     2) The physical length of the array.
//            - How many entries the array will hold before it
//              automatically "grows" larger.
//     3) The grow length of the array.
//            - How much the array will grow when required.
//
// The physical length of the array is the actual length of the
// physically allocated, but perhaps not fully used, array.
// As a point of clarification, the size in bytes of the array
// buffer for AcMapObjArray array called `myArray' would be:
//
//     sizeOf(ApiObj) * myArray.physicalLength().
//
// The physical length of the array can be zero or any positive
// integer.
//
// The logical length of the array (or just the "length()") reflects
// how many elements of AcDbObjectId have been placed into the array
// with, for example, append() or insertAt().  Many member-functions
// are only valid for indicies that are greater than or equal to
// zero AND less than length().  For example, indexing into the
// array with the operator[] is only valid for indices in this range.
//
// You can explicitly set the logical Length() to any value and
// if the physical length is not large enough the array will grow to
// that length.  Note that if the logical length is explicitly reset
// to a larger value, then all the entries from the old length up
// to the new length may contain garbage values, therefor they must be
// initialized explicitly.
//
// The logical length is always less than or equal to the physical
// length.  NOTE that the array ALWAYS starts out empty, i.e., the
// length() always starts at zero regardless of the initial physical
// length.
//
// If you add an element to the array causing the logical length
// to become greater than the physical length of the array then
// the "grow length" determines how much additional space to
// allocate, and the physical length will increase by the grow length.
//
// AcMapObjPtrArray stores pointers to the ApiObj objects in the member data.
// An application is responsible for allocating and destructing these objects.
//
// AcMapObjArray stores ApiObj objects themself in the member data. To use this
// template for collecting objects of a class, this class must have operator =
// defined.
//
#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

#include <memory.h>

const int ARRAY_GROW_LENGTH = 10 ;

template <class ApiObj> class AcMapObjPtrArray
{
public:
                        AcMapObjPtrArray() ;

    virtual             ~AcMapObjPtrArray       
                                        () ;

    // Indexing into the array.
    //
    virtual ApiObj*     operator[]      (int) const ;

    // More access to the array elements.
    //
    virtual ApiObj*     Last            () const ;
    virtual ApiObj*     First           () const ;
    virtual const ApiObj&       
                        GetAt           (int i) const ;

    // Copy operator.
    //
    virtual void        operator =      (const AcMapObjPtrArray &) ;

    // Adding array elements.
    //
    virtual void        Append          (ApiObj *) ;
    virtual void        InsertAt        (int pos, ApiObj *) ;

    // Removing array elements.
    //
    virtual void        RemoveAll       () ;
    virtual ApiObj*     Remove          (int i) ;

    // Array length.
    //
    virtual int         Length          () const ;

    // Treats as simple array of ApiObj*.
    //
    ApiObj**            AsArrayPtr      () ;
    const ApiObj**      AsArrayPtr      () const ;

protected:
    void                SetPhysicalLength       
                                        (int) ;

private:

    int                 mLogicalLength ;// Number of items in the array
    int                 mPhysicalLength ;   
                                        // Actual buffer length
    ApiObj              **mpArray ;     // Array buffer 
    
} ;

template <class ApiObj> class AcMapObjArray
{
public:
                        AcMapObjArray   () ;

    virtual             ~AcMapObjArray  () ;

    // Indexing into the array.
    //
    virtual const ApiObj&       
                        operator[]      (int) const ;

    // More access to the array elements.
    //
    virtual const ApiObj&       
                        First           () const ;
    virtual const ApiObj&       
                        Last            () const ;

    // Copy operator.
    //
    virtual void        operator =      (const AcMapObjArray &) ;

    // Adding array elements.
    //
    virtual void        Append          (const ApiObj &) ;
    virtual void        InsertAt        (int pos, const ApiObj &) ;

    // Removing array elements.
    //
    virtual void        RemoveAll       () ;
    virtual void        Remove          (int i) ;

    // Array length.
    //
    virtual int         Length          () const ;

    // Treats as simple array of ApiObj.
    //
    ApiObj*             AsArrayPtr      () ;
    const ApiObj*       AsArrayPtr      () const ;


private:
    void                SetPhysicalLength       
                                        (int) ;

    int                 mLogicalLength ;// Number of items in the array
    int                 mPhysicalLength ;   
                                        // Actual buffer length
    ApiObj              *mpArray ;      // Array buffer
    
} ;


// Template AcMapObjPtrArray
template <class ApiObj> 
inline AcMapObjPtrArray<ApiObj>::AcMapObjPtrArray()
{
    mpArray = new ApiObj *[ARRAY_GROW_LENGTH] ;
    mLogicalLength = 0 ;
    mPhysicalLength = ARRAY_GROW_LENGTH ;
}

template <class ApiObj> 
inline AcMapObjPtrArray<ApiObj>::~AcMapObjPtrArray()
{
    delete [] mpArray ;
}

template <class ApiObj> 
inline int
AcMapObjPtrArray<ApiObj>::Length()
const
{
    return mLogicalLength ;
}

template <class ApiObj> ApiObj**                
AcMapObjPtrArray<ApiObj>::AsArrayPtr() 
{
    return mpArray ;
}

template <class ApiObj> const ApiObj**              
AcMapObjPtrArray<ApiObj>::AsArrayPtr() 
const
{
    return mpArray ;
}

template <class ApiObj> 
void
AcMapObjPtrArray<ApiObj>::Append(
    ApiObj *pObj
)
{
    if (pObj)
    {
        try {

            if (mLogicalLength >= mPhysicalLength) 
            {
                SetPhysicalLength(mLogicalLength + ARRAY_GROW_LENGTH);
            }
 
            // set string value
            mpArray[mLogicalLength] = (ApiObj *)pObj ;
    
            mLogicalLength++;
        }
        catch(...)
        {}
    }
}

template <class ApiObj> 
void
AcMapObjPtrArray<ApiObj>::InsertAt(
    int pos,
    ApiObj *pObj
)
{
     if (mLogicalLength >= mPhysicalLength) 
     {
         SetPhysicalLength(mLogicalLength + ARRAY_GROW_LENGTH);
     }
 
     if (pos != mLogicalLength) 
     {
         // move elements from index right
         register ApiObj **p = (ApiObj**)(mpArray + mLogicalLength);
         register ApiObj **pStop = (ApiObj**)(mpArray + pos);
         do {
             *p = *(p-1);
         } while (--p != pStop);
     }
     
     // set string value
     mpArray[pos] = (ApiObj *)pObj ;
     
     mLogicalLength++;
}

template <class ApiObj> 
void
AcMapObjPtrArray<ApiObj>::RemoveAll()
{
    mLogicalLength = 0 ;
}

template <class ApiObj> 
ApiObj*
AcMapObjPtrArray<ApiObj>::Remove(int pos)
{
     ApiObj *pRet = NULL ;

     // Shift array elements to the left if needed.
     if (pos >= 0 && pos < mLogicalLength) 
     {
         // free index element
         pRet = mpArray[pos] ;
     
         register ApiObj **p = (ApiObj**)(mpArray + pos) ;
         register ApiObj **pStop = (ApiObj**)(mpArray + mLogicalLength - 1) ;

         while (p != pStop)
         {
             *p = *(p+1);
             p++ ;
         } 

         mLogicalLength--;                             
     }

     return pRet ;
}

template <class ApiObj> 
ApiObj *
AcMapObjPtrArray<ApiObj>::operator[](
    int index
) const 
{
    try {

        if (index < 0 || index >= mLogicalLength)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[index] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
ApiObj *
AcMapObjPtrArray<ApiObj>::First() const 
{
    try {

        if (mLogicalLength == 0)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[0] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
ApiObj *
AcMapObjPtrArray<ApiObj>::Last() const 
{
    try {

        if (mLogicalLength == 0)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[mLogicalLength-1] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
const ApiObj&       
AcMapObjPtrArray<ApiObj>::GetAt(int index) const 
{
    try {

        if (index < 0 || index >= mLogicalLength)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return *mpArray[index] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
void
AcMapObjPtrArray<ApiObj>::SetPhysicalLength(int n)
{
    if (n == mPhysicalLength) return ;
    
    ApiObj **pOldArray = mpArray;

    if (n >= mLogicalLength)
    {
        if (n == 0) 
        { 
            mpArray = NULL;
            mPhysicalLength = 0;
        } 
        else 
        {
            // Get the required amount of space.
//          mpArray = (ApiObj**)new unsigned char[n*sizeof(ApiObj*)] ;
            mpArray = new ApiObj *[n] ;
            if (mpArray == NULL) 
            {
                mPhysicalLength = 0;
            } 
            else 
            {
                if (mLogicalLength > 0) 
                {
                    memcpy(mpArray, pOldArray, ((n < mLogicalLength) ? n : mLogicalLength)
                                    * sizeof(ApiObj*));
                }
                mPhysicalLength = n;
            }
        }
    }

    if (pOldArray != NULL) 
    {
        delete[] pOldArray;
    }
}

template <class ApiObj> 
void
AcMapObjPtrArray<ApiObj>::operator = (const AcMapObjPtrArray &Source)
{
//  mpArray = (ApiObj**)new unsigned char[Source.mPhysicalLength*sizeof(ApiObj)] ;
    mpArray = new ApiObj*[Source.mPhysicalLength] ;
    mLogicalLength = Source.mLogicalLength ;
    mPhysicalLength = Source.mPhysicalLength ;
    for (int i = 0; i < Source.mLogicalLength; i++)
    {
        mpArray[i] = Source.mpArray[i] ;
    }
}

// Template AcMapObjArray
template <class ApiObj> 
inline AcMapObjArray<ApiObj>::AcMapObjArray()
{
    mpArray = new ApiObj[ARRAY_GROW_LENGTH];
    mLogicalLength = 0 ;
    mPhysicalLength = ARRAY_GROW_LENGTH ;
}

template <class ApiObj> 
inline AcMapObjArray<ApiObj>::~AcMapObjArray()
{
    delete [] mpArray ;
}

template <class ApiObj> 
inline int
AcMapObjArray<ApiObj>::Length()
const
{
    return mLogicalLength ;
}

template <class ApiObj> 
void
AcMapObjArray<ApiObj>::Append(
    const ApiObj &pObj
)
{
    try {

        if (mLogicalLength >= mPhysicalLength) 
        {
            SetPhysicalLength(mLogicalLength + ARRAY_GROW_LENGTH);
        }
 
        // set string value
        mpArray[mLogicalLength] = pObj ;
    
        mLogicalLength++;
    }
    catch(...)
    {}
}

template <class ApiObj> 
void
AcMapObjArray<ApiObj>::InsertAt(
    int pos,
    const ApiObj &pObj
)
{
     if (mLogicalLength >= mPhysicalLength) 
     {
         SetPhysicalLength(mLogicalLength + ARRAY_GROW_LENGTH);
     }
 
     if (pos != mLogicalLength) 
     {
         // move elements from index right
         register ApiObj *p = (ApiObj*)(mpArray + mLogicalLength);
         register ApiObj *pStop = (ApiObj*)(mpArray + pos);
         do {
             *p = *(p-1);
         } while (--p != pStop);
     }
     
     // set string value
     mpArray[pos] = pObj ;
     
     mLogicalLength++;
}

template <class ApiObj> 
inline void
AcMapObjArray<ApiObj>::RemoveAll()
{
    mLogicalLength = 0 ;
}


template <class ApiObj> 
inline ApiObj*
AcMapObjArray<ApiObj>::AsArrayPtr()
{
    return mpArray ;
}

template <class ApiObj> 
inline const ApiObj*
AcMapObjArray<ApiObj>::AsArrayPtr() const
{
    return mpArray ;
}

template <class ApiObj> 
const ApiObj &
AcMapObjArray<ApiObj>::operator[](
    int index
) const 
{
    try {

        if (index < 0 || index >= mLogicalLength)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[index] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
const ApiObj &
AcMapObjArray<ApiObj>::First() const 
{
    try {

        if (mLogicalLength == 0)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[0] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
const ApiObj &
AcMapObjArray<ApiObj>::Last() const 
{
    try {

        if (mLogicalLength == 0)
        {
            throw (AcMap::kErrWrongArgument) ;
        }
        return mpArray[mLogicalLength-1] ;
    }
    catch(...)
    {
        throw (AcMap::kErrWrongArgument) ;
    }
}

template <class ApiObj> 
void
AcMapObjArray<ApiObj>::Remove(int pos)
{
     // Shift array elements to the left if needed.
     if (pos >= 0 && pos < mLogicalLength) 
     {
         register ApiObj *p = (ApiObj*)(mpArray + pos) ;
         register ApiObj *pStop = (ApiObj*)(mpArray + mLogicalLength - 1) ;

         while (p != pStop)
         {
             *p = *(p+1);
             p++ ;
         } 

         mLogicalLength--;                             
     }
}

template <class ApiObj> 
void
AcMapObjArray<ApiObj>::SetPhysicalLength(int n)
{
    if (n == mPhysicalLength) return ;
    
    ApiObj *pOldArray = mpArray;

    if (n == 0) 
    { 
        mpArray = NULL;
        mPhysicalLength = 0;
    } 
    else 
    {
        // Get the required amount of space.
        mpArray = new ApiObj[n] ;

        if (mpArray == NULL) 
        {
            mPhysicalLength = 0;
        } 
        else 
        {
            if (mLogicalLength > 0) 
            {
                memcpy(mpArray, pOldArray, ((n < mLogicalLength) ? n : mLogicalLength)
                    * sizeof(ApiObj));
            }
            mPhysicalLength = n;
        }
    }

    if (pOldArray != NULL) 
    {
        delete [] pOldArray;
    }
}

template <class ApiObj>
void
AcMapObjArray<ApiObj>::operator = (const AcMapObjArray &Source)
{
    delete [] mpArray ;
    mpArray = new ApiObj[Source.mPhysicalLength];
    mLogicalLength = Source.mLogicalLength ;
    mPhysicalLength = Source.mPhysicalLength ;
    for (int i = 0; i < Source.mLogicalLength; i++)
    {
        mpArray[i].operator=(Source.mpArray[i]) ;
    }
}
#endif /* MAP_COLL_H */

