#ifndef MAP_OBJECTID_H
#define MAP_OBJECTID_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
//  This file contains the definitions of the AutoCAD object identification 
//  within ADE session class - AcMapObjectId.
//  AutoCAD ARX API introduces the AcDbObjectId class to identify an AutoCAD
//  object within one editing session, even if multiple drawing databases 
//  are loaded. On the other hand, there is an object handle, that identifies 
//  an object in the multiple editing sessions, but the scope of the handle 
//  is only the drawing database. Neither the AcDbObjectId nor the object handle 
//  is suitable for identifying an AutoCAD object within a MAP editing session, because 
//  MAP could temporarily unload the drawing databases within one editing session. 
//  To resolve this issue, the MAP ARX API introduces the AcMapObjectId class.  
//  It is similar to the AcDbObjectId class. The AcMapObjectId class stores the 
//  object handle and the Id of the Source drawing that the object belongs to. 
//
//  The AcMapObjectIdArray class has the same functionality as the AcDbObjectIdArray.
//  Some MAP classes provide an application with access to the drawing objects 
//  from the external databases. There are two variants of such a method, which take 
//  AcDbObjectId and AcMapObjectId as the parameters. Similarly, there are 
//  methods that take AcDbObjectIdArray and AcMapObjectIdArray as the parameters. 
//  If an application uses AcDbObjectId and AcDbObjectIdArray, it should not call 
//  any methods that can possibly unload the drawing database, because this will 
//  render the object Ids invalid. Methods that fall into this category are the Drawing 
//  Set methods and the methods that run a query. Use of invalid Ids to access an AutoCAD object 
//  will crash AutoCAD.

#include <adesk.h>
#include <MapConstants.h>

// Forward declarations.
class AcMapAttachedDrawing;
class AcDbObjectId;
class AcDbHandle;
class AcMapProject;

class   AcMapObjectId
{
public:
							AcMapObjectId   ();
							AcMapObjectId   (const AcDbObjectId &dbId); 
							AcMapObjectId   (const AcDbHandle &tHandle, AcMapProject *pProject, AcMapId lDwgId); 
							AcMapObjectId   (const AcMapObjectId &tSource); 

    virtual					~AcMapObjectId  (); 

    //  Copy operator.
    //
    AcMapObjectId&			operator =      (const AcMapObjectId &Source);

    //  Comparison operators.
    //
    Adesk::Boolean			operator ==     (const AcMapObjectId &Source) const; 
    Adesk::Boolean			operator !=     (const AcMapObjectId &Source) const; 
    
    // Gets the identifier of the corresponding AcMapAttachedDrawing object.
    // 0 indicates an object in the current drawing.
    //
    AcMapId					DrawingId       () const; 

    // Gets the corresponding AcDbObjectId.
    AcMap::EErrCode			GetDbId         (AcMapProject *pProject, AcDbObjectId &tId) const; 

    //  Gets the object handle.
    //
    AcMap::EErrCode			ObjectHandle    (AcDbHandle& dbHandle) const; 

    //  Object initialization.
    //
    AcMap::EErrCode			Init            (const AcDbObjectId &Id, 
											 const AcMapAttachedDrawing *pDwg); 
    AcMap::EErrCode			Init            (const AcMapObjectId &tId); 
    AcMap::EErrCode			Init            (const AcDbHandle &tHandle, 
											 AcMapProject *pProject,
											 AcMapId lDwgId); 

    //  Empty initialization.
    //
    void					NullId          ();

private:

    //  Member Data
    union {                             // Identification of the AutoCAD object
        Adesk::UInt8    handle[8];      // handle for the external object
        long            dbid;           // Id for the current drawing object
    } m_Handle;  

    AcMapId             m_DwgHandle;    // Id of the AcMapAttachedDrawing object
};
#endif /*MAP_OBJECTID_H*/
/*EOF*/
