#ifndef MAP_QUERY_H
#define MAP_QUERY_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the descriptions of ADE ARX API classes, that represent 
// the components of ADE query. They are as follows:
//
//  AcMapQueryUnit
//      AcMapQueryBranch
//      AcMapQueryCondition
//          AcMapLocationCondition
//          AcMapPropertyCondition
//          AcMapDataCondition
//          AcMapSQLCondition
//
#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

#include <adesk.h>
#include "MapConstants.h"
#include <MapColl.h>

// Forward declarations.
class AcMapLocationBoundary ;
class AcMapQueryUnit ;

typedef AcMapObjPtrArray<AcMapQueryUnit> AcMapOperandArray;

// An abstract class for the query definition utit
class AcMapQueryUnit
{
public:
    virtual					~AcMapQueryUnit         
											() ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit*	Clone           () const = 0 ;

    // Access the logical operator.
    //
    void					SetJoinOperator (AcMap::EJoinOperator kOp) ;
    AcMap::EJoinOperator    JoinOperator    () const ;

protected:
							AcMapQueryUnit  (AcMap::EJoinOperator kOp
                                                = AcMap::kOperatorAnd) ;

    AcMap::EJoinOperator	mOperator ;     // Join operator

} ;

class AcMapQueryBranch : public AcMapQueryUnit 
{
public:
							AcMapQueryBranch(AcMap::EJoinOperator kOp
                                                = AcMap::kOperatorAnd) ;

	virtual					~AcMapQueryBranch       
											() ;

    // Runtime class identification.
    //
    virtual AcMap::EClassId IsA             () const ;
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;

    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit* Clone           () const ;

    // Operands count.
    //
    int						CountOperands   () const ;

    // Access the operands.
    //
    AcMap::EErrCode         GetOperand      (const AcMapQueryUnit *&pObj,
											 int i) const ;

    // Removing the operands.
    //
    AcMap::EErrCode         RemoveOperand   (int i) ;
    virtual void			Clear           () ;

    // Adding the operand.
    //
    AcMap::EErrCode         AppendOperand   (const AcMapQueryUnit *pObj) ;
    AcMap::EErrCode         InsertOperand   (int i,
								             const AcMapQueryUnit *pObj) ;

    // Substituting the operand.
    //
    AcMap::EErrCode         SubstituteOperand
									        (int i, 
										     const AcMapQueryUnit *pObj) ;

    // Joining the specified operands into a nested branch. The join operator 
    // of the branch is the join operator of the first operand. For example:
    // A & B || C after calling Group (1,1) will be A & (B || C)
    //
    AcMap::EErrCode         Group           (int nFirst, 
								             int nNumber) ;

    // Ungrouping the operands. If the specified operand is a branch, removes it 
    // and inserts the operands of this branch instead. The join operator of the 
    // first operand is the join operator of the branch, for example:
    // A & (B || C) after calling Ungroup(1) will be A & B || C
    //
    AcMap::EErrCode         Ungroup         (int iPos) ;

    // Root branch identification.
    //
    Adesk::Boolean			IsRoot          () const ;

private:
    Adesk::Boolean			mIsRoot ;       // Flag to identify the root branch
    AcMapOperandArray		mOperands ;     // Array of operands
} ;
                                                                  
class AcMapQueryCondition : public AcMapQueryUnit
{

public:
	virtual					~AcMapQueryCondition    
								            () ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Access the .NOT. flag.
    //
    Adesk::Boolean			IsNot           () const ;
    void					SetIsNot        (Adesk::Boolean bIsNot) ;

protected:
							AcMapQueryCondition     
								            (AcMap::EJoinOperator kOp
                                                = AcMap::kOperatorAnd) ;
    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit*	Clone           () const = 0 ;

    Adesk::Boolean			mIsNot ;        // .NOT. flag
} ;

class AcMapLocationCondition : public AcMapQueryCondition
{
public:

							AcMapLocationCondition  
											(AcMap::EJoinOperator kOp
                                                = AcMap::kOperatorAnd,
											AcMap::ELocationType type 
                                                = AcMap::kLocationCrossing) ;
                                                         
    virtual					~AcMapLocationCondition 
											() ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean	IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit*	Clone           () const ;

    // Access the location type.
    //
    AcMap::EErrCode         SetLocationType (AcMap::ELocationType type) ;
    AcMap::ELocationType    LocationType    () const ;

    // Access the boundary object.
    //
    void					SetBoundary     (const AcMapLocationBoundary *pCond) ;
    const AcMapLocationBoundary*        
							Boundary        () const ;

private:
    AcMap::ELocationType	mType ;         // Location type
    AcMapLocationBoundary*  mpBoundary ;    // Boundary object
} ;

class AcMapPropertyCondition : public AcMapQueryCondition
{
public:

							AcMapPropertyCondition  
								            (AcMap::EJoinOperator kOp 
                                                = AcMap::kOperatorAnd,
									         AcMap::EPropertyType kType 
                                                = AcMap::kArea,
										     AcMap::EConditionOperator kOper 
                                                = AcMap::kCondEq) ;

							AcMapPropertyCondition  
								            (AcMap::EJoinOperator       kOp,
									         AcMap::EPropertyType       kType,
										     AcMap::EConditionOperator  kOper,
											 const char                 *pcValue) ;

    virtual					~AcMapPropertyCondition 
								            () ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit* Clone           () const ;

    // Access the property type.
    //
    void					SetPropertyType (AcMap::EPropertyType type) ;
    AcMap::EPropertyType	PropertyType    () const ;

    // Access the condition operator.
    //
    void					SetConditionOperator    
								            (AcMap::EConditionOperator type) ;
    AcMap::EConditionOperator   
							ConditionOperator       
								            () const ;

    // Access the value expression.
    //
    AcMap::EErrCode         SetValue        (const char *pcValue) ;
    const char *			Value           () const ;

protected:
    char*					mpValue ;

private:
    AcMap::EPropertyType    mPropertyType ; // Property type
    AcMap::EConditionOperator   
							mConditionOperator ;
                                        // Condition operator
} ;

// This class represents the Data Query.
class AcMapDataCondition : public AcMapQueryCondition
{
public:

							AcMapDataCondition      
								            (AcMap::EJoinOperator kJoinOp  
                                                = AcMap::kOperatorAnd,
									         AcMap::EDataQueryType kType 
                                                = AcMap::kDataIRD, 
										     AcMap::EConditionOperator kOper 
                                                = AcMap::kCondEq) ;

							AcMapDataCondition      
								            (AcMap::EJoinOperator      kJoinOp,
									         AcMap::EDataQueryType     kType, 
										     AcMap::EConditionOperator kOper,
											 const char                *pcTableName,
											const char                *pcColumnName,
											const char                *pcValue) ;

    virtual					~AcMapDataCondition     
								            () ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;
        
    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit* Clone           () const ;

    // Access the data type.
    //
    void					SetDataType     (AcMap::EDataQueryType type) ;
    AcMap::EDataQueryType   DataType        () const ;

    // Access the condition operator.
    //
    void					SetConditionOperator    
								            (AcMap::EConditionOperator type) ;
    AcMap::EConditionOperator   
							ConditionOperator
								            () const ;

    // Access the table name.
    //
    void					SetTable        (const char *pcTableName) ;
    const char*				Table           () const ;

    // Access the column name.
    //
    void					SetColumn       (const char *pcColumnName) ;
    const char *			Column          () const ;

    // Access the value expression.
    //
    void					SetValue        (const char *pcValue) ;
    const char *			Value           () const ;

private:
    char*					mpTable ;       // Table name
    char*					mpColumn ;      // Column name
    char*					mpValue ;       // Condition 
    AcMap::EDataQueryType   mDataType ;     // Data type
    AcMap::EConditionOperator   
							mConditionOperator ;
								            // Condition operation
} ;

// This class represents the SQL Query.
class AcMapSQLCondition : public AcMapQueryCondition
{
public:

							AcMapSQLCondition       
                                        (AcMap::EJoinOperator kJoinOp
                                                = AcMap::kOperatorAnd) ;

							AcMapSQLCondition
                                        (AcMap::EJoinOperator kJoinOp,
                                         const char *pcLPN,
                                         const char *pcWhereCondition) ;

    virtual					~AcMapSQLCondition      
                                        () ;

    // Runtime class identification.
    //
    virtual Adesk::Boolean  IsKindOf        (AcMap::EClassId Id) const ;
    virtual AcMap::EClassId IsA             () const ;

    // Copy operators.
    //
    virtual AcMapQueryUnit& operator =      (const AcMapQueryUnit &tSource) ;
    virtual AcMapQueryUnit* Clone           () const ;

    // Access the LPN.
    //
    void					SetLinkTemplate	(const char *pcLTName) ;
    const char*				LinkTemplate    () const ;

    // Access the WHERE condition.
    //
    void					SetWhereCondition       
								            (const char *pcWhereCondition) ;
    const char*				WhereCondition  () const ;

protected:
    char*					mpLT ;			// Link Template name
    char*					mpWhere ;       // WHERE condition
} ;

/*****************************************************************************/
/*  INLINES */
/*****************************************************************************/
inline  void                        
AcMapQueryUnit::SetJoinOperator(
    AcMap::EJoinOperator kOp
)
{
    mOperator = kOp ;
}

inline AcMap::EJoinOperator         
AcMapQueryUnit::JoinOperator() const 
{
    return mOperator ;
}

inline Adesk::Boolean
AcMapQueryBranch::IsRoot()
const
{
    return mIsRoot ;
}

inline  Adesk::Boolean              
AcMapQueryCondition::IsNot() const 
{
    return mIsNot ;
}

inline  void                        
AcMapQueryCondition::SetIsNot(
    Adesk::Boolean bIsNot
)
{
    mIsNot = bIsNot ;
}

inline const AcMapLocationBoundary*
AcMapLocationCondition::Boundary() const 
{
    return mpBoundary ;
}

inline AcMap::ELocationType
AcMapLocationCondition::LocationType() const
{
    return mType ;
}

inline void
AcMapPropertyCondition::SetConditionOperator(
    AcMap::EConditionOperator   kOper
)
{   
    mConditionOperator = kOper ;
}

inline AcMap::EConditionOperator
AcMapPropertyCondition::ConditionOperator() const
{   
    return mConditionOperator ;
}

inline void
AcMapPropertyCondition::SetPropertyType(
    AcMap::EPropertyType    kType
)
{   
    mPropertyType = kType ;
}

inline AcMap::EPropertyType
AcMapPropertyCondition::PropertyType() const
{   
    return mPropertyType ;
}

inline const char*
AcMapPropertyCondition::Value() const 
{
    return mpValue ;
}

inline const char*
AcMapDataCondition::Table() const
{
    return mpTable ;
}

inline const char*
AcMapDataCondition::Column() const
{
    return mpColumn ;
}

inline const char*
AcMapDataCondition::Value() const
{
    return mpValue ;
}

inline void
AcMapDataCondition::SetConditionOperator(
    AcMap::EConditionOperator   kOper
)
{   
    mConditionOperator = kOper ;
}

inline AcMap::EConditionOperator
AcMapDataCondition::ConditionOperator() const
{   
    return mConditionOperator ;
}

inline void
AcMapDataCondition::SetDataType(
    AcMap::EDataQueryType   kType
)
{
    mDataType = kType ;
}

inline AcMap::EDataQueryType
AcMapDataCondition::DataType() const
{
    return mDataType ;
}

inline  const char*
AcMapSQLCondition::LinkTemplate() const
{
    return mpLT ;
}

inline  const char*
AcMapSQLCondition::WhereCondition() const
{
    return mpWhere ;
}
#endif /* ADEQUERY_H */