#ifndef MAP_REACTORS_H
#define MAP_REACTORS_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the descriptions of Autodesk Map ARX API reactor classes.
// An application is notified about an Autodesk Map event if it implements the
// following operations:
//
//  a. The application derives a class from the appropriate reactor class.
//  b. One or more reactor class methods are overridden.
//  c. A reactor instance is added to the corresponding Autodesk Map resource
//     (this reactor should be removed before the resource is destroyed).
// 
//  Autodesk Map ARX API objects should generally not be modified from
//  within a notification handler.
//

#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

// INCLUDES
#include <MapConstants.h>
#include <Adesk.h>           // For Adesk::Boolean


// FORWARD DECLARATIONS
class AcMapAttachedDrawing;
class AcMapDriveAlias;
class AcMapQueryCategory;
class AcMapQueryAttribute;


//---------------------------------------------------------------------------
// AcMapDrawingSetReactor
//
// Drawing Set Reactor class
//
class AcMapDrawingSetReactor
{
protected:
	AcMapDrawingSetReactor();
	virtual ~AcMapDrawingSetReactor();

public:
	// Attach/Detach drawing events.
	//
	// Sent before attaching a drawing - return false to veto.
	virtual Adesk::Boolean DrawingToBeAttached(const char *pcAliasPath);
	// Sent to all reactors if attach is vetoed (even those that haven't
	// yet received DrawingToBeAttached).
	virtual void DrawingAttachCancelled(const char *pcAliasPath);
	virtual void DrawingAttached(const AcMapAttachedDrawing *pObj);
	virtual void DrawingDetached(
		AcMapId Id,
		const char *pcName,
		int iNestedLevel);

	// Activate/Deactivate drawing events.
	//
	// Sent before activating a drawing - return false to veto.
	virtual Adesk::Boolean DrawingToBeActivated(const AcMapAttachedDrawing *pObj);
	// Sent to all reactors if activation is vetoed (even those that haven't
	// yet received DrawingToBeActivated).
	virtual void DrawingActivationCancelled(const AcMapAttachedDrawing *pObj);
	virtual void DrawingActivated(const AcMapAttachedDrawing *pObj);
	// If the drawing is both deactivated and detached within one command or API 
	// function, this notification is suppressed.
	virtual void DrawingDeactivated(const AcMapAttachedDrawing *pObj);

	// Notifies of drawing settings modification.
	//
	virtual void DrawingSettingsModified(const AcMapAttachedDrawing *pObj);
};


//---------------------------------------------------------------------------
// AcMapAliasesReactor
//
// Drive Aliases Reactor class
//
class AcMapAliasesReactor
{
protected:
	AcMapAliasesReactor();
	virtual ~AcMapAliasesReactor();

public:

	// Adds/Removes alias event.
	//
	virtual void AliasAdded(const AcMapDriveAlias *pObj);
	virtual void AliasDeleted(const char *pcName, const char *pcPath);
};

//---------------------------------------------------------------------------
// AcMapQueryLibraryReactor
//
// Query Library Reactor
//
class AcMapQueryLibraryReactor
{
protected:
	AcMapQueryLibraryReactor();
	virtual ~AcMapQueryLibraryReactor();

public:
	// Adds/Removes query category events.
	//
	virtual void QueryCategoryAdded(const AcMapQueryCategory *pObj);
	virtual void QueryCategoryDeleted(AcMapId Id, const char *pcName);

	// Renames query category event.
	//
	virtual void QueryCategoryRenamed(const AcMapQueryCategory *pObj);
	
	// Adds/Removes query events.
	//
	virtual void QueryAdded(const AcMapQueryAttribute *pObj);
	virtual void QueryDeleted(AcMapId Id, const char *pcName);

	// Notifies of query modification and renaming events.
	//
	virtual void QueryRenamed(const AcMapQueryAttribute *pObj);
	virtual void QueryModified(const AcMapQueryAttribute *pObj);
};


//---------------------------------------------------------------------------
// AcMapProjectOptionsReactor
//
// Reactor for options stored with each project.
//
class AcMapProjectOptionsReactor
{
protected:
	AcMapProjectOptionsReactor();
	virtual ~AcMapProjectOptionsReactor();

public:
	// Option modification events.
	virtual void OptionModified(
		const char *pcName,
		const char *pcOldvalue,
		const char *pcNewValue);
	virtual void OptionModified(
		const char *pcName,
		int iOldvalue,
		int iNewValue);
};


//---------------------------------------------------------------------------
// AcMapSessionOptionsReactor
//
// Reactor for options shared for all projects.
//
class AcMapSessionOptionsReactor
{
protected:
	AcMapSessionOptionsReactor();
	virtual ~AcMapSessionOptionsReactor();

public:
	// Option modification events.
	virtual void OptionModified(
		const char *pcName,
		const char *pcOldvalue,
		const char *pcNewValue);
	virtual void OptionModified(
		const char *pcName,
		int iOldvalue,
		int iNewValue);
};


//---------------------------------------------------------------------------
// AcMapProjectReactor
//
// Interface for reactor services for project clients.  Clients can
// derive from this class and register subclasses with the project
// to receive notification.
//
// ** NOTE: This class is unsupported, and not intended for public use. **
//
class AcMapProject;
class AcMapProjectReactor
{
protected:
	AcMapProjectReactor();
	virtual ~AcMapProjectReactor();

// Public Interface
public:
	// Notifications
	virtual void BeginOpen(AcMapProject *pproject);
	virtual void EndOpen(AcMapProject *pproject);

	virtual void BeginClose(AcMapProject *pproject);
	virtual void EndClose(AcMapProject *pproject);

	virtual void BeginSave(AcMapProject *pproject);
	virtual void EndSave(AcMapProject *pproject);

	virtual void BeginDestroy(AcMapProject *pproject);
};


//---------------------------------------------------------------------------
// AcMapSessionReactor
//
// Interface for reactor services for session clients.  Clients can
// derive from this class and register subclasses with the session
// to receive notification.
//
class AcMapSession;
class AcMapSessionReactor
{
protected:
	AcMapSessionReactor();
	virtual ~AcMapSessionReactor();

// Public Interface
public:
	// Notifications
	virtual void UnloadApp();

	// Called just after Map creates and initializes a project.
	virtual void ProjectCreated(AcMapProject *pproject);
	// Called just after Map initializes drawing-specific data for a project.
	virtual void ProjectOpened(AcMapProject *pproject);
	// Called just before Map closes and destroys a project.
	virtual void ProjectBeginClose(AcMapProject *pproject);
};


#endif // MAP_REACTORS_H

// End of File.
