#ifndef MAP_STRINGARRAY_H
#define MAP_STRINGARRAY_H
//
// (C) Copyright 1993-1999 by Autodesk, Inc.
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// DESCRIPTION:
//
// This file contains the definition of AcMapStringAray - a dynamic array of 
// character string objects.
//
// "Dynamic array" means that the array can grow without bounds,
// unlike arrays of objects of type "const char *" declared in the
// usual manner.  For example declaring "const char * myArray[10]"
// is limited to holding only ten entries.
//
// To use the class AcMapStringArray, you need to understand
// a couple of simple, yet key, concepts:
//
//     1) The logical length of the array.
//            - How many entries have been placed into the array,
//              initially always zero.
//     2) The physical length of the array.
//            - How many entries the array will hold before it
//              automatically "grows" larger.
//     3) The grow length of the array.
//            - How much the array will grow when required.
//
// The physical length of the array is the actual length of the
// physically allocated, but perhaps not fully used, array.
// As a point of clarification, the size in bytes of the array
// buffer for an array called `myArray' would be:
//
//     sizeOf(const char *) * myArray.physicalLength().
//
// The physical length of the array can be zero or any positive
// integer.
//
// The logical length of the array (or just the "length()") reflects
// how many const char * elements have been placed into the array
// with, for example, append() or insertAt().  Many member functions
// are only valid for indices that are greater than or equal to
// zero AND less than length().  For example, indexing into the
// array with the operator[] is only valid for indices in this range.
//
// You can explicitly set the logical length() to any value and
// if the physical length is not large enough the array will grow to
// that length.  Note that if the logical length is explicitly reset
// to a larger value, then all the entries from the old length up
// to the new length may contain garbage values, therefore, these entries must be
// initialized explicitly.
//
// The logical length is always less than or equal to the physical
// length.  NOTE that the array ALWAYS starts out empty, i.e., the
// length() always starts at zero regardless of the initial physical
// length.
//
// If you add an element to the array causing the logical length
// to become greater than the physical length of the array then
// the "grow length" determines how much additional space to
// allocate, and the physical length will increase by the grow length.
//
// The grow length must be a positive number. Zero is an illegal
// grow length.
//
#if _MSC_VER > 1000
#pragma once // Ensure this file is only parsed once per translation unit.
#endif

#include "adesk.h"

class AcMapStringArray
{
public:
                        AcMapStringArray(int initPhysicalLength = 0, int initGrowLength = 8);

                        AcMapStringArray(const AcMapStringArray&);

                        ~AcMapStringArray
                                        ();

    // Copy operator.
    //
    AcMapStringArray&   operator =      (const AcMapStringArray&);

    // Indexes into the array.
    //
    const char *        operator []     (int);

    // Accesses array elements.
    //
    const char *        At              (int) const;
    const char *        First           () const ;
    const char *        Last            () const ;
    AcMapStringArray&   SetAt           (int, const char *);
    
    // Adds array elements.
    //
    int                 Append          (const char *);
    AcMapStringArray&   Append          (const AcMapStringArray &);
    AcMapStringArray&   InsertAt        (int, const char *);

    // Removes array elements.
    //
    AcMapStringArray&   RemoveAt        (int);
    AcMapStringArray&   RemoveFirst     ();
    AcMapStringArray&   RemoveLast      ();

    // Queries about array elements.
    //
    Adesk::Boolean      Contains        (const char *, 
                                         int start = 0,
                                         Adesk::Boolean noCase = Adesk::kFalse) const;
    Adesk::Boolean      Find            (const char *, int& foundAt,
                                         int start = 0,
                                         Adesk::Boolean noCase = Adesk::kFalse) const;

    // Array length operations.
    //
    int                 Length          () const; // Logical length.
    Adesk::Boolean      IsEmpty         () const;
    int                 LogicalLength   () const;
    AcMapStringArray&   SetLogicalLength(int);
    int                 PhysicalLength  () const;
    AcMapStringArray&   SetPhysicalLength
                                        (int);

    // Automatic resizing.
    //
    int                 GrowLength      () const;
    AcMapStringArray&   SetGrowLength   (int);

    // Utility.
    //
    AcMapStringArray&   Reverse         ();
    AcMapStringArray&   Sort            ();

    // Treat array as simple array of const char**.
    //
    const char          **AsArrayPtr    () const;

protected:
    char                **mpArray;
    int                 mPhysicalLen;   // Actual buffer length.
    int                 mLogicalLen;    // Number of items in the array.
    int                 mGrowLen;       // Buffer grows by this value.

    Adesk::Boolean      IsValid         (int) const;
    void                SetItem         (int, const char *) ;
    int                 FindPos         (const char * pStr, 
                                         int left, 
                                         int right) const;
};

// Inline methods.

inline Adesk::Boolean
AcMapStringArray::Contains(const char *value, 
                           int start, 
                           Adesk::Boolean noCase) const
{ int dummy; return Find(value, dummy, start, noCase); }

inline int
AcMapStringArray::Length() const
{ return mLogicalLen; }

inline Adesk::Boolean
AcMapStringArray::IsEmpty() const
{ return mLogicalLen == 0; }

inline int
AcMapStringArray::LogicalLength() const
{ return mLogicalLen; }

inline int
AcMapStringArray::PhysicalLength() const
{ return mPhysicalLen; }

inline int
AcMapStringArray::GrowLength() const
{ return mGrowLen; }

inline const char **
AcMapStringArray::AsArrayPtr() const
{ return (const char **)mpArray; }

inline Adesk::Boolean
AcMapStringArray::IsValid(int i) const
{ 
    if (i < 0 || i >= mLogicalLen)
    {
        throw 0 ;
        return Adesk::kFalse ;
    }
    else
    {
        return Adesk::kTrue ;
    }
}

inline const char *
AcMapStringArray::operator [] (int i)
{ IsValid(i); return mpArray[i]; }

inline const char *
AcMapStringArray::At(int i) const
{ IsValid(i); return mpArray[i]; }

inline const char *
AcMapStringArray::First() const
{ IsValid(0); return mpArray[0]; }

inline const char *
AcMapStringArray::Last() const
{ IsValid(mLogicalLen-1); return mpArray[mLogicalLen-1]; }

inline int
AcMapStringArray::Append(const char * str)
{ InsertAt(mLogicalLen, str); return mLogicalLen-1; }

inline AcMapStringArray&
AcMapStringArray::RemoveFirst()
{ IsValid(0) ; return RemoveAt(0); }

inline AcMapStringArray&
AcMapStringArray::SetGrowLength(int glen)
{ if (glen > 0)  mGrowLen = glen; return *this; }
#endif /* MAP_STRINGARRAY_H */
