{ DISQLite3 incremental blob demo. Using the incremental blob API, DISQLite3
  can conveniently handle large blobs with no strains to memory.

  Visit the DISQLite3 Internet site for latest information and updates:

    http://www.yunqa.de/delphi/

  Copyright (c) 2006-2009 Ralf Junker, The Delphi Inspiration <delphi@yunqa.de>

------------------------------------------------------------------------------ }

unit DISQLite3_Incremental_Blob_fMain;

{$I DI.inc}
{$I DISQLite3.inc}

{$IFDEF SQLITE_OMIT_INCRBLOB}
!!! This project requires the incremental blob API which unavailable in DISQLite3 Personal. !!!
!!! To compile, install DISQLite3 Pro, available from www.yunqa.de/delphi/                  !!!
{$ENDIF SQLITE_OMIT_INCRBLOB}

interface

uses
  Classes, Controls, Forms, StdCtrls, ComCtrls, ExtCtrls,
  DISQLite3_Incremental_Blob_DB;

type
  TfrmIncrementalBlob = class(TForm)
    ListView: TListView;
    pnlTop: TPanel;
    btnAdd: TButton;
    btnEdit: TButton;
    btnDelete: TButton;
    btnAddJpg: TButton;
    StatusBar: TStatusBar;
    btnSave: TButton;
    procedure btnAddClick(Sender: TObject);
    procedure btnAddJpgClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnEditClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    FDb: TBlobDatabase;
  protected
    procedure ReloadDatabase;
  published
    procedure FormCreate(Sender: TObject);
  end;

const
  APP_TITLE = 'DISQLite3 Demo: Incremental Blob';

var
  frmIncrementalBlob: TfrmIncrementalBlob;

implementation

{$R *.dfm}

//------------------------------------------------------------------------------

uses
  SysUtils, Dialogs, FileCtrl,
  DISQLite3Api, DISQLite3Database,
  DISQLite3_Incremental_Blob_fAdd, DISQLite3_Incremental_Blob_fEdit;

const
  FILE_NAME = 'Blobs.db3';

  //------------------------------------------------------------------------------
  // TfrmIncrementalBlob form
  //------------------------------------------------------------------------------

procedure TfrmIncrementalBlob.FormCreate(Sender: TObject);
begin
  Caption := APP_TITLE;

  { Create a database component and open the database. }
  FDb := TBlobDatabase.Create(nil);
  // FDb.Password := 'personal';
  FDb.DatabaseName := FILE_NAME;
  try
    FDb.Open;
  except
    on EFOpenError do
      begin
        { If the database does not yet exist, create a new one. }
        FDb.CreateDatabase;
      end;
  end;

  ReloadDatabase;
end;

//------------------------------------------------------------------------------

procedure TfrmIncrementalBlob.FormDestroy(Sender: TObject);
begin
  FDb.Free;
end;

//------------------------------------------------------------------------------

procedure TfrmIncrementalBlob.ReloadDatabase;
var
  BlobSize: Integer;
  LI: TListItem;
  Idx: Integer;
  Stmt: TDISQLite3Statement;
begin
  ListView.Items.BeginUpdate;
  try
    ListView.Items.Clear;

    Stmt := FDb.Prepare('SELECT Idx, Name, Comment FROM Blobs;');
    try
      while Stmt.Step = SQLITE_ROW do
        begin
          LI := ListView.Items.Add;
          Idx := Stmt.Column_Int64(0);
          LI.Data := Pointer(Idx);
          LI.Caption := Stmt.Column_Str16(1);
          LI.SubItems.Add(Stmt.Column_Str16(2));
          BlobSize := FDb.GetBlobSize(Idx);
          if BlobSize >= 0 then
            LI.SubItems.Add(IntToStr(BlobSize));
        end;
    finally
      Stmt.Free;
    end;
  finally
    ListView.Items.EndUpdate;
  end;
end;

//------------------------------------------------------------------------------

{ Add a new blob record to the database. Fill the blob contents from a file. }
procedure TfrmIncrementalBlob.btnAddClick(Sender: TObject);
var
  AddForm: TfrmIncrementalBlobAdd;
  BlobSize: Integer;
  {$IFNDEF COMPILER_6_UP}
  i: Integer;
  {$ENDIF !COMPILER_6_UP}
  Idx: Int64;
  LI: TListItem;
begin
  { Show form to add new blob record. }
  AddForm := TfrmIncrementalBlobAdd.Create(Self);
  try
    if AddForm.ShowModal = mrOk then
      begin
        { Add to database. }
        Idx := FDb.AddBlobFromFile(
          AddForm.edtName.Text,
          AddForm.edtComment.Text,
          AddForm.edtBlobFileName.Text);

        { Add new item to listview ... }
        LI := ListView.Items.Add;
        LI.Data := Pointer(Idx);
        LI.Caption := AddForm.edtName.Text;
        LI.SubItems.Add(AddForm.edtComment.Text);
        BlobSize := FDb.GetBlobSize(Idx);
        if BlobSize >= 0 then
          LI.SubItems.Add(IntToStr(BlobSize));

        { ... and make it visible. }
        {$IFDEF COMPILER_6_UP}
        ListView.ClearSelection;
        {$ELSE COMPILER_6_UP}
        for i := 0 to ListView.Items.Count - 1 do
          ListView.Items[i].Selected := False;
        {$ENDIF COMPILER_6_UP}
        LI.Focused := True;
        LI.Selected := True;
        LI.MakeVisible(True);
      end;
  finally
    AddForm.Free;
  end;
end;

//------------------------------------------------------------------------------

{ Search a folder for *.jpg files and add their contents to the database. }
procedure TfrmIncrementalBlob.btnAddJpgClick(Sender: TObject);
var
  Folder, FileName: String;
  SR: TSearchRec;
begin
  Folder := '';
  if Selectdirectory('Select folder with *.jpg images to add:', '', Folder) then
    begin
      Folder := Folder + '\';
      if FindFirst(Folder + '*.jpg', 0, SR) = 0 then
        try
          FDb.StartTransaction;
          try
            repeat
              FileName := Folder + SR.Name;
              StatusBar.SimpleText := FileName;
              FDb.AddBlobFromFile(FileName, '', FileName);
            until FindNext(SR) <> 0;
            FDb.Commit;
          except;
            FDb.Rollback;
            raise;
          end;
        finally
          FindClose(SR);
        end;
      ReloadDatabase;
    end;
end;

//------------------------------------------------------------------------------

{ Save a database blob to a file. }
procedure TfrmIncrementalBlob.btnSaveClick(Sender: TObject);
var
  LI: TListItem;
  SaveDlg: TSaveDialog;
begin
  LI := ListView.ItemFocused;
  if Assigned(LI) then
    begin
      SaveDlg := TSaveDialog.Create(nil);
      try
        SaveDlg.FileName := LI.Caption;
        if SaveDlg.Execute then
          begin
            FDb.SaveBlobToFile(Integer(LI.Data), SaveDlg.FileName);
          end;
      finally
        SaveDlg.Free;
      end;
    end;
end;

//------------------------------------------------------------------------------

{ Edit the focused item and update the database accordingly. }
procedure TfrmIncrementalBlob.btnEditClick(Sender: TObject);
var
  BlobUpdate: TBlobUpdate;
  l: Integer;
  LI: TListItem;
  EditForm: TfrmIncrementalBlobEdit;
  RowID: Int64;
begin
  { Show form to edit blob record. }
  LI := ListView.ItemFocused;
  if Assigned(LI) then
    begin
      EditForm := TfrmIncrementalBlobEdit.Create(Self);
      try
        EditForm.edtName.Text := LI.Caption;
        EditForm.edtComment.Text := LI.SubItems[0];

        if EditForm.ShowModal = mrOk then
          begin
            { Update database record. }
            if EditForm.rbBlobDelete.Checked then
              BlobUpdate := buDelete
            else
              if EditForm.rbBlobReplace.Checked then
                BlobUpdate := buReplace
              else
                BlobUpdate := buNoChange;

            RowID := Integer(LI.Data);
            l := FDb.UpdateBlobFromFile(
              RowID,
              EditForm.edtName.Text,
              EditForm.edtComment.Text,
              EditForm.edtBlobFileName.Text,
              BlobUpdate);

            { Update listview item. }
            LI.Caption := EditForm.edtName.Text;
            LI.SubItems[0] := EditForm.edtComment.Text;
            if l >= 0 then
              LI.SubItems[1] := IntToStr(l);
          end;
      finally
        EditForm.Free;
      end;
    end;
end;

//------------------------------------------------------------------------------

{ Delete all records selected in the listview from the database. }
procedure TfrmIncrementalBlob.btnDeleteClick(Sender: TObject);
var
  LI: TListItem;
begin
  LI := ListView.Selected;
  if Assigned(LI) and (MessageDlg('Delete selected?', mtConfirmation, mbOKCancel, 0) = mrOk) then
    try
      FDb.StartTransaction;
      repeat
        FDb.DeleteFromBlobs(Integer(LI.Data));
        LI := ListView.GetNextItem(LI, sdall, [isSelected]);
      until not Assigned(LI);
      FDb.Commit;
    except
      FDb.Rollback;
      raise;
    end;
  ReloadDatabase;
end;

//------------------------------------------------------------------------------

initialization
  { Initialize the DISQLite3 library prior to using any other DISQLite3
    functionality. See also sqlite3_shutdown() below.}
  sqlite3_initialize;

finalization
  { Deallocate any resources that were allocated by
  sqlite3_initialize() above. }
  sqlite3_shutdown;

end.

