/****************************************************************************
/                                                                           
/                     Copyright 1997 Macromedia, Inc.                       
/                                                                          
/      This material is the confidential trade secret and proprietary       
/      information of Macromedia, Inc.  It may not be reproduced, used,     
/      sold or transferred to any third party without the prior written     
/      consent of Macromedia, Inc.  All rights reserved.                    
/                                                                          
/ Filename
/   glue.cpp
/
/ Purpose
/   This file contains all calls from the NscpJavaScript directory to other
/   directories in the tree.  Currently, that list is limited to the memory
/   allocation functions.
/
/   The prtypes.h file does a #define so that all calls to malloc are
/   made to call glue_malloc instead.  Same for free and realloc.  That's
/   less error-prone than changing each call to malloc by hand.  The
/   only danger is that someone might call malloc before including prtypes.h.
/
****************************************************************************/

#define incDebugUtils

#ifdef _MAC
#pragma cplusplus on
#include "mfc2x.h"
#else
#include <afxwin.h>
#endif

#ifdef _WINDOWS
#include "WindowsCrashLogger.h"
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>

extern "C"
{
#include "jstypes.h"
#include "jsapi.h"
#include "jsarray.h"
#include "jsatom.h"
#include "jsbool.h"
#include "jscntxt.h"
#include "jsconfig.h"
#include "jsdbgapi.h"
#include "jsfun.h"
#include "jsgc.h"
#include "jsinterp.h"
#include "jslock.h"
#include "jsnum.h"
#include "jsobj.h"
#include "jsopcode.h"
#include "jsscope.h"
#include "jsscript.h"
#include "jsstr.h"
}

extern "C" JSBool JSCallNativeSafe (JSNative native, JSContext *cx, JSObject *obj, uintN argc, jsval *argv,
		      jsval *rval);

// Defined in JSInterp.cpp.
extern void AssertHaveJSErrorString();

#include <typeinfo>

// Returns whether the native call should be wrapped in a try/catch block.
static BOOL UseTryCatch()
{
	BOOL useTryCatch = TRUE;

#if defined(_DEBUG) && defined(_WINDOWS)
	// [mmorearty 2/27/03] If we're running under the debugger, then don't
	// catch the exception -- that way the debugger can catch it and drop into
	// the source code at the actual location of the exception.
	//
	// Note, this means that we have slightly different behavior when running
	// under the debugger than when running without the debugger.  That can be
	// potentially confusing, but still it is much better than not being able
	// to debug exceptions.
	if (IsDebuggerPresent() || WindowsCrashLogger::IsJitDebuggerInstalled())
		useTryCatch = FALSE;
#endif

	return useTryCatch;
}

#ifdef _WINDOWS

// If an exception happens when we call native code, then this exception
// handler will be called; we will write a crashlog to the user's disk.
LONG WINAPI JSNativeCallExceptionHandler(EXCEPTION_POINTERS* pExceptionPointers)
{
	// If the native call was wrapped in a try-catch block, then we should write
	// out a dump file, since the debugger is not going to be given the chance
	// to deal with it.
	if (UseTryCatch())
		WindowsCrashLogger::SaveCrashToDumpFile(pExceptionPointers, FALSE);

	// Whether or not we saved the crashlog, we do NOT want to enter the __except block of the caller
	// -- we want the operating system to continue looking up the exception chain for an
	// exception hander that will catch this (which will turn out to be JSCallNativeSafe)
	return EXCEPTION_CONTINUE_SEARCH;
}

#endif // _WINDOWS


static JSBool JSCallNative(JSNative native, JSContext *cx, JSObject *obj,
						   uintN argc, jsval *argv, jsval *rval)
{
	JSBool ok = JS_FALSE;

#ifdef _WINDOWS
	__try
	{
#endif

		ok = native (cx, obj, argc, argv, rval);

#ifdef _WINDOWS
	}
	// If any exception gets thrown, we call JSNativeCallExceptionHandler,
	// which will write a crashlog to disk but will *NOT* actually 'catch'
	// the exception -- it returns EXCEPTION_CONTINUE_SEARCH, which means
	// "I don't want to catch this one; keep looking up the chain of
	// exception handlers to find someone who does want to."  Since we
	// were called by JSCallNativeSafe, that function will (in most cases)
	// catch this exception and display a message to the user.
	__except( JSNativeCallExceptionHandler(GetExceptionInformation()) )
	{
		// Even if an exception is thrown, we shouldn't end up
		// here, because JSNativeCallExceptionHandler is
		// supposed to always return EXCEPTION_CONTINUE_SEARCH
		ASSERT(FALSE);
	}
#endif

	// If this assertion fires, it means a native function returned
	// false (error) without recording an error message.  This is
	// bad behavior, because it makes the bug hard to track down.
	// 
	// To get an idea of what native function we just called:
	// - Visual Studio .NET: hover the mouse over the word "native"
	//   in the function call above.  The tooltip will say the name
	//   of the native function.
	// - CodeWarrior: Right-click on "native" in the function call
	//   above, and pick "View Memory".  Change the "Display:" to
	//   "*native" (with an asterisk) and hit Enter.  Change
	//   "View:" to "Source".  You should now see the source code
	//   of the function that was called.
	// - Other: Look up the stack one level to js_Invoke.  js_Invoke
	//   has a local variable named "fun".  The expression
	//   *(void**)fun->atom->entry.key gives the address of the
	//   function name (in Unicode format).
	// You should then find the C++ implementation of that function,
	// and update it to include a call to JS_ReportError when it
	// returns false.  Search for JS_ReportError in Source/Titan/JavaScript
	// for usage examples.
	// 
	// snewman 5/23/01
	if (!ok)
		AssertHaveJSErrorString();

	return ok;
}

extern "C" JSBool JSCallNativeSafe (JSNative native, JSContext *cx, JSObject *obj, uintN argc, jsval *argv,
								    jsval *rval)
{
	JSBool	ok = JS_FALSE;

	if (UseTryCatch())
	{
		try
		{
			ok = JSCallNative(native, cx, obj, argc, argv, rval);
		}
#ifdef _DEBUG
		catch (CException *mfcEx) {
			char buf[1024];
			const char *name = typeid(*mfcEx).name();
			strcpy_s(buf, "exception thrown in native function. \"");
			strncat_s(buf, name, sizeof(buf));
			strncat_s(buf, "\": ", sizeof(buf));
			#ifndef _UNICODE
				size_t len = strlen(buf);

				mfcEx->GetErrorMessage(buf + len, sizeof(buf) - len);
			#endif
			buf[sizeof(buf) - 1] = '\0';
			JS_ReportError(cx, buf);
			mfcEx->Delete();
		}
		catch (const std::exception &stdEx) {
			char buf[1024];
			const char *what = stdEx.what();
			const char *name = typeid(stdEx).name();

			strcpy_s(buf, "exception thrown in native function. \"");
			strncat_s(buf, name, sizeof(buf));
			strncat_s(buf, "\": ", sizeof(buf));
			strncat_s(buf, what, sizeof(buf));
			buf[sizeof(buf) - 1] = '\0';
			JS_ReportError(cx, buf);
		}

		catch (char *strEx) {
			char buf[1024];

			strcpy_s(buf, "exception thrown in native function. (char *): ");
			strncat_s(buf, strEx, sizeof(buf));
			buf[sizeof(buf) - 1] = '\0';
			JS_ReportError(cx, buf);
		}
#endif
		catch (...)
		{
			JS_ReportError(cx, "Exception thrown in native function.");
		}
	}
	else // else don't use try-catch
	{
		ok = JSCallNative(native, cx, obj, argc, argv, rval);
	}

	return ok;
}
