// Copyright (c) 2008. Adobe Systems Incorporated.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of Adobe Systems Incorporated nor the names of its
//     contributors may be used to endorse or promote products derived from this
//     software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.



//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugLoadExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	loads an extension
//
// ARGUMENTS:
//		extensionid
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugLoadExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
		if (aServiceExtension)
		{
			//load extension
			retVal = aServiceExtension.loadExtension();						
		}
		
	}
	return retVal;
}


//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugExtensionCall
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//
// ARGUMENTS:
//	inExtensionId    : extension id making the call.
//	inXMLCallString  : the string sent from the extension to PlugPlug
//
// RETURNS:
//   a result string sent from PlugPlug back to the extension, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugExtensionCall(inExtensionId, inXMLCallString)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		retVal = housingPlugIn.handleExtensionCall(inExtensionId, inXMLCallString);
	}
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugNotifyStateChange
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//
// ARGUMENTS:
//	inExtensionId    : extension id making the call.
//	inEvent			 : the state change event (Open , Move)
//	eventData        : the window dimensions
//
// RETURNS:
//   a result string sent from PlugPlug back to the extension, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugNotifyStateChange(inExtensionId, inEvent, eventData)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		retVal = housingPlugIn.handleNotifyStateChange(inExtensionId, inEvent, eventData);
	}
	return retVal;
}


//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugTerminate
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	unloads all the services
//
// ARGUMENTS:
//		None
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugTerminate()
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		retVal = housingPlugIn.handleTerminate();
	}
	return retVal;
}


//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugMenuCall
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	handles a menu click (from one of the fly-out menu items of the service panel)
//
// ARGUMENTS:
//		None
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugMenuCall(inExtensionID, inMenuID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		retVal = housingPlugIn.handleMenuCall(inExtensionID, inMenuID);
	}
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugOpenExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	open as in create: the window is created and its extension loaded
//  NOTE we are not using this we are calling HideExtension instead
//
// ARGUMENTS:
//		inExtensionID - id of calling Extension
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugOpenExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();	
	if (housingPlugIn) {
	  var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
	  if (aServiceExtension)
	  {
		  //load extension
		  retVal = aServiceExtension.openExtension();						
	  }
	}		
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugCloseExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	close as in destroy: the contained extension is unloaded and the window torn down
//  NOTE we are not using this we are calling HideExtension instead
//
// ARGUMENTS:
//		inExtensionID - id of calling Extension
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugCloseExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();	
	if (housingPlugIn) {
	  var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
	  if (aServiceExtension)
	  {		
		  retVal = aServiceExtension.closeExtension();						
	  }
	}		
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugTerminateExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	PlugPlugTerminateExtension as in destroy: the contained extension is unloaded and the window torn down
//  NOTE this only gets called when the destructor for the FloatingPanel gets called.  This gets called when
//  you switch workspaces.
//
// ARGUMENTS:
//		inExtensionID - id of calling Extension
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugTerminateExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();	
	if (housingPlugIn) {
	  var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
	  if (aServiceExtension)
	  {
	    retVal = aServiceExtension.terminateExtension();		
	  }
	}		
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugHideExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	when click on a floater by clicking on the close box or selecting menu for an already open panel
//  we need to set it to be closed so menu item no longer appears
//
// ARGUMENTS:
//		inExtensionID - id of calling Extension
//
//--------------------------------------------------------------------
function PlugPlugHideExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();	
	if (housingPlugIn) {
	  var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
	  if (aServiceExtension)
	  {
		  //  notify that the Extension is close so NO checkbox appears next to menu item
		  retVal = aServiceExtension.setOpen(false);			
		  PlugPlug.NotifyStateChange(inExtensionID, "PlugPlugHide", null);						
	  }
	}		
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugShowExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//  Called when user opens an extension that is already loaded (previously opened).
//	We forward the call to HousingPlugInDoc implemented in Javascript so
//	it can check the menu item in Windows, Extensions 
//
// ARGUMENTS:
//		inExtensionID - id of calling Extension
//
//--------------------------------------------------------------------
function PlugPlugShowExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();	
	if (housingPlugIn) {
	  var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
	  if (aServiceExtension)
	  {
		  if (!aServiceExtension.isLoaded()) { // if not loaded then load the extension
			  aServiceExtension.reInitExtension();  // if the extension is not loaded we need to load it
			} else {
		    retVal = aServiceExtension.setOpen(true);			// notify that the Extension is open so checkbox appears next to menu item
		    PlugPlug.NotifyStateChange(inExtensionID, "PlugPlugShow", null);			 // need to notify the extension it is being shown
      }
	  }
	}		
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugReInitExtension
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	re-initialize an extension
//
// ARGUMENTS:
//		extensionid
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function PlugPlugReInitExtension(inExtensionID)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
		if (aServiceExtension)
		{
			//load extension
			retVal = aServiceExtension.reInitExtension();						
		}
		
	}
	return retVal;
}

//--------------------------------------------------------------------
// FUNCTION:
//   PlugPlugIsExtensionRegistered
//
// DESCRIPTION:
//	This function will return true if the extension has been registered via CSXS.  
//  Developers can create non csxs extensions using the dw.flash api directly. 
//  We need to know what type an extension is.
//	
//
// ARGUMENTS:
//		extensionid
//
// RETURNS:
//   true if the extension is registered with PlugPlug, otherwise false it must be
//   a non-csxs extension.
//--------------------------------------------------------------------
function PlugPlugIsExtensionRegistered(inExtensionID)
{
	var retVal = false;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
	  // if we find it was registered by PlugPlug
		var aServiceExtension = housingPlugIn.findExtension(inExtensionID);
		if (aServiceExtension)
		{
			retVal = true;						
		}
	}
	return retVal;
}


//---------------------------------------------------------------------
//   ExecuteScriptCall
//
// DESCRIPTION:
//	pass through-call to MM.CSXS.GetHousingPlugIn (which is guts of housing plugin lives in Shared folder)
//	the calling code is Owl Player Flash Widget (OPFW) which is in C++
//	
//	handles passing calls to Dreamweaver's javascript interpreter
//
// ARGUMENTS:
//		inExtensionID -> extension ID
//    inScript -> Javascript code to be executed
//
// RETURNS:
//   a result string sent from PlugPlug back to the application, return a success or error code.
//--------------------------------------------------------------------
function ExecuteScriptCall(inExtensionID, inScript)
{
	var retVal = null;
	var housingPlugIn = MM.CSXS.GetHousingPlugIn();		
	if (housingPlugIn)
	{
		retVal = housingPlugIn.executeScriptCall(inExtensionID, inScript);
	}
	return retVal;
}
