unit PEHeaderUnit;

interface   
         
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, ToolWin, Menus,
  JwaWinnt, UtilUnit;

type
  TPEHeader = class
  private
    FNewPEHeader: TPEHeader;
    FHdrPtr: Pointer;
    FSectionCount: Cardinal;
    FDirectoryCount: Cardinal;
    FSections: Array of PImageSectionHeader;
    FDataDirectorys: Array of PImageDataDirectory;
    FDosHeader: PImageDosHeader;
    FFileHeader: PImageFileHeader;
    FOptionalHeader32: PImageOptionalHeader32;
    function SetHeaderSize(NewSize: Cardinal): Boolean;
    procedure DumpDataDirectorys;
    procedure DumpSectionHeaders;       
    function DoAssignHeader(SrcHdrPtr: Pointer; Size: Cardinal): Boolean;
    function DumpHeader(Ptr: Pointer): Boolean;
    function GetSectionCount: Cardinal;
    function GetDirectoryCount: Cardinal;
    function GetSection(index: Cardinal): PImageSectionHeader;
    function GetDataDirectory(index: Cardinal): PImageDataDirectory;  
    function ValidatePE(Ptr: Pointer): Boolean;
    function RVAtoPtr(RVA: Cardinal): Pointer;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function AssignHeader(SrcHdrPtr: Pointer): Boolean;
    function RVAInSection(RVA: Cardinal): integer;
    function RVAtoOffset(RVA: Cardinal): Cardinal;
    function AddSection: TPEHeader;
    function CalcFileAilgnment(Len: Cardinal): Cardinal;
    function CalcSectionAilgnment(Len: Cardinal): Cardinal;
    property PEHdrPtr: Pointer read FHdrPtr;
    property SectionCount: Cardinal read GetSectionCount;
    property DirectoryCount: Cardinal read GetDirectoryCount;
    property Sections[Index: Cardinal]: PImageSectionHeader read GetSection;
    property DataDirectorys[Index: Cardinal]: PImageDataDirectory read GetDataDirectory;
    property ImageDosHeader: PImageDosHeader read FDosHeader;
    property ImageFileHeader: PImageFileHeader read FFileHeader;
    property ImageOptionalHeader32: PImageOptionalHeader32 read FOptionalHeader32;
  end;

implementation

constructor TPEHeader.Create;
begin
  inherited;
  FHdrPtr := nil;
  FNewPEHeader := nil;
end;

destructor TPEHeader.Destroy;
begin
  Clear;
  inherited;
end;

procedure TPEHeader.Clear;
begin
  if FHdrPtr <> nil then
    FreeMem(FHdrPtr);
  FHdrPtr := nil;
  FDosHeader := nil;
  FFileHeader := nil;
  FOptionalHeader32 := nil;
  if Assigned(FNewPEHeader) then
    FreeAndNil(FNewPEHeader);
end;

function TPEHeader.AssignHeader(SrcHdrPtr: Pointer): Boolean;
begin
  result := false;
  if FHdrPtr <> nil then
    Clear;
  if not ValidatePE(SrcHdrPtr) then
    exit;
  if not DumpHeader(SrcHdrPtr) then
    exit;
  GetMem(FHdrPtr, FOptionalHeader32^.SizeOfHeaders);
  CopyMemory(FHdrPtr, SrcHdrPtr, FOptionalHeader32^.SizeOfHeaders);
  result := DoAssignHeader(FHdrPtr, FOptionalHeader32^.SizeOfHeaders);
end;

function TPEHeader.DoAssignHeader(SrcHdrPtr: Pointer; Size: Cardinal): Boolean;
begin
  result := false;
  if not DumpHeader(SrcHdrPtr) then
    exit;
  DumpDataDirectorys;
  DumpSectionHeaders;
  result := true;
end;

function TPEHeader.DumpHeader(Ptr: Pointer): Boolean;
begin
  FDosHeader := Ptr;
  FFileHeader := OffsetPointer(FDosHeader , FDosHeader^.e_lfanew + 4);
  FOptionalHeader32 := OffsetPointer(FFileHeader, IMAGE_SIZEOF_FILE_HEADER);
  result := true;
end;

procedure TPEHeader.DumpDataDirectorys;
var
  i: integer;
  DDPtr: PImageDataDirectory;
begin
  FDirectoryCount := GetDirectoryCount;
  SetLength(FDataDirectorys, DirectoryCount);

  DDPtr := @ImageOptionalHeader32^.DataDirectory[0];
  for i := 0 to DirectoryCount - 1 do
  begin
    FDataDirectorys[i] := DDPtr;
    inc(DDPtr);
  end;
end;

procedure TPEHeader.DumpSectionHeaders;
var
  i: integer;
  SHPtr: PImageSectionHeader;
begin
  FSectionCount := GetSectionCount;
  SetLength(FSections, SectionCount);

  SHPtr := OffsetPointer(ImageOptionalHeader32, ImageFileHeader^.SizeOfOptionalHeader);
  for i := 0 to SectionCount - 1 do
  begin
    FSections[i] := SHPtr;
    inc(SHPtr);
  end;
end;

function TPEHeader.GetSectionCount: Cardinal;
begin
  result := ImageFileHeader^.NumberOfSections;
end;

function TPEHeader.GetDirectoryCount: Cardinal;
begin
  result := ImageOptionalHeader32^.NumberOfRvaAndSizes;
end;

function TPEHeader.GetSection(index: Cardinal): PImageSectionHeader;
begin
  if index >= FSectionCount then
    Raise Exception.Create('index out of range!');
  result := FSections[index];
end;

function TPEHeader.GetDataDirectory(index: Cardinal): PImageDataDirectory;
begin
  if index >= FDirectoryCount then
    Raise Exception.Create('index out of range!');
  result := FDataDirectorys[index];
end;

function TPEHeader.ValidatePE(Ptr: Pointer): Boolean;
label Bad;
begin
  result := true;
  try
    if IsBadReadPtr(Ptr, 2) then goto Bad;
    if PWORD(Ptr)^ <> $5A4D then goto Bad;
    Ptr := OffsetPointer(Ptr, PImageDosHeader(Ptr)^.e_lfanew);
    if IsBadReadPtr(Ptr, 2) then goto Bad;
    if PWORD(Ptr)^ <> $4550 then goto Bad;
    exit;
Bad:
    result := false;
  except
    result := false;
  end;
end;
   
function TPEHeader.RVAInSection(RVA: Cardinal): integer;
var
  i: integer;
begin
  result := -1;
  if RVA = 0 then exit;
  if integer(RVA - Sections[SectionCount - 1]^.VirtualAddress) >
       integer(Sections[SectionCount - 1]^.Misc.VirtualSize) then
  begin
    result := -2;
    exit;
  end;
  for i := 0 to SectionCount - 1 do
  begin
    if (RVA >= Sections[i]^.VirtualAddress) and
       (RVA - Sections[i]^.VirtualAddress <
              Sections[i]^.Misc.VirtualSize) then
    begin
      result := i;
      break;
    end;
  end;
end;

function TPEHeader.RVAtoOffset(RVA: Cardinal): Cardinal;
var
  SectionRVA, PointerToRawData: Cardinal;
  tmp: integer;
begin
  tmp := RVAInSection(RVA);
  case tmp of
    -1:
    begin
      SectionRVA := 0;
      PointerToRawData := 0;
    end;
    -2:
    begin
      Raise Exception.Create('RVA out of file!');
    end;
    else
    begin
      SectionRVA := Sections[tmp]^.VirtualAddress;
      PointerToRawData := Sections[tmp]^.PointerToRawData;
    end;
  end;
  result := RVA - SectionRVA + PointerToRawData;
end;

function TPEHeader.RVAtoPtr(RVA: Cardinal): Pointer;
begin
  result := Pointer(DWORD(FHdrPtr) + RVAtoOffset(RVA));
end;

function TPEHeader.SetHeaderSize(NewSize: Cardinal): Boolean;
var
  tmpPtr: Pointer;
begin
  result := false;
  if NewSize <= FOptionalHeader32^.SizeOfHeaders then
    exit;
  tmpPtr := AllocMem(NewSize);
  CopyMemory(tmpPtr, FHdrPtr, NewSize);
  if AssignHeader(tmpPtr) then
    result := true;
  DoAssignHeader(tmpPtr, NewSize);
  FreeMem(tmpPtr);
end;

function TPEHeader.CalcFileAilgnment(Len: Cardinal): Cardinal;
var
  fl: Cardinal;
begin
  result := Len;
  fl := ImageOptionalHeader32^.FileAlignment;
  if Len mod fl <> 0 then
  begin
    result := (Len div fl + 1) * fl;
  end;
end;

function TPEHeader.CalcSectionAilgnment(Len: Cardinal): Cardinal;
var
  vl: Cardinal;
begin
  result := Len;
  vl := ImageOptionalHeader32^.SectionAlignment;
  if Len mod vl <> 0 then
  begin
    result := (Len div vl + 1) * vl;
  end;
end;

function TPEHeader.AddSection: TPEHeader;
var
  TotalHdrSize: Cardinal;  
  ToBeMoveList: TList;    
  ToBeMovSize: Cardinal;    
  tmpLen: Cardinal;
  pMovDest: Pointer;
  NewHeader: TPEHeader;
  tmpDD: PImageDataDirectory;
  i: Cardinal;

  function GetToBeMoveList: Boolean;
  var
    i: Cardinal;
    f: Boolean;
  begin
    result := false;
    f := false;
    for i := 0 to NewHeader.DirectoryCount - 1 do
    begin
      if i = IMAGE_DIRECTORY_ENTRY_EXPORT then continue;
      if (NewHeader.DataDirectorys[i]^.VirtualAddress = 0) then
        Continue;
      if NewHeader.DataDirectorys[i]^.VirtualAddress < NewHeader.ImageOptionalHeader32^.SizeOfHeaders then
      begin
        ToBeMoveList.Add(NewHeader.DataDirectorys[i]);
        inc(ToBeMovSize, NewHeader.DataDirectorys[i]^.Size);
        f := true;
      end;
    end;
    if f then
      result := true;
  end;

  function CompareRVA(i1, i2: Pointer): Integer;
  begin
    Result := PImageDataDirectory(i1)^.VirtualAddress -
              PImageDataDirectory(i1)^.VirtualAddress;
  end;

begin
  result := nil;
  if Assigned(FNewPEHeader) then
  begin
    FNewPEHeader.Free;
    FNewPEHeader := nil;
  end;
  NewHeader := TPEHeader.Create;
  if not NewHeader.AssignHeader(FHdrPtr) then
  begin
    ShowMessage('AddSection Error!');
    FreeAndNil(NewHeader);
    exit;
  end;
  TotalHdrSize := NewHeader.ImageOptionalHeader32^.SizeOfHeaders;

  if TotalHdrSize < DWORD(NewHeader.DataDirectorys[DirectoryCount - 1]) - DWORD(NewHeader.ImageDosHeader) +
                    2 * SizeOf(TImageSectionHeader) then
  begin
    TotalHdrSize := DWORD(NewHeader.DataDirectorys[DirectoryCount - 1]) - DWORD(NewHeader.ImageDosHeader) +
                    2 * SizeOf(TImageSectionHeader);
  end;

  TotalHdrSize := CalcFileAilgnment(TotalHdrSize);
  ToBeMoveList := TList.Create;
  ToBeMovSize := 0;
  try                            //SetHeaderSize
    if GetToBeMoveList then
    begin
      tmpLen := CalcFileAilgnment(DWORD(NewHeader.DataDirectorys[DirectoryCount - 1]) - DWORD(NewHeader.ImageDosHeader) +
                        SizeOf(TImageSectionHeader) + ToBeMovSize);
      if TotalHdrSize < tmpLen then
      begin
        TotalHdrSize := tmpLen;
        NewHeader.SetHeaderSize(TotalHdrSize);
      end;

      ToBeMoveList.Sort(@CompareRVA);  //SORT

      pMovDest := OffsetPointer(NewHeader.DataDirectorys[DirectoryCount - 1], SizeOf(TImageDataDirectory) +
                                (NewHeader.ImageFileHeader^.NumberOfSections + 1) * SizeOf(TImageSectionHeader));
      for i := 0 to ToBeMoveList.Count - 1 do
      begin
        tmpDD := PImageDataDirectory(ToBeMoveList[i]);
        CopyMemory(pMovDest, RVAtoPtr(tmpDD^.VirtualAddress),
                    tmpDD^.Size);

        pMovDest := OffsetPointer(pMovDest, tmpDD^.Size);
        inc(PImageDataDirectory(ToBeMoveList[i])^.VirtualAddress,
            SizeOf(TImageSectionHeader));
      end;
      pMovDest := OffsetPointer(NewHeader.DataDirectorys[DirectoryCount - 1], SizeOf(TImageDataDirectory) +
                                NewHeader.ImageFileHeader^.NumberOfSections  * SizeOf(TImageSectionHeader));
      ZeroMemory(pMovDest, SizeOf(TImageSectionHeader));
    end;
  finally
    ToBeMoveList.Free;
  end;
  NewHeader.ImageOptionalHeader32^.SizeOfImage := NewHeader.ImageOptionalHeader32^.SizeOfImage +
      TotalHdrSize - NewHeader.ImageOptionalHeader32^.SizeOfHeaders;
  NewHeader.ImageOptionalHeader32^.SizeOfHeaders := TotalHdrSize;
  inc(NewHeader.ImageFileHeader^.NumberOfSections);
  NewHeader.DumpSectionHeaders;
  FNewPEHeader := NewHeader;
  result := NewHeader;
end;

{
function TPEHeader.AddSection(var pNewHdr: Pointer; var Size: Cardinal): PImageSectionHeader;
var
  HdrSize: Cardinal;
  pDosHeader: PImageDosHeader;
  pFileHeader: PImageFileHeader;
  pOptionalHeader32: PImageOptionalHeader32;
  pDataDir: Array of PImageDataDirectory;
  pSectionHdr: Array of PImageSectionHeader;
  fl: Cardinal;
  i, m: Cardinal;
  pToBeMov: Pointer;
  pMovDest: Pointer;
  ToBeMovSize: Cardinal;
  TotalHdrSize: Cardinal;
  tmpLen: Cardinal;
  ToBeMoveList: TList;

  xx: PImageDataDirectory;

  function ReAilgn(Len: Cardinal): Cardinal;
  begin
    result := Len;
    if Len mod fl <> 0 then
    begin
      result := (Len div fl + 1) * fl;
    end;
  end;

  procedure DumpDataDirectorys;
  var
    i: integer;
    DDPtr: PImageDataDirectory;
  begin
    SetLength(pDataDir, DirectoryCount);

    DDPtr := @pOptionalHeader32^.DataDirectory;
    for i := 0 to DirectoryCount - 1 do
    begin
      pDataDir[i] := DDPtr;
      inc(DDPtr);
    end;
  end;

  procedure DumpSectionHeaders(SecCount: Cardinal);
  var
    i: integer;
    SHPtr: PImageSectionHeader;
  begin
    SetLength(pSectionHdr, SecCount);

    SHPtr := OffsetPointer(pOptionalHeader32, pFileHeader^.SizeOfOptionalHeader);
    for i := 0 to SecCount - 1 do
    begin
      pSectionHdr[i] := SHPtr;
      inc(SHPtr);
    end;
  end;

  function GetToBeMoveList: Boolean;
  var
    i: Cardinal;
    f: Boolean;
  begin
    result := false;
    f := false;
    for i := 0 to DirectoryCount - 1 do
    begin
      if i = IMAGE_DIRECTORY_ENTRY_EXPORT then continue;
      if (DataDirectorys[i]^.VirtualAddress = 0) then
        Continue;
      if DataDirectorys[i]^.VirtualAddress < FImageHeaderSize then
      begin
        ToBeMoveList.Add(Pointer(i));
        inc(ToBeMovSize, DataDirectorys[i]^.Size);
        f := true;
      end;
    end;
    if f then
      result := true;
  end;

  function CompareRVA(i1, i2: Pointer): Integer;
  begin
    Result := pDataDir[DWORD(i1)]^.VirtualAddress - pDataDir[DWORD(i1)]^.VirtualAddress;
  end;

begin
  result := 0;
  pNewHdr := AllocMem(FImageHeaderSize);
  CopyMemory(pNewHdr, FImageDosHeader, FImageHeaderSize);
  pDosHeader := pNewHdr;
  pFileHeader := OffsetPointer(pDosHeader , pDosHeader^.e_lfanew + 4);
  pOptionalHeader32 := OffsetPointer(pFileHeader, IMAGE_SIZEOF_FILE_HEADER);
  DumpDataDirectorys;
  fl := pOptionalHeader32^.FileAlignment;
  TotalHdrSize := pOptionalHeader32^.SizeOfHeaders;

  if TotalHdrSize < DWORD(pDataDir[DirectoryCount - 1]) - DWORD(pDosHeader) +
                    2 * SizeOf(TImageSectionHeader) then
  begin
    TotalHdrSize := DWORD(pDataDir[DirectoryCount - 1]) - DWORD(pDosHeader) +
                    2 * SizeOf(TImageSectionHeader);
  end;

  TotalHdrSize := ReAilgn(TotalHdrSize);

  ToBeMoveList := TList.Create;
  ToBeMovSize := 0;
  try
    if GetToBeMoveList then
    begin
      tmpLen := ReAilgn(DWORD(pDataDir[DirectoryCount - 1]) - DWORD(pDosHeader) +
                        SizeOf(TImageSectionHeader) + ToBeMovSize);
      if TotalHdrSize < tmpLen then
      begin
        TotalHdrSize := tmpLen;
        ReAllocMem(pNewHdr, tmpLen);
      end;

      ToBeMoveList.Sort(@CompareRVA);  //SORT

      pMovDest := OffsetPointer(pDataDir[DirectoryCount - 1], SizeOf(TImageDataDirectory) +
                                pFileHeader^.NumberOfSections * SizeOf(TImageSectionHeader));
      for i := 0 to ToBeMoveList.Count - 1 do
      begin
        xx := DataDirectorys[DWORD(ToBeMoveList[i])];
        CopyMemory(pMovDest, RVAtoPtr(xx^.VirtualAddress),
                    xx^.Size);

        pMovDest := OffsetPointer(pMovDest, xx^.Size);
        inc(pDataDir[DWORD(ToBeMoveList[i])]^.VirtualAddress, SizeOf(TImageDataDirectory));
      end;
      pMovDest := OffsetPointer(pDataDir[DirectoryCount - 1], SizeOf(TImageDataDirectory) +
                                pFileHeader^.NumberOfSections * SizeOf(TImageSectionHeader));
      ZeroMemory(pMovDest, SizeOf(TImageSectionHeader));
    end;
  finally
    ToBeMoveList.Free;
  end;
  pOptionalHeader32^.SizeOfHeaders := TotalHdrSize;
  inc(pFileHeader^.NumberOfSections);
  Size := TotalHdrSize;
  DumpSectionHeaders(pFileHeader^.NumberOfSections);
  result := pSectionHdr[pFileHeader^.NumberOfSections - 1];
end;
}
end.
