/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 hsz Jakub Chrzanowski <jakub@hsz.mobi>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package mobi.hsz.idea.gitignore.util;

import com.intellij.notification.*;
import com.intellij.openapi.project.Project;
import mobi.hsz.idea.gitignore.IgnoreBundle;
import mobi.hsz.idea.gitignore.lang.IgnoreLanguage;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Wrapper function for showing {@link Notification}.
 *
 * @author Jakub Chrzanowski <jakub@hsz.mobi>
 * @since 1.7
 */
public class Notify {
    /**
     * Shows {@link Notification} in IGNORE_GROUP_UPDATE group.
     *
     * @param project  current project
     */
    public static void showUpdate(@NotNull Project project) {
        show(
                project,
                IgnoreBundle.message("notification.update.title", Utils.getVersion()),
                IgnoreBundle.message("notification.update.content"),
                IgnoreLanguage.GROUP + "_UPDATE",
                NotificationType.INFORMATION,
                NotificationListener.URL_OPENING_LISTENER
        );
    }

    /**
     * Shows {@link Notification} in {@link IgnoreLanguage#GROUP} group.
     *
     * @param project   current project
     * @param title     notification title
     * @param content   notification text
     * @param type      notification type
     * @param listener  optional listener
     */
    public static void show(@NotNull Project project, @NotNull String title, @NotNull String content,
                            @NotNull NotificationType type, @Nullable NotificationListener listener) {
        show(project, title, content, IgnoreLanguage.GROUP, type, listener);
    }

    /**
     * Shows {@link Notification}.
     *
     * @param project   current project
     * @param title     notification title
     * @param displayId notification group
     * @param content   notification text
     * @param type      notification type
     * @param listener  optional listener
     */
    public static void show(@NotNull Project project, @NotNull String title, @NotNull String content,
                            @NotNull String displayId, @NotNull NotificationType type,
                            @Nullable NotificationListener listener) {
        NotificationGroup group = new NotificationGroup(
                displayId,
                NotificationDisplayType.STICKY_BALLOON,
                true
        );
        Notification notification = group.createNotification(title, content, type, listener);
        Notifications.Bus.notify(notification, project);
    }
}
