/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

var EXPORTED_SYMBOLS = ["NewWindow", "console",
                        "insertButton", "destroyButton", "updateButton", "ButtonPersistence", "ProfileAlias",
                        "ErrorHandler", "ExtCompat",
                        "cookieInternalDomain", // migrateCookies
                        "Profile"
                       ];

var Cu = Components.utils;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/PrivateBrowsingUtils.jsm");
Cu.import("resource://multifox-4410e96638/new-window.js");


/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


const BrowserWindow = {

  register: function(win) {
    var profileId = Profile.getIdentity(win);

    if (Profile.isNativeProfile(profileId)) {
      console.log("BrowserWindow.register NOP");
      return;
    }

    console.log("BrowserWindow.register " + profileId);

    var ns = {}; // BUG util is undefined???
    Cu.import("resource://multifox-4410e96638/new-window.js", ns);
    if (ns.util.networkListeners.active === false) {
      // first multifox window!

      var nsActions = {};
      Components.utils.import("resource://multifox-4410e96638/actions.js", nsActions);
      nsActions.migrateCookies();


      Cookies.start();
      DocStartScriptInjection.init();
      ns.util.networkListeners.enable(httpListeners.request, httpListeners.response);
    }

    // some MultifoxContentEvent_* listeners are not called when
    // there are "unload" listeners with useCapture=true. o_O
    // But they are called if event listener is an anonymous function.
    win.addEventListener("unload", onUnloadChromeWindow, false);

    // update icon status
    win.getBrowser().tabContainer.addEventListener("TabSelect", tabSelected, false);
  },


  // should keep id for session restore?
  unregister: function(win) {
    var idw = Profile.getIdentity(win);
    console.log("BrowserWindow.unregister " + idw);

    if (Profile.isNativeProfile(idw)) {
      return; // nothing to unregister
    }

    win.removeEventListener("unload", onUnloadChromeWindow, false);
    win.getBrowser().tabContainer.removeEventListener("TabSelect", tabSelected, false);

    var ns = {}; // BUG util is undefined???
    Cu.import("resource://multifox-4410e96638/new-window.js", ns);
    if (ns.util.networkListeners.active) {
      this._checkLastWin(win);
    }
  },


  _checkLastWin: function(win) {
    var sessions = Profile.activeIdentities(win);
    var onlyNative = true;
    for (var idx = sessions.length - 1; idx > -1; idx--) {
      if (Profile.isExtensionProfile(sessions[idx])) {
        onlyNative = false;
        break;
      }
    }
    if (onlyNative) {
      var ns = {}; // BUG util is undefined???
      Cu.import("resource://multifox-4410e96638/new-window.js", ns);
      ns.util.networkListeners.disable();
      DocStartScriptInjection.stop();
      Cookies.stop();
    }
  }
};


function onUnloadChromeWindow(evt) {
  var win = evt.currentTarget;
  BrowserWindow.unregister(win);
}


function tabSelected(evt) {
  var tab = evt.originalTarget;
  ErrorHandler.updateButtonAsync(tab.linkedBrowser);
}
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


var ProfileAlias = {
  _alias: null,

  _load: function() {
    var prefs = Services.prefs.getBranch("extensions.multifox@hultmann.");
    if (prefs.prefHasUserValue("alias") === false) {
      this._alias = Object.create(null);
      return;
    }

    var a = prefs.getComplexValue("alias", Ci.nsISupportsString).data;
    try {
      this._alias = JSON.parse(a);
    } catch (ex) {
      console.log(ex + " - buggy json - " + a);
      this._alias = Object.create(null);
    }
  },


  rename: function(profileId, name) {
    if (name.length > 0) {
      this._alias[profileId] = name;
    } else {
      delete this._alias[profileId];
    }
    var ns = {}; // BUG util is undefined???
    Cu.import("resource://multifox-4410e96638/new-window.js", ns);
    ns.util.setUnicodePref("alias", JSON.stringify(this._alias));
  },


  remove: function(profileId) {
    delete this._alias[profileId];
    var ns = {}; // BUG util is undefined???
    Cu.import("resource://multifox-4410e96638/new-window.js", ns);
    ns.util.setUnicodePref("alias", JSON.stringify(this._alias));
  },

  sort: function(arr) {
    if (this._alias === null) {
      this._load();
    }
    var me = this;
    arr.sort(function(id1, id2) {
      return me.format(id1).localeCompare(me.format(id2));
    });
    return arr;
  },

  hasAlias: function(profileId) {
    if (this._alias === null) {
      this._load();
    }
    return profileId in this._alias;
  },

  format: function(profileId) {
    if (this._alias === null) {
      this._load();
    }

    var ns = {}; // BUG util is undefined???
    Cu.import("resource://multifox-4410e96638/new-window.js", ns);

    switch (profileId) {
      case Profile.DefaultIdentity:
        return ns.util.getText("button.menuitem.profile.default.label");

      case Profile.PrivateIdentity:
        return ns.util.getText("button.menuitem.profile.private.label");

      case Profile.UndefinedIdentity:
        throw new Error("unexpected Profile.UndefinedIdentity");
    }
    console.assert(Profile.isExtensionProfile(profileId), "profileId unexpected", profileId);

    if (profileId in this._alias) {
      return this._alias[profileId];
    }

    return ns.util.getText("button.menuitem.profile.extension.label", profileId);
  }
};



function updateButton(win) {
  var doc = win.document;
  var button = getButtonElem(doc);
  if (button === null) { // visible?
    return;
  }

  // show <dropmarker> (hidden by class bookmark-item)
  doc.getAnonymousElementByAttribute(button, "class", "toolbarbutton-menu-dropmarker")
     .setAttribute("style", "display:-moz-box !important");

  // update label
  var txt;
  var profileId = Profile.getIdentity(win);
  if (Profile.isExtensionProfile(profileId)) {
    txt = ProfileAlias.hasAlias(profileId) ? ProfileAlias.format(profileId)
                                           : profileId.toString();
  } else {
    txt = "";
  }

  button.setAttribute("label", txt);
}


function getButtonElem(doc) {
  return doc.getElementById("multifox-button");
}


function insertButton(doc) {
  console.assert(getButtonElem(doc) === null, "insertButton dupe");

  // restore icon after toolbar customization
  doc.defaultView.addEventListener("aftercustomization", customizeToolbar, false);

  var buttonId = "multifox-button";

  // add it to <toolbarpalette>
  doc.getElementById("navigator-toolbox")
     .palette.appendChild(createButtonElem(doc, buttonId));

  // add it to <toolbar>
  var toolbar = ButtonPersistence.findToolbar(doc, buttonId);
  if (toolbar != null) {
    if (fixToolbarBug(toolbar)) {
      return;
    }

    var button0 = ButtonPersistence.getPrecedingButton(toolbar, buttonId);
    toolbar.insertItem(buttonId, button0, null, false);
    return;
  }

  // add it to default position
  var ns = {};
  Cu.import("resource://multifox-4410e96638/new-window.js", ns); // BUG Bootstrap is undefined
  if (ns.Bootstrap.showButtonByDefault === false) {
    return; // keep button hidden
  }

  toolbar = doc.getElementById("nav-bar");
  toolbar.insertItem(buttonId);
  ButtonPersistence.saveToolbar(toolbar);
}


// bug introduced in version 2.0.0...2.0.5
function fixToolbarBug(toolbar) {
  if (toolbar.id !== "nav-bar") {
    return false;
  }
  if (toolbar.getAttribute("currentset") !== ",multifox-button") {
    return false;
  }

  toolbar.setAttribute("currentset", "");
  var doc = toolbar.ownerDocument;
  doc.persist(toolbar.id, "currentset");

  var ns = {};
  Cu.import("resource://multifox-4410e96638/new-window.js", ns);
  ns.Bootstrap.resetButton();

  doc.getElementById("cmd_newNavigator").doCommand();
  return true;
}


function createButtonElem(doc, buttonId) {
  var button = doc.createElement("toolbarbutton");
  button.setAttribute("id", buttonId);
  button.setAttribute("tab-status", "");
  button.setAttribute("type", "menu");
  button.setAttribute("class", "bookmark-item"); // show label beside its icon
  button.setAttribute("style", "list-style-image:url(chrome://multifox-4410e96638/content/favicon.ico);-moz-image-region:auto");
  button.setAttribute("label", "Multifox");
  button.setAttribute("tooltiptext", "Multifox");
  var menupopup = button.appendChild(doc.createElement("menupopup"));
  menupopup.addEventListener("popupshowing", onMenuPopupShowing, false);
  return button;
}


var ButtonPersistence = {
  saveToolbar: function(toolbar) {
    toolbar.setAttribute("currentset", toolbar.currentSet);
    toolbar.ownerDocument.persist(toolbar.id, "currentset");
  },

  removeButton: function(buttonId) {
    var enumWin = Services.wm.getEnumerator("navigator:browser");
    while (enumWin.hasMoreElements()) {
      var doc = enumWin.getNext().document;
      var toolbar = ButtonPersistence.findToolbar(doc, buttonId);
      if (toolbar !== null) {
        ButtonPersistence.saveToolbar(toolbar);
      }
    }
  },

  findToolbar: function(doc, buttonId) {
    var bars = doc.getElementsByTagName("toolbar");
    for (var idx = bars.length - 1; idx > -1; idx--) {
      var all = ButtonPersistence._getToolbarButtons(bars[idx]);
      if (all.indexOf(buttonId) > -1) {
        return bars[idx];
      }
    }
    return null;
  },

  getPrecedingButton: function(toolbar, id) {
    var all = ButtonPersistence._getToolbarButtons(toolbar);
    var idxButton = all.indexOf(id);
    if (idxButton === -1) {
      throw new Error("button not found @ " + toolbar.id);
    }
    var doc = toolbar.ownerDocument;
    for (var idx = idxButton + 1, len = all.length; idx < len; idx++) {
      var beforeNode = doc.getElementById(all[idx]);
      if (beforeNode !== null) {
        return beforeNode;
      }
    }
    return null;
  },

  _getToolbarButtons: function(toolbar) {
    // it may be empty
    return toolbar.getAttribute("currentset").split(",");
  }
};


function destroyButton(doc) {
   doc.defaultView.removeEventListener("aftercustomization", customizeToolbar, false);

  var plt = doc.getElementById("navigator-toolbox").palette;
  var button2 = plt.children.namedItem("multifox-button");
  if (button2 !== null) {
    plt.removeChild(button2);
  }

  var button = getButtonElem(doc);
  if (button !== null) {
    var menu = button.firstChild;
    console.assert(menu.tagName === "menupopup", "wrong element: " + menu.tagName)
    menu.removeEventListener("popupshowing", onMenuPopupShowing, false);
    button.parentNode.removeChild(button); // button position is persisted until uninstall
  }
}


function customizeToolbar(evt) {
  var toolbox = evt.target;
  updateButton(toolbox.ownerDocument.defaultView);
}


function onMenuPopupShowing(evt) {
  if (evt.currentTarget === evt.target) {
    var ns = {};
    Cu.import("resource://multifox-4410e96638/actions.js", ns);
    ns.menuButtonShowing(evt.target);
  }
}
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


// Add hooks to documents (cookie, localStorage, ...)

var DocStartScriptInjection = {

  _innerWindows: Object.create(null),
  _loader: null,


  init: function() {
    console.assert(this._loader === null, "this._loader is already initialized");
    this._loader = new ScriptSourceLoader();
    Services.obs.addObserver(this, "document-element-inserted", false);
    Services.obs.addObserver(this._onInnerDestroyed, "inner-window-destroyed", false);
    this._initCurrent();
  },


  stop: function() {
    Services.obs.removeObserver(this, "document-element-inserted");
    Services.obs.removeObserver(this._onInnerDestroyed, "inner-window-destroyed");
    this._loader = null;

    var innerWindows = this._innerWindows;
    this._innerWindows = Object.create(null);

    // nuke all sandboxes
    for (var id in innerWindows) {
      var sandbox = innerWindows[id];
      console.assert(sandbox !== null, "sandbox cannot be null", id);

      // avoid "can't access dead object" errors
      delete sandbox.document.cookie;
      delete sandbox.window.localStorage;
      delete sandbox.window.indexedDB;
      delete sandbox.window.mozIndexedDB;

      Cu.nukeSandbox(sandbox);
    }
  },

  _forEachWindow: function(fn, win) {
    fn(win);
    for (var idx = win.length - 1; idx > -1; idx--) {
      this._forEachWindow(fn, win[idx]);
    }
  },

  _initCurrent: function() {
    var enumWin = Services.wm.getEnumerator("navigator:browser");
    while (enumWin.hasMoreElements()) {
      var win = enumWin.getNext();
      if (Profile.isNativeProfile(Profile.getIdentity(win))) {
        continue;
      }
      var all = win.getBrowser().browsers;
      for (var idx = all.length - 1; idx > -1; idx--) {
        this._forEachWindow(DocStartScriptInjection._initWindow,
                            all[idx].contentWindow);
      }
    }
  },


  _onInnerDestroyed: {
    observe: function(subject, topic, data) {
      var id = subject.QueryInterface(Ci.nsISupportsPRUint64).data.toString();
      delete DocStartScriptInjection._innerWindows[id];
    }
  },


  observe: function(subject, topic, data) {
    var win = subject.defaultView;
    if (win !== null) { // xsl/xbl
      this._initWindow(win);
    }
  },


  _initWindow: function(win) {
    var winInfo = FindIdentity.fromContent(win);
    if (Profile.isNativeProfile(winInfo.profileNumber)) {
      return;
    }

    if (winInfo.browserElement) {
      ErrorHandler.onNewWindow(win, winInfo.browserElement);
    }

    switch (win.location.protocol) {
      case "http:":
      case "https:":
        break;
      default:
        return;
    }

    var sandbox = Cu.Sandbox(win, {sandboxName: "multifox-sandbox", wantComponents:false});
    sandbox.window = XPCNativeWrapper.unwrap(win);
    sandbox.document = XPCNativeWrapper.unwrap(win.document);
    sandbox.sendCmd = function(obj) {
      return cmdContent(obj, win.document);
    };

    var me = DocStartScriptInjection;
    var src = me._loader.getScript();
    try {
      // window.localStorage will be replaced by a Proxy object.
      // It seems it's only possible using a sandbox.
      Cu.evalInSandbox(src, sandbox);
    } catch (ex) {
      ErrorHandler.addScriptError(win, "sandbox", win.document.documentURI + " " + "//exception=" + ex);
      Cu.nukeSandbox(sandbox);
      return;
    }

    // keep a reference to Cu.nukeSandbox (Cu.getWeakReference won't work for that)
    var innerId = win.QueryInterface(Ci.nsIInterfaceRequestor)
                     .getInterface(Ci.nsIDOMWindowUtils).currentInnerWindowID.toString();
    console.assert((innerId in me._innerWindows) === false, "dupe sandbox @", innerId)
    me._innerWindows[innerId] = sandbox;
  }
};


function cmdContent(obj, contentDoc) {
  switch (obj.from) {
    case "cookie":
      return documentCookie(obj, contentDoc);
    case "localStorage":
      return windowLocalStorage(obj, contentDoc);
    case "error":
      ErrorHandler.addScriptError(contentDoc.defaultView, obj.cmd, "-");
      return undefined;
    default:
      throw obj.from;
  }
}


function ScriptSourceLoader() {
  this._src = null;
  this._load(true);
}


ScriptSourceLoader.prototype = {

  getScript: function() {
    if (this._src === null) {
      this._load(false);
    }
    return this._src;
  },

  _load: function(async) {
    var me = this;
    var xhr = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(Ci.nsIXMLHttpRequest);
    xhr.onload = function() {
      me._src = xhr.responseText;
    };
    xhr.open("GET", "chrome://multifox-4410e96638/content/content-injection.js", async);
    xhr.overrideMimeType("text/plain");
    xhr.send(null);
  }
};
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


const httpListeners = {
  request: {
    // nsIObserver
    observe: function(aSubject, aTopic, aData) {
      var httpChannel = aSubject.QueryInterface(Ci.nsIHttpChannel);
      var ctx = getLoadContext(httpChannel)
      if ((ctx === null) || ctx.usePrivateBrowsing) {
        return;
      }

      var winInfo = FindIdentity.fromContent(ctx.associatedWindow);
      if (Profile.isNativeProfile(winInfo.profileNumber)) {
        return; // default/private window, favicon, updates
      }

      if (isTopWindowChannel(httpChannel, ctx.associatedWindow)) {
        ErrorHandler.onNewWindowRequest(winInfo.browserElement);
      }
      /*
      var myHeaders = HttpHeaders.fromRequest(httpChannel);
      if (myHeaders["authorization"] !== null) {
        ErrorHandler.addNetworkError(ctx.associatedWindow, "authorization");
        return;
      }
      */

      var cook = Cookies.getCookie(false, httpChannel.URI, winInfo.profileNumber);
      httpChannel.setRequestHeader("Cookie", cook, false);
    }
  },

  response: {
    // nsIObserver
    observe: function(aSubject, aTopic, aData) {
      var httpChannel = aSubject.QueryInterface(Ci.nsIHttpChannel);
      var ctx = getLoadContext(httpChannel)
      if ((ctx === null) || ctx.usePrivateBrowsing) {
        return;
      }
      var winChannel = ctx.associatedWindow;
      var profileId = FindIdentity.fromContent(winChannel).profileNumber;
      if (Profile.isNativeProfile(profileId)) {
        return;
      }


      var myHeaders = HttpHeaders.fromResponse(httpChannel);
      if (myHeaders["www-authenticate"] !== null) {
        ErrorHandler.addNetworkError(winChannel, "www-authenticate");
        return;
      }

      // BUG cookies from multipart responses are lost

      var setCookies = myHeaders["set-cookie"];
      if (setCookies === null) {
        return;
      }

      // server sent "Set-Cookie"
      httpChannel.setResponseHeader("Set-Cookie", null, false);
      Cookies.setCookie(profileId, httpChannel.URI, setCookies, false);
    }
  }
};


const HttpHeaders = {
  visitLoop: {
    values: null,
    visitHeader: function(name, value) {
      var n = name.toLowerCase();
      if (n in this.values) {
        this.values[n] = value;
      }
    }
  },

  /*
  fromRequest: function(request) {
    var nameValues = {
      //"cookie": null, //for debug only
      "authorization": null
    }
    this.visitLoop.values = nameValues;
    request.visitRequestHeaders(this.visitLoop);
    return nameValues;
  },
  */

  fromResponse: function(response) {
    var nameValues = {
      "set-cookie": null,
      "www-authenticate": null
    }
    this.visitLoop.values = nameValues;
    response.visitResponseHeaders(this.visitLoop);
    return nameValues;
  }
};


function getLoadContext(channel) {
  if (channel.notificationCallbacks) {
    try {
      return channel
              .notificationCallbacks
              .getInterface(Ci.nsILoadContext);
    } catch (ex) {
      //console.trace("channel.notificationCallbacks ", channel.notificationCallbacks, channel.URI.spec, ex);
    }
  }

  if (channel.loadGroup && channel.loadGroup.notificationCallbacks) {
    try {
      return channel
              .loadGroup
              .notificationCallbacks
              .getInterface(Ci.nsILoadContext);
    } catch (ex) {
      console.trace("channel.loadGroup", channel.loadGroup, channel.URI.spec, ex);
    }
  }

  //var isChrome = context.associatedWindow instanceof Ci.nsIDOMChromeWindow;
  //return context.isContent ? context.associatedWindow : null;
  //console.log("LOAD CONTEXT FAIL " + channel.URI.spec);
  return null; // e.g. <link rel=prefetch> <link rel=next> ...
}


function isTopWindowChannel(channel, associatedWin) {
  if ((channel.loadFlags & Ci.nsIChannel.LOAD_DOCUMENT_URI) === 0) {
    return false;
  }
  return associatedWin === associatedWin.top;
}
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


function documentCookie(obj, contentDoc) {
  switch (obj.cmd) {
    case "set":
      documentCookieSetter(obj, contentDoc);
      return undefined;
    case "get":
      return documentCookieGetter(obj, contentDoc);
    default:
      throw obj.cmd;
  }
}


function documentCookieGetter(obj, contentDoc) {
  var profileId = FindIdentity.fromContent(contentDoc.defaultView).profileNumber;
  if (Profile.isExtensionProfile(profileId)) {
    var uri = stringToUri(contentDoc.location.href);
    var cookie2 = Cookies.getCookie(true, uri, profileId);
    var cookie = cookie2 === null ? "" : cookie2;
    return cookie; // send cookie value to content
  }
}


function documentCookieSetter(obj, contentDoc) {
  var profileId = FindIdentity.fromContent(contentDoc.defaultView).profileNumber;
  if (Profile.isExtensionProfile(profileId)) {
    var originalUri = stringToUri(contentDoc.location.href);
    Cookies.setCookie(profileId, originalUri, obj.value, true);
  }
}


const PREF_COOKIE_BEHAVIOR = "network.cookie.cookieBehavior";

const Cookies = {
  _service: null,
  _prefs: null,

  start: function() {
    this._service = Cc["@mozilla.org/cookieService;1"].getService().QueryInterface(Ci.nsICookieService);
    this._prefs = Services.prefs;
    this._prefListener.behavior = this._prefs.getIntPref(PREF_COOKIE_BEHAVIOR);
    this._prefs.addObserver(PREF_COOKIE_BEHAVIOR, this._prefListener, false);
  },

  stop: function() {
    this._service = null;
    this._prefs.removeObserver(PREF_COOKIE_BEHAVIOR, this._prefListener);
    this._prefs = null;
  },


  _prefListener: {
    behavior: -1,

    QueryInterface: XPCOMUtils.generateQI([Ci.nsIObserver]),
    // aTopic=nsPref:changed aData=network.cookie.cookieBehavior
    observe: function(aSubject, aTopic, aData) {
      this.behavior = aSubject
                      .QueryInterface(Ci.nsIPrefBranch)
                      .getIntPref(PREF_COOKIE_BEHAVIOR);
      console.log("pref! " + aSubject + aTopic + aData + this.behavior);
    }

  },

  setCookie: function(profileId, originalUri, originalCookie, fromJs) {
    var uri = toInternalUri(originalUri, profileId);
    var val = convertCookieDomain(originalCookie, profileId);

    if (this._prefListener.behavior === 0) {
      this._setCookie(fromJs, uri, val);
      return;
    }

    var p = this._prefs;
    p.removeObserver(PREF_COOKIE_BEHAVIOR, this._prefListener);
    p.setIntPref(PREF_COOKIE_BEHAVIOR, 0);
    this._setCookie(fromJs, uri, val);
    p.setIntPref(PREF_COOKIE_BEHAVIOR, this._prefListener.behavior);
    p.addObserver(PREF_COOKIE_BEHAVIOR, this._prefListener, false);
  },

  getCookie: function(fromJs, originalUri, profileId) {
    var uri = toInternalUri(originalUri, profileId);

    if (this._prefListener.behavior === 0) {
      return this._getCookie(fromJs, uri);
    }

    var p = this._prefs;
    p.removeObserver(PREF_COOKIE_BEHAVIOR, this._prefListener);
    p.setIntPref(PREF_COOKIE_BEHAVIOR, 0);
    var cookie = this._getCookie(fromJs, uri);
    p.setIntPref(PREF_COOKIE_BEHAVIOR, this._prefListener.behavior);
    p.addObserver(PREF_COOKIE_BEHAVIOR, this._prefListener, false);
    return cookie;
  },

  _setCookie: function(fromJs, uri, val) {
    if (fromJs) {
      this._service.setCookieString(uri,
                                    null,
                                    val,
                                    null);
    } else {
      //setCookieString doesn't work for httponly cookies
      this._service.setCookieStringFromHttp(uri,   // aURI
                                            null,  // aFirstURI
                                            null,  // aPrompt
                                            val,   // aCookie
                                            null,  // aServerTime
                                            null); // aChannel
    }
  },

  _getCookie: function(fromJs, uri) {
    if (fromJs) {
      return this._service.getCookieString(uri,
                                           null);
    } else {
      return this._service.getCookieStringFromHttp(uri,   // aURI
                                                   null,  // aFirstURI
                                                   null); // aChannel
    }
  }
};


function convertCookieDomain(cookieHeader, profileId) {
  var objCookies = new SetCookieParser(cookieHeader, true);
  var len = objCookies.length;
  var newCookies = new Array(len);

  for (var idx = 0; idx < len; idx++) {
    var myCookie = objCookies.getCookieByIndex(idx);
    var realDomain = myCookie.getStringProperty("domain");
    if (realDomain.length > 0) {
      myCookie.defineMeta("domain", cookieInternalDomain(realDomain, profileId));
      newCookies[idx] = myCookie.toHeaderLine();
    } else {
      newCookies[idx] = myCookie.raw;
    }
  }

  return newCookies.join("\n");//objCookies.toHeader();
}


function toInternalUri(uri, sessionId) {
  var u = uri.clone();
  if (Profile.isExtensionProfile(sessionId)) {
    u.host = cookieInternalDomain(u.host, sessionId);
  } else {
    console.trace("invalid profile", sessionId);
  }
  return u;
}


function cookieInternalDomain(domain, id) {
  // this scheme makes Multifox profiles to obey the
  // cookie limits per TLD (network.cookie.maxPerHost)
  var tld = getTldFromHost(domain).replace(".", "-", "g");
  return domain + "." + tld + "-" + id + ".multifox";
}


function getTldFromHost(hostname) {
  console.assert(typeof hostname === "string", "invalid hostname argument");
  console.assert(hostname.length > 0, "empty hostname");
  try {
    return Services.eTLD.getBaseDomainFromHost(hostname);
  } catch (ex) {
    var Cr = Components.results;
    switch (ex.result) {
      case Cr.NS_ERROR_INSUFFICIENT_DOMAIN_LEVELS: // "localhost"?
      case Cr.NS_ERROR_HOST_IS_IP_ADDRESS:         // literal ipv6? 3ffe:2a00:100:7031::1
        return hostname;                           // literal ipv4? 127.0.0.1, 0x7f.0.0.1
      case Cr.NS_ERROR_ILLEGAL_VALUE:              // ".foo.tld"?
        break;
      default:
        console.log(ex, hostname);                 // ???
        return hostname;
    }
  }

  // NS_ERROR_ILLEGAL_VALUE
  var firstDot = hostname.indexOf(".");
  //  "local.host" ==> >0 OK
  // ".localhost"  ==>  0 exception
  // ".loc.al.ho.st" ==> 0 exception
  //  "localhost" ==> -1 exception
  if (firstDot === -1) {
    console.log("NS_ERROR_ILLEGAL_VALUE firstDot=-1", hostname);
    return hostname; // ???
  }

  // firstDot=0 ("...local.host") (e.g. from cookies)
  // OBS "..local.host" returns "localhost"
  if (firstDot === 0) {
    return getTldFromHost(hostname.substr(1)); // recursive
  }
  return hostname;
}



function toLines(txt) {
  return txt.split(/\r\n|\r|\n/);
}


function SetCookieParser(cookieHeader, isSetCookie) { // TODO remove isSetCookie
  this.m_hasMeta = isSetCookie;
  this._allCookies = [];
  if (cookieHeader !== null) {
    var lines = toLines(cookieHeader);
    var len = lines.length;
    if (isSetCookie) {
      for (var idx = 0; idx < len; idx++) {
        this.parseLineSetCookie(lines[idx]);
      }
    } else {
      for (var idx = 0; idx < len; idx++) {
        this.parseLineCookie(lines[idx]);
      }
    }
  }
}

SetCookieParser.prototype = {
  parseLineSetCookie: function(headerLine) {
    var unit = new CookieUnit(headerLine);
    var items = headerLine.split(";");

    for (var idx = 0, len = items.length; idx < len; idx++) {
      var pair = splitValueName(items[idx]);
      var name = pair[0];
      var value = pair[1]; // null ==> name=HttpOnly, secure etc

      if (idx === 0) {
        if (name.length > 0 && value !== null) {
          unit.defineValue(name, value);
        } else {
          console.trace("_allCookies invalid", name, value, headerLine);
          break;
        }
      } else {
        unit.defineMeta(name, value);
      }
    }

    this._allCookies.push(unit);
  },

  parseLineCookie: function(headerLine) {
    var items = headerLine.split(";");
    for (var idx = 0, len = items.length; idx < len; idx++) {
      var unit = CookieUnit(items[idx]);
      var pair = splitValueName(items[idx]);
      unit.defineValue(pair[0], pair[1]);
      this._allCookies.push(unit);
    }
  },

  /*
  toHeader: function() {
    var allCookies = this._allCookies;
    var len = allCookies.length;
    //var buf = [];
    var buf = new Array(len);
    for (var idx = 0; idx < len; idx++) {
      //if (allCookies[idx].value !== null) {
      buf[idx] = allCookies[idx].toHeaderLine();
      //}
    }
    return this.m_hasMeta ? buf.join("\n") : buf.join(";");
  },

  getCookie: function(name) {
    var aCookie = this._allCookies;
    for (var idx = 0, len = aCookie.length; idx < len; idx++) {
      if (name === aCookie[idx].name) {
        return aCookie[idx];
      }
    }
    return null;
  },

  forEach: function(fn) {
    var c = this._allCookies;
    for (var idx = 0, len = c.length; idx < len; idx++) {
      fn(c[idx]);
    }
  }
  */

  get length() {
    return this._allCookies.length;
  },

  getCookieByIndex: function(idx) {
    return this._allCookies[idx];
  }
};

function CookieUnit(line) {
  this._data = {//instance value
    "_src": line
  };
}

CookieUnit.prototype = {
  _data: null,

  clone: function() {
    var c = new CookieUnit();
    for (var n in this._data) {
      c._data[n] = this._data[n];
    }
    return c;
  },

  get name() {
    var rv = this._data["_name"];
    return rv ? rv : "";
  },

  get value() {
    var rv = this._data["_value"];
    return rv ? rv : "";
  },

  get raw() {
    return this._data["_src"];
  },

  defineValue: function(name, val) {
    this._data["_name"] = name;
    this._data["_value"] = val;
  },

  //"secure":
  //"httponly":
  hasBooleanProperty: function(name) {
    //name = name.toLowerCase();
    return name in this._data;
  },

  setBooleanProperty: function(name, def) {
    if (def) {
      this._data[name] = null;
    } else {
      delete this._data[name];
    }
  },

  //"expires":
  //"domain":
  //"path":
  getStringProperty: function(name) {
    var rv = this._data[name];
    return rv ? rv : "";
    //return rv || "";
  },

  defineMeta: function(name, val) {
    name = name.toLowerCase();
    switch (name) {
      case "expires":
      case "domain":
      case "path":
      case "secure":
      case "httponly":
        this._data[name] = val;
        break;
    }
  },

  toHeaderLine: function() {//toString()
    var buf = [this.name + "=" + this.value];
    var props;

    props = ["secure", "httponly"];
    for (var idx = 0, len = props.length; idx < len; idx++) {
      var propName = props[idx];
      if (this.hasBooleanProperty(propName)) {
        buf.push(propName.toUpperCase());
      }
    }

    props = ["expires", "path", "domain"];
    for (var idx = 0, len = props.length; idx < len; idx++) {
      var propName = props[idx];
      var val = this.getStringProperty(propName);
      if (val.length > 0) {
        buf.push(propName.toUpperCase() + "=" + val);
      }
    }

    return buf.join(";");
  }
};


function splitValueName(cookie) {
  var idx = cookie.indexOf("=");
  if (idx === -1) {
    return [cookie.trim(), null];
  }

  // "a=bcd=e".split("=",2) returns [a,bcd]
  //   "abcde".split("=",2) returns [abcde]


  // MY =
  // 012^-----idx=3 length=4

  // MY =a:1=6
  // 012^-----idx=3 length=9

  var pair = ["", ""];
  pair[0] = cookie.substring(0, idx).trim();
  idx++;
  if (idx < cookie.length) {
    pair[1] = cookie.substring(idx);
  }

  return pair;
}
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


function windowLocalStorage(obj, contentDoc) {
  var profileId = FindIdentity.fromContent(contentDoc.defaultView).profileNumber;

  if (Profile.isNativeProfile(profileId)) {
    console.trace("windowLocalStorage", profileId);
    return;
  }


  var originalUri = stringToUri(contentDoc.location.href);
  var uri = toInternalUri(originalUri, profileId);
  var principal = Services.scriptSecurityManager.getNoAppCodebasePrincipal(uri);
  var storage = Services.domStorageManager.createStorage(principal, ""); // nsIDOMStorage

  var rv = undefined;
  var oldVal;
  var eventData = null;
  switch (obj.cmd) {
    case "clear":
      if (storage.length > 0) {
        eventData = ["", null, null];
      }
      storage.clear();
      break;
    case "removeItem":
      oldVal = storage.getItem(obj.key);
      if (oldVal !== null) {
        eventData = [obj.key, oldVal, null];
      }
      storage.removeItem(obj.key);
      break;
    case "setItem":
      oldVal = storage.getItem(obj.key);
      if (oldVal !== obj.val) {
        eventData = [obj.key, oldVal, obj.val];
      }
      storage.setItem(obj.key, obj.val); // BUG it's ignoring https
      break;
    case "getItem":
      rv = storage.getItem(obj.key);
      break;
    case "key":
      rv = storage.key(obj.index);
      break;
    case "length":
      rv = storage.length;
      break;
    default:
      throw new Error("localStorage interface unknown: " + obj.cmd);
  }

  if (eventData !== null) {
    dispatchStorageEvent(eventData, profileId, contentDoc.defaultView);
  }

  return rv;
}



function dispatchStorageEvent(data, profileId, srcWin) {

  function forEachWindow(fn, win) {
    fn(win);
    for (var idx = win.length - 1; idx > -1; idx--) {
      forEachWindow(fn, win[idx]);
    }
  }

  function dispatchStorage(win) {
    if ((win.location.origin === _origin) && (srcWin !== win)) {
      if (_evt === null) {
        _evt = srcWin.document.createEvent("StorageEvent");
        _evt.initStorageEvent("storage", false, false, data[0], data[1], data[2], srcWin.location.href, null);
      }
      win.dispatchEvent(_evt);
    }
  }

  var _origin = srcWin.location.origin;
  var _evt = null;

  var enumWin = Services.wm.getEnumerator("navigator:browser");
  while (enumWin.hasMoreElements()) {
    var win = enumWin.getNext();
    if (Profile.getIdentity(win) === profileId) {
      var tabList = win.getBrowser().tabs; // <tab> NodeList
      for (var idx = tabList.length - 1; idx > -1; idx--) {
        forEachWindow(dispatchStorage, tabList[idx].linkedBrowser.contentWindow);
      }
    }
  }

}


function stringToUri(spec) {
  try {
    return Services.io.newURI(spec, null, null);
  } catch (ex) {
    return null;
  }
}
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


var ErrorHandler = {
  _incompatibility: false,

  // incompatible extension installed
  addIncompatibilityError: function() {
    // this=undefined
    ErrorHandler._incompatibility = true;
    var enumWin = Services.wm.getEnumerator("navigator:browser");
    while (enumWin.hasMoreElements()) {
      ErrorHandler.updateButtonAsync(enumWin.getNext().getBrowser().selectedBrowser);
    }
  },


  // request/response
  addNetworkError: function(contentWin, errorCode) {
    var browser = ContentWindow.getContainerElement(contentWin);
    browser.setAttribute("multifox-tab-error-net", errorCode);
    this.updateButtonAsync(browser);
  },


  // evalInSandbox/runtime
  addScriptError: function(contentWin, errorCode, details) {
    var msg = [];
    msg.push("ERROR=" + errorCode);
    msg.push(details);
    if (contentWin.document) {
      msg.push("location=" + contentWin.location);
      if (contentWin.document.documentURIObject) {
        msg.push("uri=     " + contentWin.document.documentURIObject.spec);
      }
    }
    msg.push("title=[" + contentWin.document.title + "]");
    console.log(msg.join("\n"));

    var browser = ContentWindow.getContainerElement(contentWin);
    browser.setAttribute("multifox-tab-error-script", errorCode);
    this.updateButtonAsync(browser);
  },


  onNewWindow: function(win, browser) {
    // reset js error
    if ((win === win.top) && (browser !== null)) {
      if (browser.hasAttribute("multifox-tab-error-script")) {
        browser.removeAttribute("multifox-tab-error-script");
        this.updateButtonAsync(browser);
      }
    }
  },


  onNewWindowRequest: function(browser) {
    // reset network error - BUG new win is chrome:// (no http-on-modify-request)
    if (browser.hasAttribute("multifox-tab-error-net")) {
      browser.removeAttribute("multifox-tab-error-net");
      this.updateButtonAsync(browser);
    }
  },


  getCurrentError: function(doc) {
    var button = getButtonElem(doc);
    return button !== null ? button.getAttribute("tab-status") : "";
  },


  updateButtonAsync: function(browser) {
    browser.ownerDocument.defaultView.requestAnimationFrame(function() {
      ErrorHandler._updateButtonStatus(browser);
    });
  },


  _updateButtonStatus: function(browser) {
    if (browser.getTabBrowser().selectedBrowser !== browser) {
      return;
    }

    var button = getButtonElem(browser.ownerDocument);
    if (button === null) { // view-source?
      return;
    }

    if (this._incompatibility) {
      this._update("incompatible-extension", button);
      return;
    }

    if (browser.hasAttribute("multifox-tab-error-net")) {
      this._update(browser.getAttribute("multifox-tab-error-net"), button);
      return;
    }

    this._update(this._getJsError(browser), button);
  },


  _update: function(newStat, button) {
    var isError = newStat.length > 0;
    var showingError = button.getAttribute("tab-status").length > 0;
    if (isError === showingError) {
      return;
    }
    if (isError) {
      button.setAttribute("image", "chrome://global/skin/icons/error-16.png");
      button.setAttribute("tab-status", newStat);
    } else {
      button.setAttribute("image", "chrome://multifox-4410e96638/content/favicon.ico");
      button.setAttribute("tab-status", "");
    }
  },


  _getJsError: function(browser) {
    if (browser.hasAttribute("multifox-tab-error-script")) {
      return browser.getAttribute("multifox-tab-error-script");
    }
    // multifox-tab-error refers to a different document.
    // it seems the current is fine.
    return "";
  }

};



var ExtCompat = {

  // Extensions with known compatibility issues.
  // To update it please file a bug: https://github.com/hultmann/multifox/issues
  _incompatIds: [
    "{37fa1426-b82d-11db-8314-0800200c9a66}", // X-notifier
    "{42f25d10-4944-11e2-96c0-0b6a95a8daf0}"  // former Multifox 2
  ],


  findIncompatibleExtensions: function(onFound) {
    var jsm = {};
    Components.utils.import("resource://gre/modules/AddonManager.jsm", jsm);
    jsm.AddonManager.getAddonsByIDs(this._incompatIds, function(arr) {
      var enabled = [];
      for (var idx = arr.length - 1; idx > -1; idx--) {
        var ext = arr[idx];
        if ((ext !== null) && ext.isActive) {
          enabled.push(ext.name);
        }
      }
      if (enabled.length > 0) {
        onFound(enabled);
      }
    });
  },


  _addonListener: {
    onEnabled: function(addon) {
      if (ExtCompat._incompatIds.indexOf(addon.id) > -1) {
        ErrorHandler.addIncompatibilityError();
      }
    },

    onDisabled: function(addon) {
      if (ExtCompat._incompatIds.indexOf(addon.id) > -1) {
        ExtCompat.findIncompatibleExtensions(ErrorHandler.addIncompatibilityError);
      }
    }
  },


  installAddonListener: function() {
    var jsm = {};
    Components.utils.import("resource://gre/modules/AddonManager.jsm", jsm);
    jsm.AddonManager.addAddonListener(this._addonListener);
  },


  uninstallAddonListener: function() {
    var jsm = {};
    Components.utils.import("resource://gre/modules/AddonManager.jsm", jsm);
    jsm.AddonManager.removeAddonListener(this._addonListener);
  }

};
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

var console = {
  _prefix: "multifox ",


  assert: function console_assert(test) {
    if (test === true) {
      return;
    }
    var msg = this._format(test === false
                           ? Array.prototype.slice.call(arguments, 1)
                           : arguments);
    this._print("ASSERT ", msg + "\n" + this._stackToString(Components.stack));
    var ex =  new Error("[console.assert] " + msg);
    Cu.reportError(ex); // workaround - sometimes an exception doesn't show up in console
    throw ex;
  },


  error: function console_error(ex) {
    Cu.reportError("console.error:");
    Cu.reportError(ex);
    this._print("ERROR ", this._format(arguments)); // ex includes stacktrace
  },


  warn: function console_warn(msg) {
    var message = Cc["@mozilla.org/scripterror;1"].createInstance(Ci.nsIScriptError);
    message.init(msg,
                 null, // sourceName
                 null, // sourceLine
                 0, 0, // line, col
                 Ci.nsIScriptError.warningFlag,
                 "component javascript");
    Services.console.logMessage(message);
    this._print("warn ", this._format(arguments));
  },


  trace: function console_trace() {
    this._print("trace ", this._format(arguments) + "\n" +
                this._stackToString(Components.stack));
  },


  log: function console_log() {
    this._print("", this._format(arguments));
  },


  _print: function(name, content) {
    dump("\n-- " + this._prefix + name + this._now() +
         " -------------------------------------------------------\n" +
         content + "\n");
  },


  _format: function(args) {
    var len = args.length;
    var output = new Array(len);
    for (var idx = 0; idx < len; idx++) {
      var arg = args[idx];
      switch (typeof arg) {
        case "string":
          output[idx] = arg.length > 0 ? arg : "<empty>";
          break;
        case "number":
        case "boolean":
          output[idx] = arg.toString();
          break;
        case "object":
          output[idx] = this._formatObj(arg);
          break;
        case "undefined":
          output[idx] = "[undefined]";
          break;
        case "function":
          output[idx] = "[" + arg.toString() + "]";
          break;
        default:
          output[idx] = "[???" + arg + " (" + (typeof arg) + ")]";
          break;
      }
    }
    return output.join(" ");
  },


  _now: function() {
    var now = new Date();
    var ms = now.getMilliseconds();
    var ms2;
    if (ms < 100) {
      ms2 = ms < 10 ? "00" + ms : "0" + ms;
    } else {
      ms2 = ms.toString();
    }
    return now.toLocaleFormat("%H:%M:%S") + "." + ms2;
  },


  _formatObj: function(obj) {
    if (obj instanceof Error) {
      return "[Error: " + obj.toSource() + "]";

    } else if (obj instanceof Ci.nsIException) {
      return "[nsIException: " + obj.toString() + obj.location + "]\n" + this._stackToString(obj.location);

    } else if (obj instanceof Ci.nsIURI) {
      return "[nsIURI: " + obj.spec + "]";

    } else if (obj instanceof Ci.nsISimpleEnumerator) {
      var name = "";
      var qty = 0;
      while (obj.hasMoreElements()) {
        name += obj.getNext();
        qty++;
      }
      return "[" + obj.toString() + " " + qty + " " + name + "]";

    } else if (obj instanceof Ci.nsIDOMWindow) {
      return "[" + obj.toString() + " " + obj.location.href + "]";

    } else if (obj instanceof Ci.nsIDOMDocument) {
      return "[" + obj.toString() + " " + obj.defaultView.location.href + "]";

    } else if (obj instanceof Ci.nsIDOMNode) {
      return "[" + obj.toString() + " " +  obj.nodeName + "]";

    } else if (obj instanceof Ci.nsIDOMEvent) {
      return "[" + obj.toString() +
        "\ntype: " + obj.type +
        "\neventPhase: " + ["capture", "target", "bubble"][obj.eventPhase - 1] +
        "\ncurrentTarget:  " + this._formatObj(obj.currentTarget) +
        "\ntarget:         " + this._formatObj(obj.target) +
        "\noriginalTarget: " + this._formatObj(obj.originalTarget) + "]";

    } else if (obj instanceof Ci.nsISupports) {
      if (obj instanceof Object) {
        return "[nsISupports " + obj.toString() + " " + obj.toSource();
      } else {
        return "[nsISupports: " + obj + "]";
      }

    } else {
      try {
        return JSON.stringify(obj, null, 2);
      } catch (ex) {
        if (obj instanceof Object) {
          return obj.toString() + ex;
        } else {
          return "proto=null? " + ex;
        }
      }
    }
  },


  _stackToString: function(stack) {
    var b = [];
    for (var s = stack; s; s = s.caller) {
      b.push(s);
    }

    var padding = "";
    var t = new Array(b.length);
    for (var idx = b.length - 1; idx > -1; idx--) {
      var s = b[idx];
      var name = s.name === null ? "<anonymous>" : s.name;
      var lang = s.languageName === "JavaScript" ? "JS " : s.languageName;
      t[idx] = lang + " " + padding + s.filename + "\t\t" + name + "\t" + s.lineNumber;
      padding += " ";
    }
    return t.reverse().join("\n");
  }

};


const NewWindow = {
  newId: function(win) {
    var id;
    if (this._shouldBeDefault(win)) {
      id = Profile.DefaultIdentity;
    } else {
      id = Profile.lowerAvailableId(win);
    }
    console.log("newIdentity " + id);
    Profile.defineIdentity(win, id);
  },

  inheritId: function(newWin) {
    console.log("inheritId");
    var id;
    if (this._shouldBeDefault(newWin)) {
      id = Profile.DefaultIdentity;
    } else {
      //window.open()/fxstarting ==> opener=null
      var prevWin = newWin.opener;
      if (prevWin) {
        id = Profile.getIdentity(prevWin);
      } else {
        console.log("inheritId prevWin=" + prevWin);
        id = Profile.UndefinedIdentity;
      }
    }
    Profile.defineIdentity(newWin, id);
    console.log("/inheritId " + id);
  },

  applyRestore: function(win) {
    // restore: window is first configured by NewWindow.inheritId
    console.log("applyRestore");

    var stringId = Cc["@mozilla.org/browser/sessionstore;1"]
                    .getService(Ci.nsISessionStore)
                    .getWindowValue(win, "multifox-dom-identity-id");
    Profile.defineIdentity(win, Profile.toInt(stringId));
  },

  _shouldBeDefault: function(win) {
    // popup opened by an extension (like GTB)
    //var chromeHidden = win.document.documentElement.getAttribute("chromehidden");
    //return chromeHidden.indexOf("location") > -1;
    return false;
  }

};


const Profile = {
  UndefinedIdentity:-1,
  PrivateIdentity:   0,
  DefaultIdentity:   1,
  MaxIdentity:       999999999999999,

  defineIdentity: function(win, id) {
    console.assert(typeof id === "number", "id is not a number.", typeof id);

    if (PrivateBrowsingUtils.isWindowPrivate(win)) {
      id = Profile.PrivateIdentity;
    }

    console.log("defineIdentity " + id);
    if (id > Profile.MaxIdentity) {
      console.log("id > max " + id);
      id = Profile.MaxIdentity;
    }
    if (id < Profile.UndefinedIdentity) {
      console.log("id < UndefinedIdentity " + id);
      id = Profile.UndefinedIdentity;
    }
    var current = Profile.getIdentity(win);
    if (current === id) {
      console.log("defineIdentity NOP");
      return id;
    }
    if (this.isExtensionProfile(current)) {
      BrowserWindow.unregister(win);
    }

    this._save(win, id);
    BrowserWindow.register(win);

    return id;
  },


  isNativeProfile: function(id) { // including UndefinedIdentity
    return this.isExtensionProfile(id) === false;
  },


  isExtensionProfile: function(id) {
    return id > Profile.DefaultIdentity;
  },


  getIdentity: function(chromeWin) {
    var tabbrowser = chromeWin.getBrowser();
    if (tabbrowser === null) {
      console.log("getIdentity=DefaultIdentity, tabbrowser=null");
      return Profile.DefaultIdentity;
    }

    if (tabbrowser.hasAttribute("multifox-dom-identity-id")) {
      var profileId = this.toInt(tabbrowser.getAttribute("multifox-dom-identity-id"));
      return profileId;
    } else {
      return PrivateBrowsingUtils.isWindowPrivate(chromeWin) ? Profile.PrivateIdentity
                                                             : Profile.DefaultIdentity;
    }
  },

  _save: function(win, id) {
    console.log("save " + id);
    var node = win.getBrowser();
    if (id !== Profile.DefaultIdentity) {
      node.setAttribute("multifox-dom-identity-id", id); // UndefinedIdentity or profile
    } else {
      node.removeAttribute("multifox-dom-identity-id");
    }
    new SaveToSessionStore(win.document);

    win.requestAnimationFrame(function() {
      var ns = {}; // BUG util is undefined???
      Cu.import("resource://multifox-4410e96638/new-window.js", ns);
      var winId = ns.util.getOuterId(win).toString();
      Services.obs.notifyObservers(null, "multifox-dom-id-changed", winId);
    });
  },

  activeIdentities: function(ignoreWin) {
    var winEnum = Services.wm.getEnumerator("navigator:browser");
    var arr = [];
    while (winEnum.hasMoreElements()) {
      var win = winEnum.getNext();
      if (ignoreWin !== win) {
        var id = Profile.getIdentity(win);
        if (arr.indexOf(id) === -1) {
          arr.push(id);
        }
      }
    }
    return arr;
  },

  lowerAvailableId: function(ignoreWin) {
    var arr = this.activeIdentities(ignoreWin); //ignore win -- it doesn't have a session id yet
    var id = Profile.DefaultIdentity;
    while (arr.indexOf(id) > -1) {
      id++;
    }
    return id; // id is available
  },

  toInt: function(str) {
    var rv = parseInt(str, 10);
    return Number.isNaN(rv) ? Profile.DefaultIdentity : rv;
  },

  toString: function(id) {
    switch (id) {
      //case Profile.UndefinedIdentity:
      //  return "\u221e"; // ∞
      default:
        return id.toString();
    }
  }

};



function SaveToSessionStore(doc) {
  this._doc = Cu.getWeakReference(doc);
  this._timer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
  this._timer.init(this, 1300, Ci.nsITimer.TYPE_ONE_SHOT);
}

SaveToSessionStore.prototype = {
  QueryInterface: XPCOMUtils.generateQI([Ci.nsIObserver, Ci.nsISupportsWeakReference]),

  observe: function(aSubject, aTopic, aData) {
    var doc = this._doc.get();
    if ((doc === null) || (doc.defaultView === null)) {
      return;
    }

    // BUG extension disabled => Components is not available
    // Profile.DefaultIdentity won't be saved

    var ss = Cc["@mozilla.org/browser/sessionstore;1"].getService(Ci.nsISessionStore);
    var val = Profile.getIdentity(doc.defaultView);

    try {
      // overwrite any previous value if called twice
      ss.setWindowValue(doc.defaultView, "multifox-dom-identity-id", val);
    } catch (ex) {
      // keep trying
      console.trace("SaveToSessionStore FAIL", val, doc, doc.defaultView, doc.defaultView.state, ex);
      this._timer.init(this, 700, Ci.nsITimer.TYPE_ONE_SHOT);
      return;
    }

    if (Profile.isNativeProfile(val)) {
      ss.deleteWindowValue(doc.defaultView, "multifox-dom-identity-id");
    }
  }

};


const FindIdentity = {

  fromContent: function(contentWin) {
    if (contentWin === null) {
      return { profileNumber:  Profile.UndefinedIdentity,
               browserElement: null };
    }

    var profileId;
    var browser = ContentWindow.getContainerElement(contentWin);
    if (browser === null) {
      // source-view? -- BUG chat browser?
      profileId = this._getIdentityFromOpenerChrome(contentWin);
      return { profileNumber:  profileId,
               browserElement: null };
    }

    var chromeWin = browser.ownerDocument.defaultView;
    profileId = Profile.getIdentity(chromeWin);
    if (profileId !== Profile.UndefinedIdentity) {
      return { profileNumber:  profileId,
               browserElement: browser };
    }

    // popup via js/window.open
    profileId = this._getIdentityFromOpenerContent(contentWin, chromeWin);
    return { profileNumber:  profileId,
             browserElement: browser };
  },

  _getIdentityFromOpenerChrome: function(contentWin) {
    var chromeWin = ContentWindow.getTopLevelWindow(contentWin);
    if (chromeWin === null) {
      return Profile.UndefinedIdentity;
    }
    var tabbrowser = null;
    var type = chromeWin.document.documentElement.getAttribute("windowtype");
    if (type === "navigator:view-source") {
      var winOpener = chromeWin.opener;
      if (winOpener) {
        var type2 = winOpener.document.documentElement.getAttribute("windowtype");
        if (type2 === "navigator:browser") {
          tabbrowser = winOpener.getBrowser();
        }
      }
    }

    return tabbrowser !== null ? Profile.getIdentity(tabbrowser.ownerDocument.defaultView)
                               : Profile.UndefinedIdentity; // favicon, ...
  },

  _getIdentityFromOpenerContent: function(contentWin, chromeWin) {
    if (contentWin.opener) {
      var browserOpener = ContentWindow.getContainerElement(contentWin.opener);
      if (browserOpener) {
        var chromeOpener = browserOpener.ownerDocument.defaultView;
        var profileId = Profile.getIdentity(chromeOpener);
        if (profileId > Profile.UndefinedIdentity) {
          return Profile.defineIdentity(chromeWin, profileId);
        }
      }
    }

    return Profile.UndefinedIdentity;
  }
};


const ContentWindow = {
  getContainerElement: function(contentWin) {
    var browser = this.getParentBrowser(contentWin);
    if (browser === null) {
      return null;
    }
    // browser.xul has browser elements all over the place
    var t = browser.getAttribute("type");
    return ((t === "content-targetable") || (t === "content-primary"))
           ? browser : null;
  },


  getParentBrowser: function(win) {
    var browser = win.QueryInterface(Ci.nsIInterfaceRequestor)
                     .getInterface(Ci.nsIWebNavigation)
                     .QueryInterface(Ci.nsIDocShell)
                     .chromeEventHandler;
    if (browser === null) {
      return null;
    }
    if (browser.tagName === "xul:browser") {
      return browser;
    }
    if (browser.tagName === "browser") {
      return browser;
    }
    // e.g. <iframe> chrome://browser/content/devtools/cssruleview.xhtml
    console.log("not a browser element", browser.tagName, win, win.parent);
    return null;
  },


  getTopLevelWindow: function(win) { // content or chrome windows
    if ((!win) || (!win.QueryInterface)) {
      console.trace("getTopLevelWindow win=" + win);
      return null;
    }

    var topwin = win.QueryInterface(Ci.nsIInterfaceRequestor)
                    .getInterface(Ci.nsIWebNavigation)
                    .QueryInterface(Ci.nsIDocShellTreeItem)
                    .rootTreeItem
                    .QueryInterface(Ci.nsIInterfaceRequestor)
                    .getInterface(Ci.nsIDOMWindow);

    console.assert(topwin !== null, "getTopLevelWindow null", win);
    console.assert(topwin !== undefined, "getTopLevelWindow undefined", win);
    console.assert(topwin === topwin.top, "getTopLevelWindow should return a top window");
    // unwrapped object allows access to gBrowser etc
    return XPCNativeWrapper.unwrap(topwin);
  }

};
