/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */


"use strict";

var EXPORTED_SYMBOLS = ["Cc", "Ci", "util", "console", "Bootstrap", "newPendingWindow", "onXulCommand"];

const Cc = Components.classes;
const Ci = Components.interfaces;

Components.utils.import("resource://gre/modules/XPCOMUtils.jsm");
Components.utils.import("resource://gre/modules/Services.jsm");
Components.utils.import("resource://multifox-4410e96638/main.js");


var m_pendingNewWindows = [];


function newPendingWindow(profileId) {
  if (profileId === undefined) {
    profileId = Profile.UndefinedIdentity;
  }
  m_pendingNewWindows.push(profileId);
}

var m_docObserver = null;

var Bootstrap = {

  extensionStartup: function(firstRun, reinstall) {
    console.assert(m_docObserver === null, "m_docObserver should be null");
    console.assert(m_pendingNewWindows.length === 0, "m_pendingNewWindows should be empty");

    var prefs = Services.prefs.getBranch("extensions.multifox@hultmann.");
    if (prefs.prefHasUserValue("button-added") === false) {
      prefs.setBoolPref("button-added", true);
      this._showButtonByDefault = true;
    }

    if (firstRun || reinstall) {
      var desc = util.getTextFrom("extensions.multifox@hultmann.description", "about-multifox");
      util.setUnicodePref("description", desc);
    }

    m_docObserver = new DocObserver();

    Services.obs.addObserver(UpdateUI, "multifox-dom-id-changed", false);

    var enumWin = Services.wm.getEnumerator(null);
    while (enumWin.hasMoreElements()) {
      forEachChromeWindow(addOverlay, enumWin.getNext());
    }

    enumWin = Services.wm.getEnumerator("navigator:browser");
    while (enumWin.hasMoreElements()) {
      BrowserOverlay.add(enumWin.getNext());
    }

    this._incompatibilityCheck();
  },

  get showButtonByDefault() {
    return this._showButtonByDefault;
  },

  resetButton: function() {
    var prefs = Services.prefs.getBranch("extensions.multifox@hultmann.");
    if (prefs.prefHasUserValue("button-added") === false) {
      prefs.clearUserPref("button-added");
    }
    this._showButtonByDefault = true;
  },

  _timer: null,
  _showButtonByDefault: false,

  _incompatibilityCheck: function() {
    this._timer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
    this._timer.initWithCallback({
      notify: function() {
        delete Bootstrap._timer;
        ExtCompat.findIncompatibleExtensions(ErrorHandler.addIncompatibilityError);
        ExtCompat.installAddonListener();
      }
    }, 8000, Ci.nsITimer.TYPE_ONE_SHOT);
  },

  extensionShutdown: function() {
    Services.obs.removeObserver(UpdateUI, "multifox-dom-id-changed");
    m_docObserver.shutdown();
    ExtCompat.uninstallAddonListener();
    var enumWin = Services.wm.getEnumerator(null);
    while (enumWin.hasMoreElements()) {
      forEachChromeWindow(disableExtension, enumWin.getNext());
    }
  },


  extensionUninstall: function() {
    var enumWin = Services.wm.getEnumerator("navigator:browser");
    while (enumWin.hasMoreElements()) {
      removeState(enumWin.getNext());
    }

    // prefs
    Services.prefs.getBranch("extensions.multifox@hultmann.").deleteBranch("");

    var ns = {};
    Components.utils.import("resource://multifox-4410e96638/actions.js", ns);
    ns.removeData();
  }

};


function forEachChromeWindow(fn, win) {
  if (win instanceof Ci.nsIDOMChromeWindow) {
    fn(win);
    for (var idx = win.length - 1; idx > -1; idx--) {
      forEachChromeWindow(fn, win[idx]);
    }
  }
}


var UpdateUI = {
  observe: function(subject, topic, data) {
    var win = Services.wm.getOuterWindowWithId(parseInt(data, 10));
    updateButton(win);
  }
}


function DocObserver() {
  Services.obs.addObserver(this, "chrome-document-global-created", false);
}


DocObserver.prototype = {
  shutdown: function() {
    Services.obs.removeObserver(this, "chrome-document-global-created");
  },

  observe: function(win, topic, data) {
    // win.location=about:blank
    win.addEventListener("DOMContentLoaded", onDOMContentLoaded, false);
  }
};


function onDOMContentLoaded(evt) {
  var win = evt.currentTarget;
  if (win.document === evt.target) {
    // avoid bubbled DOMContentLoaded events
    win.removeEventListener("DOMContentLoaded", onDOMContentLoaded, false);
    addOverlay(win);
  }
}


// DOMContentLoaded is too early for #navigator-toolbox.palette
// load might be a bit late for network listeners
function onBrowserWinLoad(evt) {
  var win = evt.currentTarget;
  win.removeEventListener("load", onBrowserWinLoad, false);
  win.requestAnimationFrame(function() {
    BrowserOverlay.add(win);
  });
}


function addOverlay(win) {
  switch (win.location.href) {
    case "chrome://browser/content/browser.xul":
      setWindowProfile(win); // enable netwok listeners
      win.addEventListener("load", onBrowserWinLoad, false);
      break;
    case "chrome://browser/content/history/history-panel.xul":
    case "chrome://browser/content/bookmarks/bookmarksPanel.xul":
      PlacesOverlay.add(win);
      break;
    case "chrome://browser/content/places/places.xul":
      // BUG removed to avoid bugs with private window
      //PlacesOverlay.add(win);
      break;
    case "chrome://mozapps/content/extensions/about.xul":
      AboutOverlay.add(win);
      break;
  }
}


function disableExtension(win) {
  switch (win.location.href) {
    case "chrome://browser/content/browser.xul":
      var node = win.getBrowser();
      if (node.hasAttribute("multifox-dom-identity-id")) {
        var id = node.getAttribute("multifox-dom-identity-id");
        node.setAttribute("multifox-dom-identity-id-tmp", id);
      } else {
        node.removeAttribute("multifox-dom-identity-id-tmp");
      }

      Profile.defineIdentity(win, Profile.DefaultIdentity);
      BrowserOverlay.remove(win);
      break;

    case "chrome://browser/content/history/history-panel.xul":
    case "chrome://browser/content/bookmarks/bookmarksPanel.xul":
      PlacesOverlay.remove(win);
      break;

    case "chrome://browser/content/places/places.xul":
      break;
    case "chrome://mozapps/content/extensions/about.xul":
      break;
  }
}


function removeState(win) { // uninstalling
  console.assert(win.location.href === "chrome://browser/content/browser.xul", "win should be a browser window");

  var node = win.getBrowser();
  node.removeAttribute("multifox-dom-identity-id");
  node.removeAttribute("multifox-dom-identity-id-tmp");

  var ss = Cc["@mozilla.org/browser/sessionstore;1"].getService(Ci.nsISessionStore);
  ss.deleteWindowValue(win, "multifox-dom-identity-id");
}


const BrowserOverlay = {
  add: function(win) {
    console.assert(win.location.href === "chrome://browser/content/browser.xul",
                   "win should be a browser window", win.location.href);

    win.addEventListener("unload", BrowserOverlay._unload, false);


    var doc = win.document;
    //if ((doc instanceof Ci.nsIDOMDocument) === false) {

    // detect session restore
    doc.addEventListener("SSTabRestoring", onTabRestoring, false);
    doc.addEventListener("SSTabRestored", onTabRestored, false);

    // commands
    appendXulCommands(doc);

    // key
    var key = doc.getElementById("mainKeyset").appendChild(doc.createElement("key"));
    key.setAttribute("id", "key_multifox-dom-new-identity");
    key.setAttribute("modifiers", Services.appinfo.OS === "Darwin" ? "control,alt" : "accel,alt");
    key.setAttribute("key", "M");
    key.setAttribute("command", "multifox:cmd_new_profile");

    // menus
    addMenuListeners(doc);

    // insert into toolbar
    insertButton(doc);
    updateButton(doc.defaultView);
  },


  _unload: function(evt) {
    // do not set window to DefaultIdentity, it can be restored
    var win = evt.currentTarget;
    BrowserOverlay.remove(win);
  },

  remove: function(win) {
    win.removeEventListener("unload", BrowserOverlay._unload, false);

    var doc = win.document;
    removeMenuListeners(doc);

    // key
    var key = doc.getElementById("key_multifox-dom-new-identity");
    key.parentNode.removeChild(key);

    // commands
    removeXulCommands(doc);

    destroyButton(doc);
  }
};


function appendXulCommands(doc) {
  var commands = [
    "multifox:cmd_show_error",
    "multifox:cmd_new_profile",
    "multifox:cmd_rename_profile_prompt",
    "multifox:cmd_delete_profile_prompt",
    "multifox:cmd_delete_profile",
    "multifox:cmd_select_window",
    "multifox:cmd_set_profile_window"
  ];

  var js = "var jsm={};Cu.import('resource://multifox-4410e96638/new-window.js',jsm);jsm.onXulCommand(event)";
  var cmdset = doc.documentElement.appendChild(doc.createElement("commandset"));

  for (var idx = commands.length - 1; idx > -1; idx--) {
    var cmd = cmdset.appendChild(doc.createElement("command"));
    cmd.setAttribute("id", commands[idx]);
    cmd.setAttribute("oncommand", js);
  }
}


function removeXulCommands(doc) {
  var cmdset = doc.getElementById("multifox:cmd_new_profile").parentNode;
  cmdset.parentNode.removeChild(cmdset);
}


function onXulCommand(evt) {
  var ns = {};
  Components.utils.import("resource://multifox-4410e96638/actions.js", ns);
  ns.xulCommand(evt);
  Components.utils.unload("resource://multifox-4410e96638/actions.js");
}


var PlacesOverlay = {
  add: function(win) {
    var popup = win.document.getElementById("placesContext");
    popup.addEventListener("popupshowing", PlacesOverlay._listener, false);
  },

  remove: function(win) {
    var popup = win.document.getElementById("placesContext");
    popup.removeEventListener("popupshowing", PlacesOverlay._listener, false);
  },

  _listener: function(evt) {
    var ns = {};
    Components.utils.import("resource://multifox-4410e96638/menus.js", ns);
    ns.menuShowing(evt);
  }
};


var AboutOverlay = {
  add: function(win) {
    if (win.arguments[0].id !== "multifox@hultmann") {
      return;
    }

    var browserWin = Services.wm.getMostRecentWindow("navigator:browser");
    if (browserWin === null) {
      return;
    }

    var uri = Services.io.newURI("about:multifox", null, null);
    var where = Ci.nsIBrowserDOMWindow.OPEN_NEWTAB;
    browserWin.browserDOMWindow.openURI(uri, null, where, null);

    win.close();

    // hide window to avoid flickering
    var root = win.document.documentElement;
    root.setAttribute("hidechrome", "true");
    root.setAttribute("hidden", "true");
  }
};


function setWindowProfile(newWin) {
  var node = newWin.getBrowser();
  var nameTmp = "multifox-dom-identity-id-tmp";

  if (node.hasAttribute(nameTmp)) {
    // updating/enabling the extension
    var tmp = node.getAttribute(nameTmp);
    node.removeAttribute(nameTmp);
    var savedId = Profile.toInt(tmp);
    console.assert(Profile.isExtensionProfile(savedId), "not a profile id", savedId);
    Profile.defineIdentity(newWin, savedId);

  } else if (m_pendingNewWindows.length > 0) {
    var profileId = m_pendingNewWindows.pop();
    if (profileId !== Profile.UndefinedIdentity) {
      Profile.defineIdentity(newWin, profileId);
    } else {
      // new identity profile
      NewWindow.newId(newWin);
    }

  } else {
    // inherit identity profile
    if (util.networkListeners.active) {
      NewWindow.inheritId(newWin);
    } else {
      // no Multifox window
      console.log("setWindowProfile NOP => util.networkListeners.active=false");
    }
  }
}


function onTabRestoring(evt) {
  var doc = evt.currentTarget;
  var win = doc.defaultView;

  var stringId = Cc["@mozilla.org/browser/sessionstore;1"]
                  .getService(Ci.nsISessionStore)
                  .getWindowValue(win, "multifox-dom-identity-id");

  if (util.networkListeners.active === false && stringId.length === 0) {
    // default scenario
    console.log("first tab restoring NOP");
    return;
  }

  console.log("first tab restoring", stringId);

  // add icon; sync id — override any previous profile id
  NewWindow.applyRestore(win);
}


function onTabRestored(evt) {
  var doc = evt.currentTarget;
  var win = doc.defaultView;
  var tab = evt.originalTarget;

  // we need to [re]configure identity window id,
  // only the first restored tab is necessary.
  if (tab.linkedBrowser.currentURI.spec !== "about:sessionrestore") {
    console.log("removeEventListener SSTabRestored+SSTabRestoring");
    doc.removeEventListener("SSTabRestoring", onTabRestoring, false);
    doc.removeEventListener("SSTabRestored", onTabRestored, false);
  }
}


function addMenuListeners(doc) {
  var ids = ["contentAreaContextMenu", "placesContext", "menu_FilePopup"];
  for (var idx = ids.length - 1; idx > -1; idx--) {
    doc.getElementById(ids[idx]).addEventListener("popupshowing", onMenuPopupShowing, false);
  }

  doc.getElementById("tabContextMenu").addEventListener("popupshowing", onMenuPopupShowing, false);

  var windowMenu = doc.getElementById("appmenu_newNavigator");
  if (windowMenu === null) {
    return;
  }
  var newTabPopup = windowMenu.parentNode;
  newTabPopup.addEventListener("popupshowing", onMenuPopupShowing, false);
  newTabPopup.setAttribute("multifox-id", "app-menu");
}


function removeMenuListeners(doc) {
  var ids = ["contentAreaContextMenu", "placesContext", "menu_FilePopup"];
  for (var idx = ids.length - 1; idx > -1; idx--) {
    doc.getElementById(ids[idx]).removeEventListener("popupshowing", onMenuPopupShowing, false);
  }

  doc.getElementById("tabContextMenu").removeEventListener("popupshowing", onMenuPopupShowing, false);

  var windowMenu = doc.getElementById("appmenu_newNavigator");
  if (windowMenu === null) {
    return;
  }
  var newTabPopup = windowMenu.parentNode;
  newTabPopup.removeEventListener("popupshowing", onMenuPopupShowing, false);
  newTabPopup.removeAttribute("multifox-id");
}


function onMenuPopupShowing(evt) {
  var ns = {};
  Components.utils.import("resource://multifox-4410e96638/menus.js", ns);
  ns.menuShowing(evt);
}


const util = {
  setUnicodePref: function(name, val) {
    var CiS = Ci.nsISupportsString;
    var str = Cc["@mozilla.org/supports-string;1"].createInstance(CiS);
    str.data = val;
    Services.prefs.getBranch("extensions.multifox@hultmann.").setComplexValue(name, CiS, str);
  },

  getOuterId: function(win) {
    return win.QueryInterface(Ci.nsIInterfaceRequestor)
              .getInterface(Ci.nsIDOMWindowUtils)
              .outerWindowID;
  },

  getText: function(name) {
    return this._getTextCore(name, "extension", arguments, 1);
  },

  getTextFrom: function(name, filename) {
    return this._getTextCore(name, filename, arguments, 2);
  },

  _getTextCore: function(name, filename, args, startAt) {
    var bundle = Services.strings.createBundle("chrome://multifox-4410e96638/locale/" + filename + ".properties");

    if (args.length === startAt) {
      return bundle.GetStringFromName(name);
    } else {
      var args2 = Array.prototype.slice.call(args, startAt, args.length);
      console.assert(args2.length > 0, "_getTextCore");
      return bundle.formatStringFromName(name, args2, args2.length)
    }
  },

  networkListeners: {
    _observers: null,

    get active() {
      return this._observers !== null;
    },

    _cookieRejectedListener: {
      observe: function(aSubject, aTopic, aData) {
        console.log("cookie-rejected\n", aSubject, "\n", aTopic, "\n", aData, "\n", aSubject.QueryInterface(Ci.nsIURI).spec);
      }
    },

    enable: function(onRequest, onResponse) {
      console.log("networkListeners enable");
      if (this._observers !== null) {
        throw "networkListeners.enable ==> this._observers=true";
      }
      this._observers = [onRequest, onResponse];

      var obs = Services.obs;
      obs.addObserver(this._observers[0], "http-on-modify-request", false);
      obs.addObserver(this._observers[1], "http-on-examine-response", false);
      obs.addObserver(this._cookieRejectedListener, "cookie-rejected", false);
    },

    disable: function() {
      console.log("networkListeners disable");
      var obs = Services.obs;
      obs.removeObserver(this._observers[0], "http-on-modify-request");
      obs.removeObserver(this._observers[1], "http-on-examine-response");
      this._observers = null;
      obs.removeObserver(this._cookieRejectedListener, "cookie-rejected");
    }
  }
};
