/*
Creative Commons License: Attribution-No Derivative Works 3.0 Unported
http://creativecommons.org/licenses/by-nd/3.0/
(c) 2010-2013 Michael Koch
*/

Components.utils.import("resource://viewmarks/global.jsm"); 

var ViewMarksData = {

	dbConn: null,

	bookmarks: Components.classes["@mozilla.org/browser/nav-bookmarks-service;1"]
		.getService(Components.interfaces.nsINavBookmarksService),

	history: Components.classes["@mozilla.org/browser/nav-history-service;1"]
		.getService(Components.interfaces.nsINavHistoryService),
		
	tagging: Components.classes["@mozilla.org/browser/tagging-service;1"]
		.getService(Components.interfaces.nsITaggingService),		

	livemarks: null,

	annotation: Components.classes["@mozilla.org/browser/annotation-service;1"]
		.getService(Components.interfaces.nsIAnnotationService),
		
	uuidGenerator: Components.classes["@mozilla.org/uuid-generator;1"]
		.getService(Components.interfaces.nsIUUIDGenerator),		

	mostVisitedFolder: -1,
	lastVisitedFolder: -2,
	historyFolder: -3,

	//TN_DEFAULT_WIDTH: 256,
	//TN_DEFAULT_HEIGHT: 192,
	TN_DEFAULT_WIDTH: 384,
	TN_DEFAULT_HEIGHT: 288,

	SAVE_TO_PLACES: 1,
	SAVE_TO_DATABASE: 2,
	LOAD_FROM_PLACES: 4,
	LOAD_FROM_DATABASE: 8,

	ANNO_VMKSREF: "bookmarkProperties/vmksref",


	getPageMeta: function(win, name)
	{
		var meta = win.document.getElementsByTagName('meta');
		name = name.toLowerCase();
		for (var i=0; i<meta.length; i++) {
			if (meta[i].name.toLowerCase() == name) {
				return meta[i].content;
			}
		}
		return "";
	},


	copyText: function(text)
	{
		Components.classes["@mozilla.org/widget/clipboardhelper;1"]
			.getService(Components.interfaces.nsIClipboardHelper).copyString(text);  
	},


	copyImageFromData: function(data)
	{
		var img = document.createElementNS("http://www.w3.org/1999/xhtml","html:img");		
		img.onload = function() {
			document.popupNode = img;
			var command = "cmd_copyImageContents";
			var controller = document.commandDispatcher.getControllerForCommand(command);
			if (controller && controller.isCommandEnabled(command)) {
				controller.doCommand(command);
			}
			document.popupNode = null;
		}
		img.src = data;
	},


	saveImageFromData: function(data)
	{
  	internalSave(data, null, "code.png", "inline", "image/png", false, "SaveImageTitle", null, null, false, null);
	},


	getRecentlyUsedFolder: function()
	{
		var list = ViewMarksPref.recently_used_folder.split(",");
		var result = new Array();
		for (var i=0; i<list.length; i++) {
			var itemId = parseInt(list[i]);
			if (itemId == this.bookmarks.bookmarksMenuFolder) continue;
			if (this.isExistingFolder(itemId)) {
				result.push(itemId);
			}
		}
		return result;
	},


	updateRecentlyUsedFolder: function(itemId)
	{
		if (itemId == this.bookmarks.bookmarksMenuFolder) return;
		var list = this.getRecentlyUsedFolder();
		var value = ""+itemId;
		var count = 1;
		for (var i=0; i<list.length; i++) {
			if (list[i] == itemId) continue;
			value = value + "," + list[i];
			count++;
			if (count == 10) break;
		}
		ViewMarksPref.prefs.setCharPref("recently_used_folder", value);
	},


	getFavicon: function(url, callback)
	{
		if (ViewMarksUtil.startsWith(url, "about:viewmarks")) {
			callback(ViewMarksUtil.getURIFromString("chrome://viewmarks/skin/viewmarks.ico"));
		}
		url = url.split("#")[0];
		var nsIURI = ViewMarksUtil.getURIFromString(url);
		var fs = Components.classes["@mozilla.org/browser/favicon-service;1"]
			.getService(Components.interfaces.nsIFaviconService);
		try {
			fs.getFaviconURLForPage(nsIURI, callback);
		} catch (e) {
			try {
				setTimeout(function() {
					let fsURI = fs.getFaviconImageForPage(nsIURI);
					callback(fsURI);
					}, 0);
			} catch (e) {}
		}
	},


	countFolderItems: function(itemId)
	{
		if (itemId == 0) itemId = this.bookmarks.bookmarksMenuFolder;
		var query = this.history.getNewQuery();
		query.setFolders([itemId], 1);
		var result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		result.root.containerOpen = true;
		var count = result.root.childCount;
		result.root.containerOpen = false;
		return count;
	},


	refreshFolderCount: function()
	{
		this.toolbarCount = this.countFolderItems(this.bookmarks.toolbarFolder);
		this.unfiledCount = this.countFolderItems(this.bookmarks.unfiledBookmarksFolder);
	},


	enumFolderItems: function(itemId, callback)
	{
		if (itemId == 0) itemId = this.bookmarks.bookmarksMenuFolder;
		let query = this.history.getNewQuery();
		query.setFolders([itemId], 1);
		let result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		let folderNode = result.root;
		folderNode.containerOpen = true;
		let count = 0;
		for (var i=0; i<folderNode.childCount; i++) {
			let child = folderNode.getChild(i);
			let itemType = this.getItemType(child.itemId);
			if (itemType != this.bookmarks.TYPE_FOLDER && itemType != this.bookmarks.TYPE_BOOKMARK) continue;
			if (itemType == this.bookmarks.TYPE_BOOKMARK && child.uri && child.uri.indexOf("place:") == 0) continue;
			callback(child);
			count++;
		}
		folderNode.containerOpen = false;
		return count;
	},


	SORT_VISIT: 1,
	SORT_DATE: 2,
	
	enumVisitedItems: function(sort, count, callback)
	{
		var query = this.history.getNewQuery();
		var options = this.history.getNewQueryOptions();
		if (sort == this.SORT_VISIT) options.sortingMode = options.SORT_BY_VISITCOUNT_DESCENDING;
		if (sort == this.SORT_DATE) options.sortingMode = options.SORT_BY_DATE_DESCENDING;
		options.maxResults = count;
		options.queryType = options.QUERY_TYPE_BOOKMARKS;
		var result = this.history.executeQuery(query, options);
		var folderNode = result.root;
		folderNode.containerOpen = true;
		for (var i = 0; i < folderNode.childCount; i++) {
			var child = folderNode.getChild(i);
			callback({
				itemId:child.itemId,
				itemType:this.bookmarks.TYPE_BOOKMARK,
				title:child.title,
				uri:child.uri,
				accessCount:child.accessCount,
				time:child.time
			});
		}
		folderNode.containerOpen = false;
		return count;
	},


	readBookmarkList: function(aParentId, list)
	{
		if (aParentId == null) {
			aParentId = this.bookmarks.placesRoot;
			list = new Array();
		}
		if (aParentId == this.bookmarks.tagsFolder) return null;

		var query = this.history.getNewQuery();
		query.setFolders([aParentId], 1);

		var result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		var folderNode = result.root;
		folderNode.containerOpen = true;
	
		for (var i=0; i<folderNode.childCount; i++) {
			var childNode = folderNode.getChild(i);
			var itemType = ViewMarksData.getItemType(childNode.itemId);
			if (itemType == this.bookmarks.TYPE_FOLDER) {
				this.readBookmarkList(childNode.itemId, list);
			} else if (itemType == this.bookmarks.TYPE_BOOKMARK) {
				if (ViewMarksUtil.startsWith(childNode.uri, "place:")) continue;
				if (ViewMarksUtil.startsWith(childNode.uri, "javascript:")) continue;
				if (ViewMarksUtil.startsWith(childNode.uri, "about:")) continue;
				list.push(childNode);
			}
		}

		folderNode.containerOpen = false;
		return list;
	},


	sortBookmarkArray: function(array, order)
	{
		if (order == 0) return;
		array.sort(
			function(a, b) {
				if (a.itemType != ViewMarksData.bookmarks.TYPE_BOOKMARK) {
					if (b.itemType != ViewMarksData.bookmarks.TYPE_BOOKMARK) {
						return a.title.toString().localeCompare(b.title);
					} else {
						return -1;
					}
				}
				if (b.itemType != ViewMarksData.bookmarks.TYPE_BOOKMARK) {
					return 1;
				}
				if (order == 1) {
					return b.time - a.time;
				}
				if (order == 2) {
					var diff = b.accessCount - a.accessCount;
					if (diff != 0) return diff;
					return b.time - a.time;
				}
				return b.time - a.time;
			}
		);
	},
		

	getBookmarkIdsForURL: function(url)
	{
		var aURI = ViewMarksUtil.getURIFromString(url);
		var origURI = ViewMarksData.bookmarks.getBookmarkedURIFor(aURI);
		if (origURI) aURI = origURI;
		return ViewMarksData.bookmarks.getBookmarkIdsForURI(aURI, {});
	},

	
	getFolderIconUrl: function(itemId)
	{
		var url = "chrome://viewmarks/skin/folder.png";
		if (itemId == this.bookmarks.bookmarksMenuFolder) {
			url = "chrome://viewmarks/skin/bookmarksMenu.png";
		} else if (itemId == this.bookmarks.toolbarFolder) {
			url = "chrome://viewmarks/skin/bookmarksToolbar.png";
		} else if (itemId == this.bookmarks.unfiledBookmarksFolder) {
			url = "chrome://viewmarks/skin/unsortedBookmarks.png";
		} else if (itemId == this.lastVisitedFolder) {
			url = "chrome://viewmarks/skin/history.png";
		} else if (itemId == this.mostVisitedFolder) {
			url = "chrome://viewmarks/skin/history.png";
		} else if (itemId == this.historyFolder) {
			url = "chrome://viewmarks/skin/calendar.png";
		} else if (itemId == ViewMarksPref.trash_folder_id) {
			url = "chrome://viewmarks/skin/trash.png";
		} else if (this.isLivemark(itemId)) {		
			url = "chrome://viewmarks/skin/feed16.png";
		}
		return url;
	},


	getFolderInsertIndex: function(folderId)
	{
		if (ViewMarksPref.insert_folder_before_bookmarks == false) return this.bookmarks.DEFAULT_INDEX;
		var query = this.history.getNewQuery();
		query.setFolders([folderId], 1);
		var result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		var folderNode = result.root;
		folderNode.containerOpen = true;
		for (var i=0; i<folderNode.childCount; i++) {
			var childNode = folderNode.getChild(i);
			var itemType = this.getItemType(childNode.itemId);
			if (itemType != this.bookmarks.TYPE_BOOKMARK) continue;
			if (ViewMarksUtil.startsWith(childNode.uri, "place:")) continue;
			return this.bookmarks.getItemIndex(childNode.itemId);
		}
		return this.bookmarks.DEFAULT_INDEX;
	},


	readFolderItemIds: function(itemId)
	{
		let list = new Array();
		let query = this.history.getNewQuery();
		query.setFolders([itemId], 1);
		let result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		let folderNode = result.root;
		folderNode.containerOpen = true;
		for (var i=0; i<folderNode.childCount; i++) {
			list.push(folderNode.getChild(i).itemId);
		}
		folderNode.containerOpen = false;
		return list;
	},


	copyFolder: function(aItemId, aNewParentId, aIndex)
	{
		let newItemId = 0;
		try {
			newItemId = this.bookmarks.createFolder(aNewParentId, this.bookmarks.getItemTitle(aItemId), aIndex); 
		} catch (e) {
			return null;
		}
		this.bookmarks.setItemDateAdded(newItemId, this.bookmarks.getItemDateAdded(aItemId));
		this.bookmarks.setItemLastModified(newItemId, this.bookmarks.getItemLastModified(aItemId));
		this.annotation.copyItemAnnotations(aItemId, newItemId, true);
		this.removeItemAnnotation(newItemId, this.ANNO_VMKSREF);
		return newItemId;
	},


	copyBookmark: function(aItemId, aNewParentId, aIndex)
	{
		let newItemId = 0;
		try {
			newItemId = this.bookmarks.insertBookmark(aNewParentId, this.bookmarks.getBookmarkURI(aItemId),     
				aIndex, this.bookmarks.getItemTitle(aItemId)); 
		} catch (e) {
			return null;
		}
		this.setItemKeywords(newItemId, this.getItemKeywords(aItemId));
		this.setItemDescription(newItemId, this.getItemDescription(aItemId));
		// tbd: copy tags
		this.bookmarks.setItemDateAdded(newItemId, this.bookmarks.getItemDateAdded(aItemId));
		this.bookmarks.setItemLastModified(newItemId, this.bookmarks.getItemLastModified(aItemId));
		this.annotation.copyItemAnnotations(aItemId, newItemId, true);
		this.removeItemAnnotation(newItemId, this.ANNO_VMKSREF);
		var tn = this.getItemThumbnail(aItemId);
		if (tn) this.setItemThumbnail(newItemId, tn.data, tn.width, tn.height);
		return newItemId;
	},


	copyLivemark: function(aItemId, aNewParentId, aIndex)
	{
		if (this.livemarks == null) {
			try {
				this.livemarks = Components.classes["@mozilla.org/browser/livemark-service;2"].getService(Components.interfaces.nsILivemarkService);
			} catch(e) {
				return null;
			}
		}
		let title = this.bookmarks.getItemTitle(aItemId);
		let siteuri = this.livemarks.getSiteURI(aItemId);
		let feeduri = this.livemarks.getFeedURI(aItemId);
		let newItemId = this.livemarks.createLivemarkFolderOnly(aNewParentId, title, siteuri, feeduri, aIndex);
		this.annotation.copyItemAnnotations(aItemId, newItemId, true);
		this.removeItemAnnotation(newItemId, this.ANNO_VMKSREF);
		return newItemId;
	},


	copyItem: function(aItemId, aNewParentId, aIndex, shallow)
	{
		if (this.isLivemark(aItemId)) {
			return this.copyLivemark(aItemId, aNewParentId, aIndex);
		}
		let itemType = this.getItemType(aItemId);
		if (itemType == this.bookmarks.TYPE_BOOKMARK) {
			return this.copyBookmark(aItemId, aNewParentId, aIndex);
		}		
		if (itemType == this.bookmarks.TYPE_FOLDER) {
			let newItemId = this.copyFolder(aItemId, aNewParentId, aIndex);
			if (shallow === undefined || shallow == false) {
				let list = this.readFolderItemIds(aItemId);
				for (var i=0; i<list.length; i++) {
					this.copyItem(list[i], newItemId, this.bookmarks.DEFAULT_INDEX);
				}
			}
			return newItemId;
		}		
		return null;
	},


	removeItem: function(itemId)
	{
		try {this.bookmarks.removeItem(itemId);} catch (e) {}
	},
	

	isLivemark: function(itemId)
	{
		//return this.livemarks.isLivemark(itemId);
		if (this.isExistingFolder(itemId) == false) return false;
		var feeduri = this.getItemAnnotation(itemId, "livemark/feedURI");
		if (feeduri.length == 0) return false;
		return true;
	},


	isSpecialFolder: function(itemId)
	{
		if (itemId < 0) return true;
		if (itemId == this.bookmarks.placesRoot) return true;
		if (itemId == this.bookmarks.bookmarksMenuFolder) return true;
		if (itemId == this.bookmarks.toolbarFolder) return true;
		if (itemId == this.bookmarks.unfiledBookmarksFolder) return true;
		if (itemId == this.bookmarks.tagsFolder) return true;
		if (itemId == ViewMarksPref.trash_folder_id) return true;
		return false;
	},


	isExistingFolder: function(itemId)
	{
		try {
			if (this.getItemType(itemId) == this.bookmarks.TYPE_FOLDER) return true;
		} catch(e) {
			return false;
		}
		return false;
	},


	getParentFolderId: function(itemId)
	{
		if (ViewMarksData.isSpecialFolder(itemId)) return this.bookmarks.bookmarksMenuFolder;
		try {
			return this.bookmarks.getFolderIdForItem(itemId);
		} catch (e) {
			return null;
		}
	},


	getRootParentId: function(itemId)
	{
		if (this.isSpecialFolder(itemId)) return itemId;				
		return this.getRootParentId(this.bookmarks.getFolderIdForItem(itemId));
	},


	getBookmarkByURL: function(url)
	{
		try {
			var aURI = ViewMarksUtil.getURIFromString(url);
			var origURI = this.bookmarks.getBookmarkedURIFor(aURI);
			if (origURI) {
				aURI = origURI;
				//ViewMarksUtil.logmsg("getBookmarkByURL: redir from "+aURI.spec);
			}
			var bmkIds = this.bookmarks.getBookmarkIdsForURI(aURI, {});
			for (var i=0; i<bmkIds.length; i++) {
				var root = this.getRootParentId(bmkIds[i]);
				if (root == this.bookmarks.tagsFolder) continue;
				return bmkIds[i];
			}
		} catch (e) {}
		return 0;
	},


	getBookmarksByURL: function(url)
	{
		var aURI = ViewMarksUtil.getURIFromString(url);
		var origURI = this.bookmarks.getBookmarkedURIFor(aURI);
		if (origURI) {
			aURI = origURI;
			//ViewMarksUtil.logmsg("getBookmarkByURL: redir from "+aURI.spec);
		}
		return this.bookmarks.getBookmarkIdsForURI(aURI, {});
	},


	getItemPath: function(itemId)
	{
		if (itemId <= 0) return "";
		if (ViewMarksData.isSpecialFolder(itemId)) return "";
		var parent = this.getParentFolderId(itemId);
		var path = this.getItemPath(parent);
		var title = this.getItemTitle(parent);
		if (path.length > 0) {
			return path + " » " + title;
		} else {
			return title;
		}
	},
	

	getItemDateDeleted: function(itemId)
	{
		if (itemId <= 0) return "";
		let date = this.getItemAnnotation(itemId, "bookmarkProperties/dateDeleted");
		if (date == null || date == "") return "";
		return (new Date(date)).toLocaleString();
	},
	

	getItemDateAdded: function(itemId)
	{
		if (itemId <= 0) return "";
		return (new Date(this.bookmarks.getItemDateAdded(itemId)/1000)).toLocaleString();
	},
	

	getItemLastModified: function(itemId)
	{
		if (itemId <= 0) return "";
		return (new Date(this.bookmarks.getItemLastModified(itemId)/1000)).toLocaleString();
	},
	

	getItemType: function(itemId)
	{
		if (itemId == this.mostVisitedFolder) return this.bookmarks.TYPE_FOLDER;
		if (itemId == this.lastVisitedFolder) return this.bookmarks.TYPE_FOLDER;
		if (itemId == this.historyFolder) return this.bookmarks.TYPE_FOLDER;
		try {
			return this.bookmarks.getItemType(itemId);
		} catch(e) {
			return 0;
		}
	},
	
	
	isFolder: function(itemId)
	{
		return (this.getItemType(itemId) == this.bookmarks.TYPE_FOLDER);
	},


	isBookmark: function(itemId)
	{
		return (this.getItemType(itemId) == this.bookmarks.TYPE_BOOKMARK);
	},
			

	getItemTitle: function(itemId)
	{
		if (itemId == this.mostVisitedFolder) return ViewMarksUtil.getString("most_visited_folder");
		if (itemId == this.lastVisitedFolder) return ViewMarksUtil.getString("last_visited_folder");
		if (itemId == this.historyFolder) return ViewMarksUtil.getString("history_folder");
		if (itemId == ViewMarksPref.trash_folder_id) return ViewMarksUtil.getString("trash_folder");
		try {
			return this.bookmarks.getItemTitle(itemId);
		} catch(e) {
			//alert(e+"\nitemId="+itemId+"\nfrom"+arguments.callee.caller);
			return "?";
		}
	},


	getFolderNameFromId: function(folderId)
	{
		if (folderId == ViewMarksData.mostVisitedFolder) return "mostvisited";
		if (folderId == ViewMarksData.lastVisitedFolder) return "lastvisited";
		if (folderId == ViewMarksData.historyFolder) return "history";
		if (folderId == ViewMarksPref.trash_folder_id) return "trash";
		if (folderId == this.bookmarks.bookmarksMenuFolder) return "bookmarks";
		if (folderId == this.bookmarks.toolbarFolder) return "toolbar";
		if (folderId == this.bookmarks.unfiledBookmarksFolder) return "unsorted";
		return folderId;
	},


	getItemKeywords: function(itemId)
	{
		if (itemId <= 0) return "";
		var value = this.bookmarks.getKeywordForBookmark(itemId)
		if (value == null) value = "";
		return value;
	},


	setItemKeywords: function(itemId, value)
	{
		this.bookmarks.setKeywordForBookmark(itemId, value);
	},


	itemHasAnnotation: function(itemId, name)
	{
		try {
			return this.annotation.itemHasAnnotation(itemId, name);
		} catch (e) {	
			return false;
		}
	},


	getItemAnnotation: function(itemId, name)
	{
		try {
			var value = this.annotation.getItemAnnotation(itemId, name);
			if (value == null) value = "";
			return value;
		} catch (e) {	
			return "";
		}
	},


	setItemAnnotation: function(itemId, name, value)
	{
		this.annotation.setItemAnnotation(itemId, name, value, 0, this.annotation.EXPIRE_NEVER);
	},


	removeItemAnnotation: function(itemId, name)
	{
		this.annotation.removeItemAnnotation(itemId, name);
	},


	getItemsWithAnnotation: function(name)
	{
		try {
			return this.annotation.getItemsWithAnnotation(name, {});
		} catch(e) {
			return [];
		}
	},


	getItemDescription: function(itemId)
	{
		if (itemId <= 0) return "";
		return this.getItemAnnotation(itemId, "bookmarkProperties/description");
	},


	setItemDescription: function(itemId, value)
	{
		this.setItemAnnotation(itemId, "bookmarkProperties/description", value);
	},


	setItemTags: function(itemId, tags)
	{
		let uri = this.bookmarks.getBookmarkURI(itemId);
		this.tagging.tagURI(uri, tags);		
	},


	removeItemTags: function(itemId, tags)
	{
		let uri = this.bookmarks.getBookmarkURI(itemId);
		this.tagging.untagURI(uri, tags);		
	},


	getItemTags: function(itemId)
	{
		let uri = this.bookmarks.getBookmarkURI(itemId);
		return this.tagging.getTagsForURI(uri, {});	
	},


	setUrlTags: function(url, tags)
	{
		let uri = ViewMarksUtil.getURIFromString(url);
		return this.tagging.tagURI(uri, tags);	
	},


	removeUrlTags: function(url, tags)
	{
		let uri = ViewMarksUtil.getURIFromString(url);
		return this.tagging.untagURI(uri, tags);	
	},


	getUrlTags: function(url)
	{
		let uri = ViewMarksUtil.getURIFromString(url);
		return this.tagging.getTagsForURI(uri, {});	
	},


	getAllTags: function()
	{
		return this.tagging.allTags;
	},


	getLocalFile: function(filename)
	{
		var file = Components.classes["@mozilla.org/file/directory_service;1"]
			.getService(Components.interfaces.nsIProperties)  
			.get("ProfD", Components.interfaces.nsIFile);  
		file.append("viewmarks");  
		if( !file.exists() || !file.isDirectory() ) {   
			file.create(Components.interfaces.nsIFile.DIRECTORY_TYPE, 0x1ff);  
		}  
		if (filename && filename.length > 0) file.append(filename);
		return file;  
	},


	readObjectFromFile: function(file)
	{
		try {
			var fin = Components.classes["@mozilla.org/network/file-input-stream;1"]
			.createInstance(Components.interfaces.nsIFileInputStream);
			fin.init(file, -1, -1, 0);
			var fsin = Components.classes["@mozilla.org/scriptableinputstream;1"]
				.createInstance(Components.interfaces.nsIScriptableInputStream);
			fsin.init(fin);
			var data = fsin.read(file.fileSize);
			fsin.close();
			var obj = JSON.parse(data);
			return obj;
		} catch(e) {
			ViewMarksUtil.alert(e);
			return null;
		}
	},


	writeObjectToFile: function(file, obj)
	{
		try {
			var fout = Components.classes["@mozilla.org/network/file-output-stream;1"]
				.createInstance(Components.interfaces.nsIFileOutputStream);
			fout.init(file, -1, -1, 0);
			var data = JSON.stringify(obj);
			fout.write(data, data.length);
			fout.close();
		} catch(e) {
			ViewMarksUtil.logmsg("writeObjectToFile("+file+"): "+e);
		}
	},


	openDatabase: function()
	{
		let file = this.getLocalFile("viewmarks.sqlite");  
		let storageService = Components.classes["@mozilla.org/storage/service;1"]  
			.getService(Components.interfaces.mozIStorageService);  
		let dbConn = storageService.openDatabase(file); 
		if (!dbConn.tableExists("thumbnail")) {
			dbConn.executeSimpleSQL("CREATE TABLE thumbnail (id INTEGER UNIQUE, data, width INTEGER, height INTEGER, ref TEXT)");  
		} else {
			try {
				dbConn.executeSimpleSQL("ALTER TABLE thumbnail ADD COLUMN ref TEXT");  
				ViewMarksUtil.logmsg("update database");
				let dest = this.getLocalFile(null);  
				file.copyTo(dest, "viewmarks.sqlite.backup");
				let statement = dbConn.createStatement("SELECT * FROM thumbnail");
				statement.executeAsync({
					handleResult: function(aResultSet) {
						let row = aResultSet.getNextRow(); 
						while (row) {
							let itemId = row.getResultByName("id");
							let uuid = ViewMarksData.uuidGenerator.generateUUID().toString();
							try {
								ViewMarksData.setItemAnnotation(itemId, ViewMarksData.ANNO_VMKSREF, uuid);
								let statement = dbConn.createStatement("UPDATE thumbnail SET ref=:uuid WHERE id=:id");
								statement.params.uuid = uuid;
								statement.params.id = itemId;
								statement.execute();
							} catch(e) {
								let statement = dbConn.createStatement("DELETE FROM thumbnail WHERE id=:id");
								statement.params.id = itemId;
								statement.execute();
								ViewMarksUtil.logmsg("update: item "+itemId+" is missing, removing thumbnail");
							}
							row = aResultSet.getNextRow();
						} 
					},
					handleError: function(aError) {
						ViewMarksUtil.logmsg("handleError "+aReason);
					},
					handleCompletion: function(aReason) {
						ViewMarksUtil.logmsg("database updated");
					}
				});



			} catch(e) {}
		}
		return dbConn;
	},


	closeDatabase: function()
	{
		if (this.dbConn.asyncClose) {
			this.dbConn.asyncClose();
		} else {
			this.dbConn.close();
		}
	},


	exportDatabase: function(file)
	{
		if (this.dbConn == null) return;
		let tn_array = new Array();
		let list = this.getItemsWithAnnotation(this.ANNO_VMKSREF);
		for (var i=0; i<list.length; i++) {
			let ref = this.getItemAnnotation(list[i], this.ANNO_VMKSREF);
			let uri = this.bookmarks.getBookmarkURI(list[i]);
			let statement = this.dbConn.createStatement("SELECT * FROM thumbnail WHERE ref=:ref");
			try{statement.params.ref = ref;} catch(e) {}
			while (statement.executeStep()) {
				let thumbnail = {};
				thumbnail.url = uri.spec;
				thumbnail.data = statement.row.data;
				thumbnail.width = statement.row.width;
				thumbnail.height = statement.row.height;
				tn_array.push(thumbnail);
			}
			statement.reset();
			statement.finalize();
		}
		this.writeObjectToFile(file, tn_array);
	},


	setThumbnailFromMimeType: function(data, itemId, obj, callback)
	{
		var canvas = document.createElementNS("http://www.w3.org/1999/xhtml","html:canvas");
	  canvas.width = ViewMarksData.TN_DEFAULT_WIDTH;
	  canvas.height = ViewMarksData.TN_DEFAULT_HEIGHT;
	  var context = canvas.getContext("2d");
		var img = new Image();
		img.onload = function() {
			context.drawImage(img, 0, 0);
			var data = canvas.toDataURL();
			callback(obj, data, img.width, img.height, true);
		}
		img.src = "chrome://viewmarks/skin/docs/doc_"+data.replace("/", "_")+".png";
	},


	getThumbnailItemIdList: function(callback)
	{
		if (ViewMarksPref.data_strategy & this.LOAD_FROM_DATABASE) {
			if (this.dbConn == null) {
				callback([]);
				return;
			}
			var itemIdList = new Array();
			var statement = this.dbConn.createStatement("SELECT id FROM thumbnail");
			statement.executeAsync({
				handleResult: function(aResultSet) {
					var row = aResultSet.getNextRow(); 
					while (row) {
						let id = row.getResultByName("id");
						itemIdList.push(id);
						row = aResultSet.getNextRow();
					} 
				},
				handleError: function(aError) {
					callback([]);
				},
				handleCompletion: function(aReason) {
					if (aReason == Components.interfaces.mozIStorageStatementCallback.REASON_FINISHED) {
						callback(itemIdList);
					} else {
						callback([]);
					}
				}
			});
		}
	},	


	getItemThumbnail: function(itemId, obj, callback)
	{
		if (ViewMarksPref.data_strategy & this.LOAD_FROM_PLACES) {
			var enc = this.getItemAnnotation(itemId, "bookmarkProperties/viewmark");
			if (enc != "") {
				var arr = enc.split("*");
				if (callback === undefined || callback == null) {
					var thumbnail = {};
					thumbnail.data = arr[2];
					thumbnail.width = arr[0];
					thumbnail.height = arr[1];
					return thumbnail;
				} else {
					callback(obj, arr[2], arr[0], arr[1]);
					return null;
				}
			}
		}

		if (ViewMarksPref.data_strategy & this.LOAD_FROM_DATABASE) {
			if (this.dbConn == null) return null;
			if (itemId == 0) return null;
/*
			let ref = this.getItemAnnotation(itemId, this.ANNO_VMKSREF);
			if (ref == "") {
				var statement = this.dbConn.createStatement("SELECT * FROM thumbnail WHERE id=:id");
				try{statement.params.id = itemId;} catch(e) {}
			} else {
				var statement = this.dbConn.createStatement("SELECT * FROM thumbnail WHERE ref=:ref");
				try{statement.params.ref = ref;} catch(e) {}
			}
*/
			var statement = this.dbConn.createStatement("SELECT * FROM thumbnail WHERE id=:id");
			try{statement.params.id = itemId;} catch(e) {}

			if (callback === undefined || callback == null) {
				var thumbnail = {};
				if (statement.executeStep()) {
					thumbnail.data = statement.row.data;
					thumbnail.width = statement.row.width;
					thumbnail.height = statement.row.height;
				} else {
					thumbnail = null;
				}
				statement.reset();
				statement.finalize();
				return thumbnail;
			} else {
				statement.executeAsync({
					handleResult: function(aResultSet) {
						var row = aResultSet.getNextRow(); 
						while (row) {
							let data = row.getResultByName("data");
							if (data == "" || data.substr(0, 5) == "data:") {						
								var width = row.getResultByName("width");
								var height = row.getResultByName("height");
								callback(obj, data, width, height);
							} else {
								ViewMarksData.setThumbnailFromMimeType(data, itemId, obj, callback);
							}
							row = aResultSet.getNextRow();
						} 
					},
					handleError: function(aError) {
						ViewMarksUtil.logmsg("handleError "+aReason);
					},
					handleCompletion: function(aReason) {
						if (ViewMarksView.load_start) {
							ViewMarksView.load_count++;
							if (ViewMarksView.load_count == ViewMarksView.boxList.length) {
								ViewMarksUtil.logmsg("view load time = "+(performance.now() - ViewMarksView.load_start));
							}
						} 
						if (aReason != 0) ViewMarksUtil.logmsg("handleCompletion "+aReason);
					}
				});
				return null;
			}
		}
		
		return null;
	},	

	
	setItemThumbnail: function(itemId, data, width, height)
	{
		if (ViewMarksPref.data_strategy & this.SAVE_TO_PLACES) {
			var enc = ""+width+"*"+height+"*"+data;
			this.setItemAnnotation(itemId, "bookmarkProperties/viewmark", enc);
		}
	
		if (ViewMarksPref.data_strategy & this.SAVE_TO_DATABASE) {
			if (this.dbConn == null) return;

			let uuid = this.getItemAnnotation(itemId, this.ANNO_VMKSREF);
			if (uuid == null || uuid.length == 0) {
				uuid = this.uuidGenerator.generateUUID().toString();
				this.setItemAnnotation(itemId, this.ANNO_VMKSREF, uuid);
			}
			try {
				let statement = this.dbConn.createStatement("INSERT INTO thumbnail VALUES (:id,:data,:width,:height,:uuid)");
				statement.params.id = itemId;
				statement.params.data = data;
				statement.params.width = width;
				statement.params.height = height;
				statement.params.uuid = uuid;
				statement.execute();
			} catch (e) {
				//ViewMarksUtil.alert(e);
				try {
					let statement = this.dbConn.createStatement("UPDATE thumbnail SET data=:data,width=:width,height=:height,ref=:uuid WHERE id=:id");
					statement.params.id = itemId;
					statement.params.data = data;
					statement.params.width = width;
					statement.params.height = height;
					statement.params.uuid = uuid;
					statement.execute();
				} catch (e) {
					ViewMarksUtil.alert(e);
				}
			}

		}
	},	


	deleteItemThumbnail: function(itemId)
	{
		let ref = this.getItemAnnotation(itemId, this.ANNO_VMKSREF);
		this.removeItemAnnotation(itemId, "bookmarkProperties/viewmark");
		this.removeItemAnnotation(itemId, this.ANNO_VMKSREF);
		if (this.dbConn == null) return;
		try {
			if (ref == "") {
				let statement = this.dbConn.createStatement("DELETE FROM thumbnail WHERE id=:id");
				statement.params.id = itemId;
				statement.execute();
			} else {
				let statement = this.dbConn.createStatement("DELETE FROM thumbnail WHERE ref=:ref");
				statement.params.ref = ref;
				statement.execute();
			}
		} catch (e) {
			ViewMarksUtil.logerror(e);
		}
	},


	createFolderListItem: function(label, value, depth)
	{
		var imgurl = ViewMarksData.getFolderIconUrl(value);

		var item = document.createElement("menuitem");
		item.setAttribute("label", label);
		item.setAttribute("value", value);
		item.setAttribute("image", imgurl);
		item.setAttribute("class", "menuitem-iconic bookmark-item");
		item.setAttribute("style", "padding-left:"+(depth*10)+"px");

		var image = document.createElement("image");		
		image.setAttribute("src", imgurl);
		item.appendChild(image);

		var nlabel = document.createElement("label");
		nlabel.setAttribute("value", label);
		item.appendChild(nlabel);
		
		return item;
	},


	createFolderList: function(aItemId, popup, depth)
	{
		ViewMarksData.enumFolderItems(aItemId, function(node) {
			var type = ViewMarksData.getItemType(node.itemId);
			if (type == ViewMarksData.bookmarks.TYPE_FOLDER) {
				var item = ViewMarksData.createFolderListItem(node.title, node.itemId, depth);
				popup.appendChild(item);
				ViewMarksData.createFolderList(node.itemId, popup, depth+1);
			}
		});
	},


	createFolderTree: function(popup, readonly)
	{
		var folderid;

		if (readonly) {
			folderid = ViewMarksData.mostVisitedFolder;
			var root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("most_visited_folder"), folderid, 0);
			popup.appendChild(root);
			folderid = ViewMarksData.lastVisitedFolder;
			var root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("last_visited_folder"), folderid, 0);
			popup.appendChild(root);
			if (ViewMarksPref.create_history) {
				folderid = ViewMarksData.historyFolder;
				var root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("history_folder"), folderid, 0);
				popup.appendChild(root);
			}
			if (ViewMarksPref.create_trash) {
				folderid = ViewMarksPref.trash_folder_id;
				var root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("trash_folder"), folderid, 0);
				popup.appendChild(root);
			}
		}

		folderid = ViewMarksData.bookmarks.bookmarksMenuFolder;
		var root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("bookmarksmenu"), folderid, 0);
		popup.appendChild(root);
		ViewMarksData.createFolderList(folderid, popup, 1);

		folderid = ViewMarksData.bookmarks.toolbarFolder;
		root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("bookmarkstoolbar"), folderid, 0);
		popup.appendChild(root);
		ViewMarksData.createFolderList(folderid, popup, 1);

		folderid = ViewMarksData.bookmarks.unfiledBookmarksFolder;
		root = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("bookmarksunfiled"), folderid, 0);
		popup.appendChild(root);
		ViewMarksData.createFolderList(folderid, popup, 1);
	}


};


ViewMarksData.dbConn = ViewMarksData.openDatabase();

