/*
Creative Commons License: Attribution-No Derivative Works 3.0 Unported
http://creativecommons.org/licenses/by-nd/3.0/
(c) 2010-2013 Michael Koch
*/



var ViewMarksEdit = {

	TN_EDIT_WIDTH: 256,
	TN_EDIT_HEIGHT: 192,

	DF_CREATE_BOOKMARK: 1,
	DF_EDIT_BOOKMARK: 2,
	DF_CREATE_FOLDER: 3,
	DF_EDIT_FOLDER: 4,

	bookmarks: Components.classes["@mozilla.org/browser/nav-bookmarks-service;1"]
		.getService(Components.interfaces.nsINavBookmarksService),

	history: Components.classes["@mozilla.org/browser/nav-history-service;1"]
		.getService(Components.interfaces.nsINavHistoryService),
		
	appShellService: Components.classes["@mozilla.org/appshell/appShellService;1"]
  	.getService(Components.interfaces.nsIAppShellService),

	quicksave: false,
	extended: false,

	TNF_NONE: 0,
	TNF_PRESENT: 1,
	TNF_SAVE: 2,
	TNF_DELETE: 3,
	TNF_MIME: 4,
	thumbnail_action: 0,
	mime_type: null,

	srcCanvas: null,
	dstCanvas: null,

	tagsUrl: null,
	tagsList: null,
	tagsChanged: false,
	tagsListAll: null,

	dialogInitial: null,

	dialogList: [
		["title", "vmks_edit_name"],
		["url", "vmks_edit_url"],
		["description", "vmks_edit_description"],
		["keyword", "vmks_edit_keyword"],
		["folder", "vmks_dir_list"],
		["subfolder", "vmks_edit_subfolder"],
	],

	dialogObject: function()
	{
		for (var i=0; i<ViewMarksEdit.dialogList.length; i++) {
			var name = ViewMarksEdit.dialogList[i][0];
			this[name] = {};
			this[name].id = ViewMarksEdit.dialogList[i][1];
			this[name].node = document.getElementById(this[name].id);
			this[name].value = this[name].node.value;
		}
		this.title.value = this.title.value.trim();
		this.subfolder.value = this.subfolder.value.trim();
		this.url.value = ViewMarksUtil.getURIFromString(this.url.value);
		this.folder.value = this.folder.node.selectedItem.getAttribute("value");
	},


	checkDialogChanges: function(dialog)
	{
		if (this.dialogInitial == null) return;
		for (var i=0; i<ViewMarksEdit.dialogList.length; i++) {
			var name = ViewMarksEdit.dialogList[i][0];
			if (dialog[name].value != this.dialogInitial[name].value) {
				dialog[name].node.changed = true;
			}
		}
	},


	resetChangeFlags: function()
	{
		for (var i=0; i<ViewMarksEdit.dialogList.length; i++) {
			var node = document.getElementById(ViewMarksEdit.dialogList[i][1]);
			node.setAttribute("onchange", "this.changed=true;");
			node.changed = false;
		}
	},


	resizeCanvas: function(canvas, width, height)
	{
		var fResize = (canvas.height != height);
		canvas.width = width;
		canvas.height = height;
		if (fResize && window.name == "dlgViewMarksEdit") {
			window.sizeToContent();
			//ViewMarksUtil.alert(window.name);
		}
	},


	paintThumbnail: function(dstCanvas, srcCanvas, timeout)
	{
		var mainWindow = ViewMarksUtil.getMainWindow();
		var outerHeight = mainWindow.outerHeight;
		if (dstCanvas.timer) clearTimeout(dstCanvas.timer);
		var height = Math.round(dstCanvas.width*srcCanvas.sh/srcCanvas.sw);
		this.resizeCanvas(dstCanvas, dstCanvas.width, height);
		if (timeout > 0) {
		  var dstContext = dstCanvas.getContext("2d");
			dstContext.drawImage(srcCanvas, srcCanvas.sx, srcCanvas.sy, srcCanvas.sw, srcCanvas.sh, 
				0, 0, dstCanvas.width, dstCanvas.height);
			dstCanvas.timer = setTimeout(function() {
				var elapsed = (new Date()).getTime();	
				ViewMarksBitmap.createThumbnail(dstCanvas, 0, 0, dstCanvas.width, dstCanvas.height, 
					srcCanvas, srcCanvas.sx, srcCanvas.sy, srcCanvas.sw, srcCanvas.sh);
				elapsed = (new Date()).getTime() - elapsed;	
				ViewMarksUtil.logmsg("paint = "+elapsed+"ms");
			}, timeout);
		} else {
			var elapsed = (new Date()).getTime();	
			ViewMarksBitmap.createThumbnail(dstCanvas, 0, 0, dstCanvas.width, dstCanvas.height, 
				srcCanvas, srcCanvas.sx, srcCanvas.sy, srcCanvas.sw, srcCanvas.sh);
			elapsed = (new Date()).getTime() - elapsed;	
			ViewMarksUtil.logmsg("paint = "+elapsed+"ms");
		}
	},


	open: function(action, itemId)
	{
		var argObj = {};
		argObj.action = action;
		argObj.itemId = itemId;
		argObj.win = ViewMarksUtil.getBrowser().contentWindow;
		window.openDialog("chrome://viewmarks/content/editmark.xul", "dlgViewMarksEdit", "centerscreen,chrome,modal", argObj).focus();
	},


	onAccept: function(event) 
	{
		try {

			var dialog = new this.dialogObject();
			var canvas = document.getElementById("vmks_edit_canvas");
			var itemId = window.arguments[0].itemId; 
	
			if (dialog.title.value == "" && ViewMarksPref.allow_unnamed == false) {
				event.preventDefault();
				return false;
			}

			this.checkDialogChanges(dialog);
				
			if (dialog.subfolder.value != "") {
				var index =  ViewMarksData.getFolderInsertIndex(dialog.folder.value);
				dialog.folder.value = this.bookmarks.createFolder(dialog.folder.value, dialog.subfolder.value, index);
			}
	
			if (ViewMarksEdit.action == this.DF_CREATE_BOOKMARK) {
				itemId = this.bookmarks.insertBookmark(dialog.folder.value, dialog.url.value, this.bookmarks.DEFAULT_INDEX, 
					dialog.title.value);  
				ViewMarksData.setItemDescription(itemId, dialog.description.value);
				ViewMarksData.setItemKeywords(itemId, dialog.keyword.value)
				ViewMarksData.updateRecentlyUsedFolder(dialog.folder.value);
				if (this.tagsUrl && this.tagsList && this.tagsList.length > 0) {
					ViewMarksData.setUrlTags(this.tagsUrl, this.tagsList);
				}
			}
	
			if (ViewMarksEdit.action == this.DF_EDIT_BOOKMARK) {
				if (dialog.title.node.changed) this.bookmarks.setItemTitle(itemId, dialog.title.value);
				if (dialog.url.node.changed) {
					try {this.bookmarks.changeBookmarkURI(itemId, dialog.url.value);} catch (e) {/* catch illegal url values */}
				}
				if (dialog.description.node.changed) ViewMarksData.setItemDescription(itemId, dialog.description.value);
				if (dialog.keyword.node.changed) ViewMarksData.setItemKeywords(itemId, dialog.keyword.value)
				if (this.bookmarks.getFolderIdForItem(itemId) != dialog.folder.value) {
					this.bookmarks.moveItem(itemId, dialog.folder.value, this.bookmarks.DEFAULT_INDEX);
					ViewMarksTrash.removeTrashInformation(itemId);
					ViewMarksData.updateRecentlyUsedFolder(dialog.folder.value);
				}
				if (this.tagsChanged || dialog.url.node.changed) {
					ViewMarksData.removeUrlTags(this.tagsUrl, null);
					if (this.tagsList.length) {
						ViewMarksData.setUrlTags(dialog.url.value.spec, this.tagsList);
					}
				}
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
			}
	
			if (ViewMarksEdit.action == this.DF_CREATE_FOLDER) {
				var index =  ViewMarksData.getFolderInsertIndex(dialog.folder.value);
				itemId = this.bookmarks.createFolder(dialog.folder.value, dialog.title.value, index);
				ViewMarksData.updateRecentlyUsedFolder(dialog.folder.value);
			}
	
			if (ViewMarksEdit.action == this.DF_EDIT_FOLDER) {
				if (dialog.title.node.changed) this.bookmarks.setItemTitle(itemId, dialog.title.value);
				if (this.bookmarks.getFolderIdForItem(itemId) != dialog.folder.value) {
					var index = ViewMarksData.getFolderInsertIndex(dialog.folder.value);
					this.bookmarks.moveItem(itemId, dialog.folder.value, index);
					ViewMarksTrash.removeTrashInformation(itemId);
					ViewMarksData.updateRecentlyUsedFolder(dialog.folder.value);
				}
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
			}
	
			if (this.thumbnail_action == this.TNF_SAVE) {
				let saveCanvas = document.createElementNS("http://www.w3.org/1999/xhtml","html:canvas");
				saveCanvas.width = ViewMarksData.TN_DEFAULT_WIDTH;
				saveCanvas.height = ViewMarksData.TN_DEFAULT_HEIGHT;
				this.paintThumbnail(saveCanvas, this.srcCanvas, 0);
				//ViewMarksData.setItemThumbnail(itemId, canvas.toDataURL(), canvas.width, canvas.height);
				ViewMarksData.setItemThumbnail(itemId, saveCanvas.toDataURL(), saveCanvas.width, saveCanvas.height);
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000);
			}
	
			if (this.thumbnail_action == this.TNF_DELETE) {
				ViewMarksData.deleteItemThumbnail(itemId);
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
			}
	
			if (this.thumbnail_action == this.TNF_MIME) {
				ViewMarksData.setItemThumbnail(itemId, this.mime_type, ViewMarksData.TN_DEFAULT_WIDTH, ViewMarksData.TN_DEFAULT_HEIGHT);
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
			}
	
			return true;
		} catch (e) {
			ViewMarksUtil.logerror(e);
			ViewMarksReport.openDialog(e);
			return true;
		}
	},


	onHelp: function() 
	{
	},


	createFolderListItem: function(label, value, depth)
	{
		var imgurl = ViewMarksData.getFolderIconUrl(value);

		var item = document.createElement("menuitem");
		item.setAttribute("label", label);
		item.setAttribute("value", value);
		item.setAttribute("image", imgurl);
		item.setAttribute("class", "menuitem-iconic bookmark-item");
		item.setAttribute("style", "padding-left:"+(depth*10)+"px");

		var image = document.createElement("image");		
		image.setAttribute("src", imgurl);
		item.appendChild(image);

		var nlabel = document.createElement("label");
		nlabel.setAttribute("value", label);
		item.appendChild(nlabel);
		
		return item;
	},


	createFolderList: function(aItemId, popup, depth)
	{
		ViewMarksData.enumFolderItems(aItemId, function(node) {
			let type = ViewMarksData.getItemType(node.itemId);
			if (type == ViewMarksData.bookmarks.TYPE_FOLDER) {
				var item = ViewMarksEdit.createFolderListItem(node.title, node.itemId, depth);
				popup.appendChild(item);
				ViewMarksEdit.createFolderList(node.itemId, popup, depth+1);
			}
		});
	},


	onQuickSave: function()
	{
		let button = document.getElementById("vmks_toggle_quicksave");
		let extpane = document.getElementById("vmks_edit_quicksave");
		if (this.quicksave == true) {
			extpane.setAttribute("collapsed", "true");
			button.setAttribute("src", "chrome://viewmarks/skin/quicksave_open.png");
			button.setAttribute("tooltiptext", ViewMarksUtil.getString("tooltip_edit_open_quicksave"));
			window.sizeToContent();
			this.quicksave = false;
		} else {
			//if (extpane.getAttribute("collapsed") == true) {
				extpane.removeAttribute("collapsed");
				button.setAttribute("src", "chrome://viewmarks/skin/quicksave_close.png");
				button.setAttribute("tooltiptext", ViewMarksUtil.getString("tooltip_edit_close_quicksave"));
				window.sizeToContent();
			//}
			this.quicksave = true;
		}
	},


	onExtend: function()
	{
		let button = document.getElementById("vmks_toggle_extend");
		let extpane = document.getElementById("vmks_edit_extended");
		if (this.extended == true) {
			extpane.setAttribute("collapsed", "true");
			button.setAttribute("src", "chrome://viewmarks/skin/extended_open.png");
			button.setAttribute("tooltiptext", ViewMarksUtil.getString("tooltip_edit_extend"));
			window.sizeToContent();
			this.extended = false;
		} else {
			//if (extpane.getAttribute("collapsed") == true) {
				extpane.removeAttribute("collapsed");
				button.setAttribute("src", "chrome://viewmarks/skin/extended_close.png");
				button.setAttribute("tooltiptext", ViewMarksUtil.getString("tooltip_edit_collapse"));
				window.sizeToContent();
			//}
			this.extended = true;
		}
	},

	
	loadIcon: function(canvas, name) {
	  let context = canvas.getContext("2d");
		let img = new Image();
		img.onload = function() {
			var height = ViewMarksEdit.TN_EDIT_WIDTH*img.height/img.width;
			ViewMarksEdit.resizeCanvas(canvas, ViewMarksEdit.TN_EDIT_WIDTH, ViewMarksEdit.TN_EDIT_HEIGHT);
			context.drawImage(img, 0, 0, ViewMarksEdit.TN_EDIT_WIDTH, height);
		}
		img.src = "chrome://viewmarks/skin/docs/"+name;
	},


	drawDataURL: function(canvas, data) {
	  let context = canvas.getContext("2d");
		let img = new Image();
		img.onload = function() {
			var height = ViewMarksEdit.TN_EDIT_WIDTH*img.height/img.width;
			ViewMarksEdit.resizeCanvas(canvas, ViewMarksEdit.TN_EDIT_WIDTH, height);
			context.drawImage(img, 0, 0, ViewMarksEdit.TN_EDIT_WIDTH, height);
		}
		if (data == "" || data.substr(0, 5) == "data:") {						
			img.src = data;
		} else {
			img.src = "chrome://viewmarks/skin/docs/doc_"+data.replace("/", "_")+".png";
		}
	},


	getThumbnailFromDataURL: function(data)
	{
		this.srcCanvas = document.createElementNS("http://www.w3.org/1999/xhtml", "html:canvas");
		let canvas = this.srcCanvas;
	  let context = canvas.getContext("2d");
		let img = new Image();
		img.onload = function() {
			let height = ViewMarksData.TN_DEFAULT_WIDTH*img.height/img.width;
			ViewMarksEdit.resizeCanvas(canvas, ViewMarksData.TN_DEFAULT_WIDTH, height);
			canvas.sx = canvas.sy = 0;
			canvas.sw = canvas.width;
			canvas.sh = canvas.height;
			context.drawImage(img, 0, 0, ViewMarksData.TN_DEFAULT_WIDTH, height);
		}
		img.src = data;
	},


	getThumbnail: function(win, dstCanvas, timeout)
	{
		this.srcCanvas = document.createElementNS("http://www.w3.org/1999/xhtml", "html:canvas");
		let doCrop = (timeout == 0 ? ViewMarksGlobal.nativeDetectBorders : true);
		if (ViewMarksCapture.snapshot(this.srcCanvas, win, doCrop) == false) {
			this.srcCanvas = null;
			return false;
		}
		this.paintThumbnail(dstCanvas, this.srcCanvas, timeout);
		return true;
	},


	onNewFolderClick: function()
	{
		var button = document.getElementById("vmks_edit_subfolder_button");
		var input = document.getElementById("vmks_edit_subfolder");
		if (input.getAttribute("collapsed") == "true") {
			input.removeAttribute("collapsed");
			input.value = ViewMarksUtil.getString("new_subfolder");
			input.select();
			button.setAttribute("src", "chrome://viewmarks/skin/nofolder.png");
			button.setAttribute("tooltiptext", ViewMarksUtil.getString("no_subfolder_tooltip"));
		} else {
			input.setAttribute("collapsed", true);
			input.value = "";
			button.setAttribute("src", "chrome://viewmarks/skin/newfolder.png");
			button.setAttribute("tooltiptext", ViewMarksUtil.getString("new_subfolder_tooltip"));
		}
		window.sizeToContent();
	},


	onReloadThumbnail: function(event) 
	{
		if (event.target.disabled) return;
		var canvas = document.getElementById("vmks_edit_canvas");
		if (this.getThumbnail(window.arguments[0].win, canvas, 10) /*&& canvas.written*/) {
			this.thumbnail_action = this.TNF_SAVE;
			this.updateZoomButtons();
			ViewMarksUtil.enableNode("vmks_delete_thumbnail", true);
		} else {
			let type = ViewMarksUtil.getDocumentMimeType(window.arguments[0].win.document);
			if (type) {
				this.loadIcon(canvas, "doc_"+type.replace("/", "_")+".png");
				this.mime_type = type;
				this.thumbnail_action = this.TNF_MIME;
				ViewMarksUtil.enableNode("vmks_delete_thumbnail", true);
			}
		}
	},


	onDeleteThumbnail: function(event) 
	{
		if (event.target.disabled) return;
		var canvas = document.getElementById("vmks_edit_canvas");
		canvas.height = canvas.height;
		this.thumbnail_action = this.TNF_DELETE;
		this.srcCanvas = null;
		this.loadIcon(canvas, "generic.png");
		this.updateZoomButtons();
		ViewMarksUtil.enableNode("vmks_delete_thumbnail", false);
	},


	updateZoomButtons: function()
	{
		if (this.srcCanvas == null) {
			ViewMarksUtil.enableNode("vmks_zoomin_thumbnail", false);
			ViewMarksUtil.enableNode("vmks_zoomout_thumbnail", false);
			this.dstCanvas.setAttribute("editable", false);		
		} else {
			ViewMarksUtil.enableNode("vmks_zoomin_thumbnail", this.srcCanvas.sw > ViewMarksData.TN_DEFAULT_WIDTH);
			ViewMarksUtil.enableNode("vmks_zoomout_thumbnail", this.srcCanvas.sw < this.srcCanvas.width);
			this.dstCanvas.setAttribute("editable", (this.srcCanvas.sw < this.srcCanvas.width) 
				|| (this.srcCanvas.sh < this.srcCanvas.height));		
		}
	},


	limitCanvasPosition: function()
	{
		if (this.srcCanvas.sx < 0) this.srcCanvas.sx = 0;
		if (this.srcCanvas.sx + this.srcCanvas.sw > this.srcCanvas.width) {
			this.srcCanvas.sx = this.srcCanvas.width - this.srcCanvas.sw;
		}
		if (this.srcCanvas.sy < 0) this.srcCanvas.sy = 0;
		if (this.srcCanvas.sy + this.srcCanvas.sh > this.srcCanvas.height) {
			this.srcCanvas.sy = this.srcCanvas.height - this.srcCanvas.sh;
		}
	},


	onLoadImage: function() {
		var nsIFilePicker = Components.interfaces.nsIFilePicker;
	  var fp = Components.classes["@mozilla.org/filepicker;1"].createInstance(nsIFilePicker);
	  fp.init(window, ViewMarksUtil.getString("load_image"), nsIFilePicker.modeOpen);
	  fp.appendFilters(nsIFilePicker.filterImages);
		var last_load_dir = ViewMarksPref.getCharPref("last_load_dir");
		if (last_load_dir != "") {
		  var localFile = Components.classes["@mozilla.org/file/local;1"].createInstance(Components.interfaces.nsILocalFile);
			localFile.initWithPath(ViewMarksPref.getCharPref("last_load_dir"));
	  	fp.displayDirectory = localFile;
	  }
	  var res = fp.show();
	  if (res == nsIFilePicker.returnOK) {
			try {ViewMarksPref.prefs.setCharPref("last_load_dir", fp.file.parent.target);} catch(e){}
			var ios = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);  
			var URL = ios.newFileURI(fp.file);  
			var img = new Image();
			img.onload = function() {
				ViewMarksEdit.srcCanvas = document.createElementNS("http://www.w3.org/1999/xhtml", "html:canvas");
				var canvas = ViewMarksEdit.srcCanvas;
				canvas.width = img.width;
				canvas.height = img.height;
			  var context = canvas.getContext("2d");
				context.drawImage(img, 0, 0);
				canvas.sx = canvas.sy = 0;
				canvas.sw = canvas.width;
				canvas.sh = canvas.height;
				if (canvas.sh > ViewMarksData.TN_DEFAULT_HEIGHT*canvas.sw/ViewMarksData.TN_DEFAULT_WIDTH) {
					canvas.sh = Math.round(ViewMarksData.TN_DEFAULT_HEIGHT*canvas.sw/ViewMarksData.TN_DEFAULT_WIDTH);
				}
				ViewMarksEdit.paintThumbnail(ViewMarksEdit.dstCanvas, canvas, 10);
				ViewMarksEdit.thumbnail_action = ViewMarksEdit.TNF_SAVE;
				ViewMarksEdit.updateZoomButtons();
				ViewMarksUtil.enableNode("vmks_delete_thumbnail", true);
			}
			img.src = URL.spec;
	  }  
	},


	onZoominThumbnail: function(event)
	{
		if (event.target.disabled) return;
		if (this.srcCanvas.sw == ViewMarksData.TN_DEFAULT_WIDTH) return;
		var cx = this.srcCanvas.sx + Math.round(this.srcCanvas.sw / 2);
		this.srcCanvas.sw = Math.round(0.8 * this.srcCanvas.sw);
		if (this.srcCanvas.sw <= ViewMarksData.TN_DEFAULT_WIDTH) this.srcCanvas.sw = ViewMarksData.TN_DEFAULT_WIDTH;
		this.srcCanvas.sx = cx - Math.round(this.srcCanvas.sw / 2);
		var cy = this.srcCanvas.sy + Math.round(this.srcCanvas.sh / 2);
		this.srcCanvas.sh = Math.round(0.8 * this.srcCanvas.sh);
		if (this.srcCanvas.sh < ViewMarksData.TN_DEFAULT_HEIGHT) this.srcCanvas.sh = ViewMarksData.TN_DEFAULT_HEIGHT;
		var height = Math.round(this.srcCanvas.sw * ViewMarksData.TN_DEFAULT_HEIGHT / ViewMarksData.TN_DEFAULT_WIDTH);
		if (this.srcCanvas.sh < height) this.srcCanvas.sh = height;
		this.srcCanvas.sy = cy - Math.round(this.srcCanvas.sh / 2);
		ViewMarksEdit.paintThumbnail(this.dstCanvas, this.srcCanvas, 100);
		this.updateZoomButtons();
	},


	onZoomoutThumbnail: function(event)
	{
		if (event.target.disabled) return;
		if (this.srcCanvas.sw == this.srcCanvas.width) return;
		var cx = this.srcCanvas.sx + Math.round(this.srcCanvas.sw / 2);
		this.srcCanvas.sw = Math.round(1.25 * this.srcCanvas.sw);
		if (this.srcCanvas.sw >= this.srcCanvas.width) this.srcCanvas.sw = this.srcCanvas.width;
		this.srcCanvas.sx = cx - Math.round(this.srcCanvas.sw / 2);
		var cy = this.srcCanvas.sy + Math.round(this.srcCanvas.sh / 2);
		this.srcCanvas.sh = Math.round(1.25 * this.srcCanvas.sh);
		if (this.srcCanvas.sh > this.srcCanvas.height) this.srcCanvas.sh = this.srcCanvas.height;
		this.srcCanvas.sy = cy - Math.round(this.srcCanvas.sh / 2);
		this.limitCanvasPosition();
		ViewMarksEdit.paintThumbnail(this.dstCanvas, this.srcCanvas, 100);
		this.updateZoomButtons();
	},


	onDragStart: function(event)
	{
		if (ViewMarksEdit.srcCanvas.sw == ViewMarksEdit.srcCanvas.width && 
			ViewMarksEdit.srcCanvas.sh == ViewMarksEdit.srcCanvas.height) return;
		event.dataTransfer.setData("preview/viewmark", true);
		event.dataTransfer.setDragImage(document.getElementById("vmks_edit_drag_image"), 0, 0);
		event.dataTransfer.effectAllowed = "all";
		event.target.x = event.screenX;
		event.target.y = event.screenY;
	},
	

	onDragEnter: function(event)
	{
		if (event.dataTransfer.getData("preview/viewmark") == "") return;
		event.preventDefault();
	},


	onDragOver: function(event)
	{
		try {
		var dx = event.screenX - event.target.x;
		var dy = event.screenY - event.target.y;
		if (dx != 0 || dy != 0) {
			//ViewMarksUtil.logmsg("move "+dx+" "+dy);
			var scale = ViewMarksEdit.srcCanvas.sw / ViewMarksEdit.TN_EDIT_WIDTH;
			ViewMarksEdit.srcCanvas.sx -= Math.round(scale*dx);
			ViewMarksEdit.srcCanvas.sy -= Math.round(scale*dy);
			event.target.x = event.screenX;
			event.target.y = event.screenY;
			ViewMarksEdit.limitCanvasPosition();
			ViewMarksEdit.paintThumbnail(ViewMarksEdit.dstCanvas, ViewMarksEdit.srcCanvas, 1000);
		}
		event.preventDefault();
		} catch(e) {ViewMarksUtil.alert(e);}
	},

		
	onDrop: function(event)
	{
		if (event.dataTransfer.getData("preview/viewmark") == "") return;
		ViewMarksEdit.paintThumbnail(ViewMarksEdit.dstCanvas, ViewMarksEdit.srcCanvas, 10);
		event.preventDefault();
	},


	onQuickButton: function(itemId)
	{
		var node = document.getElementById("vmks_dir_list");
		var popup = document.getElementById("vmks_dir_popup");

		for (var i=0; i<popup.childNodes.length; i++) {
			var id = popup.childNodes[i].getAttribute("value");
			if (id == itemId) {
				node.selectedIndex = i;
				if (ViewMarksPref.instant_quicksave) {
					document.getElementById("dlgViewMarksEdit").acceptDialog();
				}
				break;
			}
		}
	},


	popupEnsureSelectedItemVisible: function(id)
	{
	/*
		var node = document.getElementById(id);
		ViewMarksUtil.logmsg("item "+(100*node.selectedIndex/node.itemCount));
		node.scrollTop = 50;
		node.scrollBy(0, 50);
		ViewMarksUtil.logmsg("scroll "+node.scrollTop);
	*/
	},


	setDirListTooltip: function() 
	{
		let node = document.getElementById("vmks_dir_list");
		let id = node.selectedItem.getAttribute("value");
		let title = ViewMarksData.getItemTitle(id);
		let path = ViewMarksData.getItemPath(id);
		let tooltip = (path.length > 0 ? path + " » " : "") + title;
		node.setAttribute("tooltiptext", tooltip);
	},


	updateTagsPopup: function()
	{
		var tagsPopup = document.getElementById("vmks_edit_tags_popup");
		for (var i=0; i<tagsPopup.childNodes.length; i++) {
			var node = tagsPopup.childNodes[i];
			node.removeAttribute("collapsed");
			var label = node.label.toLowerCase();
			for (var j=0; j<this.tagsList.length; j++) {
				if (this.tagsList[j].toLowerCase() == label) {
					node.setAttribute("collapsed", "true");
					break;
				}
			}
		}
	},


	updateTags: function()
	{
		var editTags = document.getElementById("vmks_edit_tags");
		while (editTags.childNodes.length) {
			editTags.removeChild(editTags.firstChild);
		}
		for (var i=0; i<this.tagsList.length; i++) {
			var span = document.createElement("span");
			span.setAttribute("class", "vmks_tag");
			span.setAttribute("tooltiptext", ViewMarksUtil.getString("remove_tag")+" '"+this.tagsList[i]+"'");
			span.setAttribute("onclick", "ViewMarksEdit.removeTag('"+escape(this.tagsList[i])+"')");
			var text = document.createTextNode(this.tagsList[i]);
			span.appendChild(text);
			editTags.appendChild(span);
		}
	},


	addTag: function(tagsInput)
	{
		if (tagsInput === "undefined" || tagsInput == null || tagsInput.length == 0) {
			var input = document.getElementById("vmks_edit_tags_list");
			tagsInput = input.value;	
			if (tagsInput.length == 0) return;
			input.value = "";
		}
		tagsInput = tagsInput.split(",");
		for (var i=0; i<tagsInput.length; i++) {
			tagsInput[i] = tagsInput[i].trim();
			//tagsInput[i] = ViewMarksUtil.removeChars(tagsInput[i], '<>&"');
		}
		for (var i=0; i<this.tagsList.length; i++) {
			for (var j=0; j<tagsInput.length; j++) {
				if (this.tagsList[i].toLowerCase() == tagsInput[j].toLowerCase()) {
					if (this.tagsList[i] != tagsInput[j]) {
						this.tagsList[i] = tagsInput[j];
						tagsInput[j] = "";
						this.tagsChanged = true;
					}
				}
			}
		}
		for (var i=0; i<tagsInput.length; i++) {
			if (tagsInput[i].length) {
				this.tagsList.push(tagsInput[i]);
				this.tagsChanged = true;
			}
		}
		this.updateTags();
		this.updateTagsPopup();
	},


	removeTag: function(tag)
	{
		tag = unescape(tag);
		var index = this.tagsList.indexOf(tag);
		if (index < 0) return;
		this.tagsList.splice(index, 1);
		this.updateTags();
		this.updateTagsPopup();
		this.tagsChanged = true;
	},


	onTagKeyDown: function(event)
	{
		if (event.keyCode != 13) return;
		event.preventDefault();
		ViewMarksEdit.addTag();
	},
	

	init: function() 
	{
		try {
			ViewMarksPref.init();
			let bookmark_exists = false;
			let win = window.arguments[0].win;
			let description = document.getElementById("vmks_edit_description");
			let keyword = document.getElementById("vmks_edit_keyword");

			let winhref = (win.location ? win.location.href : "");
			
			let dstCanvas = document.createElementNS("http://www.w3.org/1999/xhtml","html:canvas");
			this.dstCanvas = dstCanvas;
			dstCanvas.setAttribute("id", "vmks_edit_canvas");
			document.getElementById("vmks_edit_canvas_parent").appendChild(dstCanvas);
			this.resizeCanvas(dstCanvas, this.TN_EDIT_WIDTH, this.TN_EDIT_HEIGHT);
	
			dstCanvas.addEventListener("dragstart", ViewMarksEdit.onDragStart, true);
			dstCanvas.addEventListener("dragenter", ViewMarksEdit.onDragEnter, true);
			dstCanvas.addEventListener("dragover", ViewMarksEdit.onDragOver, true);
			dstCanvas.addEventListener("drop", ViewMarksEdit.onDrop, true);
	
			let favicon = document.getElementById("vmks_edit_favicon");
			let dialog_title = document.getElementById("dlgViewMarksEdit");
			let parentId = this.bookmarks.bookmarksMenuFolder;
	
			this.action = window.arguments[0].action;
	
			if (this.action == this.DF_CREATE_BOOKMARK) {
				var itemId = ViewMarksData.getBookmarkByURL(winhref);
				if (itemId > 0) {
					window.arguments[0].itemId = itemId;
					this.action = this.DF_EDIT_BOOKMARK;
					bookmark_exists = true;
				}
			}
	
			if (this.action == this.DF_CREATE_BOOKMARK) {
				this.extended = !ViewMarksPref.last_edit_extended;
				this.onExtend();
				this.quicksave = !ViewMarksPref.last_edit_quicksave;
				this.onQuickSave();
				document.getElementById("vmks_edit_name").value = win.document.title;
				document.getElementById("vmks_edit_url").value = winhref;
				this.tagsUrl = winhref;
				if (ViewMarksPref.fetch_description) description.value = ViewMarksData.getPageMeta(win, "description");
				if (ViewMarksPref.fetch_keywords) keyword.value = ViewMarksData.getPageMeta(win, "keywords");
				document.getElementById("vmks_edit_added").value = (new Date()).toLocaleString();
				document.getElementById("vmks_edit_modified").value = "-";
				if (win.document.body === undefined) ViewMarksUtil.enableNode("vmks_reload_thumbnail", false);
				dialog_title.setAttribute("title", ViewMarksUtil.getString("title_create_bookmark"));
				ViewMarksData.getFavicon(winhref, 
					function(aURI) {
						favicon.setAttribute("style", "list-style-image:url("+aURI.spec+")");				
						}
					);
				parentId = this.bookmarks.bookmarksMenuFolder;
				if (win.thumbnail) {
					this.getThumbnailFromDataURL(win.thumbnail);
					this.drawDataURL(dstCanvas, win.thumbnail); 
					this.thumbnail_action = this.TNF_SAVE;
				} else if (win.document.body === undefined) {
					this.loadIcon(dstCanvas, "generic.png");
				} else {
					delete dstCanvas.written;
					if (this.getThumbnail(win, dstCanvas, 10) /*&& dstCanvas.written*/) {
						this.thumbnail_action = this.TNF_SAVE;
					} else {
						let type = ViewMarksUtil.getDocumentMimeType(win.document);
						if (type) {
							this.loadIcon(dstCanvas, "doc_"+type.replace("/", "_")+".png");
							this.mime_type = type;
							this.thumbnail_action = this.TNF_MIME;
							// why zoom buttons?
						}
					}
				}
			}
	
			if (this.action == this.DF_EDIT_BOOKMARK) {
				this.extended = !ViewMarksPref.last_edit_extended;
				this.onExtend();
				this.quicksave = !ViewMarksPref.last_edit_quicksave;
				this.onQuickSave();
				var itemId = window.arguments[0].itemId;
				document.getElementById("vmks_edit_name").value = ViewMarksData.getItemTitle(itemId);
				var href = this.bookmarks.getBookmarkURI(itemId).spec;
				document.getElementById("vmks_edit_url").value = href;
				this.tagsUrl = href;
				description.value = ViewMarksData.getItemDescription(itemId);
				keyword.value = ViewMarksData.getItemKeywords(itemId);
				var date_added = ViewMarksData.getItemDateAdded(itemId);
				document.getElementById("vmks_edit_added").value = date_added;
				var date_modified = ViewMarksData.getItemLastModified(itemId);
				if (date_added == date_modified) {
					document.getElementById("vmks_edit_modified").value = "-";
				} else {
					document.getElementById("vmks_edit_modified").value = date_modified;
				}
				if (bookmark_exists == false) ViewMarksUtil.enableNode("vmks_reload_thumbnail", false);
				dialog_title.setAttribute("title", ViewMarksUtil.getString("title_edit_bookmark"));
				ViewMarksData.getFavicon(href, 
					function(aURI) {
						favicon.setAttribute("style", "list-style-image:url("+aURI.spec+")");				
						}
					);
				parentId = this.bookmarks.getFolderIdForItem(itemId);
				var thumbnail = ViewMarksData.getItemThumbnail(itemId);
				if (thumbnail) {
					this.drawDataURL(dstCanvas, thumbnail.data);
					this.thumbnail_action = this.TNF_PRESENT;
				} else {
					if (!bookmark_exists || win.document.body === undefined) {
						this.loadIcon(dstCanvas, "generic.png");
					} else {
						delete dstCanvas.written;
						if (this.getThumbnail(win, dstCanvas, 10) /*&& dstCanvas.written*/) {
							this.thumbnail_action = this.TNF_SAVE;
						} else {
							this.loadIcon(dstCanvas, "generic.png");
						}
					}
				}
			}
	
			if (this.action == this.DF_CREATE_FOLDER) {
				this.extended = true;
				this.onExtend();
				this.quicksave = true;
				this.onQuickSave();
				favicon.setAttribute("style", "list-style-image:url(chrome://viewmarks/skin/folder.png)");
				document.getElementById("vmks_edit_name").value = ViewMarksUtil.getString("new_folder");
				document.getElementById("vmks_edit_tools").setAttribute("collapsed", "true");
				document.getElementById("vmks_edit_subfolder_button").setAttribute("disabled", "true");
				dialog_title.setAttribute("title", ViewMarksUtil.getString("title_create_folder"));
				parentId = window.arguments[0].itemId;
				this.loadIcon(dstCanvas, "folder.png");
			}
	
			if (this.action == this.DF_EDIT_FOLDER) {
				this.extended = true;
				this.onExtend();
				this.quicksave = true;
				this.onQuickSave();
				var itemId = window.arguments[0].itemId;
				document.getElementById("vmks_edit_name").value = ViewMarksData.getItemTitle(itemId);
				favicon.setAttribute("style", "list-style-image:url(chrome://viewmarks/skin/folder.png)");
				document.getElementById("vmks_edit_tools").setAttribute("collapsed", "true");
				dialog_title.setAttribute("title", ViewMarksUtil.getString("title_edit_folder"));
				parentId = this.bookmarks.getFolderIdForItem(itemId);
				var thumbnail = ViewMarksData.getItemThumbnail(itemId);
				if (thumbnail) {
					this.drawDataURL(dstCanvas, thumbnail.data);
					this.thumbnail_action = this.TNF_PRESENT;
				} else {
					this.loadIcon(dstCanvas, "folder.png");
				}
			}
	
			if (this.tagsUrl) {
				this.tagsList = ViewMarksData.getUrlTags(this.tagsUrl);
				this.tagsListAll = ViewMarksData.getAllTags();
				var tagsPopup = document.getElementById("vmks_edit_tags_popup");
				for (var i=0; i<this.tagsListAll.length; i++) {
					var item = document.createElement("menuitem");
					item.setAttribute("label", this.tagsListAll[i]);
					tagsPopup.appendChild(item);
				}
				this.updateTagsPopup();
				this.updateTags();
			}
	
	
			if (this.thumbnail_action == this.TNF_NONE) ViewMarksUtil.enableNode("vmks_delete_thumbnail", false);
	
			var popup = document.createElement("menupopup");
			popup.setAttribute("id", "vmks_dir_popup");
			popup.setAttribute("class", "menu-iconic bookmark-item");
	
			ViewMarksData.createFolderTree(popup, false);
	
			if (parentId == ViewMarksPref.trash_folder_id) {
				let item = ViewMarksData.createFolderListItem(ViewMarksUtil.getString("trash_folder"), ViewMarksPref.trash_folder_id, 0);
				popup.appendChild(item);
			}
	
			var node = document.getElementById("vmks_dir_list");
			var old = document.getElementById("vmks_dir_popup");
			node.replaceChild(popup, old);
	
			for (var i=0; i<popup.childNodes.length; i++) {
				var id = popup.childNodes[i].getAttribute("value");
				if (id == parentId) {
					node.selectedIndex = i;
					break;
				}
			}
	
			this.setDirListTooltip();
			popup.setAttribute("onpopuphidden", "ViewMarksEdit.setDirListTooltip()");
	
			this.resetChangeFlags();
	
			if (bookmark_exists) {
				if (ViewMarksPref.fetch_description && description.value == "") {
					description.value = ViewMarksData.getPageMeta(win, "description");
				}
				if (ViewMarksPref.fetch_keywords && keyword.value == "") {
					keyword.value = ViewMarksData.getPageMeta(win, "keywords");
				}
			}
	
			this.updateZoomButtons();
	
			var related = new Array();
	
			if (ViewMarksPref.show_related_folder) {
				var arrFilter = null;
				if (this.action == this.DF_CREATE_BOOKMARK) {
					arrFilter = [this.bookmarks.bookmarksMenuFolder];
				} else if (this.action == this.DF_EDIT_BOOKMARK) {
					arrFilter = [ViewMarksData.getParentFolderId(window.arguments[0].itemId)];
					var itemIdList = ViewMarksData.getBookmarkIdsForURL(winhref);
					for (var i=0; i<itemIdList.length; i++) {
						let pid = ViewMarksData.getParentFolderId(itemIdList[i]);
						if (arrFilter.indexOf(pid) < 0) arrFilter.push(pid);
					}
				}
				if (arrFilter) {
					related = ViewMarksModel.getRelatedFolderList(
						arrFilter,
						document.getElementById("vmks_edit_url").value,
						document.getElementById("vmks_edit_name").value,
						document.getElementById("vmks_edit_description").value + " " +
						document.getElementById("vmks_edit_keyword").value
					);
				} 
			}
	
			var relcount = 0;
			
			for (var i=0; i<4; i++) {
				var obj = document.getElementById("vmks_rel_btn" + (i+1));
				if (i<related.length) {
					obj.removeAttribute("collapsed");
					var title = ViewMarksData.getItemTitle(related[i]);
					if (title.length > 64) title = title.substr(0, 64) + "...";
					obj.childNodes[1].textContent = title;
					obj.childNodes[0].setAttribute("src", ViewMarksData.getFolderIconUrl(related[i]));
					obj.setAttribute("onclick", "ViewMarksEdit.onQuickButton("+Math.floor(related[i])+")");
					var path = ViewMarksData.getItemPath(related[i]);
					var tooltip = (path.length > 0 ? path + " » " : "") + title;
					obj.setAttribute("tooltiptext", tooltip);
					relcount++;
				} else {
					obj.setAttribute("collapsed", true);
				}
			}
	
			var label =	document.getElementById("vmks_quicksave_related");
			if (relcount > 0) {
				label.removeAttribute("collapsed");
			} else {
				label.setAttribute("collapsed", true);
				relcount = -1;
			}
		
			var mru = ViewMarksData.getRecentlyUsedFolder();
			for (var i=0; i<9; i++) {
				var obj = document.getElementById("vmks_mru_btn" + (i+1));
				if (i<mru.length && i<(8 - relcount)) {
					obj.removeAttribute("collapsed");
					var title = ViewMarksData.getItemTitle(mru[i]);
					if (title.length > 64) title = title.substr(0, 64) + "...";
					obj.childNodes[1].textContent = title;
					obj.childNodes[0].setAttribute("src", ViewMarksData.getFolderIconUrl(mru[i]));
					obj.setAttribute("onclick", "ViewMarksEdit.onQuickButton("+Math.floor(mru[i])+")");
					var path = ViewMarksData.getItemPath(mru[i]);
					var tooltip = (path.length > 0 ? path + " » " : "") + title;
					obj.setAttribute("tooltiptext", tooltip);
				} else {
					obj.setAttribute("collapsed", true);
				}
			}
	
			window.sizeToContent();

			this.dialogInitial = new this.dialogObject();			
		
		} catch (e) {
			ViewMarksUtil.logerror(e)
			ViewMarksReport.openDialog(e);
			document.getElementById("dlgViewMarksEdit").cancelDialog();
		}	
	},

	
	close: function()
	{
		if (this.action == this.DF_CREATE_BOOKMARK || this.action == this.DF_EDIT_BOOKMARK) {
			ViewMarksPref.prefs.setBoolPref("last_edit_extended", this.extended);
			ViewMarksPref.prefs.setBoolPref("last_edit_quicksave", this.quicksave);
		}
	},


	setQuickThumbnail: function(win, itemId)
	{
		if (win.document.body != undefined && ViewMarksData.getItemThumbnail(itemId) == null) {
			//ViewMarksUtil.logmsg("get thumbnail for "+win.location.href);
			//ViewMarksUtil.alert("get thumbnail for "+win.location.href);
			var canvas = document.createElementNS("http://www.w3.org/1999/xhtml","html:canvas");
		  canvas.width = ViewMarksData.TN_DEFAULT_WIDTH;
		  canvas.height = ViewMarksData.TN_DEFAULT_HEIGHT;
			if (this.getThumbnail(win, canvas, 0) && canvas.written) {
				ViewMarksData.setItemThumbnail(itemId, canvas.toDataURL(), canvas.width, canvas.height);
				this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
			} else {
				let type = ViewMarksUtil.getDocumentMimeType(win.document);
				if (type) {
					ViewMarksData.setItemThumbnail(itemId, type, ViewMarksData.TN_DEFAULT_WIDTH, ViewMarksData.TN_DEFAULT_HEIGHT);
					this.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
				}
			}
		}
	},


	setQuickBookmark: function(win, itemId, folderId)
	{
		if (itemId == null) {
			if (win.location == null) return;
			itemId = this.bookmarks.insertBookmark(folderId, ViewMarksUtil.getURIFromString(win.location.href), 
				this.bookmarks.DEFAULT_INDEX, win.document.title);
		}
		if (ViewMarksPref.fetch_description && ViewMarksData.getItemDescription(itemId) == "") {	
			var description = ViewMarksData.getPageMeta(win, "description");		  
			if (description != "") ViewMarksData.setItemDescription(itemId, description);
		}
		if (ViewMarksPref.fetch_keywords && ViewMarksData.getItemKeywords(itemId) == "") {
			var keywords = ViewMarksData.getPageMeta(win, "keywords");
			if (keywords != "") ViewMarksData.setItemKeywords(itemId, keywords);
		}
		//ViewMarksUtil.logmsg("itemId = "+itemId);	
		//ViewMarksUtil.logmsg("itemTN = "+ViewMarksData.getItemThumbnail(itemId));
		this.setQuickThumbnail(win, itemId);
	},


	updateBookmark: function(itemId)
	{
		var url = this.bookmarks.getBookmarkURI(itemId).spec;
		if (url.substr(0, 4) != "http") return;

		//var win = this.appShellService.hiddenDOMWindow;
		var win = this.appShellService.hiddenDOMWindow;

		var XUL_NS = "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul";
		var XHTML_NS = "http://www.w3.org/1999/xhtml";
		
    var frame = win.document.createElementNS(XUL_NS, "iframe");
    frame.setAttribute("type", "content");
		frame.width = 960;
		frame.height = 720;

		frame.addEventListener("load", function(event) {
			try {
			event.currentTarget.removeEventListener(event.type, arguments.callee, false);
			//ViewMarksUtil.alert(frame.contentDocument);
			if (ViewMarksPref.fetch_description && ViewMarksData.getItemDescription(itemId) == "") {
				var description = ViewMarksData.getPageMeta(win, "description");		  
				if (description != "") ViewMarksData.setItemDescription(itemId, description);
			}
			if (ViewMarksPref.fetch_keywords && ViewMarksData.getItemKeywords(itemId) == "") {
				var keywords = ViewMarksData.getPageMeta(win, "keywords");
				if (keywords != "") ViewMarksData.setItemKeywords(itemId, keywords);
			}
			if (frame.contentDocument && frame.contentDocument.body) {
				if (ViewMarksData.getItemThumbnail(itemId) == null) {
					var canvas = document.createElementNS("http://www.w3.org/1999/xhtml","html:canvas");
				  canvas.width = ViewMarksData.TN_DEFAULT_WIDTH;
				  canvas.height = ViewMarksData.TN_DEFAULT_HEIGHT;
					if (ViewMarksEdit.getThumbnail(frame.contentWindow, canvas, 0) && canvas.written) {
						ViewMarksData.setItemThumbnail(itemId, canvas.toDataURL(), canvas.width, canvas.height);
						ViewMarksEdit.bookmarks.setItemLastModified(itemId, (new Date()).getTime() * 1000)
					}
				}
			}
			} catch(e) {ViewMarksUtil.alert(e);}
		}, false);

		frame.setAttribute("src", url);
    win.document.documentElement.appendChild(frame);
	},

	
};



