/*
Creative Commons License: Attribution-No Derivative Works 3.0 Unported
http://creativecommons.org/licenses/by-nd/3.0/
(c) 2010-2013 Michael Koch
*/

Components.utils.import("resource://viewmarks/global.jsm"); 


var ViewMarksView = {

	win: null,
	view: null,

	VIEW_FOLDER: 1,
	VIEW_SEARCH: 2,
	viewType: 0,

	uri: "about:viewmarks",

	SF_TITLE: 1,
	SF_URL: 2,
	SF_KEYWORDS: 4,
	SF_DESCRIPTION: 8,
	SF_TAGS: 16,

	bookmarks: Components.classes["@mozilla.org/browser/nav-bookmarks-service;1"]
		.getService(Components.interfaces.nsINavBookmarksService),

	history: Components.classes["@mozilla.org/browser/nav-history-service;1"]
		.getService(Components.interfaces.nsINavHistoryService),

	zoomLevelList: [384, 352, 320, 288, 256, 224, 192, 160, 128, 96, 64, 48, 32],
	//zoomLevelList: [256, 224, 192, 160, 128, 96, 64, 48, 32],
	zoomLevel: 2,

	folderId: null,
	folderList: null,
	folderCount: 0,
	boxList: null,

	dragIndex: -1,
	dropIndex: 0,
	
	scrollTimer: null,

	load_all: false,

	RESIZE: 1,
	EXTEND: 2,
	REFRESH: 4,


	open: function()
	{
		ViewMarksUtil.openInNewTab("about:viewmarks", false);
		//content.document.location.href = "about:viewmarks";
		//content.document.location.href = ViewMarksPref.empty_tab_url;
	},
	
	
	getHistorySearchResultArray: function(patterns, flags)
	{
		var results = new Array();
		for (var j=0; j < patterns.length; j++) patterns[j] = patterns[j].toLowerCase();
		if (flags == 0) flags = this.SF_TITLE | this.SF_URL | this.SF_KEYWORDS | this.SF_DESCRIPTION;

		for (var i=0; i<ViewMarksGlobal.historyList.length; i++) {
			var node = ViewMarksGlobal.historyList[i];
			var count;
			if ((flags & this.SF_TITLE) && node.title) {
				var title = node.title.toLowerCase();
				for (var j=0, count=0; j<patterns.length; j++) {
					if (title.indexOf(patterns[j]) >= 0) count++;
				}
				if (count == patterns.length) {results.push(node); continue;}
			}
			if (flags & this.SF_URL) {
				var uri = node.uri.toLowerCase();
				for (var j=0, count=0; j<patterns.length; j++) {
					if (uri.indexOf(patterns[j]) >= 0) count++;
				}
				if (count == patterns.length) {results.push(node); continue;}
			}
			if (flags & this.SF_DESCRIPTION) {
				if (node.description && node.description.length) {
					var description = node.description.toLowerCase();
					for (var j=0, count=0; j<patterns.length; j++) {
						if (description.indexOf(patterns[j]) >= 0) count++;
					}
					if (count == patterns.length) {results.push(node); continue;}
				}
			}
			if (flags & this.SF_KEYWORDS) {
				if (node.keywords) {
					var keywords = node.keywords.toLowerCase();
					for (var j=0, count=0; j<patterns.length; j++) {
						if (keywords.indexOf(patterns[j]) >= 0) count++;
					}
					if (count == patterns.length) {results.push(node); continue;}
				}
			}
		}

		return results;
	},


	createSpecialFolderItem: function(itemId)
	{
		var item = {};
		item.itemId = itemId;
		item.itemType = ViewMarksData.getItemType(itemId);
		item.title = ViewMarksData.getItemTitle(itemId);
		return item;
	},


	getFolderArray: function(aItemId)
	{
		var array = new Array();

		if (aItemId == this.bookmarks.bookmarksMenuFolder) {
			ViewMarksData.refreshFolderCount();
			if (ViewMarksData.toolbarCount > 0) {
				array.push(this.createSpecialFolderItem(this.bookmarks.toolbarFolder));
			}
			if (ViewMarksData.unfiledCount > 0) {
				array.push(this.createSpecialFolderItem(this.bookmarks.unfiledBookmarksFolder));
			}
			if (ViewMarksPref.show_most_visited) array.push({
				itemId:ViewMarksData.mostVisitedFolder,
				itemType:this.bookmarks.TYPE_FOLDER,
				title: ViewMarksData.getItemTitle(ViewMarksData.mostVisitedFolder)
			});
			if (ViewMarksPref.show_last_visited) array.push({
				itemId:ViewMarksData.lastVisitedFolder,
				itemType:this.bookmarks.TYPE_FOLDER,
				title: ViewMarksData.getItemTitle(ViewMarksData.lastVisitedFolder)
			});
			if (ViewMarksPref.create_history) array.push({
				itemId:ViewMarksData.historyFolder,
				itemType:this.bookmarks.TYPE_FOLDER,
				title: ViewMarksData.getItemTitle(ViewMarksData.historyFolder)
			});
			if (ViewMarksPref.create_trash) array.push({
				itemId:ViewMarksPref.trash_folder_id,
				itemType:this.bookmarks.TYPE_FOLDER,
				title: ViewMarksData.getItemTitle(ViewMarksPref.trash_folder_id)
			}); 
		}

		if (aItemId == ViewMarksData.lastVisitedFolder) {
			var count = ViewMarksData.enumVisitedItems(ViewMarksData.SORT_DATE, ViewMarksPref.last_visited_count, 
				function(item){array.push(item);});
		} else if (aItemId == ViewMarksData.mostVisitedFolder) {
			var count = ViewMarksData.enumVisitedItems(ViewMarksData.SORT_VISIT, ViewMarksPref.most_visited_count, 
				function(item){array.push(item);});
		} else if (aItemId == ViewMarksData.historyFolder) {
			array = ViewMarksGlobal.historyList;
		} else {
			var count = ViewMarksData.enumFolderItems(aItemId, function(item) {
				var itemType = ViewMarksData.getItemType(item.itemId);
				var entry = {
					itemId:item.itemId, 
					itemType:itemType, 
					title:item.title, 
					uri:item.uri,
					accessCount:item.accessCount,
					time:item.time
				};
				array.push(entry);
			});
			//document.getElementById("vmks_sortbutton").removeAttribute("collapsed");
			ViewMarksData.sortBookmarkArray(array, this.sortOrder);
		}
		
		ViewMarksUtil.logmsg("ViewMarks: read folder "+aItemId+" with "+array.length+" items");
		return array;
	},


	swapChildNodes: function(node1, node2)
	{
		for (var i=0; i<node1.childNodes.length; i++) {
			var tmp1 = node1.childNodes[i].cloneNode(true);
			var tmp2 = node2.childNodes[i].cloneNode(true);
			node1.replaceChild(tmp2, node1.childNodes[i]);
			node2.replaceChild(tmp1, node2.childNodes[i]);
		}
	},


	updateDropView: function(mouseIndex)
	{
		if (mouseIndex == this.dropIndex) return;


		var delta = 1;
		if (mouseIndex < this.dropIndex) delta = -1;
		var index = this.dropIndex;
		while (index != mouseIndex) {
			this.swapChildNodes(this.boxList[index], this.boxList[index+delta]);
			index += delta;
		}
		this.dropIndex = mouseIndex;
	},


	createDragPreview: function(node)
	{
		let dup = node.cloneNode(true);
		dup.height = node.boxObject.height;
		document.getElementById("vmks_dragged").appendChild(dup);

		node.childNodes[1].setAttribute("src", "chrome://viewmarks/skin/docs/drop_target.png");
		let imgwidth = this.zoomLevelList[4+this.zoomLevel];
		let imgheight = (imgwidth * 3) / 4;
		node.childNodes[1].setAttribute("style", "width:"+imgwidth+"px;height:"+imgheight+"px");
		node.childNodes[1].removeAttribute("class");
		node.childNodes[3].setAttribute("value", "");

/*
		while (node.childNodes.length > 0) node.removeChild(node.firstChild);

		var spacer = document.createElement("vspacer");
		spacer.setAttribute("flex", "1");
		node.appendChild(spacer);

		var img = document.createElement("image");
		img.setAttribute("src", "chrome://viewmarks/skin/docs/drop_target.png");
		var imgwidth = this.zoomLevelList[4+this.zoomLevel];
		var imgheight = (imgwidth * 3) / 4;
		img.setAttribute("style", "width:"+imgwidth+"px;height:"+imgheight+"px");
		node.appendChild(img);

		spacer = document.createElement("vspacer");
		spacer.setAttribute("flex", "1");
		node.appendChild(spacer);

		var label = document.createElement("label");
		label.setAttribute("value", "");
		node.appendChild(label);
*/
		return dup;
	},


	addDragDropEventHandler: function(box, entry, index) 
	{
		var scrollRim = 32;
		var scrollSpeed = 0.9;
		if (ViewMarksPref.xulAppInfo.version > "4.0") scrollSpeed = 0.5;
	
		box.index = index;
		box.itemId = entry.itemId;
		if (entry.itemType == this.bookmarks.TYPE_BOOKMARK) box.uri = entry.uri;
		box.addEventListener("dragstart", function(event) {
				//ViewMarksUtil.logmsg("dragstart");
				if (this.childNodes.length > 4) {
					this.removeChild(this.childNodes[0]);
				}
				event.dataTransfer.setData("image/viewmark", this.itemId);
				if (this.uri) {
					event.dataTransfer.setData("text/uri-list", this.uri);
					event.dataTransfer.setData("text/plain", this.uri);
				}
				event.dataTransfer.effectAllowed = "copyMove";
				ViewMarksView.dragIndex = this.index;
				ViewMarksView.dropIndex = this.index;
				let relX = event.screenX - this.boxObject.screenX - 6;
				let relY = event.screenY - this.boxObject.screenY - 8;
				let preview = ViewMarksView.createDragPreview(this);
				event.dataTransfer.setDragImage(preview, relX, relY);
			}, true);
		box.addEventListener("dragend", function(event) {
				//ViewMarksUtil.logmsg("dragend "+event.dataTransfer.dropEffect);
				if (event.dataTransfer.dropEffect != "move") ViewMarksView.refreshGrid();
				ViewMarksView.dragIndex = -1;
			}, false, true);
		box.addEventListener("dragenter", function(event) {
				//ViewMarksUtil.logmsg("dragenter");
				if (event.dataTransfer.types.contains("image/viewmark") && ViewMarksView.dragIndex >= 0) {
					ViewMarksView.updateDropView(this.index);
					event.preventDefault();
				}
			}, true);
		box.addEventListener("dragover", function(event) {
				//ViewMarksUtil.logmsg("dragover");
				if (event.dataTransfer.types.contains("image/viewmark") && ViewMarksView.dragIndex >= 0) {
					ViewMarksView.updateDropView(this.index);
					event.preventDefault();
				}
				if (event.pageY - ViewMarksView.headerHeight < scrollRim) {
					ViewMarksView.scrollBy(scrollSpeed * ((event.pageY - ViewMarksView.headerHeight) - scrollRim));
				}
				if (window.innerHeight - event.pageY < scrollRim)  {
					ViewMarksView.scrollBy(scrollSpeed * (event.pageY - window.innerHeight + scrollRim));
				}
			}, true);
		box.addEventListener("drop", function(event) {
				//ViewMarksUtil.logmsg("drop");
				var data = event.dataTransfer.getData("image/viewmark");
				if (data.length > 0) {
					var itemId = parseInt(data);
					var srcIndex = ViewMarksView.bookmarks.getItemIndex(itemId);
					var dstIndex = ViewMarksView.bookmarks.getItemIndex(this.itemId); 
					if (dstIndex != srcIndex) {
						if (dstIndex > srcIndex) dstIndex++;
						ViewMarksUtil.logmsg("ViewMarks: move item "+itemId+" from index "+srcIndex+" to "+dstIndex);
						ViewMarksView.bookmarks.moveItem(itemId, ViewMarksView.folderId, dstIndex);
					}
					ViewMarksView.refreshGrid();
					ViewMarksView.dragIndex = -1;
					event.preventDefault();
				}
			}, true);
	},


	createRatingBar: function(itemId)
	{
		let bar = document.createElement("hbox");	
		bar.setAttribute("class", "vmks_rate_bar");
		bar.rating = ViewMarksData.getItemAnnotation(itemId, "bookmarkProperties/vmksRating");
		if (bar.rating == null || bar.rating.length == 0) bar.rating = 0;
		bar.itemId = itemId;
		bar.style.backgroundPosition="16px -"+(bar.rating*16)+"px";
		bar.setAttribute("onmouseout", "this.style.backgroundPosition='16px -'+(this.rating*16)+'px'");
		for (var i=0; i<=5; i++) {
			let node = document.createElement("box");
			node.rating = i;
			node.setAttribute("class", (i == 0 ? "vmks_rate_box_none" : "vmks_rate_box"));
			node.setAttribute("onmouseover", "this.parentNode.style.backgroundPosition='16px -"+(i*16)+"px'");
			node.setAttribute("tooltiptext", (i == 0 ? ViewMarksUtil.getString("rating_none") : ViewMarksUtil.getString("rating_x") + " " + i));
			node.addEventListener("click", function(event) {
				event.preventDefault();
				event.stopPropagation();
				node.parentNode.rating = node.rating;
				ViewMarksData.setItemAnnotation(node.parentNode.itemId, "bookmarkProperties/vmksRating", node.rating);
			}, false);
			bar.appendChild(node);
		}	
		return bar;						
	},



	addMouseOverEventHandler: function(box, entry, index) 
	{
		box.addEventListener("mouseover", function(event) {
			let iconsize = 16;
			let target = event.currentTarget;
			if (target.childNodes.length > 4) return;
			//ViewMarksUtil.logmsg("mouseover:"+target.tagName+" childs:"+target.childNodes.length);

			let box = document.createElement("hbox");
			box.setAttribute("class", "vmks_overlay");

			let node = document.createElement("hbox");
			node.setAttribute("class", "vmks_overlay_box");

			let image = document.createElement("image");
			image.setAttribute("src", "chrome://viewmarks/skin/overlay_edit_"+iconsize+".png");
			image.setAttribute("class", "vmks_overlay_button");
			if (target.itemType == ViewMarksView.bookmarks.TYPE_FOLDER) {
				image.addEventListener("click", function(event) {
					event.preventDefault();
					event.stopPropagation();
					ViewMarksEdit.open(ViewMarksEdit.DF_EDIT_FOLDER,Math.floor(target.itemId));
					target.removeChild(target.childNodes[0]);
				}, false);
				image.setAttribute("tooltiptext", ViewMarksUtil.getString("edit_folder"));
			}
			if (target.itemType == ViewMarksView.bookmarks.TYPE_BOOKMARK) {
				image.addEventListener("click", function(event) {
					event.preventDefault();
					event.stopPropagation();
					ViewMarksEdit.open(ViewMarksEdit.DF_EDIT_BOOKMARK,Math.floor(target.itemId));
					target.removeChild(target.childNodes[0]);
				}, false);
				image.setAttribute("tooltiptext", ViewMarksUtil.getString("edit_bookmark"));
			}
			node.appendChild(image);


			let spacer = document.createElement("hspacer");
			spacer.setAttribute("flex", "1");
			node.appendChild(spacer);
			if (ViewMarksPref.show_rating && 
				target.itemType == ViewMarksView.bookmarks.TYPE_BOOKMARK && 
				target.boxObject.width-2*iconsize-14 > 96) {
				node.appendChild(ViewMarksView.createRatingBar(target.itemId));
				spacer = document.createElement("hspacer");
				spacer.setAttribute("flex", "1");
				node.appendChild(spacer);
			}

			image = document.createElement("image");
			if (ViewMarksPref.create_trash == true && ViewMarksView.folderId != ViewMarksPref.trash_folder_id) {
				image.setAttribute("src", "chrome://viewmarks/skin/overlay_trash_"+iconsize+".png");
			} else {
				image.setAttribute("src", "chrome://viewmarks/skin/overlay_delete_"+iconsize+".png");
			}
			image.setAttribute("class", "vmks_overlay_button");
			image.addEventListener("click", function(event) {
				event.preventDefault();
				event.stopPropagation();
				ViewMarksView.onDelete(Math.floor(target.itemId));
				if (target.childNodes.length > 4) target.removeChild(target.childNodes[0]);
				if (ViewMarksView.viewType == ViewMarksView.VIEW_SEARCH) {
					window.location.reload();
				}
			}, false);
			if (ViewMarksPref.create_trash == true && ViewMarksView.folderId != ViewMarksPref.trash_folder_id) {
				image.setAttribute("tooltiptext", ViewMarksUtil.getString("move_to_trash"));
			} else {
				image.setAttribute("tooltiptext", ViewMarksUtil.getString("delete"));
			}
			node.appendChild(image);

			box.appendChild(node);
			
			target.insertBefore(box, target.childNodes[0]);
			//target.insertBefore(node, target.childNodes[0]);
			if (ViewMarksView.overlayNode && ViewMarksView.overlayNode.childNodes.length > 4) {
				ViewMarksView.overlayNode.removeChild(ViewMarksView.overlayNode.childNodes[0]);
			}
			ViewMarksView.overlayNode = target;
		}, true);

		box.addEventListener("mouseout", function(event) {
			let target = event.currentTarget;
			let x = ViewMarksView.view.scrollLeft + event.clientX;
			let y = ViewMarksView.view.scrollTop + event.clientY;
			if (ViewMarksUtil.isOverElement(x, y, target)) return;
			if (target.childNodes.length > 4) {
				target.removeChild(target.childNodes[0]);
			}
			if (ViewMarksView.overlayNode == target) {
				ViewMarksView.overlayNode = null;
			}
		}, true);

	},


	paintGrid: function(flags)
	{
		if (this.folderList == null) return;
		let entry = this.folderList;
		let win = ViewMarksUtil.getBrowser().contentWindow;

		let imgwidth = this.zoomLevelList[4+this.zoomLevel];
		let imgheight = (imgwidth * 3) / 4;
		let boxwidth = imgwidth + 2*8;
		let boxheight = imgheight + 2*8 + 3 + 20;
		let pagerows = Math.floor((win.innerHeight + boxheight - 1) / boxheight);

		//let displaywidth = win.innerWidth - 8;
		let displaywidth = document.getElementById("vmks_viewheader").clientWidth - 20;
		ViewMarksUtil.logmsg("paintGrid: displaywidth:"+displaywidth+" flags:"+flags);
		

		let gridwidth = Math.floor(displaywidth/boxwidth);
		if (gridwidth === this.gridwidth && (flags & this.RESIZE)) return;
		this.gridwidth = gridwidth;

		var maxrows = Math.floor((entry.length + gridwidth - 1) / gridwidth);

		if (flags & this.EXTEND) {
			var rows = document.getElementById("vmks_viewrows");
			if (this.visible_rows == maxrows) return;
			nrows = this.visible_rows + pagerows;
			if (nrows > maxrows) nrows = maxrows;
		} else {
			var grid = document.createElement("grid");
			grid.setAttribute("id", "vmks_viewgrid");
			var rows = document.createElement("rows");
			rows.setAttribute("id", "vmks_viewrows");
			grid.appendChild(rows);
			this.boxList = new Array();
			if (flags & this.REFRESH) {
				var nrows = this.visible_rows;
			} else if (this.load_all) {
				var nrows = maxrows;
			} else {
				var nrows = 2 * pagerows;
			}
			this.visible_rows = 0; 
		}

		var row;

		if (entry.length == 0) {
			row = document.createElement("row");		
			rows.appendChild(row);
			var label = document.createElement("label");
			label.setAttribute("value", "");
			label.setAttribute("class", "vmks_title_label");
			row.appendChild(label);
		}


		for (var nrow = this.visible_rows; nrow < nrows; nrow++) {

			row = document.createElement("row");		
			rows.appendChild(row);

			for (var ncol = 0; ncol < gridwidth; ncol++) {

				var index = nrow * gridwidth + ncol;
				if (index >= entry.length) {
					var spacer = document.createElement("hspacer");
					spacer.setAttribute("style", "width:"+boxwidth+"px");
					row.appendChild(spacer);
					continue;
				}

				var item = entry[index];
				var box = document.createElement("vbox");
				box.setAttribute("class", "vmks_view_box");
				box.setAttribute("context", "vmks_view_context");
				box.itemId = item.itemId;
				box.itemType = item.itemType;
				if (item.itemType == this.bookmarks.TYPE_FOLDER) {
					if (this.folderId != ViewMarksPref.trash_folder_id) {
						box.setAttribute("onclick", "ViewMarksView.onFolderClick(event,"+Math.floor(item.itemId)+")");
					}
				} else {
					box.setAttribute("onclick", "ViewMarksView.onEntryClick(event,"+String.quote(item.uri)+")");
				}
	
				var img = document.createElement("image");
				img.width = imgwidth;
				img.index = index;
				img.setAttribute("tooltip", "vmks_view_tooltip");
				if (this.folderId == ViewMarksData.historyFolder) {
					img.setAttribute("src", "chrome://viewmarks/skin/docs/generic.png");
					img.setAttribute("class", "vmks_thumbnail_generic"); 
					img.setAttribute("style", "width:"+imgwidth+"px;height:"+imgheight+"px");
					img.height = imgheight;
				} else {
					if (item.itemType == this.bookmarks.TYPE_FOLDER) {
						if (item.itemId == this.bookmarks.toolbarFolder) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_toolbar.png");
						} else if (item.itemId == ViewMarksData.lastVisitedFolder) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_history.png");
						} else if (item.itemId == ViewMarksData.mostVisitedFolder) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_history.png");
						} else if (item.itemId == this.bookmarks.unfiledBookmarksFolder) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_unfiled.png");
						} else if (item.itemId == ViewMarksData.historyFolder) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_url_history.png");
						} else if (item.itemId == ViewMarksPref.trash_folder_id) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_trash.png");
						} else if (ViewMarksData.isLivemark(item.itemId)) {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder_livemark.png");
						} else {
							img.setAttribute("src", "chrome://viewmarks/skin/docs/folder.png");
						}
						img.setAttribute("class", "vmks_folder"); 
						img.loaded = true;
					} else {
						img.setAttribute("src", "chrome://viewmarks/skin/docs/generic.png");
						img.setAttribute("class", "vmks_thumbnail_generic"); 
					}
					img.setAttribute("style", "width:"+imgwidth+"px;height:"+imgheight+"px");
					img.height = imgheight;
				}
				if (item.itemType == this.bookmarks.TYPE_BOOKMARK) {
					img.setAttribute("onmouseover", "self.status="+String.quote(item.uri)+";return true;"); 
					img.setAttribute("onmouseout", "self.status='';return true;"); 
				}
				if (this.folderId == ViewMarksPref.trash_folder_id) {
					img.setAttribute("deleted", "true"); 
				}
	
				var height = imgheight - img.height;
				var spacer = document.createElement("vspacer");
				spacer.setAttribute("flex", "1");
				box.appendChild(spacer);
				box.appendChild(img);
				spacer = document.createElement("vspacer");
				spacer.setAttribute("flex", "1");
				box.appendChild(spacer);
	
				var label = document.createElement("label");
				label.index = index;
				label.setAttribute("value", ((item.title == null || item.title == "") ? item.uri : item.title));
				label.setAttribute("tooltip", "vmks_view_tooltip");
				label.setAttribute("style", "max-width:"+imgwidth+"px");
				label.setAttribute("class", "vmks_title_label vmks_label");
				label.setAttribute("crop", "end");
				box.appendChild(label);
	
				if (this.viewType == this.VIEW_FOLDER && 
					!ViewMarksData.isSpecialFolder(item.itemId) && 
					//!ViewMarksData.isSpecialFolder(ViewMarksView.folderId) && 
					ViewMarksView.folderId > 0) {
						this.addDragDropEventHandler(box, item, index);
						this.addMouseOverEventHandler(box, item, index);
				}
				if (this.viewType == this.VIEW_SEARCH && ViewMarksView.folderId > 0) {
					this.addMouseOverEventHandler(box, item, index);
				}
	
				row.appendChild(box);
				this.boxList.push(box); 
			}
		}

		this.visible_rows = nrows;

/*
		if (entry.length < gridwidth) {
			var width = (gridwidth - entry.length) * boxwidth;
			var spacer = document.createElement("hspacer");
			spacer.setAttribute("width", width);
			row.appendChild(spacer);
		}
*/

		if ((flags & this.EXTEND) == 0) {
			var node = document.getElementById("vmks_viewgrid");
			node.parentNode.replaceChild(grid, node);
			if (this.load_all) {
				this.load_count = 0;
				this.load_start = performance.now();
				this.paintView();
			} else {
				this.refreshScrolledView(true);
			}
		}

		if (this.folderId == ViewMarksData.historyFolder) {
			ViewMarksUtil.enableNode("vmks_clear_history_button", this.folderList.length > 0);
		}

		if (this.folderId == ViewMarksPref.trash_folder_id) {
			ViewMarksUtil.enableNode("vmks_empty_trash_button", this.folderList.length > 0);
		}
	},


	refreshGrid: function()
	{
		this.folderList = ViewMarksView.getFolderArray(ViewMarksView.folderId);
		this.paintGrid(this.REFRESH);
	},


	refreshImageCallback: function(obj, data, width, height, transparent) 
	{
		if (data == null || data == "") return;
		var imgwidth = ViewMarksView.zoomLevelList[4+ViewMarksView.zoomLevel];
		obj.setAttribute("src", data);
		obj.setAttribute("class", "vmks_thumbnail"); 
		if (typeof transparent === "undefined") transparent = false;
		obj.setAttribute("tn_border", transparent ? "false" : ViewMarksPref.tn_border); 
		obj.setAttribute("tn_shadow", transparent ? "false" : ViewMarksPref.tn_shadow); 
		obj.height = imgwidth * (height / width);
		obj.setAttribute("style", "width:"+imgwidth+"px;height:"+height+"px");
		obj.loaded = true;
	},


	refreshScrolledView: function(initial)
	{
		var wy0 = this.view.scrollTop;
		var wy1 = wy0 + window.innerHeight + window.innerHeight/2;
		for (var i=0; i<this.boxList.length; i++) {
			var box = this.boxList[i];
			var y0 = box.boxObject.y;
			var y1 = y0 + box.boxObject.height;
			if (y0 < wy1 && y1 > wy0) {
				var img = box.childNodes[box.childNodes.length-3];
				if (img.loaded) continue;
				if (!initial) img.setAttribute("fadein", "true"); 
				if (this.folderId == ViewMarksData.historyFolder) {
					ViewMarksHistory.getItemThumbnail(box.itemId, img, ViewMarksView.refreshImageCallback);
				} else {
					ViewMarksData.getItemThumbnail(box.itemId, img, ViewMarksView.refreshImageCallback);
				}
			}
		}

		var grid = document.getElementById("vmks_viewgrid");
		if (wy1 + 128 > grid.clientHeight) {
			this.paintGrid(this.EXTEND);
		}
	},


	paintView: function()
	{
		for (var i=0; i<this.boxList.length; i++) {
			var box = this.boxList[i];
			var img = box.childNodes[box.childNodes.length-3];
			if (img.loaded) continue;
			if (this.folderId == ViewMarksData.historyFolder) {
				ViewMarksHistory.getItemThumbnail(box.itemId, img, ViewMarksView.refreshImageCallback);
			} else {
				ViewMarksData.getItemThumbnail(box.itemId, img, ViewMarksView.refreshImageCallback);
			}
		}
	},


	onScroll: function(event)
	{
		if (ViewMarksView.load_all) return;
		if (ViewMarksView.scrollTimer) clearTimeout(ViewMarksView.scrollTimer);
		ViewMarksView.scrollTimer = setTimeout(function(){ViewMarksView.refreshScrolledView();}, 100);
	},


	onEntryClick: function(event, url)
	{
		var modifier = (ViewMarksUtil.platform == "Darwin" ? event.metaKey : event.ctrlKey)  
		var button = event.button;
		if (button == 0 && modifier) button = 1;
		if (button == 0) {
			switch(ViewMarksPref.standard_click_action) {
				case 0: window.location.href = url; break;
				case 1: ViewMarksUtil.openInNewTab(url, true); break;
				case 2: window.open(url, "newwindow"); break;
			}
		}
		if (button == 1) ViewMarksUtil.openInNewTab(url, true);
	},


	onFolderClick: function(event, id)
	{
		//if (this.folderId == ViewMarksPref.trash_folder_id) return;
		if (id == ViewMarksData.bookmarks.bookmarksMenuFolder) id = "bookmarks";
		if (id == ViewMarksData.bookmarks.toolbarFolder) id = "toolbar";
		if (id == ViewMarksData.bookmarks.unfiledBookmarksFolder) id = "unsorted";
		if (id == ViewMarksData.lastVisitedFolder) id = "lastvisited";
		if (id == ViewMarksData.mostVisitedFolder) id = "mostvisited";
		if (id == ViewMarksData.historyFolder) id = "history";
		if (id == ViewMarksPref.trash_folder_id) id = "trash";
		var modifier = (ViewMarksUtil.platform == "Darwin" ? event.metaKey : event.ctrlKey)  
		var button = event.button;
		if (button == 0 && event.ctrlKey) button = 1;
		if (button == 0) window.location.href = "about:viewmarks?folder=" + id;
		if (button == 1) ViewMarksUtil.openInNewTab("about:viewmarks?folder=" + id, true);
	},


	onZoomClick: function(step)
	{
		if (step > 0) {
			document.getElementById("vmks_zoomscale").decrease();
		} else {
			document.getElementById("vmks_zoomscale").increase();
		}
	},
	
	
	onZoom: function(level)
	{
		ViewMarksPref.prefs.setIntPref("zoom", level);
		if (this.zoomLevel != level) {
			this.zoomLevel = level;
			this.paintGrid(this.REFRESH);
		}
	},


	onRestore: function(itemId)
	{
		let parentId = ViewMarksData.getItemAnnotation(itemId, "bookmarkProperties/originalParentId");
		ViewMarksTrash.removeTrashInformation(itemId);
		try {
			this.bookmarks.moveItem(itemId, parentId, this.bookmarks.DEFAULT_INDEX);
		} catch(e) {
			this.bookmarks.moveItem(itemId, this.bookmarks.bookmarksMenuFolder, this.bookmarks.DEFAULT_INDEX);
			ViewMarksPopup.open(
				ViewMarksUtil.getString("restore_item"), 
				ViewMarksUtil.getString("restore_missing_parent"),
				ViewMarksPopup.MB_OK|ViewMarksPopup.MB_ICON_WARNING);
		}
	},


	onDelete: function(itemId)
	{
		try {
			let result = true;
			if (!ViewMarksPref.create_trash || this.folderId == ViewMarksPref.trash_folder_id) {
				var msg, title;
				let itemType = ViewMarksData.getItemType(itemId);
				if (itemType == this.bookmarks.TYPE_FOLDER) {
					title = ViewMarksUtil.getString("delete_folder");
					msg = ViewMarksUtil.getString("ask_delete_folder");
				}
				if (itemType == this.bookmarks.TYPE_BOOKMARK) {
					title = ViewMarksUtil.getString("delete_bookmark");
					msg = ViewMarksUtil.getString("ask_delete_bookmark");
				}
				result = ViewMarksPopup.open(title, msg, 
					ViewMarksPopup.MB_OK|ViewMarksPopup.MB_CANCEL|ViewMarksPopup.MB_ICON_WARNING);
			}
			if (result == true) {
				if (ViewMarksPref.create_trash && this.folderId != ViewMarksPref.trash_folder_id) {
					ViewMarksTrash.moveItemToTrash(itemId);
				}
				try {this.bookmarks.removeItem(itemId);} catch (e) {}
				ViewMarksData.deleteItemThumbnail(itemId);
			}
		} catch (e) {
			ViewMarksUtil.logerror(e)
			ViewMarksReport.openDialog(e);
		}
	},


	onViewContext: function(event)
	{
		if (document.popupNode == null) return;
		let index = document.popupNode.index;
		let itemId = ViewMarksView.folderList[index].itemId;
		let itemType = ViewMarksView.folderList[index].itemType;

		let open = document.getElementById("vmks_view_context_open");
		let openintab = document.getElementById("vmks_view_context_openintab");
		let openinwin = document.getElementById("vmks_view_context_openinwin");
		let createbookmark = document.getElementById("vmks_view_context_createbookmark");
		let edit = document.getElementById("vmks_view_context_edit");
		let del = document.getElementById("vmks_view_context_delete");
		let restore = document.getElementById("vmks_view_context_restore");
		let copylink = document.getElementById("vmks_view_context_copylink");
		let copyimage = document.getElementById("vmks_view_context_copyimage");
		let copytofolder = document.getElementById("vmks_view_context_copytofolder");
		let movetofolder = document.getElementById("vmks_view_context_movetofolder");
		let qrcodeshow = document.getElementById("vmks_view_context_code");
		let qrcodeimage = document.getElementById("vmks_view_context_code_image");

		open.setAttribute("default", ViewMarksPref.standard_click_action == 0);
		openintab.setAttribute("default", ViewMarksPref.standard_click_action == 1);
		openinwin.setAttribute("default", ViewMarksPref.standard_click_action == 2);

		if (ViewMarksPref.create_trash == true && ViewMarksView.folderId != ViewMarksPref.trash_folder_id) {
			del.setAttribute("label", ViewMarksUtil.getString("move_to_trash"));
		} else {
			del.setAttribute("label", ViewMarksUtil.getString("delete"));
		}

		if (ViewMarksPref.show_qrcode_context_menu == true) {
			qrcodeshow.removeAttribute("collapsed");
		} else {
			qrcodeshow.setAttribute("collapsed", true);
		}

		restore.setAttribute("collapsed", true);

		if (itemType == this.bookmarks.TYPE_FOLDER) {
			var url = "about:viewmarks?folder="+itemId;
			if (ViewMarksData.isSpecialFolder(itemId)) {
				edit.setAttribute("disabled", "true");
				del.setAttribute("disabled", "true");
				copytofolder.setAttribute("disabled", "true");
				movetofolder.setAttribute("disabled", "true");
			} else {
				edit.removeAttribute("disabled");
				edit.setAttribute("oncommand", "ViewMarksEdit.open(ViewMarksEdit.DF_EDIT_FOLDER,"+Math.floor(itemId)+")");
				del.removeAttribute("disabled");
				copytofolder.removeAttribute("disabled");
				movetofolder.removeAttribute("disabled");
			}
			if (ViewMarksView.folderId == ViewMarksPref.trash_folder_id) {
				edit.setAttribute("disabled", "true");
				restore.removeAttribute("collapsed");
				copytofolder.setAttribute("disabled", "true");
			}
			copylink.setAttribute("disabled", true);
			copyimage.setAttribute("disabled", true);
			qrcodeshow.setAttribute("disabled", true);
		} else {
			var url = ViewMarksView.folderList[index].uri;
			copylink.removeAttribute("disabled");
			copylink.setAttribute("oncommand", "ViewMarksData.copyText("+String.quote(url)+")");
			if (document.popupNode.src && document.popupNode.src.indexOf("data:") == 0) {
				copyimage.removeAttribute("disabled");
				copyimage.addEventListener("command", function() {
						ViewMarksData.copyImageFromData(document.popupNode.src);
					}, true);
			} else {
				copyimage.setAttribute("disabled", true);
			}
			if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
				createbookmark.removeAttribute("collapsed");
				createbookmark.setAttribute("oncommand", "ViewMarksHistory.createBookmark("+Math.floor(itemId)+")");
				edit.setAttribute("disabled", true);
				del.setAttribute("disabled", true);
				copytofolder.setAttribute("disabled", true);
				movetofolder.setAttribute("disabled", true);
			} else if (ViewMarksView.folderId == ViewMarksPref.trash_folder_id) {
				edit.setAttribute("disabled", true);
				del.removeAttribute("disabled");
				restore.removeAttribute("collapsed");
				copytofolder.setAttribute("disabled", "true");
				movetofolder.removeAttribute("disabled");
			} else {
				edit.removeAttribute("disabled");
				edit.setAttribute("oncommand", "ViewMarksEdit.open(ViewMarksEdit.DF_EDIT_BOOKMARK,"+Math.floor(itemId)+")");
				del.removeAttribute("disabled");
				copytofolder.removeAttribute("disabled");
				movetofolder.removeAttribute("disabled");
			}
			if (ViewMarksPref.show_qrcode_context_menu == true) {
				qrcodeshow.removeAttribute("disabled");
				qrcodeimage.setAttribute("src", ViewMarksQRCode.createQRCodeLink(url));
				qrcodeimage.setAttribute("tooltiptext", url);
				qrcodeimage.style.padding = (4*ViewMarksPref.qrcode_cell_size)+"px";
			} 
		}

		open.removeAttribute("disabled");
		openintab.removeAttribute("disabled");
		openinwin.removeAttribute("disabled");

		open.setAttribute("oncommand", "window.location.href="+String.quote(url));
		openintab.setAttribute("oncommand", "ViewMarksUtil.openInNewTab("+String.quote(url)+",true)");
		if (ViewMarksPref.browserPrefs.getIntPref("link.open_newwindow") == 2) {
			openinwin.removeAttribute("disabled");
			openinwin.setAttribute("oncommand", "window.open("+String.quote(url)+")");
		} else {
			openinwin.setAttribute("disabled", "true");
		}

		if (itemType == this.bookmarks.TYPE_FOLDER && ViewMarksView.folderId == ViewMarksPref.trash_folder_id) {
			open.setAttribute("disabled", true);
			openintab.setAttribute("disabled", true);
			openinwin.setAttribute("disabled", true);
		} 

		del.setAttribute("oncommand", "ViewMarksView.onDelete("+Math.floor(itemId)+")");
		restore.setAttribute("oncommand", "ViewMarksView.onRestore("+Math.floor(itemId)+")");
	},


	hiliteSearchPatterns: function(node)
	{
		let text = node.data;
		let lctext = text.toLowerCase();
		let minindex = text.length;
		let patindex = -1;
		for (var i=0; i<this.searchPatterns.length; i++) {
			let index = lctext.indexOf(this.searchPatterns[i]);
			if (index >= 0 && index < minindex) {
				minindex = index;
				patindex = i;
			}
		}
		if (patindex >= 0) {
			let patlength = this.searchPatterns[patindex].length;
			let node_prev = document.createTextNode(text.substr(0, minindex));
			let node_after = document.createTextNode(text.substr(minindex+patlength));
			let node_span = document.createElementNS("http://www.w3.org/1999/xhtml", "html:span");
			node_span.setAttribute("class", "searchpattern");
			let node_pattern = document.createTextNode(text.substr(minindex, patlength));
			node_span.appendChild(node_pattern);
			let parentNode = node.parentNode;
			parentNode.replaceChild(node_after, node);
			parentNode.insertBefore(node_span, node_after);
			parentNode.insertBefore(node_prev, node_span);
			this.hiliteSearchPatterns(node_after);
		}
	},


	setTooltipText: function(id, value)
	{
		var node = document.getElementById(id);
		if (value == "") {
			node.parentNode.setAttribute("collapsed", true);
		} else {
			node.parentNode.removeAttribute("collapsed");
			while (node.childNodes.length) node.removeChild(node.firstChild);
			let textnode = document.createTextNode(value);
			node.appendChild(textnode);
			if (this.viewType == this.VIEW_SEARCH && this.searchPatterns) {
				this.hiliteSearchPatterns(textnode);
			}
		}
	},


	onTooltip: function(event)
	{
		try {
		var index = document.tooltipNode.index;
		var itemId = ViewMarksView.folderList[index].itemId;
		var itemType = ViewMarksView.folderList[index].itemType;
		var title = ViewMarksUtil.insertWordBreaks(ViewMarksView.folderList[index].title);

		this.setTooltipText("vmks_tooltip_title", title);
		if ((this.viewType == this.VIEW_SEARCH || this.folderId < 0) && this.folderId != ViewMarksData.historyFolder) {
			let parentId = ViewMarksData.getParentFolderId(itemId);
			this.setTooltipText("vmks_tooltip_parent", (parentId ? ViewMarksData.getItemTitle(parentId) : ""));
		} else {
			this.setTooltipText("vmks_tooltip_parent", "");
		}
		if (this.folderId == ViewMarksPref.trash_folder_id) {
			let parentId = ViewMarksData.getItemAnnotation(itemId, "bookmarkProperties/originalParentId");
			this.setTooltipText("vmks_tooltip_old_parent", ViewMarksData.getItemTitle(parentId));
		} else {
			this.setTooltipText("vmks_tooltip_old_parent", "");
		}
		if (itemType == this.bookmarks.TYPE_BOOKMARK) {
			var url = ViewMarksView.folderList[index].uri;
			url = ViewMarksUtil.insertWordBreaks(url);
			if (url.length > 512) url = url.substr(0, 512) + " ...";
			this.setTooltipText("vmks_tooltip_url", ViewMarksUtil.insertWordBreaks(url));
			var accessCount = ViewMarksView.folderList[index].accessCount;
			if (accessCount == 1) {
				var accessed = "1 "+ViewMarksUtil.getString("time");
			} else {
				var accessed = accessCount+" "+ViewMarksUtil.getString("times");
			}
			this.setTooltipText("vmks_tooltip_accessed", accessed);
			if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
				this.setTooltipText("vmks_tooltip_description", ViewMarksView.folderList[index].description);
				this.setTooltipText("vmks_tooltip_keywords", ViewMarksView.folderList[index].keywords);
				var tags = ViewMarksData.getUrlTags(ViewMarksView.folderList[index].url).join(", ");
				this.setTooltipText("vmks_tooltip_tags", tags);
			} else {
				var description = ViewMarksUtil.insertWordBreaks(ViewMarksData.getItemDescription(itemId), 512, 576);
				this.setTooltipText("vmks_tooltip_description", description);
				var keywords = ViewMarksUtil.insertWordBreaks(ViewMarksData.getItemKeywords(itemId), 512, 576);
				this.setTooltipText("vmks_tooltip_keywords", keywords);
				var tags = ViewMarksData.getItemTags(itemId).join(", ");
				this.setTooltipText("vmks_tooltip_tags", tags);
			}
		} else {
			this.setTooltipText("vmks_tooltip_url", "");
			this.setTooltipText("vmks_tooltip_accessed", "");
			this.setTooltipText("vmks_tooltip_description", "");
			this.setTooltipText("vmks_tooltip_keywords", "");
			this.setTooltipText("vmks_tooltip_tags", "");
			this.setTooltipText("vmks_tooltip_lastvisit", "");
		}
		if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
			this.setTooltipText("vmks_tooltip_added", "");
			this.setTooltipText("vmks_tooltip_modified", "");
			this.setTooltipText("vmks_tooltip_lastvisit", new Date(ViewMarksView.folderList[index].time*1000).toLocaleString());
		} else {
			let date_added = ViewMarksData.getItemDateAdded(itemId);
			this.setTooltipText("vmks_tooltip_added", date_added);
			let date_modified = ViewMarksData.getItemLastModified(itemId);
			if (date_modified == date_added) date_modified = "";
			this.setTooltipText("vmks_tooltip_modified", date_modified);
			this.setTooltipText("vmks_tooltip_lastvisit", "");
		}
		if (this.folderId == ViewMarksPref.trash_folder_id) {
			let date_modified = ViewMarksData.getItemLastModified(itemId);
			let date_deleted = ViewMarksData.getItemDateDeleted(itemId);
			this.setTooltipText("vmks_tooltip_deleted", date_deleted);
			if (date_modified == date_deleted) {
				this.setTooltipText("vmks_tooltip_modified", "");
			}
		} else {
			this.setTooltipText("vmks_tooltip_deleted", "");
		}

		var image = document.getElementById("vmks_view_tooltip_image");
		image.setAttribute("collapsed", true);
		if (ViewMarksPref.tn_tooltips && itemType == this.bookmarks.TYPE_BOOKMARK) { 
			var box = this.boxList[index];
			var img = box.childNodes[box.childNodes.length-3];
			if (img.loaded) {
				image.src = img.src;
				image.removeAttribute("collapsed");
			}
		}
		} catch(e) {
			ViewMarksUtil.logerror(e)
		}
	},
	

	onNewFolderClick: function()
	{
		ViewMarksEdit.open(ViewMarksEdit.DF_CREATE_FOLDER, this.folderId)
	},


	alignmentIcons: [
		"align_left.png",
		"align_center.png",
		"align_right.png"
	],

	setAlignment: function(align)
	{
		var image = document.getElementById("vmks_alignment_icon");
		image.setAttribute("src", "chrome://viewmarks/skin/"+this.alignmentIcons[align]);
		ViewMarksPref.prefs.setIntPref("view_alignment", align);
		ViewMarksUtil.showElement("vmks_gridspacer_left", align > 0);
		ViewMarksUtil.showElement("vmks_gridspacer_right", align < 2);
	},


	setSortOrder: function(order)
	{
		if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
			ViewMarksPref.prefs.setIntPref("history_sort_order", order);
			ViewMarksHistory.setSortOrder(order);
			this.refreshGrid();
		}
		if (ViewMarksView.folderId >= 0) {
			ViewMarksPref.prefs.setIntPref("sort_order", order);
			this.sortOrder = order;
			ViewMarksData.sortBookmarkArray(this.folderList, order);
			this.refreshGrid();
		}
	},


	onSearch: function(extended)
	{
		try {
			if (extended === undefined) extended = false;
			let value = "";
			let flags = 0;
			if (extended) {
				if (ViewMarksSearch.openDialog() == false) return;
				value = ViewMarksPref.search_value;			
				value = value.replace(/^\s+|\s+$/g, '');
				let patterns = value.split(/\s+/);
				let uri = "search=" + patterns.join("+");
				flags = ViewMarksPref.search_flags;			
				if ((flags & ViewMarksSearch.SF_LOCAL) && ViewMarksView.folderId) {
					uri += "&folder=" + ViewMarksData.getFolderNameFromId(ViewMarksView.folderId);
				}
				uri += "&flags=" + flags;
				if (ViewMarksUtil.startsWith(window.location.href, "chrome")) {
					ViewMarksUtil.openInNewTab(this.uri + "?" + uri);
				} else {
					window.location.href = this.uri + "?" + uri;
				}
			} else {
				value = document.getElementById("vmks_searchbox").value;
				value = value.replace(/^\s+|\s+$/g, '');
				if (value.length == 0) return;
				let patterns = value.split(/\s+/);
				if (patterns.length == 0) return;
				if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
					window.location.href = "about:viewmarks?folder=history&search=" + patterns.join("+");
				} else if (ViewMarksView.folderId == ViewMarksPref.trash_folder_id) {
					window.location.href = "about:viewmarks?folder=trash&search=" + patterns.join("+");
				} else {
					window.location.href = "about:viewmarks?search=" + patterns.join("+");
				}
			}
		} catch (e) {
			ViewMarksUtil.logerror(e);
			ViewMarksReport.openDialog(e);
		}
	},


	scrollBy: function(dy)
	{
		this.view.scrollTop += dy;
	},
	

	scrollTo: function(y)
	{
		this.view.scrollTop = y;
	},


	onMouseMove: function(event)
	{
		if (event.buttons) {
			if ((event.buttons & 1) == 0 && ViewMarksView.dragIndex >= 0) {
				ViewMarksView.refreshGrid(ViewMarksView.REFRESH);
				ViewMarksView.dragIndex = -1;
			}
		} else {
			if ((event.button & 1) == 0 && ViewMarksView.dragIndex >= 0) {
				ViewMarksView.refreshGrid(ViewMarksView.REFRESH);
				ViewMarksView.dragIndex = -1;
			}
		}
	},

	onKeyDown: function(event)
	{
		if (event.keyCode == KeyEvent.DOM_VK_HOME) ViewMarksView.scrollTo(0);
		else if (event.keyCode == KeyEvent.DOM_VK_END) ViewMarksView.scrollTo(ViewMarksView.view.boxObject.height);
		else if (event.keyCode == KeyEvent.DOM_VK_PAGE_UP) ViewMarksView.scrollBy(-ViewMarksView.view.scrollHeight);
		else if (event.keyCode == KeyEvent.DOM_VK_PAGE_DOWN) ViewMarksView.scrollBy(ViewMarksView.view.scrollHeight);
		else if (event.keyCode == KeyEvent.DOM_VK_UP) ViewMarksView.scrollBy(-20);
		else if (event.keyCode == KeyEvent.DOM_VK_DOWN) ViewMarksView.scrollBy(20);
		else {
			document.getElementById("vmks_searchbox").focus();
		}
	},


	onVisitedCountChanged: function(value)
	{
		if (this.folderId == ViewMarksData.mostVisitedFolder) ViewMarksPref.prefs.setIntPref("most_visited_count", value);
		if (this.folderId == ViewMarksData.lastVisitedFolder) ViewMarksPref.prefs.setIntPref("last_visited_count", value);
		window.location.reload();
	},


	onHeaderClick: function(event)
	{
		if (event.target.id && event.target.id == "vmks_viewheader") {
			ViewMarksView.view.scrollTop = 0
		}
	},
	

	createSepPopupItem: function(id, title, highlightId)
	{
		if (title == null) title = ViewMarksData.getItemTitle(id);
		var item = document.createElement("menuitem");
		item.setAttribute("label", title);
		item.setAttribute("class", "menuitem-iconic bookmark-item");
		item.setAttribute("image", ViewMarksData.getFolderIconUrl(id));
		item.setAttribute("onclick", "ViewMarksView.onFolderClick(event,"+Math.floor(id)+")");
		if (highlightId == id) item.setAttribute("style", "font-weight:bold");
		return item;
	},


	showSepPopup: function(event, parentId, selectedId)
	{
		var popup = document.getElementById("vmks_breadcrumb_sep_popup");
		while (popup.childNodes.length) popup.removeChild(popup.firstChild);

		if (parentId == this.bookmarks.bookmarksMenuFolder) {
			ViewMarksData.refreshFolderCount();
			if (ViewMarksData.toolbarCount > 0) {
				popup.appendChild(this.createSepPopupItem(this.bookmarks.toolbarFolder, null, selectedId));
			}
			if (ViewMarksData.unfiledCount > 0) {
				popup.appendChild(this.createSepPopupItem(this.bookmarks.unfiledBookmarksFolder, null, selectedId));
			}
			if (ViewMarksPref.show_most_visited) {
				popup.appendChild(this.createSepPopupItem(ViewMarksData.mostVisitedFolder, null, selectedId));
			}
			if (ViewMarksPref.show_last_visited) {
				popup.appendChild(this.createSepPopupItem(ViewMarksData.lastVisitedFolder, null, selectedId));
			}
			if (ViewMarksPref.create_history) {
				popup.appendChild(this.createSepPopupItem(ViewMarksData.historyFolder, null, selectedId));
			}
			if (ViewMarksPref.create_trash) {
				popup.appendChild(this.createSepPopupItem(ViewMarksPref.trash_folder_id, null, selectedId));
			}
			popup.appendChild(document.createElement("menuseparator"));
		}
		
		var query = this.history.getNewQuery();
		query.setFolders([parentId], 1);
		var result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		var folderNode = result.root;
		folderNode.containerOpen = true;
	
		for (var i=0; i<folderNode.childCount; i++) {
			var childNode = folderNode.getChild(i);
			var type = ViewMarksData.getItemType(childNode.itemId);
			if (type == this.bookmarks.TYPE_FOLDER || type == this.bookmarks.TYPE_DYNAMIC_CONTAINER) {
				var item = this.createSepPopupItem(childNode.itemId, childNode.title, selectedId);
				popup.appendChild(item);
			}
		}
		popup.openPopup(event.target, "end_before", 0, event.target.boxObject.height, true, false);
	},


	copyOrMove: function(event, action)
	{
		var index = document.popupNode.index;
		var srcId = ViewMarksView.folderList[index].itemId;
		var dstId = event.target.itemId;
		ViewMarksUtil.logmsg("srcId="+srcId+" dstId="+dstId);
		if (this.bookmarks.getFolderIdForItem(srcId) == dstId) {
			title = ViewMarksUtil.getString("error");
			msg = ViewMarksUtil.getString("identical_target_folder");
			ViewMarksPopup.open(title, msg, ViewMarksPopup.MB_OK|ViewMarksPopup.MB_ICON_ERROR);
		} else {
			ViewMarksUtil.logmsg("ViewMarks: "+action+" item "+srcId+" to folder "+dstId);
			if (action == "move") {
				if (ViewMarksData.getItemType(srcId) == this.bookmarks.TYPE_FOLDER) {
					var index =  ViewMarksData.getFolderInsertIndex(dstId);
					this.bookmarks.moveItem(srcId, dstId, index);
				} else {
					this.bookmarks.moveItem(srcId, dstId, this.bookmarks.DEFAULT_INDEX);
				}
				ViewMarksData.updateRecentlyUsedFolder(dstId);
				if (this.folderId == ViewMarksPref.trash_folder_id) {
					ViewMarksTrash.removeTrashInformation(itemId);
				}
			}
			if (action == "copy") {
				//ViewMarksData.copyBookmark(srcId, dstId, this.bookmarks.DEFAULT_INDEX);
				let newItemId = ViewMarksData.copyItem(srcId, dstId, this.bookmarks.DEFAULT_INDEX);
				ViewMarksData.updateRecentlyUsedFolder(dstId);
				if (this.folderId == ViewMarksPref.trash_folder_id) {
					ViewMarksTrash.removeTrashInformation(newItemId);
				}
			}
		}
		document.getElementById('vmks_view_context').hidePopup();
	},


	createFolderTreeItem: function(id, title, subtree)
	{
		if (title == null) title = ViewMarksData.getItemTitle(id);
		if (subtree) {
			var item = document.createElement("menu");
			item.setAttribute("class", "menu-iconic bookmark-item");
		} else {
			var item = document.createElement("menuitem");
			item.setAttribute("class", "menuitem-iconic bookmark-item");
		}
		item.setAttribute("label", title);
		item.setAttribute("image", ViewMarksData.getFolderIconUrl(id));
		item.itemId = id;
		if (subtree) item.appendChild(subtree);
		return item;
	},


	createFolderTree: function(itemId)
	{
		var popup = document.createElement("menupopup");

		if (itemId == null) {
			var subtree = null;
			var subtree = this.createFolderTree(this.bookmarks.bookmarksMenuFolder);
			var item = this.createFolderTreeItem(this.bookmarks.bookmarksMenuFolder, null, subtree);
			popup.appendChild(item);
			subtree = this.createFolderTree(this.bookmarks.toolbarFolder);
			item = this.createFolderTreeItem(this.bookmarks.toolbarFolder, null, subtree);
			popup.appendChild(item);
			subtree = this.createFolderTree(this.bookmarks.unfiledBookmarksFolder);
			item = this.createFolderTreeItem(this.bookmarks.unfiledBookmarksFolder, null, subtree);
			popup.appendChild(item);
			if (!ViewMarksData.isSpecialFolder(this.folderId)) {
				popup.appendChild(document.createElement("menuseparator"));
				try {
					var parentId = this.bookmarks.getFolderIdForItem(this.folderId);
					if (parentId != 1) {
						var title = ViewMarksUtil.getString("parent_folder");
						item = this.createFolderTreeItem(parentId, title, null);
						item.setAttribute("image", "chrome://viewmarks/skin/undo.png");
						popup.appendChild(item);
					}
				} catch (e) {}
				var entry = this.folderList;
				for (var i=0; i<entry.length; i++) {
					if (entry[i].itemType != this.bookmarks.TYPE_FOLDER) continue;
					item = this.createFolderTreeItem(entry[i].itemId, null, null);
					popup.appendChild(item);
				}
			}
			return popup;
		}

		var query = this.history.getNewQuery();
		query.setFolders([itemId], 1);
		var result = this.history.executeQuery(query, this.history.getNewQueryOptions());
		var folderNode = result.root;
		folderNode.containerOpen = true;
	
		for (var i=0; i<folderNode.childCount; i++) {
			var childNode = folderNode.getChild(i);
			var type = ViewMarksData.getItemType(childNode.itemId);
			if (type == this.bookmarks.TYPE_FOLDER || type == this.bookmarks.TYPE_DYNAMIC_CONTAINER) {
				var subtree = this.createFolderTree(childNode.itemId);
				var item = this.createFolderTreeItem(childNode.itemId, childNode.title, subtree);
				popup.appendChild(item);
			}
		}

		if (popup.childNodes.length > 0) return popup;
		return null;
	},


	onHistoryAddItem: function(event)
	{
		ViewMarksUtil.logmsg("event history-changed "+window.location.href);
		setTimeout(
			function(){
				ViewMarksView.refreshGrid();
			}, 0);	
	},


	initViewWithFolder: function(folderId)
	{
		if (folderId == 0) folderId = this.bookmarks.bookmarksMenuFolder; 
		this.folderId = folderId;
		this.sortOrder = ViewMarksPref.sort_order;
		this.folderList = this.getFolderArray(this.folderId);

		let title = ViewMarksData.getItemTitle(folderId) + " - ViewMarks";
		this.win.setAttribute("title", title);

		let breadcrumb = document.createElement("hbox");
		breadcrumb.setAttribute("id", "vmks_breadcrumb");

		let id = this.folderId;
		
		let count = 0;
		for (var i=0; i<this.folderList.length; i++) {
			if (this.folderList[i].itemType == this.bookmarks.TYPE_FOLDER) {
				count++;
			}
		}
		this.folderCount = count;

		let label = null;

		if (this.folderCount > 0 && id != ViewMarksPref.trash_folder_id) {
			label = document.createElement("label");
			//label.setAttribute("value", "˅");
			label.setAttribute("class", "vmks_button vmks_breadcrumb_sep");
			label.setAttribute("onclick", "ViewMarksView.showSepPopup(event,"+Math.floor(id)+",0)");
			let image = document.createElement("image");
			image.setAttribute("src", "chrome://viewmarks/skin/arrow-down.png");
			label.appendChild(image);
			breadcrumb.appendChild(label);
		}

		label = document.createElement("label");
		label.setAttribute("value", ViewMarksData.getItemTitle(id));
		label.setAttribute("class", "vmks_breadcrumb_last");
		//breadcrumb.appendChild(label);
		breadcrumb.insertBefore(label, breadcrumb.firstChild);

		while (id != this.bookmarks.bookmarksMenuFolder) {
			var parentId = ViewMarksData.getParentFolderId(id);
			label = document.createElement("label");
			label.setAttribute("value", "»");
			label.setAttribute("class", "vmks_button vmks_breadcrumb_sep");
			label.setAttribute("onclick", "ViewMarksView.showSepPopup(event,"+Math.floor(parentId)+","+Math.floor(id)+")");
			breadcrumb.insertBefore(label, breadcrumb.firstChild);
			label = document.createElement("label");
			label.setAttribute("value", ViewMarksData.getItemTitle(parentId));
			label.setAttribute("onclick", "ViewMarksView.onFolderClick(event,"+Math.floor(parentId)+")");
			label.setAttribute("class", "vmks_breadcrumb_link vmks_button");
			breadcrumb.insertBefore(label, breadcrumb.firstChild);
			id = parentId;
		}
		var node = document.getElementById("vmks_breadcrumb");
		node.parentNode.replaceChild(breadcrumb, node);

		if (folderId < 0 || folderId == ViewMarksPref.trash_folder_id) {
			document.getElementById("vmks_newfolder_button").setAttribute("collapsed", "true");
		}

		if (folderId == ViewMarksData.historyFolder) {
			document.getElementById("vmks_sortbutton_history").removeAttribute("collapsed");
			var sort_order = ViewMarksPref.history_sort_order;
			var popup = document.getElementById("vmks_sortpopup_history");
			for (var i=0; i<popup.childNodes.length; i++) {
				if (popup.childNodes[i].getAttribute("value") == sort_order) {
					popup.childNodes[i].setAttribute("checked", true)
				}
			}
		} else {
			var popup = document.getElementById("vmks_sortpopup");
			for (var i=0; i<popup.childNodes.length; i++) {
				if (popup.childNodes[i].getAttribute("value") == this.sortOrder) {
					popup.childNodes[i].setAttribute("checked", true)
				}
			}
		}

		ViewMarksUtil.showElement("vmks_clear_history_button", folderId == ViewMarksData.historyFolder);
		ViewMarksUtil.showElement("vmks_empty_trash_button", folderId == ViewMarksPref.trash_folder_id);
		ViewMarksUtil.showElement("vmks_search_ext_button", folderId >= 0);

		this.load_all = ViewMarksPref.load_all;
		if (folderId == ViewMarksData.historyFolder) {
			this.load_all = false;
		}

		this.paintGrid(0);

		this.bookmarks.addObserver(ViewMarksView.bookmarkObserver, false);
		if (folderId == ViewMarksData.historyFolder) {
			try {
				var doc = ViewMarksUtil.getMainWindow().document;
				doc.addEventListener("history-changed", ViewMarksView.onHistoryAddItem, false, true);
			} catch(e) {ViewMarksUtil.alert(e);}
		}
	},


	initViewWithSearch: function(folderId, patterns, flags)
	{
		if (folderId == 0) folderId = this.bookmarks.bookmarksMenuFolder; 
		this.folderId = folderId;

		this.win.setAttribute("title", ViewMarksUtil.getString("search")+" - ViewMarks");

		var breadcrumb = document.createElement("hbox");
		breadcrumb.setAttribute("id", "vmks_breadcrumb");

		var label = document.createElement("label");
		var homeId = this.bookmarks.bookmarksMenuFolder;
		label.setAttribute("value", ViewMarksData.getItemTitle(homeId));
		label.setAttribute("onclick", "ViewMarksView.onFolderClick(event,"+Math.floor(homeId)+")");
		label.setAttribute("class", "vmks_breadcrumb_link vmks_button");
		breadcrumb.appendChild(label);

		label = document.createElement("label");
		label.setAttribute("value", "»");
		label.setAttribute("class", "vmks_breadcrumb_sep vmks_button");
		label.setAttribute("onclick", "ViewMarksView.showSepPopup(event,"+Math.floor(homeId)+",0)");
		breadcrumb.appendChild(label);

		label = document.createElement("label");
		if (folderId == ViewMarksData.historyFolder) {
			label.setAttribute("value", ViewMarksUtil.getString("search_history_for")+" \""+patterns.join(" ")+"\"");
		} else if (folderId == ViewMarksPref.trash_folder_id) {
			label.setAttribute("value", ViewMarksUtil.getString("search_trash_for")+" \""+patterns.join(" ")+"\"");
		} else {
			label.setAttribute("value", ViewMarksUtil.getString("search_bookmarks_for")+" \""+patterns.join(" ")+"\"");
		}
		label.setAttribute("class", "vmks_breadcrumb_last");
		breadcrumb.appendChild(label);

		var node = document.getElementById("vmks_breadcrumb");
		node.parentNode.replaceChild(breadcrumb, node);

		ViewMarksUtil.showElement("vmks_clear_history_button", false);
		ViewMarksUtil.showElement("vmks_empty_trash_button", false);
		ViewMarksUtil.showElement("vmks_newfolder_button", false);
		document.getElementById("vmks_searchbox").value = patterns.join(" ");
		document.getElementById("vmks_searchbox").select();

		try {
		if (folderId == ViewMarksData.historyFolder) {
			this.folderList = this.getHistorySearchResultArray(patterns, flags);
		} else if (folderId == ViewMarksPref.trash_folder_id) {
			this.folderList = ViewMarksSearch.getBookmarksSearchResultArray(ViewMarksPref.trash_folder_id, null, patterns, flags);
		} else {
			this.folderList = ViewMarksSearch.getBookmarksSearchResultArray(null, null, patterns, flags);
		}
		this.paintGrid(0);
		} catch(e) {ViewMarksUtil.alert(e);}
	},
	

	initView: function() 
	{
		try {
			ViewMarksPref.init();
			this.win = document.getElementById("dlgViewMarksView");
			this.view = document.getElementById("vmks_viewbox");
			this.win.setAttribute("skin", ViewMarksPref.skin);
			this.win.addEventListener("keydown", ViewMarksView.onKeyDown, true);
			this.win.addEventListener("mousemove", ViewMarksView.onMouseMove, true);
			this.win.addEventListener("scroll", ViewMarksView.onScroll, true);
			//this.win.focus();
			this.zoomLevel = ViewMarksPref.zoom;
			this.setAlignment(ViewMarksPref.view_alignment);
			document.getElementById("vmks_zoomscale").setAttribute("value", this.zoomLevel);
	
			let patterns = null;
			let flags = 0;
			let folderId = this.bookmarks.placesRoot;
			this.viewType = this.VIEW_FOLDER;
			
			var argv = decodeURI(window.location.href).split('?');
			if (argv.length == 1) {
				this.initViewWithFolder(0);
			} else {
				argv = argv[1].split("&");
				for (var i=0; i<argv.length; i++) {
					var arg = argv[i].split("=");
					if (arg[0] == "folder") {
						folderId = parseInt(arg[1]);
						if (arg[1] == "mostvisited") {
							document.getElementById("vmks_visited_box").removeAttribute("collapsed");
							document.getElementById("vmks_visited_count").value = ViewMarksPref.most_visited_count;
							folderId = ViewMarksData.mostVisitedFolder;
						}
						if (arg[1] == "lastvisited") {
							document.getElementById("vmks_visited_box").removeAttribute("collapsed");
							document.getElementById("vmks_visited_count").value = ViewMarksPref.last_visited_count;
							folderId = ViewMarksData.lastVisitedFolder;
						}
						if (arg[1] == "history") folderId = ViewMarksData.historyFolder;
						if (arg[1] == "trash") folderId = ViewMarksPref.trash_folder_id;
						if (arg[1] == "bookmarks") folderId = this.bookmarks.bookmarksMenuFolder;
						if (arg[1] == "toolbar") folderId = this.bookmarks.toolbarFolder;
						if (arg[1] == "unsorted") folderId = this.bookmarks.unfiledBookmarksFolder;
					}
					if (arg[0] == "search") {
						patterns = arg[1].split(/\+/);
						this.viewType = this.VIEW_SEARCH;
						if (arg[1].length > 0) {
							this.searchPatterns = patterns;
						} else {
							this.searchPatterns = null;
						}
					}
					if (arg[0] == "flags") {
						flags = parseInt(arg[1]);
					}
				}
				if (this.viewType == this.VIEW_FOLDER) {
					this.initViewWithFolder(folderId);
				}
				if (this.viewType == this.VIEW_SEARCH) {
	/*
					if (folderId == ViewMarksData.historyFolder) {
						this.initViewWithSearch(ViewMarksData.historyFolder, patterns, flags);
					} else if (folderId == ViewMarksPref.trash_folder_id) {
						this.initViewWithSearch(ViewMarksPref.trash_folder_id, patterns, flags);
					} else {
						this.initViewWithSearch(this.bookmarks.placesRoot, patterns, flags);
					}
	*/
					this.initViewWithSearch(folderId, patterns, flags);
				}
			}
	
			var box = document.getElementById("vmks_searchbox");
			if (ViewMarksView.folderId == ViewMarksData.historyFolder) {
				var attr = ViewMarksUtil.getString("search_history_placeholder");
			} else if (folderId == ViewMarksPref.trash_folder_id) {
				var attr = ViewMarksUtil.getString("search_trash_placeholder");
			} else {
				var attr = ViewMarksUtil.getString("search_bookmarks_placeholder");
			}
			box.setAttribute("placeholder", attr);
			box.setAttribute("emptytext", attr);
	
			document.getElementById("vmks_view_context_copytofolder").appendChild(this.createFolderTree(null));
			document.getElementById("vmks_view_context_movetofolder").appendChild(this.createFolderTree(null));
	
			//ViewMarksUtil.getMainWindow().document.getElementById("urlbar").select();
			this.headerHeight = document.getElementById("vmks_viewheader").boxObject.height
		} catch (e) {
			ViewMarksUtil.logerror(e)
			ViewMarksReport.openDialog(e);
		}
	},


	closeView: function() 
	{
		try {
			var doc = ViewMarksUtil.getMainWindow().document;
			doc.removeEventListener("history-changed", ViewMarksView.onHistoryAddItem, false);
		} catch (e) {ViewMarksUtil.alert(e);}
		try {
			this.bookmarks.removeObserver(ViewMarksView.bookmarkObserver, false);
		} catch (e) {}
	},


	bookmarkObserver: {
		timer: null,
	  inBatch: false,
	  doRefresh: function() {
	  	if (!this.inBatch) {
	  		if (this.timer) clearTimeout(this.timer);
				this.timer = setTimeout(function(){ViewMarksView.refreshGrid();}, 500);
	  	}
	  },
	  onBeginUpdateBatch: function() {this.inBatch = true;},
	  onEndUpdateBatch: function() {this.inBatch = false;this.doRefresh();},
	  onBeforeItemRemoved: function(id, type) {},
	  onItemAdded: function(id, folder, index) {if (folder == ViewMarksView.folderId) this.doRefresh();},
	  onItemRemoved: function(id, folder, index) {if (folder == ViewMarksView.folderId) this.doRefresh();},
	  onItemChanged: function(id, property, isAnnotationProperty, value, aLastModified, aItemType, aParentId, aGUID, aParentGUID) {
	  	ViewMarksUtil.logmsg("onItemChanged: "+id+" "+property+" "+value+" type="+aItemType+" parent="+aParentId);
	  	if (property == "title" || property == "lastModified") {
		  	if (aParentId && aParentId != ViewMarksView.folderId) return; 
		  	this.doRefresh();
		  }
	  },
	  onItemVisited: function(id, visitID, time) {},
	  onItemMoved: function(id, oldParent, oldIndex, newParent, newIndex) {
	  	if (oldParent == ViewMarksView.folderId || newParent == ViewMarksView.folderId) this.doRefresh();
	  },
	  QueryInterface: function(iid) {
	    if (iid.equals(Components.interfaces.nsINavBookmarkObserver) ||
	        iid.equals(Components.interfaces.nsISupports)) {
	      return this;
	    }
	    throw Components.results.NS_ERROR_NO_INTERFACE;
	  },
	},


	init: function() 
	{
	},


};


