"use strict";
var AutoCollectConsole = require("./AutoCollection/Console");
var AutoCollectExceptions = require("./AutoCollection/Exceptions");
var AutoCollectPerformance = require("./AutoCollection/Performance");
var AutoCollectClientRequests = require("./AutoCollection/ClientRequests");
var AutoCollectServerRequests = require("./AutoCollection/ServerRequests");
var Client = require("./Library/Client");
var Logging = require("./Library/Logging");
/**
 * The singleton meta class for the default client of the client. This class is used to setup/start and configure
 * the auto-collection behavior of the application insights module.
 */
var ApplicationInsights = (function () {
    function ApplicationInsights() {
    }
    /**
     * Initializes a client with the given instrumentation key, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights/Client} a new client
     */
    ApplicationInsights.getClient = function (instrumentationKey) {
        return new Client(instrumentationKey);
    };
    /**
     * Initializes the default client of the client and sets the default configuration
     * @param instrumentationKey the instrumentation key to use. Optional, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setup = function (instrumentationKey) {
        if (!ApplicationInsights.client) {
            ApplicationInsights.client = ApplicationInsights.getClient(instrumentationKey);
            ApplicationInsights._console = new AutoCollectConsole(ApplicationInsights.client);
            ApplicationInsights._exceptions = new AutoCollectExceptions(ApplicationInsights.client);
            ApplicationInsights._performance = new AutoCollectPerformance(ApplicationInsights.client);
            ApplicationInsights._serverRequests = new AutoCollectServerRequests(ApplicationInsights.client);
            ApplicationInsights._clientRequests = new AutoCollectClientRequests(ApplicationInsights.client);
        }
        else {
            Logging.info("The default client is already setup");
        }
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(ApplicationInsights._isOfflineMode);
        }
        return ApplicationInsights;
    };
    /**
     * Starts automatic collection of telemetry. Prior to calling start no telemetry will be collected
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.start = function () {
        if (!!this.client) {
            ApplicationInsights._isStarted = true;
            ApplicationInsights._console.enable(ApplicationInsights._isConsole);
            ApplicationInsights._exceptions.enable(ApplicationInsights._isExceptions);
            ApplicationInsights._performance.enable(ApplicationInsights._isPerformance);
            ApplicationInsights._serverRequests.enable(ApplicationInsights._isRequests);
            ApplicationInsights._clientRequests.enable(ApplicationInsights._isDependencies);
        }
        else {
            Logging.warn("Start cannot be called before setup");
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of console tracking (enabled by default)
     * @param value if true console activity will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectConsole = function (value) {
        ApplicationInsights._isConsole = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._console.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of exception tracking (enabled by default)
     * @param value if true uncaught exceptions will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectExceptions = function (value) {
        ApplicationInsights._isExceptions = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._exceptions.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of performance tracking (enabled by default)
     * @param value if true performance counters will be collected every second and sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectPerformance = function (value) {
        ApplicationInsights._isPerformance = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._performance.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of request tracking (enabled by default)
     * @param value if true requests will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectRequests = function (value) {
        ApplicationInsights._isRequests = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._serverRequests.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of dependency tracking (enabled by default)
     * @param value if true dependencies will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectDependencies = function (value) {
        ApplicationInsights._isDependencies = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._clientRequests.enable(value);
        }
        return ApplicationInsights;
    };
    /**
    * Enable or disable offline mode to cache events when client is offline (disabled by default)
    * @param value if true events that occured while client is offline will be cached on disk
    * @param resendInterval. The wait interval for resending cached events.
    * @returns {ApplicationInsights} this class
    */
    ApplicationInsights.setOfflineMode = function (value, resendInterval) {
        ApplicationInsights._isOfflineMode = value;
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(value, resendInterval);
        }
        return ApplicationInsights;
    };
    /**
     * Enables verbose debug logging
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.enableVerboseLogging = function () {
        Logging.enableDebug = true;
        return ApplicationInsights;
    };
    /**
      * Disposes the default client and all the auto collectors so they can be reinitialized with different configuration
      */
    ApplicationInsights.dispose = function () {
        ApplicationInsights.client = null;
        ApplicationInsights._isStarted = false;
        if (ApplicationInsights._console) {
            ApplicationInsights._console.dispose();
        }
        if (ApplicationInsights._exceptions) {
            ApplicationInsights._exceptions.dispose();
        }
        if (ApplicationInsights._performance) {
            ApplicationInsights._performance.dispose();
        }
        if (ApplicationInsights._serverRequests) {
            ApplicationInsights._serverRequests.dispose();
        }
        if (ApplicationInsights._clientRequests) {
            ApplicationInsights._clientRequests.dispose();
        }
    };
    return ApplicationInsights;
}());
ApplicationInsights._isConsole = true;
ApplicationInsights._isExceptions = true;
ApplicationInsights._isPerformance = true;
ApplicationInsights._isRequests = true;
ApplicationInsights._isDependencies = true;
ApplicationInsights._isOfflineMode = false;
ApplicationInsights._isStarted = false;
module.exports = ApplicationInsights;
