(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "assert", "../parser/htmlScanner"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    var assert = require("assert");
    var htmlScanner_1 = require("../parser/htmlScanner");
    suite('HTML Scanner', function () {
        function assertTokens(tests) {
            var scannerState = htmlScanner_1.ScannerState.WithinContent;
            for (var _i = 0, tests_1 = tests; _i < tests_1.length; _i++) {
                var t = tests_1[_i];
                var scanner = htmlScanner_1.createScanner(t.input, 0, scannerState);
                var tokenType = scanner.scan();
                var actual = [];
                while (tokenType !== htmlScanner_1.TokenType.EOS) {
                    var actualToken = { offset: scanner.getTokenOffset(), type: tokenType };
                    if (tokenType === htmlScanner_1.TokenType.StartTag || tokenType === htmlScanner_1.TokenType.EndTag) {
                        actualToken.content = t.input.substr(scanner.getTokenOffset(), scanner.getTokenLength());
                    }
                    actual.push(actualToken);
                    tokenType = scanner.scan();
                }
                assert.deepEqual(actual, t.tokens);
                scannerState = scanner.getScannerState();
            }
        }
        test('Open Start Tag #1', function () {
            assertTokens([{
                    input: '<abc',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' }
                    ]
                }
            ]);
        });
        test('Open Start Tag #2', function () {
            assertTokens([{
                    input: '<input',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'input' }
                    ]
                }
            ]);
        });
        test('Open Start Tag with Invalid Tag', function () {
            assertTokens([{
                    input: '< abc',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 2, type: htmlScanner_1.TokenType.StartTag, content: 'abc' }
                    ]
                }
            ]);
        });
        test('Open Start Tag #3', function () {
            assertTokens([{
                    input: '< abc>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 2, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 5, type: htmlScanner_1.TokenType.StartTagClose },
                    ]
                }
            ]);
        });
        test('Open Start Tag #4', function () {
            assertTokens([{
                    input: 'i <len;',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.Content },
                        { offset: 2, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 3, type: htmlScanner_1.TokenType.StartTag, content: 'len' },
                        { offset: 6, type: htmlScanner_1.TokenType.Unknown }
                    ]
                }
            ]);
        });
        test('Open Start Tag #5', function () {
            assertTokens([{
                    input: '<',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen }
                    ]
                }
            ]);
        });
        test('Open End Tag', function () {
            assertTokens([{
                    input: '</a',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 2, type: htmlScanner_1.TokenType.EndTag, content: 'a' }
                    ]
                }
            ]);
        });
        test('Complete Start Tag', function () {
            assertTokens([{
                    input: '<abc>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Complete Start Tag with Whitespace', function () {
            assertTokens([{
                    input: '<abc >',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('bug 9809 - Complete Start Tag with Namespaceprefix', function () {
            assertTokens([{
                    input: '<foo:bar>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'foo:bar' },
                        { offset: 8, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Complete End Tag', function () {
            assertTokens([{
                    input: '</abc>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 2, type: htmlScanner_1.TokenType.EndTag, content: 'abc' },
                        { offset: 5, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Complete End Tag with Whitespace', function () {
            assertTokens([{
                    input: '</abc  >',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 2, type: htmlScanner_1.TokenType.EndTag, content: 'abc' },
                        { offset: 5, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 7, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Empty Tag', function () {
            assertTokens([{
                    input: '<abc />',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.StartTagSelfClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #1', function () {
            assertTokens([{
                    input: '<script type="text/javascript">var i= 10;</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 31, type: htmlScanner_1.TokenType.Script },
                        { offset: 41, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 43, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 49, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #2', function () {
            assertTokens([{
                    input: '<script type="text/javascript">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }, {
                    input: 'var i= 10;',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.Script }
                    ]
                }, {
                    input: '</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 2, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 8, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #3', function () {
            assertTokens([{
                    input: '<script type="text/javascript">var i= 10;',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 31, type: htmlScanner_1.TokenType.Script }
                    ]
                }, {
                    input: '</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 2, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 8, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #4', function () {
            assertTokens([{
                    input: '<script type="text/javascript">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }, {
                    input: 'var i= 10;</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.Script },
                        { offset: 10, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 12, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 18, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #5', function () {
            assertTokens([{
                    input: '<script type="text/plain">a\n<a</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 25, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 26, type: htmlScanner_1.TokenType.Script },
                        { offset: 30, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 32, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 38, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #6', function () {
            assertTokens([{
                    input: '<script>a</script><script>b</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 9, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 11, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 17, type: htmlScanner_1.TokenType.EndTagClose },
                        { offset: 18, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 19, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 25, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 26, type: htmlScanner_1.TokenType.Script },
                        { offset: 27, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 29, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 35, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #7', function () {
            assertTokens([{
                    input: '<script type="text/javascript"></script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 31, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 33, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 39, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #8', function () {
            assertTokens([{
                    input: '<script>var i= 10;</script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 18, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 20, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 26, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #9', function () {
            assertTokens([{
                    input: '<script type="text/javascript" src="main.js"></script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 8, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 30, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 31, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 34, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 35, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 44, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 45, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 47, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 53, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #10', function () {
            assertTokens([{
                    input: '<script><!-- alert("<script></script>"); --></script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 44, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 46, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 52, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #11', function () {
            assertTokens([{
                    input: '<script><!-- alert("<script></script>"); </script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 41, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 43, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 49, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #12', function () {
            assertTokens([{
                    input: '<script><!-- alert("</script>"); </script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 20, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 22, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 28, type: htmlScanner_1.TokenType.EndTagClose },
                        { offset: 29, type: htmlScanner_1.TokenType.Content },
                        { offset: 33, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 35, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 41, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Embedded Content #13', function () {
            assertTokens([{
                    input: '<script> alert("<script></script>"); </script>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script },
                        { offset: 24, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 26, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 32, type: htmlScanner_1.TokenType.EndTagClose },
                        { offset: 33, type: htmlScanner_1.TokenType.Content },
                        { offset: 37, type: htmlScanner_1.TokenType.EndTagOpen },
                        { offset: 39, type: htmlScanner_1.TokenType.EndTag, content: 'script' },
                        { offset: 45, type: htmlScanner_1.TokenType.EndTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Attribute', function () {
            assertTokens([{
                    input: '<abc foo="bar">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 14, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Empty Attribute Value', function () {
            assertTokens([{
                    input: '<abc foo=\'bar\'>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 14, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with empty attributes', function () {
            assertTokens([{
                    input: '<abc foo="">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 11, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Attributes', function () {
            assertTokens([{
                    input: '<abc foo="bar" bar=\'foo\'>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 14, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 15, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 18, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 19, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 24, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Attributes, no quotes', function () {
            assertTokens([{
                    input: '<abc foo=bar bar=help-me>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 12, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 13, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 16, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 17, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 24, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Attributes, no quotes, self close', function () {
            assertTokens([{
                    input: '<abc foo=bar/>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 12, type: htmlScanner_1.TokenType.StartTagSelfClose }
                    ]
                }
            ]);
        });
        test('Tag with Attribute And Whitespace', function () {
            assertTokens([{
                    input: '<abc foo=  "bar">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 11, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 16, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Attribute And Whitespace #2', function () {
            assertTokens([{
                    input: '<abc foo = "bar">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 9, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 10, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 11, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 16, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Name-Only-Attribute #1', function () {
            assertTokens([{
                    input: '<abc foo>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Name-Only-Attribute #2', function () {
            assertTokens([{
                    input: '<abc foo bar>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 12, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Interesting Attribute Name', function () {
            assertTokens([{
                    input: '<abc foo!@#="bar">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 11, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 12, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 17, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Angular Attribute Name', function () {
            assertTokens([{
                    input: '<abc #myinput (click)="bar" [value]="someProperty" *ngIf="someCondition">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 13, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 14, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 21, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 22, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 27, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 28, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 35, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 36, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 50, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 51, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 56, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 57, type: htmlScanner_1.TokenType.AttributeValue },
                        { offset: 72, type: htmlScanner_1.TokenType.StartTagClose }
                    ]
                }
            ]);
        });
        test('Tag with Invalid Attribute Value', function () {
            assertTokens([{
                    input: '<abc foo=">',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'abc' },
                        { offset: 4, type: htmlScanner_1.TokenType.Whitespace },
                        { offset: 5, type: htmlScanner_1.TokenType.AttributeName },
                        { offset: 8, type: htmlScanner_1.TokenType.DelimiterAssign },
                        { offset: 9, type: htmlScanner_1.TokenType.AttributeValue }
                    ]
                }
            ]);
        });
        test('Simple Comment 1', function () {
            assertTokens([{
                    input: '<!--a-->',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartCommentTag },
                        { offset: 4, type: htmlScanner_1.TokenType.Comment },
                        { offset: 5, type: htmlScanner_1.TokenType.EndCommentTag }
                    ]
                }
            ]);
        });
        test('Simple Comment 2', function () {
            assertTokens([{
                    input: '<!--a>foo bar</a -->',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartCommentTag },
                        { offset: 4, type: htmlScanner_1.TokenType.Comment },
                        { offset: 17, type: htmlScanner_1.TokenType.EndCommentTag }
                    ]
                }
            ]);
        });
        test('Multiline Comment', function () {
            assertTokens([{
                    input: '<!--a>\nfoo \nbar</a -->',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartCommentTag },
                        { offset: 4, type: htmlScanner_1.TokenType.Comment },
                        { offset: 19, type: htmlScanner_1.TokenType.EndCommentTag }
                    ]
                }
            ]);
        });
        test('Simple Doctype', function () {
            assertTokens([{
                    input: '<!Doctype a>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartDoctypeTag },
                        { offset: 9, type: htmlScanner_1.TokenType.Doctype },
                        { offset: 11, type: htmlScanner_1.TokenType.EndDoctypeTag }
                    ]
                }
            ]);
        });
        test('Simple Doctype #2', function () {
            assertTokens([{
                    input: '<!doctype a>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartDoctypeTag },
                        { offset: 9, type: htmlScanner_1.TokenType.Doctype },
                        { offset: 11, type: htmlScanner_1.TokenType.EndDoctypeTag }
                    ]
                }
            ]);
        });
        test('Simple Doctype #4', function () {
            assertTokens([{
                    input: '<!DOCTYPE a\n"foo" \'bar\'>',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartDoctypeTag },
                        { offset: 9, type: htmlScanner_1.TokenType.Doctype },
                        { offset: 23, type: htmlScanner_1.TokenType.EndDoctypeTag }
                    ]
                }
            ]);
        });
        test('Incomplete', function () {
            assertTokens([{
                    input: '    ',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.Content }
                    ]
                }]);
            assertTokens([{
                    input: '<!---   ',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartCommentTag },
                        { offset: 4, type: htmlScanner_1.TokenType.Comment }
                    ]
                }]);
            assertTokens([{
                    input: '<style>color:red',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'style' },
                        { offset: 6, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 7, type: htmlScanner_1.TokenType.Styles }
                    ]
                }]);
            assertTokens([{
                    input: '<script>alert("!!")',
                    tokens: [
                        { offset: 0, type: htmlScanner_1.TokenType.StartTagOpen },
                        { offset: 1, type: htmlScanner_1.TokenType.StartTag, content: 'script' },
                        { offset: 7, type: htmlScanner_1.TokenType.StartTagClose },
                        { offset: 8, type: htmlScanner_1.TokenType.Script }
                    ]
                }]);
        });
    });
});
//# sourceMappingURL=scanner.test.js.map