/// <reference path="thenable.d.ts" />
/// <reference types="node" />
import { RequestType, RequestHandler, NotificationType, NotificationHandler, ResponseError, ErrorCodes, MessageReader, DataCallback, StreamMessageReader, IPCMessageReader, MessageWriter, StreamMessageWriter, IPCMessageWriter, Event } from 'vscode-jsonrpc';
import { TextDocument, TextDocumentChangeEvent, TextDocumentContentChangeEvent, Range, Position, Location, Diagnostic, DiagnosticSeverity, Command, TextEdit, WorkspaceEdit, WorkspaceChange, TextEditChange, TextDocumentIdentifier, CompletionItemKind, CompletionItem, CompletionList, Hover, MarkedString, SignatureHelp, SignatureInformation, ParameterInformation, Definition, CodeActionContext, DocumentHighlight, DocumentHighlightKind, SymbolInformation, SymbolKind, CodeLens, FormattingOptions, DocumentLink } from 'vscode-languageserver-types';
import { InitializeParams, InitializeResult, InitializeError, ServerCapabilities, MessageActionItem, DidChangeConfigurationParams, DidOpenTextDocumentParams, DidChangeTextDocumentParams, DidCloseTextDocumentParams, DidSaveTextDocumentParams, DidChangeWatchedFilesParams, FileEvent, FileChangeType, PublishDiagnosticsParams, TextDocumentPositionParams, TextDocumentSyncKind, CompletionOptions, ReferenceParams, DocumentSymbolParams, WorkspaceSymbolParams, CodeActionParams, CodeLensOptions, CodeLensRequest, CodeLensParams, CodeLensResolveRequest, DocumentFormattingRequest, DocumentFormattingParams, DocumentRangeFormattingRequest, DocumentRangeFormattingParams, DocumentOnTypeFormattingRequest, DocumentOnTypeFormattingParams, RenameRequest, RenameParams, DocumentLinkRequest, DocumentLinkResolveRequest, DocumentLinkParams } from './protocol';
export { RequestType, RequestHandler, NotificationType, NotificationHandler, ResponseError, ErrorCodes, MessageReader, DataCallback, StreamMessageReader, IPCMessageReader, MessageWriter, StreamMessageWriter, IPCMessageWriter, MessageActionItem, InitializeParams, InitializeResult, InitializeError, ServerCapabilities, DidChangeConfigurationParams, DidChangeWatchedFilesParams, FileEvent, FileChangeType, DidOpenTextDocumentParams, DidChangeTextDocumentParams, TextDocumentContentChangeEvent, DidCloseTextDocumentParams, DidSaveTextDocumentParams, PublishDiagnosticsParams, Diagnostic, DiagnosticSeverity, Range, Position, Location, TextDocumentIdentifier, TextDocumentPositionParams, TextDocumentSyncKind, Hover, MarkedString, CompletionOptions, CompletionItemKind, CompletionItem, CompletionList, TextDocument, TextEdit, WorkspaceEdit, WorkspaceChange, TextEditChange, SignatureHelp, SignatureInformation, ParameterInformation, Definition, ReferenceParams, DocumentHighlight, DocumentHighlightKind, SymbolInformation, SymbolKind, DocumentSymbolParams, WorkspaceSymbolParams, CodeActionParams, CodeActionContext, Command, CodeLensRequest, CodeLensParams, CodeLensResolveRequest, CodeLens, CodeLensOptions, DocumentFormattingRequest, DocumentFormattingParams, DocumentRangeFormattingRequest, DocumentRangeFormattingParams, DocumentOnTypeFormattingRequest, DocumentOnTypeFormattingParams, FormattingOptions, RenameRequest, RenameParams, DocumentLink, DocumentLinkRequest, DocumentLinkResolveRequest };
export { Event };
import * as fm from './files';
export declare namespace Files {
    let uriToFilePath: typeof fm.uriToFilePath;
    let resolveModule: typeof fm.resolveModule;
    let resolveModule2: typeof fm.resolveModule2;
    let resolveModulePath: typeof fm.resolveModulePath;
}
/**
 * A manager for simple text documents
 */
export declare class TextDocuments {
    private _documents;
    private _onDidChangeContent;
    private _onDidOpen;
    private _onDidClose;
    private _onDidSave;
    /**
     * Create a new text document manager.
     */
    constructor();
    /**
     * Returns the [TextDocumentSyncKind](#TextDocumentSyncKind) used by
     * this text document manager.
     */
    readonly syncKind: TextDocumentSyncKind;
    /**
     * An event that fires when a text document managed by this manager
     * has been opened or the content changes.
     */
    readonly onDidChangeContent: Event<TextDocumentChangeEvent>;
    /**
     * An event that fires when a text document managed by this manager
     * has been opened.
     */
    readonly onDidOpen: Event<TextDocumentChangeEvent>;
    /**
     * An event that fires when a text document managed by this manager
     * has been closed.
     */
    readonly onDidClose: Event<TextDocumentChangeEvent>;
    /**
     * An event that fires when a text document managed by this manager
     * has been closed.
     */
    readonly onDidSave: Event<TextDocumentChangeEvent>;
    /**
     * Returns the document for the given URI. Returns undefined if
     * the document is not mananged by this instance.
     *
     * @param uri The text document's URI to retrieve.
     * @return the text document or `undefined`.
     */
    get(uri: string): TextDocument;
    /**
     * Returns all text documents managed by this instance.
     *
     * @return all text documents.
     */
    all(): TextDocument[];
    /**
     * Returns the URIs of all text documents managed by this instance.
     *
     * @return the URI's of all text documents.
     */
    keys(): string[];
    /**
     * Listens for `low level` notification on the given connection to
     * update the text documents managed by this instance.
     *
     * @param connection The connection to listen on.
     */
    listen(connection: IConnection): void;
}
/**
 * Helps tracking error message. Equal occurences of the same
 * message are only stored once. This class is for example
 * usefull if text documents are validated in a loop and equal
 * error message should be folded into one.
 */
export declare class ErrorMessageTracker {
    private messages;
    constructor();
    /**
     * Add a message to the tracker.
     *
     * @param message The message to add.
     */
    add(message: string): void;
    /**
     * Send all tracked messages to the conenction's window.
     *
     * @param connection The connection establised between client and server.
     */
    sendErrors(connection: {
        window: RemoteWindow;
    }): void;
}
/**
 * The RemoteConsole interface contains all functions to interact with
 * the developer console of VS Code.
 */
export interface RemoteConsole {
    /**
     * Show an error message.
     *
     * @param message The message to show.
     */
    error(message: string): any;
    /**
     * Show a warning message.
     *
     * @param message The message to show.
     */
    warn(message: string): any;
    /**
     * Show an information message.
     *
     * @param message The message to show.
     */
    info(message: string): any;
    /**
     * Log a message.
     *
     * @param message The message to log.
     */
    log(message: string): any;
}
/**
 * The RemoteWindow interface contains all functions to interact with
 * the visual window of VS Code.
 */
export interface RemoteWindow {
    /**
     * Show an error message.
     *
     * @param message The message to show.
     */
    showErrorMessage(message: string): any;
    showErrorMessage<T extends MessageActionItem>(message: string, ...actions: T[]): Thenable<T>;
    /**
     * Show a warning message.
     *
     * @param message The message to show.
     */
    showWarningMessage(message: string): any;
    showWarningMessage<T extends MessageActionItem>(message: string, ...actions: T[]): Thenable<T>;
    /**
     * Show an information message.
     *
     * @param message The message to show.
     */
    showInformationMessage(message: string): any;
    showInformationMessage<T extends MessageActionItem>(message: string, ...actions: T[]): Thenable<T>;
}
/**
 * Interface to log telemetry events. The events are actually send to the client
 * and the client needs to feed the event into a propert telemetry system.
 */
export interface Telemetry {
    /**
     * Log the given data to telemetry.
     *
     * @param data The data to log. Must be a JSON serializable object.
     */
    logEvent(data: any): void;
}
/**
 * Interface to log traces to the client. The events are sent to the client and the
 * client needs to log the trace events.
 */
export interface Tracer {
    /**
     * Log the given data to the trace Log
     */
    log(message: string, verbose?: string): void;
}
/**
 * Interface to describe the shape of the server connection.
 */
export interface IConnection {
    /**
     * Start listening on the input stream for messages to process.
     */
    listen(): void;
    /**
     * Installs a request handler described by the given [RequestType](#RequestType).
     *
     * @param type The [RequestType](#RequestType) describing the request.
     * @param handler The handler to install
     */
    onRequest<P, R, E>(type: RequestType<P, R, E>, handler: RequestHandler<P, R, E>): void;
    /**
     * Installs a notification handler described by the given [NotificationType](#NotificationType).
     *
     * @param type The [NotificationType](#NotificationType) describing the notification.
     * @param handler The handler to install
     */
    onNotification<P>(type: NotificationType<P>, handler: NotificationHandler<P>): void;
    /**
     * Send a notification to the client.
     *
     * @param type The [NotificationType](#NotificationType) describing the notification.
     * @param params The notification's parameters.
     */
    sendNotification<P>(type: NotificationType<P>, params?: P): void;
    /**
     * Send a request to the client.
     *
     * @param type The [RequestType](#RequestType) describing the request.
     * @param params The request's parameters.
     */
    sendRequest<P, R, E>(type: RequestType<P, R, E>, params?: P): Thenable<R>;
    /**
     * Installs a handler for the intialize request.
     *
     * @param handler The initialize handler.
     */
    onInitialize(handler: RequestHandler<InitializeParams, InitializeResult, InitializeError>): void;
    /**
     * Installs a handler for the shutdown request.
     *
     * @param handler The initialize handler.
     */
    onShutdown(handler: RequestHandler<void, void, void>): void;
    /**
     * Installs a handler for the exit notification.
     *
     * @param handler The exit handler.
     */
    onExit(handler: NotificationHandler<void>): void;
    /**
     * A proxy for VSCode's development console. See [RemoteConsole](#RemoteConsole)
     */
    console: RemoteConsole;
    /**
     * A proxy for VSCode's window. See [RemoteWindow](#RemoteWindow)
     */
    window: RemoteWindow;
    /**
     * A proxy to send telemetry events to the client.
     */
    telemetry: Telemetry;
    /**
     * A proxy to send trace events to the client.
     */
    tracer: Tracer;
    /**
     * Installs a handler for the `DidChangeConfiguration` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidChangeConfiguration(handler: NotificationHandler<DidChangeConfigurationParams>): void;
    /**
     * Installs a handler for the `DidChangeWatchedFiles` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidChangeWatchedFiles(handler: NotificationHandler<DidChangeWatchedFilesParams>): void;
    /**
     * Installs a handler for the `DidOpenTextDocument` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidOpenTextDocument(handler: NotificationHandler<DidOpenTextDocumentParams>): void;
    /**
     * Installs a handler for the `DidChangeTextDocument` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidChangeTextDocument(handler: NotificationHandler<DidChangeTextDocumentParams>): void;
    /**
     * Installs a handler for the `DidCloseTextDocument` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidCloseTextDocument(handler: NotificationHandler<DidCloseTextDocumentParams>): void;
    /**
     * Installs a handler for the `DidSaveTextDocument` notification.
     *
     * @param handler The corresponding handler.
     */
    onDidSaveTextDocument(handler: NotificationHandler<DidSaveTextDocumentParams>): void;
    /**
     * Sends diagnostics computed for a given document to VSCode to render them in the
     * user interface.
     *
     * @param params The diagnostic parameters.
     */
    sendDiagnostics(params: PublishDiagnosticsParams): void;
    /**
     * Installs a handler for the `Hover` request.
     *
     * @param handler The corresponding handler.
     */
    onHover(handler: RequestHandler<TextDocumentPositionParams, Hover, void>): void;
    /**
     * Installs a handler for the `Completion` request.
     *
     * @param handler The corresponding handler.
     */
    onCompletion(handler: RequestHandler<TextDocumentPositionParams, CompletionItem[] | CompletionList, void>): void;
    /**
     * Installs a handler for the `CompletionResolve` request.
     *
     * @param handler The corresponding handler.
     */
    onCompletionResolve(handler: RequestHandler<CompletionItem, CompletionItem, void>): void;
    /**
     * Installs a handler for the `SignatureHelp` request.
     *
     * @param handler The corresponding handler.
     */
    onSignatureHelp(handler: RequestHandler<TextDocumentPositionParams, SignatureHelp, void>): void;
    /**
     * Installs a handler for the `Definition` request.
     *
     * @param handler The corresponding handler.
     */
    onDefinition(handler: RequestHandler<TextDocumentPositionParams, Definition, void>): void;
    /**
     * Installs a handler for the `References` request.
     *
     * @param handler The corresponding handler.
     */
    onReferences(handler: RequestHandler<ReferenceParams, Location[], void>): void;
    /**
     * Installs a handler for the `DocumentHighlight` request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentHighlight(handler: RequestHandler<TextDocumentPositionParams, DocumentHighlight[], void>): void;
    /**
     * Installs a handler for the `DocumentSymbol` request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentSymbol(handler: RequestHandler<DocumentSymbolParams, SymbolInformation[], void>): void;
    /**
     * Installs a handler for the `WorkspaceSymbol` request.
     *
     * @param handler The corresponding handler.
     */
    onWorkspaceSymbol(handler: RequestHandler<WorkspaceSymbolParams, SymbolInformation[], void>): void;
    /**
     * Installs a handler for the `CodeAction` request.
     *
     * @param handler The corresponding handler.
     */
    onCodeAction(handler: RequestHandler<CodeActionParams, Command[], void>): void;
    /**
     * Compute a list of [lenses](#CodeLens). This call should return as fast as possible and if
     * computing the commands is expensive implementors should only return code lens objects with the
     * range set and handle the resolve request.
     *
     * @param handler The corresponding handler.
     */
    onCodeLens(handler: RequestHandler<CodeLensParams, CodeLens[], void>): void;
    /**
     * This function will be called for each visible code lens, usually when scrolling and after
     * the onCodeLens has been called.
     *
     * @param handler The corresponding handler.
     */
    onCodeLensResolve(handler: RequestHandler<CodeLens, CodeLens, void>): void;
    /**
     * Installs a handler for the document formatting request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentFormatting(handler: RequestHandler<DocumentFormattingParams, TextEdit[], void>): void;
    /**
     * Installs a handler for the document range formatting request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentRangeFormatting(handler: RequestHandler<DocumentRangeFormattingParams, TextEdit[], void>): void;
    /**
     * Installs a handler for the document on type formatting request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentOnTypeFormatting(handler: RequestHandler<DocumentOnTypeFormattingParams, TextEdit[], void>): void;
    /**
     * Installs a handler for the rename request.
     *
     * @param handler The corresponding handler.
     */
    onRenameRequest(handler: RequestHandler<RenameParams, WorkspaceEdit, void>): void;
    /**
     * Installs a handler for the document links request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentLinks(handler: RequestHandler<DocumentLinkParams, DocumentLink[], void>): void;
    /**
     * Installs a handler for the document links resolve request.
     *
     * @param handler The corresponding handler.
     */
    onDocumentLinkResolve(handler: RequestHandler<DocumentLink, DocumentLink, void>): void;
    /**
     * Disposes the connection
     */
    dispose(): void;
}
/**
 * Creates a new connection using a the given streams.
 *
 * @param inputStream The stream to read messages from.
 * @param outputStream The stream to write messages to.
 * @return a [connection](#IConnection)
 */
export declare function createConnection(inputStream: NodeJS.ReadableStream, outputStream: NodeJS.WritableStream): IConnection;
/**
 * Creates a new connection.
 *
 * @param reader The message reader to read messages from.
 * @param writer The message writer to write message to.
 */
export declare function createConnection(reader: MessageReader, writer: MessageWriter): IConnection;
/**
 * Creates a new connection based on the processes command line arguments:
 * --ipc : connection using the node  process ipc
 *
 * @param reader The message reader to read messages from.
 * @param writer The message writer to write message to.
 */
export declare function createConnection(): IConnection;
