/**
 * Uniform Resource Identifier (Uri) http://tools.ietf.org/html/rfc3986.
 * This class is a simple parser which creates the basic component paths
 * (http://tools.ietf.org/html/rfc3986#section-3) with minimal validation
 * and encoding.
 *
 *       foo://example.com:8042/over/there?name=ferret#nose
 *       \_/   \______________/\_________/ \_________/ \__/
 *        |           |            |            |        |
 *     scheme     authority       path        query   fragment
 *        |   _____________________|__
 *       / \ /                        \
 *       urn:example:animal:ferret:nose
 *
 *
 */
export default class Uri {
    private static _empty;
    private static _slash;
    private static _regexp;
    private static _driveLetterPath;
    private static _upperCaseDrive;
    private _scheme;
    private _authority;
    private _path;
    private _query;
    private _fragment;
    private _formatted;
    private _fsPath;
    private constructor();
    /**
     * scheme is the 'http' part of 'http://www.msft.com/some/path?query#fragment'.
     * The part before the first colon.
     */
    readonly scheme: string;
    /**
     * authority is the 'www.msft.com' part of 'http://www.msft.com/some/path?query#fragment'.
     * The part between the first double slashes and the next slash.
     */
    readonly authority: string;
    /**
     * path is the '/some/path' part of 'http://www.msft.com/some/path?query#fragment'.
     */
    readonly path: string;
    /**
     * query is the 'query' part of 'http://www.msft.com/some/path?query#fragment'.
     */
    readonly query: string;
    /**
     * fragment is the 'fragment' part of 'http://www.msft.com/some/path?query#fragment'.
     */
    readonly fragment: string;
    /**
     * Returns a string representing the corresponding file system path of this Uri.
     * Will handle UNC paths and normalize windows drive letters to lower-case. Also
     * uses the platform specific path separator. Will *not* validate the path for
     * invalid characters and semantics. Will *not* look at the scheme of this Uri.
     */
    readonly fsPath: string;
    /**
     * Derive a new Uri from this Uri.
     *
     * @param change An object that describes a change to this Uri.
     * @return A new Uri that reflects the given change. Will return `this` Uri if the change
     *  is not changing anything.
     * @sample ```
        let file = Uri.parse('before:some/file/path');
        let other = file.with({ scheme: 'after' });
        assert.ok(other.toString() === 'after:some/file/path');
        * ```
        */
    with(change: {
        scheme?: string;
        authority?: string;
        path?: string;
        query?: string;
        fragment?: string;
    }): Uri;
    /**
     * Create an Uri from uri components.
     *
     * @param components An object containing the Uri components
     * @return A new Uri instance
     */
    static from(components: {
        scheme?: string;
        authority?: string;
        path?: string;
        query?: string;
        fragment?: string;
    }): Uri;
    /**
     * Create an Uri from a string. Will throw if the given value is not
     * valid.
     *
     * @param value The string value of an Uri.
     * @return A new Uri instance.
     */
    static parse(value: string): Uri;
    /**
     * Create an Uri from a file system path. The [scheme](#Uri.scheme)
     * will be `file`.
     *
     * @param path A file system or UNC path.
     * @return A new Uri instance.
     */
    static file(path: string): Uri;
    private static _parseComponents(value);
    private static _validate(ret);
    /**
     *
     * @param skipEncoding Do not encode the result, default is `false`
     */
    toString(skipEncoding?: boolean): string;
    private static _asFormatted(uri, skipEncoding);
    toJSON(): any;
}
