(function (factory) {
    if (typeof module === 'object' && typeof module.exports === 'object') {
        var v = factory(require, exports); if (v !== undefined) module.exports = v;
    }
    else if (typeof define === 'function' && define.amd) {
        define(["require", "exports"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    function _encode(ch) {
        return '%' + ch.charCodeAt(0).toString(16).toUpperCase();
    }
    // see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/encodeURIComponent
    function encodeURIComponent2(str) {
        return encodeURIComponent(str).replace(/[!'()*]/g, _encode);
    }
    function encodeNoop(str) {
        return str;
    }
    /**
     * Uniform Resource Identifier (Uri) http://tools.ietf.org/html/rfc3986.
     * This class is a simple parser which creates the basic component paths
     * (http://tools.ietf.org/html/rfc3986#section-3) with minimal validation
     * and encoding.
     *
     *       foo://example.com:8042/over/there?name=ferret#nose
     *       \_/   \______________/\_________/ \_________/ \__/
     *        |           |            |            |        |
     *     scheme     authority       path        query   fragment
     *        |   _____________________|__
     *       / \ /                        \
     *       urn:example:animal:ferret:nose
     *
     *
     */
    var Uri = (function () {
        function Uri() {
            this._scheme = Uri._empty;
            this._authority = Uri._empty;
            this._path = Uri._empty;
            this._query = Uri._empty;
            this._fragment = Uri._empty;
            this._formatted = null;
            this._fsPath = null;
        }
        Object.defineProperty(Uri.prototype, "scheme", {
            /**
             * scheme is the 'http' part of 'http://www.msft.com/some/path?query#fragment'.
             * The part before the first colon.
             */
            get: function () {
                return this._scheme;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Uri.prototype, "authority", {
            /**
             * authority is the 'www.msft.com' part of 'http://www.msft.com/some/path?query#fragment'.
             * The part between the first double slashes and the next slash.
             */
            get: function () {
                return this._authority;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Uri.prototype, "path", {
            /**
             * path is the '/some/path' part of 'http://www.msft.com/some/path?query#fragment'.
             */
            get: function () {
                return this._path;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Uri.prototype, "query", {
            /**
             * query is the 'query' part of 'http://www.msft.com/some/path?query#fragment'.
             */
            get: function () {
                return this._query;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Uri.prototype, "fragment", {
            /**
             * fragment is the 'fragment' part of 'http://www.msft.com/some/path?query#fragment'.
             */
            get: function () {
                return this._fragment;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(Uri.prototype, "fsPath", {
            // ---- filesystem path -----------------------
            /**
             * Returns a string representing the corresponding file system path of this Uri.
             * Will handle UNC paths and normalize windows drive letters to lower-case. Also
             * uses the platform specific path separator. Will *not* validate the path for
             * invalid characters and semantics. Will *not* look at the scheme of this Uri.
             */
            get: function () {
                if (!this._fsPath) {
                    var value;
                    if (this._authority && this.scheme === 'file') {
                        // unc path: file://shares/c$/far/boo
                        value = "//" + this._authority + this._path;
                    }
                    else if (Uri._driveLetterPath.test(this._path)) {
                        // windows drive letter: file:///c:/far/boo
                        value = this._path[1].toLowerCase() + this._path.substr(2);
                    }
                    else {
                        // other path
                        value = this._path;
                    }
                    if (isWindows) {
                        value = value.replace(/\//g, '\\');
                    }
                    this._fsPath = value;
                }
                return this._fsPath;
            },
            enumerable: true,
            configurable: true
        });
        // ---- modify to new -------------------------
        /**
         * Derive a new Uri from this Uri.
         *
         * @param change An object that describes a change to this Uri.
         * @return A new Uri that reflects the given change. Will return `this` Uri if the change
         *  is not changing anything.
         * @sample ```
            let file = Uri.parse('before:some/file/path');
            let other = file.with({ scheme: 'after' });
            assert.ok(other.toString() === 'after:some/file/path');
            * ```
            */
        Uri.prototype.with = function (change) {
            if (!change) {
                return this;
            }
            var scheme = change.scheme || this.scheme;
            var authority = change.authority || this.authority;
            var path = change.path || this.path;
            var query = change.query || this.query;
            var fragment = change.fragment || this.fragment;
            if (scheme === this.scheme
                && authority === this.authority
                && path === this.path
                && query === this.query
                && fragment === this.fragment) {
                return this;
            }
            var ret = new Uri();
            ret._scheme = scheme;
            ret._authority = authority;
            ret._path = path;
            ret._query = query;
            ret._fragment = fragment;
            Uri._validate(ret);
            return ret;
        };
        // ---- parse & validate ------------------------
        /**
         * Create an Uri from uri components.
         *
         * @param components An object containing the Uri components
         * @return A new Uri instance
         */
        Uri.from = function (components) {
            if (!components) {
                throw new Error();
            }
            return new Uri().with(components);
        };
        /**
         * Create an Uri from a string. Will throw if the given value is not
         * valid.
         *
         * @param value The string value of an Uri.
         * @return A new Uri instance.
         */
        Uri.parse = function (value) {
            var ret = new Uri();
            var data = Uri._parseComponents(value);
            ret._scheme = data.scheme;
            ret._authority = decodeURIComponent(data.authority);
            ret._path = decodeURIComponent(data.path);
            ret._query = decodeURIComponent(data.query);
            ret._fragment = decodeURIComponent(data.fragment);
            Uri._validate(ret);
            return ret;
        };
        /**
         * Create an Uri from a file system path. The [scheme](#Uri.scheme)
         * will be `file`.
         *
         * @param path A file system or UNC path.
         * @return A new Uri instance.
         */
        Uri.file = function (path) {
            var ret = new Uri();
            ret._scheme = 'file';
            // normalize to fwd-slashes
            path = path.replace(/\\/g, Uri._slash);
            // check for authority as used in UNC shares
            // or use the path as given
            if (path[0] === Uri._slash && path[0] === path[1]) {
                var idx = path.indexOf(Uri._slash, 2);
                if (idx === -1) {
                    ret._authority = path.substring(2);
                }
                else {
                    ret._authority = path.substring(2, idx);
                    ret._path = path.substring(idx);
                }
            }
            else {
                ret._path = path;
            }
            // Ensure that path starts with a slash
            // or that it is at least a slash
            if (ret._path[0] !== Uri._slash) {
                ret._path = Uri._slash + ret._path;
            }
            Uri._validate(ret);
            return ret;
        };
        Uri._parseComponents = function (value) {
            var ret = {
                scheme: Uri._empty,
                authority: Uri._empty,
                path: Uri._empty,
                query: Uri._empty,
                fragment: Uri._empty,
            };
            var match = Uri._regexp.exec(value);
            if (match) {
                ret.scheme = match[2] || ret.scheme;
                ret.authority = match[4] || ret.authority;
                ret.path = match[5] || ret.path;
                ret.query = match[7] || ret.query;
                ret.fragment = match[9] || ret.fragment;
            }
            return ret;
        };
        Uri._validate = function (ret) {
            // validation
            // path, http://tools.ietf.org/html/rfc3986#section-3.3
            // If a Uri contains an authority component, then the path component
            // must either be empty or begin with a slash ("/") character.  If a Uri
            // does not contain an authority component, then the path cannot begin
            // with two slash characters ("//").
            if (ret.authority && ret.path && ret.path[0] !== '/') {
                throw new Error('[UriError]: If a Uri contains an authority component, then the path component must either be empty or begin with a slash ("/") character');
            }
            if (!ret.authority && ret.path.indexOf('//') === 0) {
                throw new Error('[UriError]: If a Uri does not contain an authority component, then the path cannot begin with two slash characters ("//")');
            }
        };
        // ---- printing/externalize ---------------------------
        /**
         *
         * @param skipEncoding Do not encode the result, default is `false`
         */
        Uri.prototype.toString = function (skipEncoding) {
            if (skipEncoding === void 0) { skipEncoding = false; }
            if (!skipEncoding) {
                if (!this._formatted) {
                    this._formatted = Uri._asFormatted(this, false);
                }
                return this._formatted;
            }
            else {
                // we don't cache that
                return Uri._asFormatted(this, true);
            }
        };
        Uri._asFormatted = function (uri, skipEncoding) {
            var encoder = !skipEncoding
                ? encodeURIComponent2
                : encodeNoop;
            var result = '';
            var scheme = uri.scheme, authority = uri.authority, path = uri.path, query = uri.query, fragment = uri.fragment;
            if (scheme) {
                result += scheme;
                result += ':';
            }
            if (authority || scheme === 'file') {
                result += '//';
            }
            if (authority) {
                authority = authority.toLowerCase();
                var idx = authority.indexOf(':');
                if (idx === -1) {
                    result += encoder(authority);
                }
                else {
                    result += encoder(authority.substr(0, idx));
                    result += authority.substr(idx);
                }
            }
            if (path) {
                // lower-case windown drive letters in /C:/fff
                var m = Uri._upperCaseDrive.exec(path);
                if (m) {
                    path = m[1] + m[2].toLowerCase() + path.substr(m[1].length + m[2].length);
                }
                // encode every segement but not slashes
                // make sure that # and ? are always encoded
                // when occurring in paths - otherwise the result
                // cannot be parsed back again
                var lastIdx = 0;
                while (true) {
                    var idx = path.indexOf(Uri._slash, lastIdx);
                    if (idx === -1) {
                        result += encoder(path.substring(lastIdx)).replace(/[#?]/, _encode);
                        break;
                    }
                    result += encoder(path.substring(lastIdx, idx)).replace(/[#?]/, _encode);
                    result += Uri._slash;
                    lastIdx = idx + 1;
                }
                ;
            }
            if (query) {
                result += '?';
                result += encoder(query);
            }
            if (fragment) {
                result += '#';
                result += encoder(fragment);
            }
            return result;
        };
        Uri.prototype.toJSON = function () {
            return {
                scheme: this.scheme,
                authority: this.authority,
                path: this.path,
                fsPath: this.fsPath,
                query: this.query,
                fragment: this.fragment
            };
        };
        Uri._empty = '';
        Uri._slash = '/';
        Uri._regexp = /^(([^:/?#]+?):)?(\/\/([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/;
        Uri._driveLetterPath = /^\/[a-zA-z]:/;
        Uri._upperCaseDrive = /^(\/)?([A-Z]:)/;
        return Uri;
    }());
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.default = Uri;
    var isWindows;
    if (typeof process === 'object') {
        isWindows = process.platform === 'win32';
    }
    else if (typeof navigator === 'object') {
        var userAgent = navigator.userAgent;
        isWindows = userAgent.indexOf('Windows') >= 0;
    }
});
