(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "vscode-languageserver-types"], factory);
    }
})(function (require, exports) {
    /*---------------------------------------------------------------------------------------------
     *  Copyright (c) Microsoft Corporation. All rights reserved.
     *  Licensed under the MIT License. See License.txt in the project root for license information.
     *--------------------------------------------------------------------------------------------*/
    'use strict';
    var vscode_languageserver_types_1 = require("vscode-languageserver-types");
    var JSONValidation = (function () {
        function JSONValidation(jsonSchemaService, promiseConstructor) {
            this.jsonSchemaService = jsonSchemaService;
            this.promise = promiseConstructor;
            this.validationEnabled = true;
        }
        JSONValidation.prototype.configure = function (raw) {
            if (raw) {
                this.validationEnabled = raw.validate;
            }
        };
        JSONValidation.prototype.doValidation = function (textDocument, jsonDocument) {
            if (!this.validationEnabled) {
                return this.promise.resolve([]);
            }
            return this.jsonSchemaService.getSchemaForResource(textDocument.uri, jsonDocument).then(function (schema) {
                if (schema) {
                    if (schema.errors.length && jsonDocument.root) {
                        var astRoot = jsonDocument.root;
                        var property = astRoot.type === 'object' ? astRoot.getFirstProperty('$schema') : null;
                        if (property) {
                            var node = property.value || property;
                            jsonDocument.warnings.push({ location: { start: node.start, end: node.end }, message: schema.errors[0] });
                        }
                        else {
                            jsonDocument.warnings.push({ location: { start: astRoot.start, end: astRoot.start + 1 }, message: schema.errors[0] });
                        }
                    }
                    else {
                        jsonDocument.validate(schema.schema);
                    }
                }
                var diagnostics = [];
                var added = {};
                jsonDocument.errors.concat(jsonDocument.warnings).forEach(function (error, idx) {
                    // remove duplicated messages
                    var signature = error.location.start + ' ' + error.location.end + ' ' + error.message;
                    if (!added[signature]) {
                        added[signature] = true;
                        var range = {
                            start: textDocument.positionAt(error.location.start),
                            end: textDocument.positionAt(error.location.end)
                        };
                        diagnostics.push({
                            severity: idx >= jsonDocument.errors.length ? vscode_languageserver_types_1.DiagnosticSeverity.Warning : vscode_languageserver_types_1.DiagnosticSeverity.Error,
                            range: range,
                            message: error.message
                        });
                    }
                });
                return diagnostics;
            });
        };
        return JSONValidation;
    }());
    exports.JSONValidation = JSONValidation;
});
//# sourceMappingURL=jsonValidation.js.map