/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
/// <reference path="./thenable.ts" />
'use strict';
const vscode_jsonrpc_1 = require('vscode-jsonrpc');
exports.ResponseError = vscode_jsonrpc_1.ResponseError;
exports.ErrorCodes = vscode_jsonrpc_1.ErrorCodes;
exports.StreamMessageReader = vscode_jsonrpc_1.StreamMessageReader;
exports.IPCMessageReader = vscode_jsonrpc_1.IPCMessageReader;
exports.StreamMessageWriter = vscode_jsonrpc_1.StreamMessageWriter;
exports.IPCMessageWriter = vscode_jsonrpc_1.IPCMessageWriter;
exports.Event = vscode_jsonrpc_1.Event;
const vscode_languageserver_types_1 = require('vscode-languageserver-types');
exports.TextDocument = vscode_languageserver_types_1.TextDocument;
exports.Range = vscode_languageserver_types_1.Range;
exports.Position = vscode_languageserver_types_1.Position;
exports.Location = vscode_languageserver_types_1.Location;
exports.Diagnostic = vscode_languageserver_types_1.Diagnostic;
exports.Command = vscode_languageserver_types_1.Command;
exports.TextEdit = vscode_languageserver_types_1.TextEdit;
exports.WorkspaceChange = vscode_languageserver_types_1.WorkspaceChange;
exports.TextDocumentIdentifier = vscode_languageserver_types_1.TextDocumentIdentifier;
exports.CompletionItem = vscode_languageserver_types_1.CompletionItem;
exports.CompletionList = vscode_languageserver_types_1.CompletionList;
exports.MarkedString = vscode_languageserver_types_1.MarkedString;
exports.SignatureInformation = vscode_languageserver_types_1.SignatureInformation;
exports.ParameterInformation = vscode_languageserver_types_1.ParameterInformation;
exports.CodeActionContext = vscode_languageserver_types_1.CodeActionContext;
exports.DocumentHighlight = vscode_languageserver_types_1.DocumentHighlight;
exports.SymbolInformation = vscode_languageserver_types_1.SymbolInformation;
exports.CodeLens = vscode_languageserver_types_1.CodeLens;
exports.FormattingOptions = vscode_languageserver_types_1.FormattingOptions;
exports.DocumentLink = vscode_languageserver_types_1.DocumentLink;
const protocol_1 = require('./protocol');
exports.FileChangeType = protocol_1.FileChangeType;
exports.TextDocumentSyncKind = protocol_1.TextDocumentSyncKind;
exports.CodeLensRequest = protocol_1.CodeLensRequest;
exports.CodeLensResolveRequest = protocol_1.CodeLensResolveRequest;
exports.DocumentFormattingRequest = protocol_1.DocumentFormattingRequest;
exports.DocumentRangeFormattingRequest = protocol_1.DocumentRangeFormattingRequest;
exports.DocumentOnTypeFormattingRequest = protocol_1.DocumentOnTypeFormattingRequest;
exports.RenameRequest = protocol_1.RenameRequest;
exports.DocumentLinkRequest = protocol_1.DocumentLinkRequest;
exports.DocumentLinkResolveRequest = protocol_1.DocumentLinkResolveRequest;
const Is = require('./utils/is');
// ------------- Reexport the API surface of the language worker API ----------------------
const fm = require('./files');
const net = require('net');
const stream = require('stream');
var Files;
(function (Files) {
    Files.uriToFilePath = fm.uriToFilePath;
    Files.resolveModule = fm.resolveModule;
    Files.resolveModule2 = fm.resolveModule2;
    Files.resolveModulePath = fm.resolveModulePath;
})(Files = exports.Files || (exports.Files = {}));
/**
 * A manager for simple text documents
 */
class TextDocuments {
    /**
     * Create a new text document manager.
     */
    constructor() {
        this._documents = Object.create(null);
        this._onDidChangeContent = new vscode_jsonrpc_1.Emitter();
        this._onDidOpen = new vscode_jsonrpc_1.Emitter();
        this._onDidClose = new vscode_jsonrpc_1.Emitter();
        this._onDidSave = new vscode_jsonrpc_1.Emitter();
    }
    /**
     * Returns the [TextDocumentSyncKind](#TextDocumentSyncKind) used by
     * this text document manager.
     */
    get syncKind() {
        return protocol_1.TextDocumentSyncKind.Full;
    }
    /**
     * An event that fires when a text document managed by this manager
     * has been opened or the content changes.
     */
    get onDidChangeContent() {
        return this._onDidChangeContent.event;
    }
    /**
     * An event that fires when a text document managed by this manager
     * has been opened.
     */
    get onDidOpen() {
        return this._onDidOpen.event;
    }
    /**
     * An event that fires when a text document managed by this manager
     * has been closed.
     */
    get onDidClose() {
        return this._onDidClose.event;
    }
    /**
     * An event that fires when a text document managed by this manager
     * has been closed.
     */
    get onDidSave() {
        return this._onDidSave.event;
    }
    /**
     * Returns the document for the given URI. Returns undefined if
     * the document is not mananged by this instance.
     *
     * @param uri The text document's URI to retrieve.
     * @return the text document or `undefined`.
     */
    get(uri) {
        return this._documents[uri];
    }
    /**
     * Returns all text documents managed by this instance.
     *
     * @return all text documents.
     */
    all() {
        return Object.keys(this._documents).map(key => this._documents[key]);
    }
    /**
     * Returns the URIs of all text documents managed by this instance.
     *
     * @return the URI's of all text documents.
     */
    keys() {
        return Object.keys(this._documents);
    }
    /**
     * Listens for `low level` notification on the given connection to
     * update the text documents managed by this instance.
     *
     * @param connection The connection to listen on.
     */
    listen(connection) {
        connection.__textDocumentSync = protocol_1.TextDocumentSyncKind.Full;
        connection.onDidOpenTextDocument((event) => {
            let td = event.textDocument;
            let document = vscode_languageserver_types_1.TextDocument.create(td.uri, td.languageId, td.version, td.text);
            this._documents[td.uri] = document;
            this._onDidOpen.fire({ document });
            this._onDidChangeContent.fire({ document });
        });
        connection.onDidChangeTextDocument((event) => {
            let td = event.textDocument;
            let changes = event.contentChanges;
            let last = changes.length > 0 ? changes[changes.length - 1] : null;
            if (last) {
                let document = this._documents[td.uri];
                if (document && Is.func(document['update'])) {
                    document.update(last, td.version);
                    this._onDidChangeContent.fire({ document });
                }
            }
        });
        connection.onDidCloseTextDocument((event) => {
            let document = this._documents[event.textDocument.uri];
            if (document) {
                delete this._documents[event.textDocument.uri];
                this._onDidClose.fire({ document });
            }
        });
        connection.onDidSaveTextDocument((event) => {
            let document = this._documents[event.textDocument.uri];
            if (document) {
                this._onDidSave.fire({ document });
            }
        });
    }
}
exports.TextDocuments = TextDocuments;
// ------------------------- implementation of the language server protocol ---------------------------------------------
/**
 * Helps tracking error message. Equal occurences of the same
 * message are only stored once. This class is for example
 * usefull if text documents are validated in a loop and equal
 * error message should be folded into one.
 */
class ErrorMessageTracker {
    constructor() {
        this.messages = Object.create(null);
    }
    /**
     * Add a message to the tracker.
     *
     * @param message The message to add.
     */
    add(message) {
        let count = this.messages[message];
        if (!count) {
            count = 0;
        }
        count++;
        this.messages[message] = count;
    }
    /**
     * Send all tracked messages to the conenction's window.
     *
     * @param connection The connection establised between client and server.
     */
    sendErrors(connection) {
        Object.keys(this.messages).forEach(message => {
            connection.window.showErrorMessage(message);
        });
    }
}
exports.ErrorMessageTracker = ErrorMessageTracker;
class ConnectionLogger {
    constructor() {
    }
    attach(connection) {
        this.connection = connection;
    }
    error(message) {
        this.send(protocol_1.MessageType.Error, message);
    }
    warn(message) {
        this.send(protocol_1.MessageType.Warning, message);
    }
    info(message) {
        this.send(protocol_1.MessageType.Info, message);
    }
    log(message) {
        this.send(protocol_1.MessageType.Log, message);
    }
    send(type, message) {
        if (this.connection) {
            this.connection.sendNotification(protocol_1.LogMessageNotification.type, { type, message });
        }
    }
}
class RemoteWindowImpl {
    constructor(connection) {
        this.connection = connection;
    }
    showErrorMessage(message, ...actions) {
        return this.connection.sendRequest(protocol_1.ShowMessageRequest.type, { type: protocol_1.MessageType.Error, message, actions });
    }
    showWarningMessage(message, ...actions) {
        return this.connection.sendRequest(protocol_1.ShowMessageRequest.type, { type: protocol_1.MessageType.Warning, message, actions });
    }
    showInformationMessage(message, ...actions) {
        return this.connection.sendRequest(protocol_1.ShowMessageRequest.type, { type: protocol_1.MessageType.Info, message, actions });
    }
}
class TracerImpl {
    constructor(connection) {
        this.connection = connection;
        this._trace = vscode_jsonrpc_1.Trace.Off;
    }
    set trace(value) {
        this._trace = value;
    }
    log(message, verbose) {
        if (this._trace === vscode_jsonrpc_1.Trace.Off) {
            return;
        }
        this.connection.sendNotification(vscode_jsonrpc_1.LogTraceNotification.type, {
            message: message,
            verbose: this._trace === vscode_jsonrpc_1.Trace.Verbose ? verbose : null
        });
    }
}
class TelemetryImpl {
    constructor(connection) {
        this.connection = connection;
    }
    logEvent(data) {
        this.connection.sendNotification(protocol_1.TelemetryEventNotification.type, data);
    }
}
function createConnection(input, output) {
    if (!input && !output && process.argv.length > 2) {
        let port = void 0;
        let argv = process.argv.slice(2);
        for (let i = 0; i < argv.length; i++) {
            let arg = argv[i];
            if (arg === '--node-ipc') {
                input = new vscode_jsonrpc_1.IPCMessageReader(process);
                output = new vscode_jsonrpc_1.IPCMessageWriter(process);
            }
            else if (arg === '--stdio') {
                input = process.stdin;
                output = process.stdout;
            }
            else if (arg === '--socket') {
                port = parseInt(argv[i + 1]);
                i++;
            }
            else {
                var args = arg.split('=');
                if (args[0] === '--socket') {
                    port = parseInt(args[1]);
                }
            }
        }
        if (port) {
            output = new stream.PassThrough();
            input = new stream.PassThrough();
            let server = net.createServer(socket => {
                server.close();
                socket.pipe(output);
                input.pipe(socket);
            }).listen(port);
        }
    }
    var commandLineMessage = "Use arguments of createConnection or set command line parameters: '--node-ipc', '--stdio' or '--socket={number}'";
    if (!input) {
        throw new Error("Connection input stream is not set. " + commandLineMessage);
    }
    if (!output) {
        throw new Error("Connection output stream is not set. " + commandLineMessage);
    }
    let shutdownReceived;
    // Backwards compatibility
    if (Is.func(input.read) && Is.func(input.on)) {
        let inputStream = input;
        inputStream.on('end', () => {
            process.exit(shutdownReceived ? 0 : 1);
        });
        inputStream.on('close', () => {
            process.exit(shutdownReceived ? 0 : 1);
        });
    }
    const logger = new ConnectionLogger();
    const connection = vscode_jsonrpc_1.createMessageConnection(input, output, logger);
    logger.attach(connection);
    const remoteWindow = new RemoteWindowImpl(connection);
    const telemetry = new TelemetryImpl(connection);
    const tracer = new TracerImpl(connection);
    function asThenable(value) {
        if (Is.thenable(value)) {
            return value;
        }
        else {
            return Promise.resolve(value);
        }
    }
    let shutdownHandler = null;
    let initializeHandler = null;
    let exitHandler = null;
    let protocolConnection = {
        listen: () => connection.listen(),
        sendRequest: (type, params) => connection.sendRequest(type, params),
        onRequest: (type, handler) => connection.onRequest(type, handler),
        sendNotification: (type, params) => connection.sendNotification(type, params),
        onNotification: (type, handler) => connection.onNotification(type, handler),
        onInitialize: (handler) => initializeHandler = handler,
        onShutdown: (handler) => shutdownHandler = handler,
        onExit: (handler) => exitHandler = handler,
        get console() { return logger; },
        get window() { return remoteWindow; },
        get telemetry() { return telemetry; },
        get tracer() { return tracer; },
        onDidChangeConfiguration: (handler) => connection.onNotification(protocol_1.DidChangeConfigurationNotification.type, handler),
        onDidChangeWatchedFiles: (handler) => connection.onNotification(protocol_1.DidChangeWatchedFilesNotification.type, handler),
        __textDocumentSync: undefined,
        onDidOpenTextDocument: (handler) => connection.onNotification(protocol_1.DidOpenTextDocumentNotification.type, handler),
        onDidChangeTextDocument: (handler) => connection.onNotification(protocol_1.DidChangeTextDocumentNotification.type, handler),
        onDidCloseTextDocument: (handler) => connection.onNotification(protocol_1.DidCloseTextDocumentNotification.type, handler),
        onDidSaveTextDocument: (handler) => connection.onNotification(protocol_1.DidSaveTextDocumentNotification.type, handler),
        sendDiagnostics: (params) => connection.sendNotification(protocol_1.PublishDiagnosticsNotification.type, params),
        onHover: (handler) => connection.onRequest(protocol_1.HoverRequest.type, handler),
        onCompletion: (handler) => connection.onRequest(protocol_1.CompletionRequest.type, handler),
        onCompletionResolve: (handler) => connection.onRequest(protocol_1.CompletionResolveRequest.type, handler),
        onSignatureHelp: (handler) => connection.onRequest(protocol_1.SignatureHelpRequest.type, handler),
        onDefinition: (handler) => connection.onRequest(protocol_1.DefinitionRequest.type, handler),
        onReferences: (handler) => connection.onRequest(protocol_1.ReferencesRequest.type, handler),
        onDocumentHighlight: (handler) => connection.onRequest(protocol_1.DocumentHighlightRequest.type, handler),
        onDocumentSymbol: (handler) => connection.onRequest(protocol_1.DocumentSymbolRequest.type, handler),
        onWorkspaceSymbol: (handler) => connection.onRequest(protocol_1.WorkspaceSymbolRequest.type, handler),
        onCodeAction: (handler) => connection.onRequest(protocol_1.CodeActionRequest.type, handler),
        onCodeLens: (handler) => connection.onRequest(protocol_1.CodeLensRequest.type, handler),
        onCodeLensResolve: (handler) => connection.onRequest(protocol_1.CodeLensResolveRequest.type, handler),
        onDocumentFormatting: (handler) => connection.onRequest(protocol_1.DocumentFormattingRequest.type, handler),
        onDocumentRangeFormatting: (handler) => connection.onRequest(protocol_1.DocumentRangeFormattingRequest.type, handler),
        onDocumentOnTypeFormatting: (handler) => connection.onRequest(protocol_1.DocumentOnTypeFormattingRequest.type, handler),
        onRenameRequest: (handler) => connection.onRequest(protocol_1.RenameRequest.type, handler),
        onDocumentLinks: (handler) => connection.onRequest(protocol_1.DocumentLinkRequest.type, handler),
        onDocumentLinkResolve: (handler) => connection.onRequest(protocol_1.DocumentLinkResolveRequest.type, handler),
        dispose: () => connection.dispose()
    };
    connection.onRequest(protocol_1.InitializeRequest.type, (params) => {
        if (Is.number(params.processId)) {
            // We received a parent process id. Set up a timer to periodically check
            // if the parent is still alive.
            setInterval(() => {
                try {
                    process.kill(params.processId, 0);
                }
                catch (ex) {
                    // Parent process doesn't exist anymore. Exit the server.
                    process.exit(shutdownReceived ? 0 : 1);
                }
            }, 3000);
        }
        if (Is.string(params.trace)) {
            tracer.trace = vscode_jsonrpc_1.Trace.fromString(params.trace);
        }
        if (initializeHandler) {
            let result = initializeHandler(params, new vscode_jsonrpc_1.CancellationTokenSource().token);
            return asThenable(result).then((value) => {
                if (value instanceof vscode_jsonrpc_1.ResponseError) {
                    return value;
                }
                let result = value;
                if (!result) {
                    result = { capabilities: {} };
                }
                let capabilities = result.capabilities;
                if (!capabilities) {
                    capabilities = {};
                    result.capabilities = {};
                }
                if (!Is.number(capabilities.textDocumentSync)) {
                    capabilities.textDocumentSync = Is.number(protocolConnection.__textDocumentSync) ? protocolConnection.__textDocumentSync : protocol_1.TextDocumentSyncKind.None;
                }
                return result;
            });
        }
        else {
            let result = { capabilities: { textDocumentSync: protocol_1.TextDocumentSyncKind.None } };
            return result;
        }
    });
    connection.onRequest(protocol_1.ShutdownRequest.type, (params) => {
        shutdownReceived = true;
        if (shutdownHandler) {
            return shutdownHandler(params, new vscode_jsonrpc_1.CancellationTokenSource().token);
        }
        else {
            return undefined;
        }
    });
    connection.onNotification(protocol_1.ExitNotification.type, (params) => {
        try {
            if (exitHandler) {
                exitHandler(params);
            }
        }
        finally {
            if (shutdownReceived) {
                process.exit(0);
            }
            else {
                process.exit(1);
            }
        }
    });
    connection.onNotification(vscode_jsonrpc_1.SetTraceNotification.type, (params) => {
        tracer.trace = vscode_jsonrpc_1.Trace.fromString(params.value);
        ;
    });
    return protocolConnection;
}
exports.createConnection = createConnection;
