/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const WebSocket = require("ws");
const errors = require("../errors");
const utils = require("../utils");
const logger = require("../logger");
const chromeTargetDiscoveryStrategy_1 = require("./chromeTargetDiscoveryStrategy");
const noice_json_rpc_1 = require("noice-json-rpc");
/**
 * A subclass of WebSocket that logs all traffic
 */
class LoggingSocket extends WebSocket {
    constructor(address, protocols, options) {
        super(address, protocols, options);
        this.on('error', e => {
            logger.log('Websocket error: ' + e.toString());
        });
        this.on('close', () => {
            logger.log('Websocket closed');
        });
        this.on('message', msgStr => {
            let msgObj;
            try {
                msgObj = JSON.parse(msgStr);
            }
            catch (e) {
                logger.error(`Invalid JSON from target: (${e.message}): ${msgStr}`);
                return;
            }
            if (msgObj
                && !(msgObj.method === 'Debugger.scriptParsed' && msgObj.params && msgObj.params.isContentScript)
                && !(msgObj.params && msgObj.params.url && msgObj.params.url.indexOf('extensions::') === 0)) {
                // Not really the right place to examine the content of the message, but don't log annoying extension script notifications.
                logger.verbose('From target: ' + msgStr);
            }
        });
    }
    send(data, cb) {
        super.send.apply(this, arguments);
        const msgStr = JSON.stringify(data);
        logger.verbose('To target: ' + msgStr);
    }
}
/**
 * Connects to a target supporting the Chrome Debug Protocol and sends and receives messages
 */
class ChromeConnection {
    constructor(targetDiscovery, targetFilter) {
        this._targetFilter = targetFilter;
        this._targetDiscoveryStrategy = targetDiscovery || chromeTargetDiscoveryStrategy_1.getChromeTargetWebSocketURL;
    }
    get isAttached() { return !!this._client; }
    get api() {
        return this._client.api();
    }
    /**
     * Attach the websocket to the first available tab in the chrome instance with the given remote debugging port number.
     */
    attach(address = '127.0.0.1', port = 9222, targetUrl) {
        return this._attach(address, port, targetUrl)
            .then(() => { });
    }
    _attach(address, port, targetUrl, timeout = ChromeConnection.ATTACH_TIMEOUT) {
        return utils.retryAsync(() => this._targetDiscoveryStrategy(address, port, this._targetFilter, targetUrl), timeout, /*intervalDelay=*/ 200)
            .catch(err => Promise.reject(errors.runtimeConnectionTimeout(timeout, err.message)))
            .then(wsUrl => {
            this._socket = new LoggingSocket(wsUrl);
            this._client = new noice_json_rpc_1.Client(this._socket);
            this._client.on('error', e => logger.error('Error handling message from target: ' + e.message));
        });
    }
    run() {
        // This is a CDP version difference which will have to be handled more elegantly with others later...
        // For now, we need to send both messages and ignore a failing one.
        return Promise.all([
            this.api.Runtime.runIfWaitingForDebugger(),
            this.api.Runtime.run()
        ])
            .then(() => { }, e => { });
    }
    close() {
        this._socket.close();
    }
    onClose(handler) {
        this._socket.on('close', handler);
    }
}
ChromeConnection.ATTACH_TIMEOUT = 10000; // ms
exports.ChromeConnection = ChromeConnection;

//# sourceMappingURL=chromeConnection.js.map
