/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const vscode_debugadapter_1 = require("vscode-debugadapter");
const chromeConnection_1 = require("./chromeConnection");
const ChromeUtils = require("./chromeUtils");
const variables_1 = require("./variables");
const Variables = require("./variables");
const consoleHelper_1 = require("./consoleHelper");
const errors = require("../errors");
const utils = require("../utils");
const logger = require("../logger");
const telemetry = require("../telemetry");
const lineNumberTransformer_1 = require("../transformers/lineNumberTransformer");
const remotePathTransformer_1 = require("../transformers/remotePathTransformer");
const eagerSourceMapTransformer_1 = require("../transformers/eagerSourceMapTransformer");
const path = require("path");
class ChromeDebugAdapter {
    constructor({ chromeConnection, lineColTransformer, sourceMapTransformer, pathTransformer }, session) {
        this._currentStep = Promise.resolve();
        this._nextUnboundBreakpointId = 0;
        this._smartStepCount = 0;
        this._initialSourceMapsP = Promise.resolve();
        telemetry.setupEventHandler(e => session.sendEvent(e));
        this._session = session;
        this._chromeConnection = new (chromeConnection || chromeConnection_1.ChromeConnection)();
        this._frameHandles = new vscode_debugadapter_1.Handles();
        this._variableHandles = new vscode_debugadapter_1.Handles();
        this._breakpointIdHandles = new utils.ReverseHandles();
        this._sourceHandles = new utils.ReverseHandles();
        this._pendingBreakpointsByUrl = new Map();
        this._hitConditionBreakpointsById = new Map();
        this._lineColTransformer = new (lineColTransformer || lineNumberTransformer_1.LineColTransformer)(this._session);
        this._sourceMapTransformer = new (sourceMapTransformer || eagerSourceMapTransformer_1.EagerSourceMapTransformer)(this._sourceHandles);
        this._pathTransformer = new (pathTransformer || remotePathTransformer_1.RemotePathTransformer)();
        this.clearTargetContext();
    }
    get chrome() {
        return this._chromeConnection.api;
    }
    /**
     * Called on 'clearEverything' or on a navigation/refresh
     */
    clearTargetContext() {
        this._sourceMapTransformer.clearTargetContext();
        this._scriptsById = new Map();
        this._scriptsByUrl = new Map();
        this._committedBreakpointsByUrl = new Map();
        this._setBreakpointsRequestQ = Promise.resolve();
        this._pathTransformer.clearTargetContext();
    }
    initialize(args) {
        if (args.pathFormat !== 'path') {
            return Promise.reject(errors.pathFormat());
        }
        // because session bypasses dispatchRequest
        if (typeof args.linesStartAt1 === 'boolean') {
            this._clientLinesStartAt1 = args.linesStartAt1;
        }
        if (typeof args.columnsStartAt1 === 'boolean') {
            this._clientColumnsStartAt1 = args.columnsStartAt1;
        }
        // This debug adapter supports two exception breakpoint filters
        return {
            exceptionBreakpointFilters: [
                {
                    label: 'All Exceptions',
                    filter: 'all',
                    default: false
                },
                {
                    label: 'Uncaught Exceptions',
                    filter: 'uncaught',
                    default: true
                }
            ],
            supportsConfigurationDoneRequest: true,
            supportsSetVariable: true,
            supportsConditionalBreakpoints: true,
            supportsCompletionsRequest: true,
            supportsHitConditionalBreakpoints: true
        };
    }
    configurationDone() {
        return Promise.resolve();
    }
    launch(args) {
        this.commonArgs(args);
        this._sourceMapTransformer.launch(args);
        this._pathTransformer.launch(args);
        telemetry.reportEvent('debugStarted', { request: 'launch' });
        return Promise.resolve();
    }
    attach(args) {
        this._attachMode = true;
        this.commonArgs(args);
        this._sourceMapTransformer.attach(args);
        this._pathTransformer.attach(args);
        if (args.port == null) {
            return utils.errP('The "port" field is required in the attach config.');
        }
        telemetry.reportEvent('debugStarted', { request: 'attach' });
        return this.doAttach(args.port, args.url, args.address, args.timeout);
    }
    commonArgs(args) {
        const minLogLevel = args.verboseDiagnosticLogging ?
            logger.LogLevel.Verbose :
            args.diagnosticLogging ?
                logger.LogLevel.Log :
                logger.LogLevel.Error;
        logger.setMinLogLevel(minLogLevel);
        this._launchAttachArgs = args;
    }
    shutdown() {
        this._inShutdown = true;
        this._session.shutdown();
    }
    /**
     * Chrome is closing, or error'd somehow, stop the debug session
     */
    terminateSession(reason, restart) {
        if (!this._hasTerminated) {
            telemetry.reportEvent('debugStopped', { reason });
            this._hasTerminated = true;
            if (this._clientAttached) {
                this._session.sendEvent(new vscode_debugadapter_1.TerminatedEvent(restart));
            }
            if (this._chromeConnection.isAttached) {
                this._chromeConnection.close();
            }
        }
        process.exit(0)
    }
   
    /**
     * Hook up all connection events
     */
    hookConnectionEvents() {
        this.chrome.Debugger.onPaused(params => this.onPaused(params));
        this.chrome.Debugger.onResumed(() => this.onResumed());
        this.chrome.Debugger.onScriptParsed(params => this.onScriptParsed(params));
        this.chrome.Debugger.onBreakpointResolved(params => this.onBreakpointResolved(params));
        this.chrome.Console.onMessageAdded(params => this.onMessageAdded(params));
        this.chrome.Runtime.onConsoleAPICalled(params => this.onConsoleAPICalled(params));
        this.chrome.Runtime.onExecutionContextsCleared(() => this.onExecutionContextsCleared());
        this.chrome.Inspector.onDetached(() => this.terminateSession('Debug connection detached'));
        this._chromeConnection.onClose(() => this.terminateSession('websocket closed'));
    }
    /**
     * Enable clients and run connection
     */
    runConnection() {
        return [
            this.chrome.Console.enable()
                .catch(e => { }),
            this.chrome.Debugger.enable(),
            this.chrome.Runtime.enable(),
            this._chromeConnection.run()
        ];
    }
    doAttach(port, targetUrl, address, timeout) {
        // Client is attaching - if not attached to the chrome target, create a connection and attach
        this._clientAttached = true;
        if (!this._chromeConnection.isAttached) {
            return this._chromeConnection.attach(address, port, targetUrl)
                .then(e => {
                this.hookConnectionEvents();
                let patterns = [];
                if (this._launchAttachArgs.skipFiles) {
                    const skipFilesArgs = this._launchAttachArgs.skipFiles.filter(glob => {
                        if (glob.startsWith('!')) {
                            logger.log(`Warning: skipFiles entries starting with '!' aren't supported and will be ignored. ("${glob}")`, /*forceLog=*/ true);
                            return false;
                        }
                        return true;
                    });
                    patterns = skipFilesArgs.map(glob => utils.pathGlobToBlackboxedRegex(glob));
                }
                if (this._launchAttachArgs.skipFileRegExps) {
                    patterns = patterns.concat(this._launchAttachArgs.skipFileRegExps);
                }
                if (patterns.length) {
                    this._blackboxedRegexes = patterns.map(pattern => new RegExp(pattern, 'i'));
                    this.chrome.Debugger.setBlackboxPatterns({ patterns });
                }
                return Promise.all(this.runConnection());
            })
                .then(() => this.sendInitializedEvent());
        }
        else {
            return Promise.resolve();
        }
    }
    /**
     * This event tells the client to begin sending setBP requests, etc. Some consumers need to override this
     * to send it at a later time of their choosing.
     */
    sendInitializedEvent() {
        // Wait to finish loading sourcemaps from the initial scriptParsed events
        this._initialSourceMapsP.then(() => {
            this._session.sendEvent(new vscode_debugadapter_1.InitializedEvent());
            this._initialSourceMapsP = null;
        });
    }
    /**
     * e.g. the target navigated
     */
    onExecutionContextsCleared() {
        this.clearTargetContext();
    }
    onPaused(notification) {
        this._variableHandles.reset();
        this._frameHandles.reset();
        this._exception = undefined;
        this._currentStack = notification.callFrames;
        // We can tell when we've broken on an exception. Otherwise if hitBreakpoints is set, assume we hit a
        // breakpoint. If not set, assume it was a step. We can't tell the difference between step and 'break on anything'.
        let reason;
        let smartStepP = Promise.resolve(false);
        if (notification.reason === 'exception') {
            reason = 'exception';
            this._exception = notification.data;
        }
        else if (notification.hitBreakpoints && notification.hitBreakpoints.length) {
            reason = 'breakpoint';
            // Did we hit a hit condition breakpoint?
            for (let hitBp of notification.hitBreakpoints) {
                if (this._hitConditionBreakpointsById.has(hitBp)) {
                    // Increment the hit count and check whether to pause
                    const hitConditionBp = this._hitConditionBreakpointsById.get(hitBp);
                    hitConditionBp.numHits++;
                    // Only resume if we didn't break for some user action (step, pause button)
                    if (!this._expectingStopReason && !hitConditionBp.shouldPause(hitConditionBp.numHits)) {
                        this.chrome.Debugger.resume();
                        return;
                    }
                }
            }
        }
        else if (this._expectingStopReason) {
            // If this was a step, check whether to smart step
            reason = this._expectingStopReason;
            if (this._launchAttachArgs.smartStep) {
                smartStepP = this.shouldSmartStep(this._currentStack[0]);
            }
        }
        else {
            reason = 'debugger';
        }
        this._expectingStopReason = undefined;
        smartStepP.then(should => {
            if (should) {
                this._smartStepCount++;
                return this.stepIn();
            }
            else {
                if (this._smartStepCount > 0) {
                    logger.log(`SmartStep: Skipped ${this._smartStepCount} steps`);
                    this._smartStepCount = 0;
                }
                // Enforce that the stopped event is not fired until we've send the response to the step that induced it.
                // Also with a timeout just to ensure things keep moving
                const sendStoppedEvent = () => this._session.sendEvent(new vscode_debugadapter_1.StoppedEvent(this.stopReasonText(reason), /*threadId=*/ ChromeDebugAdapter.THREAD_ID));
                return utils.promiseTimeout(this._currentStep, /*timeoutMs=*/ 300)
                    .then(sendStoppedEvent, sendStoppedEvent);
            }
        }).catch(err => logger.error('Problem while smart stepping: ' + (err && err.stack) ? err.stack : err));
    }
    shouldSmartStep(frame) {
        if (!this._launchAttachArgs.sourceMaps)
            return Promise.resolve(false);
        const stackFrame = this.callFrameToStackFrame(frame);
        const clientPath = this._pathTransformer.getClientPathFromTargetPath(stackFrame.source.path) || stackFrame.source.path;
        return this._sourceMapTransformer.mapToAuthored(clientPath, frame.location.lineNumber, frame.location.columnNumber).then(mapping => {
            return !mapping;
        });
    }
    stopReasonText(reason) {
        const comment = ['https://github.com/Microsoft/vscode/issues/4568'];
        switch (reason) {
            case 'entry':
                return utils.localize({ key: 'reason.entry', comment }, "entry");
            case 'exception':
                return utils.localize({ key: 'reason.exception', comment }, "exception");
            case 'breakpoint':
                return utils.localize({ key: 'reason.breakpoint', comment }, "breakpoint");
            case 'debugger':
                return utils.localize({ key: 'reason.debugger_statement', comment }, "debugger statement");
            case 'frame_entry':
                return utils.localize({ key: 'reason.restart', comment }, "frame entry");
            case 'step':
                return utils.localize({ key: 'reason.step', comment }, "step");
            case 'user_request':
                return utils.localize({ key: 'reason.user_request', comment }, "user request");
            default:
                return reason;
        }
    }
    onResumed() {
        this._currentStack = null;
        if (!this._expectingResumedEvent) {
            let resumedEvent = new vscode_debugadapter_1.ContinuedEvent(ChromeDebugAdapter.THREAD_ID);
            this._session.sendEvent(resumedEvent);
        }
        else {
            this._expectingResumedEvent = false;
        }
    }
    onScriptParsed(script) {
        // Totally ignore extension scripts, internal Chrome scripts, and so on
        if (this.shouldIgnoreScript(script)) {
            return;
        }
        if (script.url) {
            script.url = utils.fixDriveLetter(script.url);
        }
        else {
            script.url = ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL + script.scriptId;
        }
        this._scriptsById.set(script.scriptId, script);
        this._scriptsByUrl.set(script.url, script);
        const resolvePendingBPs = source => {
            if (this._pendingBreakpointsByUrl.has(source)) {
                this.resolvePendingBreakpoint(this._pendingBreakpointsByUrl.get(source))
                    .then(() => this._pendingBreakpointsByUrl.delete(source));
            }
        };
        const mappedUrl = this._pathTransformer.scriptParsed(script.url);
        const sourceMapsP = this._sourceMapTransformer.scriptParsed(mappedUrl, script.sourceMapURL).then(sources => {
            if (sources) {
                sources.forEach(resolvePendingBPs);
            }
            resolvePendingBPs(mappedUrl);
            this.resolveSkipFiles(script.scriptId, mappedUrl, sources);
        });
        if (this._initialSourceMapsP) {
            this._initialSourceMapsP = Promise.all([this._initialSourceMapsP, sourceMapsP]);
        }
    }
    resolveSkipFiles(scriptId, mappedUrl, sources) {
        if (this.shouldSkipFile(mappedUrl)) {
            // set the whole script as lib code
            this.chrome.Debugger.setBlackboxedRanges({
                scriptId: scriptId,
                positions: [{ lineNumber: 0, columnNumber: 0 }]
            });
        }
        else if (sources) {
            const librarySources = new Set(sources.filter(sourcePath => this.shouldSkipFile(sourcePath)));
            if (librarySources.size) {
                this._sourceMapTransformer.allSourcePathDetails(mappedUrl).then(details => {
                    const libPositions = [];
                    let inLibRange = false;
                    details.forEach((detail, i) => {
                        const isSkippedFile = librarySources.has(detail.inferredPath);
                        if ((isSkippedFile && !inLibRange) || (!isSkippedFile && inLibRange)) {
                            libPositions.push({
                                lineNumber: detail.startPosition.line,
                                columnNumber: detail.startPosition.column
                            });
                            inLibRange = !inLibRange;
                        }
                    });
                    this.chrome.Debugger.setBlackboxedRanges({
                        scriptId: scriptId,
                        positions: libPositions
                    });
                });
            }
        }
    }
    shouldSkipFile(sourcePath) {
        return this._blackboxedRegexes && this._blackboxedRegexes.some(regex => {
            return regex.test(sourcePath);
        });
    }
    resolvePendingBreakpoint(pendingBP) {
        return this.setBreakpoints(pendingBP.args, pendingBP.requestSeq, pendingBP.ids).then(response => {
            response.breakpoints.forEach((bp, i) => {
                bp.id = pendingBP.ids[i];
                this._session.sendEvent(new vscode_debugadapter_1.BreakpointEvent('new', bp));
            });
        });
    }
    onBreakpointResolved(params) {
        const script = this._scriptsById.get(params.location.scriptId);
        if (!script) {
            // Breakpoint resolved for a script we don't know about
            return;
        }
        const committedBps = this._committedBreakpointsByUrl.get(script.url) || [];
        committedBps.push(params.breakpointId);
        this._committedBreakpointsByUrl.set(script.url, committedBps);
        const bp = {
            id: this._breakpointIdHandles.lookup(params.breakpointId),
            verified: true,
            line: params.location.lineNumber,
            column: params.location.columnNumber
        };
        const scriptPath = this._pathTransformer.breakpointResolved(bp, script.url);
        this._sourceMapTransformer.breakpointResolved(bp, scriptPath);
        this._lineColTransformer.breakpointResolved(bp);
        this._session.sendEvent(new vscode_debugadapter_1.BreakpointEvent('new', bp));
    }
    onConsoleAPICalled(params) {
        const formattedMessage = consoleHelper_1.formatConsoleMessage(params);
        if (formattedMessage) {
            this._session.sendEvent(new vscode_debugadapter_1.OutputEvent(formattedMessage.text + '\n', formattedMessage.isError ? 'stderr' : 'stdout'));
        }
    }
    /**
     * For backcompat, also listen to Console.messageAdded, only if it looks like the old format.
     */
    onMessageAdded(params) {
        // message.type is undefined when Runtime.consoleAPICalled is being sent
        if (params && params.message && params.message.type) {
            const onConsoleAPICalledParams = {
                type: params.message.type,
                timestamp: params.message.timestamp,
                args: params.message.parameters || [{ type: 'string', value: params.message.text }],
                stackTrace: params.message.stack,
                executionContextId: 1
            };
            this.onConsoleAPICalled(onConsoleAPICalledParams);
        }
    }
    disconnect() {
        this.shutdown();
        return this.terminateSession('Got disconnect request');
    }
    setBreakpoints(args, requestSeq, ids) {
        this.reportBpTelemetry(args);
        return this.validateBreakpointsPath(args)
            .then(() => {
            this._lineColTransformer.setBreakpoints(args);
            this._sourceMapTransformer.setBreakpoints(args, requestSeq);
            this._pathTransformer.setBreakpoints(args);
            let targetScriptUrl;
            if (args.source.path) {
                targetScriptUrl = args.source.path;
            }
            else if (args.source.sourceReference) {
                const handle = this._sourceHandles.get(args.source.sourceReference);
                const targetScript = this._scriptsById.get(handle.scriptId);
                if (targetScript) {
                    targetScriptUrl = targetScript.url;
                }
            }
            if (targetScriptUrl) {
                // DebugProtocol sends all current breakpoints for the script. Clear all breakpoints for the script then add all of them
                const setBreakpointsPFailOnError = this._setBreakpointsRequestQ
                    .then(() => this.clearAllBreakpoints(targetScriptUrl))
                    .then(() => this.addBreakpoints(targetScriptUrl, args.breakpoints))
                    .then(responses => ({ breakpoints: this.chromeBreakpointResponsesToODPBreakpoints(targetScriptUrl, responses, args.breakpoints, ids) }));
                const setBreakpointsPTimeout = utils.promiseTimeout(setBreakpointsPFailOnError, ChromeDebugAdapter.SET_BREAKPOINTS_TIMEOUT, 'Set breakpoints request timed out');
                // Do just one setBreakpointsRequest at a time to avoid interleaving breakpoint removed/breakpoint added requests to Crdp.
                // Swallow errors in the promise queue chain so it doesn't get blocked, but return the failing promise for error handling.
                this._setBreakpointsRequestQ = setBreakpointsPTimeout.catch(() => undefined);
                return setBreakpointsPTimeout.then(body => {
                    this._sourceMapTransformer.setBreakpointsResponse(body, requestSeq);
                    this._lineColTransformer.setBreakpointsResponse(body);
                    return body;
                });
            }
            else {
                return Promise.resolve(this.unverifiedBpResponse(args, requestSeq, utils.localize('bp.fail.noscript', `Can't find script for breakpoint request`)));
            }
        }, e => this.unverifiedBpResponse(args, requestSeq, e.message));
    }
    reportBpTelemetry(args) {
        let fileExt = '';
        if (args.source.path) {
            fileExt = path.extname(args.source.path);
        }
        telemetry.reportEvent('setBreakpointsRequest', { fileExt });
    }
    validateBreakpointsPath(args) {
        if (!args.source.path)
            return Promise.resolve();
        return this._sourceMapTransformer.getGeneratedPathFromAuthoredPath(args.source.path).then(mappedPath => {
            if (!mappedPath) {
                return utils.errP(utils.localize('sourcemapping.fail.message', "Breakpoint ignored because generated code not found (source map problem?)."));
            }
            const targetPath = this._pathTransformer.getTargetPathFromClientPath(mappedPath);
            if (!targetPath) {
                return utils.errP('Breakpoint ignored because target path not found');
            }
            return undefined;
        });
    }
    unverifiedBpResponse(args, requestSeq, message) {
        const breakpoints = args.breakpoints.map(bp => {
            return {
                verified: false,
                line: bp.line,
                column: bp.column,
                message,
                id: this._breakpointIdHandles.create(this._nextUnboundBreakpointId++ + '')
            };
        });
        if (args.source.path) {
            const ids = breakpoints.map(bp => bp.id);
            this._pendingBreakpointsByUrl.set(args.source.path, { args, ids, requestSeq });
        }
        return { breakpoints };
    }
    clearAllBreakpoints(url) {
        if (!this._committedBreakpointsByUrl.has(url)) {
            return Promise.resolve();
        }
        // Remove breakpoints one at a time. Seems like it would be ok to send the removes all at once,
        // but there is a chrome bug where when removing 5+ or so breakpoints at once, it gets into a weird
        // state where later adds on the same line will fail with 'breakpoint already exists' even though it
        // does not break there.
        return this._committedBreakpointsByUrl.get(url).reduce((p, breakpointId) => {
            return p.then(() => this.chrome.Debugger.removeBreakpoint({ breakpointId })).then(() => { });
        }, Promise.resolve()).then(() => {
            this._committedBreakpointsByUrl.delete(url);
        });
    }
    /**
     * Makes the actual call to either Debugger.setBreakpoint or Debugger.setBreakpointByUrl, and returns the response.
     * Responses from setBreakpointByUrl are transformed to look like the response from setBreakpoint, so they can be
     * handled the same.
     */
    addBreakpoints(url, breakpoints) {
        let responsePs;
        if (url.startsWith(ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL)) {
            // eval script with no real url - use debugger_setBreakpoint
            const scriptId = utils.lstrip(url, ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL);
            responsePs = breakpoints.map(({ line, column = 0, condition }, i) => this.chrome.Debugger.setBreakpoint({ location: { scriptId, lineNumber: line, columnNumber: column }, condition }));
        }
        else {
            // script that has a url - use debugger_setBreakpointByUrl so that Chrome will rebind the breakpoint immediately
            // after refreshing the page. This is the only way to allow hitting breakpoints in code that runs immediately when
            // the page loads.
            const script = this._scriptsByUrl.get(url);
            const urlRegex = utils.pathToRegex(url);
            responsePs = breakpoints.map(({ line, column = 0, condition }, i) => {
                return this.chrome.Debugger.setBreakpointByUrl({ urlRegex, lineNumber: line, columnNumber: column, condition }).then(result => {
                    // Now convert the response to a SetBreakpointResponse so both response types can be handled the same
                    const locations = result.locations;
                    return {
                        breakpointId: result.breakpointId,
                        actualLocation: locations[0] && {
                            lineNumber: locations[0].lineNumber,
                            columnNumber: locations[0].columnNumber,
                            scriptId: script.scriptId
                        }
                    };
                }, err => ({})); // Ignore errors, return an empty object
            });
        }
        // Join all setBreakpoint requests to a single promise
        return Promise.all(responsePs);
    }
    chromeBreakpointResponsesToODPBreakpoints(url, responses, requestBps, ids) {
        // Don't cache errored responses
        const committedBpIds = responses
            .filter(response => !!response.breakpointId)
            .map(response => response.breakpointId);
        // Cache successfully set breakpoint ids from chrome in committedBreakpoints set
        this._committedBreakpointsByUrl.set(url, committedBpIds);
        // Map committed breakpoints to DebugProtocol response breakpoints
        return responses
            .map((response, i) => {
            // The output list needs to be the same length as the input list, so map errors to
            // unverified breakpoints.
            if (!response) {
                return {
                    verified: false
                };
            }
            let bpId;
            if (ids && ids[i]) {
                bpId = ids[i];
                this._breakpointIdHandles.set(bpId, response.breakpointId);
            }
            else {
                bpId = this._breakpointIdHandles.lookup(response.breakpointId) ||
                    this._breakpointIdHandles.create(response.breakpointId);
            }
            if (!response.actualLocation) {
                return {
                    id: bpId,
                    verified: false
                };
            }
            const thisBpRequest = requestBps[i];
            if (thisBpRequest.hitCondition) {
                if (!this.addHitConditionBreakpoint(thisBpRequest, response)) {
                    return {
                        id: bpId,
                        message: 'Invalid hit condition: ' + thisBpRequest.hitCondition,
                        verified: false
                    };
                }
            }
            return {
                id: bpId,
                verified: true,
                line: response.actualLocation.lineNumber,
                column: response.actualLocation.columnNumber
            };
        });
    }
    addHitConditionBreakpoint(requestBp, response) {
        const result = ChromeDebugAdapter.HITCONDITION_MATCHER.exec(requestBp.hitCondition.trim());
        if (result && result.length >= 3) {
            let op = result[1] || '>=';
            if (op === '=')
                op = '==';
            const value = result[2];
            const expr = op === '%'
                ? `return (numHits % ${value}) === 0;`
                : `return numHits ${op} ${value};`;
            // eval safe because of the regex, and this is only a string that the current user will type in
            /* tslint:disable:no-function-constructor-with-string-args */
            const shouldPause = new Function('numHits', expr);
            /* tslint:enable:no-function-constructor-with-string-args */
            this._hitConditionBreakpointsById.set(response.breakpointId, { numHits: 0, shouldPause });
            return true;
        }
        else {
            return false;
        }
    }
    setExceptionBreakpoints(args) {
        let state;
        if (args.filters.indexOf('all') >= 0) {
            state = 'all';
        }
        else if (args.filters.indexOf('uncaught') >= 0) {
            state = 'uncaught';
        }
        else {
            state = 'none';
        }
        return this.chrome.Debugger.setPauseOnExceptions({ state })
            .then(() => { });
    }
    /**
     * internal -> suppress telemetry
     */
    continue(internal = false) {
        if (!internal)
            telemetry.reportEvent('continueRequest');
        this._expectingResumedEvent = true;
        return this._currentStep = this.chrome.Debugger.resume()
            .then(() => { });
    }
    next() {
        telemetry.reportEvent('nextRequest');
        this._expectingStopReason = 'step';
        this._expectingResumedEvent = true;
        return this._currentStep = this.chrome.Debugger.stepOver()
            .then(() => { });
    }
    stepIn() {
        telemetry.reportEvent('stepInRequest');
        this._expectingStopReason = 'step';
        this._expectingResumedEvent = true;
        return this._currentStep = this.chrome.Debugger.stepInto()
            .then(() => { });
    }
    stepOut() {
        telemetry.reportEvent('stepOutRequest');
        this._expectingStopReason = 'step';
        this._expectingResumedEvent = true;
        return this._currentStep = this.chrome.Debugger.stepOut()
            .then(() => { });
    }
    pause() {
        telemetry.reportEvent('pauseRequest');
        this._expectingStopReason = 'user_request';
        return this._currentStep = this.chrome.Debugger.pause()
            .then(() => { });
    }
    stackTrace(args) {
        // Only process at the requested number of frames, if 'levels' is specified
        if (!this._currentStack) {
        }
        let stack = this._currentStack;
        if (args.levels) {
            stack = this._currentStack.filter((_, i) => i < args.levels);
        }
        const stackTraceResponse = {
            stackFrames: stack.map(frame => this.callFrameToStackFrame(frame))
        };
        this._pathTransformer.stackTraceResponse(stackTraceResponse);
        this._sourceMapTransformer.stackTraceResponse(stackTraceResponse);
        this._lineColTransformer.stackTraceResponse(stackTraceResponse);
        return stackTraceResponse;
    }
    callFrameToStackFrame(frame) {
        const { location, functionName } = frame;
        const line = location.lineNumber;
        const column = location.columnNumber;
        const script = this._scriptsById.get(location.scriptId);
        try {
            // When the script has a url and isn't one we're ignoring, send the name and path fields. PathTransformer will
            // attempt to resolve it to a script in the workspace. Otherwise, send the name and sourceReference fields.
            const source = script && !this.shouldIgnoreScript(script) ?
                {
                    name: path.basename(script.url),
                    path: script.url,
                    sourceReference: this.getSourceReferenceForScriptId(script.scriptId)
                } :
                {
                    name: script && path.basename(script.url),
                    path: ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL + location.scriptId,
                    sourceReference: this.getSourceReferenceForScriptId(script.scriptId)
                };
            // If the frame doesn't have a function name, it's either an anonymous function
            // or eval script. If its source has a name, it's probably an anonymous function.
            const frameName = functionName || (script.url ? '(anonymous function)' : '(eval code)');
            return {
                id: this._frameHandles.create(frame),
                name: frameName,
                source,
                line: line,
                column
            };
        }
        catch (e) {
            // Some targets such as the iOS simulator behave badly and return nonsense callFrames.
            // In these cases, return a dummy stack frame
            return {
                id: this._frameHandles.create({}),
                name: 'Unknown',
                source: { name: 'eval:Unknown', path: ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL + 'Unknown' },
                line,
                column
            };
        }
    }
    /**
     * Get the existing handle for this script, identified by runtime scriptId, or create a new one
     */
    getSourceReferenceForScriptId(scriptId) {
        return this._sourceHandles.lookupF(container => container.scriptId === scriptId) ||
            this._sourceHandles.create({ scriptId });
    }
    scopes(args) {
        const currentFrame = this._frameHandles.get(args.frameId);
        const currentScript = this._scriptsById.get(currentFrame.location.scriptId);
        const currentScriptUrl = currentScript && currentScript.url;
        const currentScriptPath = (currentScriptUrl && this._pathTransformer.getClientPathFromTargetPath(currentScriptUrl)) || currentScriptUrl;
        const scopes = currentFrame.scopeChain.map((scope, i) => {
            // The first scope should include 'this'. Keep the RemoteObject reference for use by the variables request
            const thisObj = i === 0 && currentFrame.this;
            const returnValue = i === 0 && currentFrame.returnValue;
            const variablesReference = this._variableHandles.create(new variables_1.ScopeContainer(currentFrame.callFrameId, i, scope.object.objectId, thisObj, returnValue));
            const resultScope = {
                name: scope.type.substr(0, 1).toUpperCase() + scope.type.substr(1),
                variablesReference,
                expensive: scope.type === 'global'
            };
            if (scope.startLocation && scope.endLocation) {
                resultScope.column = scope.startLocation.columnNumber;
                resultScope.line = scope.startLocation.lineNumber;
                resultScope.endColumn = scope.endLocation.columnNumber;
                resultScope.endLine = scope.endLocation.lineNumber;
            }
            return resultScope;
        });
        if (this._exception) {
            scopes.unshift({
                name: utils.localize('scope.exception', "Exception"),
                variablesReference: this._variableHandles.create(variables_1.ExceptionContainer.create(this._exception))
            });
        }
        const scopesResponse = { scopes };
        if (currentScriptPath) {
            this._sourceMapTransformer.scopesResponse(currentScriptPath, scopesResponse);
            this._lineColTransformer.scopeResponse(scopesResponse);
        }
        return scopesResponse;
    }
    variables(args) {
        const handle = this._variableHandles.get(args.variablesReference);
        if (!handle) {
            return Promise.resolve(undefined);
        }
        return handle.expand(this, args.filter, args.start, args.count).then(variables => {
            return { variables };
        });
    }
    propertyDescriptorToVariable(propDesc, owningObjectId, parentEvaluateName) {
        if (propDesc.get) {
            // Getter
            const grabGetterValue = 'function remoteFunction(propName) { return this[propName]; }';
            return this.chrome.Runtime.callFunctionOn({
                objectId: owningObjectId,
                functionDeclaration: grabGetterValue,
                arguments: [{ value: propDesc.name }]
            }).then(response => {
                if (response.exceptionDetails) {
                    // Not an error, getter could be `get foo() { throw new Error('bar'); }`
                    const exceptionMessage = ChromeUtils.errorMessageFromExceptionDetails(response.exceptionDetails);
                    logger.verbose('Exception thrown evaluating getter - ' + exceptionMessage);
                    return { name: propDesc.name, value: exceptionMessage, variablesReference: 0 };
                }
                else {
                    return this.remoteObjectToVariable(propDesc.name, response.result, parentEvaluateName);
                }
            }, error => {
                logger.error('Error evaluating getter - ' + error.toString());
                return { name: propDesc.name, value: error.toString(), variablesReference: 0 };
            });
        }
        else if (propDesc.set) {
            // setter without a getter, unlikely
            return Promise.resolve({ name: propDesc.name, value: 'setter', variablesReference: 0 });
        }
        else {
            // Non getter/setter
            return this.internalPropertyDescriptorToVariable(propDesc, parentEvaluateName);
        }
    }
    getVariablesForObjectId(objectId, evaluateName, filter, start, count) {
        if (typeof start === 'number' && typeof count === 'number') {
            return this.getFilteredVariablesForObject(objectId, evaluateName, filter, start, count);
        }
        filter = filter === 'indexed' ? 'all' : filter;
        return Promise.all([
            // Need to make two requests to get all properties
            this.chrome.Runtime.getProperties({ objectId, ownProperties: false, accessorPropertiesOnly: true, generatePreview: true }),
            this.chrome.Runtime.getProperties({ objectId, ownProperties: true, accessorPropertiesOnly: false, generatePreview: true })
        ]).then(getPropsResponses => {
            // Sometimes duplicates will be returned - merge all descriptors by name
            const propsByName = new Map();
            const internalPropsByName = new Map();
            getPropsResponses.forEach(response => {
                if (response) {
                    response.result.forEach(propDesc => propsByName.set(propDesc.name, propDesc));
                    if (response.internalProperties) {
                        response.internalProperties.forEach(internalProp => {
                            internalPropsByName.set(internalProp.name, internalProp);
                        });
                    }
                }
            });
            // Convert Chrome prop descriptors to DebugProtocol vars
            const variables = [];
            propsByName.forEach(propDesc => {
                if (!filter || filter === 'all' || (variables_1.isIndexedPropName(propDesc.name) === (filter === 'indexed'))) {
                    variables.push(this.propertyDescriptorToVariable(propDesc, objectId, evaluateName));
                }
            });
            internalPropsByName.forEach(internalProp => {
                if (!filter || filter === 'all' || (variables_1.isIndexedPropName(internalProp.name) === (filter === 'indexed'))) {
                    variables.push(Promise.resolve(this.internalPropertyDescriptorToVariable(internalProp, evaluateName)));
                }
            });
            return Promise.all(variables);
        }).then(variables => {
            // Sort all variables properly
            return variables.sort((var1, var2) => ChromeUtils.compareVariableNames(var1.name, var2.name));
        });
    }
    internalPropertyDescriptorToVariable(propDesc, parentEvaluateName) {
        return this.remoteObjectToVariable(propDesc.name, propDesc.value, parentEvaluateName);
    }
    getFilteredVariablesForObject(objectId, evaluateName, filter, start, count) {
        // No ES6, in case we talk to an old runtime
        const getIndexedVariablesFn = `
            function getIndexedVariables(start, count) {
                var result = [];
                for (var i = start; i < (start + count); i++) result[i] = this[i];
                return result;
            }`;
        // TODO order??
        const getNamedVariablesFn = `
            function getNamedVariablesFn(start, count) {
                var result = [];
                var ownProps = Object.getOwnPropertyNames(this);
                for (var i = start; i < (start + count); i++) result[i] = ownProps[i];
                return result;
            }`;
        const getVarsFn = filter === 'indexed' ? getIndexedVariablesFn : getNamedVariablesFn;
        return this.getFilteredVariablesForObjectId(objectId, evaluateName, getVarsFn, filter, start, count);
    }
    getFilteredVariablesForObjectId(objectId, evaluateName, getVarsFn, filter, start, count) {
        return this.chrome.Runtime.callFunctionOn({
            objectId,
            functionDeclaration: getVarsFn,
            arguments: [{ value: start }, { value: count }],
            silent: true
        }).then(evalResponse => {
            if (evalResponse.exceptionDetails) {
                const errMsg = ChromeUtils.errorMessageFromExceptionDetails(evalResponse.exceptionDetails);
                return Promise.reject(errors.errorFromEvaluate(errMsg));
            }
            else {
                // The eval was successful and returned a reference to the array object. Get the props, then filter
                // out everything except the index names.
                return this.getVariablesForObjectId(evalResponse.result.objectId, evaluateName, filter)
                    .then(variables => variables.filter(variable => variables_1.isIndexedPropName(variable.name)));
            }
        }, error => Promise.reject(errors.errorFromEvaluate(error.message)));
    }
    source(args) {
        const handle = this._sourceHandles.get(args.sourceReference);
        if (!handle) {
            return Promise.reject(errors.sourceRequestIllegalHandle());
        }
        // Have inlined content?
        if (handle.contents) {
            return Promise.resolve({
                content: handle.contents
            });
        }
        // If not, should have scriptId
        return this.chrome.Debugger.getScriptSource({ scriptId: handle.scriptId }).then(response => {
            return {
                content: response.scriptSource,
                mimeType: 'text/javascript'
            };
        });
    }
    threads() {
        return {
            threads: [
                {
                    id: ChromeDebugAdapter.THREAD_ID,
                    name: 'Thread ' + ChromeDebugAdapter.THREAD_ID
                }
            ]
        };
    }
    evaluate(args) {
        if (args.expression.startsWith(ChromeDebugAdapter.SCRIPTS_COMMAND)) {
            return this.handleScriptsCommand(args);
        }
        // These two responses are shaped exactly the same
        let evalPromise;
        if (typeof args.frameId === 'number') {
            const callFrameId = this._frameHandles.get(args.frameId).callFrameId;
            evalPromise = this.chrome.Debugger.evaluateOnCallFrame({ callFrameId, expression: args.expression, silent: true, generatePreview: true });
        }
        else {
            evalPromise = this.globalEvaluate({ expression: args.expression, silent: true, generatePreview: true });
        }
        return evalPromise.then(evalResponse => {
            // Convert to a Variable object then just copy the relevant fields off
            return this.remoteObjectToVariable('', evalResponse.result, /*parentEvaluateName=*/ undefined, /*stringify=*/ undefined, args.context).then(variable => {
                if (evalResponse.exceptionDetails) {
                    let resultValue = variable.value;
                    if (resultValue && resultValue.startsWith('ReferenceError: ') && args.context !== 'repl') {
                        resultValue = utils.localize('eval.not.available', "not available");
                    }
                    return utils.errP(resultValue);
                }
                return {
                    result: variable.value,
                    variablesReference: variable.variablesReference,
                    indexedVariables: variable.indexedVariables,
                    namedVariables: variable.namedVariables
                };
            });
        });
    }
    /**
     * Handle the .scripts command, which can be used as `.scripts` to return a list of all script details,
     * or `.scripts <url>` to show the contents of the given script.
     */
    handleScriptsCommand(args) {
        let outputStringP;
        const scriptsRest = utils.lstrip(args.expression, ChromeDebugAdapter.SCRIPTS_COMMAND).trim();
        if (scriptsRest) {
            // `.scripts <url>` was used, look up the script by url
            const requestedScript = this._scriptsByUrl.get(scriptsRest);
            if (requestedScript) {
                outputStringP = this.chrome.Debugger.getScriptSource({ scriptId: requestedScript.scriptId })
                    .then(result => {
                    const maxLength = 1e5;
                    return result.scriptSource.length > maxLength ?
                        result.scriptSource.substr(0, maxLength) + '[⋯]' :
                        result.scriptSource;
                });
            }
            else {
                outputStringP = Promise.resolve(`No runtime script with url: ${scriptsRest}`);
            }
        }
        else {
            outputStringP = this.getAllScriptsString();
        }
        return outputStringP.then(scriptsStr => {
            this._session.sendEvent(new vscode_debugadapter_1.OutputEvent(scriptsStr));
            return {
                result: '',
                variablesReference: 0
            };
        });
    }
    getAllScriptsString() {
        const runtimeScripts = Array.from(this._scriptsByUrl.keys())
            .sort();
        return Promise.all(runtimeScripts.map(script => this.getOneScriptString(script))).then(strs => {
            return strs.join('\n');
        });
    }
    getOneScriptString(runtimeScriptPath) {
        let result = '› ' + runtimeScriptPath;
        const clientPath = this._pathTransformer.getClientPathFromTargetPath(runtimeScriptPath);
        if (clientPath && clientPath !== runtimeScriptPath)
            result += ` (${clientPath})`;
        return this._sourceMapTransformer.allSourcePathDetails(clientPath || runtimeScriptPath).then(sourcePathDetails => {
            let mappedSourcesStr = sourcePathDetails.map(details => `    - ${details.originalPath} (${details.inferredPath})`).join('\n');
            if (sourcePathDetails.length)
                mappedSourcesStr = '\n' + mappedSourcesStr;
            return result + mappedSourcesStr;
        });
    }
    /**
     * Allow consumers to override just because of https://github.com/nodejs/node/issues/8426
     */
    globalEvaluate(args) {
        return this.chrome.Runtime.evaluate(args);
    }
    setVariable(args) {
        const handle = this._variableHandles.get(args.variablesReference);
        if (!handle) {
            return Promise.reject(errors.setValueNotSupported());
        }
        return handle.setValue(this, args.name, args.value)
            .then(value => ({ value }));
    }
    setVariableValue(callFrameId, scopeNumber, variableName, value) {
        let evalResultObject;
        return this.chrome.Debugger.evaluateOnCallFrame({ callFrameId, expression: value, silent: true }).then(evalResponse => {
            if (evalResponse.exceptionDetails) {
                const errMsg = ChromeUtils.errorMessageFromExceptionDetails(evalResponse.exceptionDetails);
                return Promise.reject(errors.errorFromEvaluate(errMsg));
            }
            else {
                evalResultObject = evalResponse.result;
                const newValue = ChromeUtils.remoteObjectToCallArgument(evalResultObject);
                return this.chrome.Debugger.setVariableValue({ callFrameId, scopeNumber, variableName, newValue });
            }
        }, error => Promise.reject(errors.errorFromEvaluate(error.message)))
            .then(setVarResponse => ChromeUtils.remoteObjectToValue(evalResultObject).value);
    }
    setPropertyValue(objectId, propName, value) {
        const setPropertyValueFn = `function() { return this["${propName}"] = ${value} }`;
        return this.chrome.Runtime.callFunctionOn({
            objectId, functionDeclaration: setPropertyValueFn,
            silent: true
        }).then(response => {
            if (response.exceptionDetails) {
                const errMsg = ChromeUtils.errorMessageFromExceptionDetails(response.exceptionDetails);
                return Promise.reject(errors.errorFromEvaluate(errMsg));
            }
            else {
                // Temporary, Microsoft/vscode#12019
                return ChromeUtils.remoteObjectToValue(response.result).value;
            }
        }, error => Promise.reject(errors.errorFromEvaluate(error.message)));
    }
    remoteObjectToVariable(name, object, parentEvaluateName, stringify = true, context = 'variables') {
        let value = '';
        if (object) {
            if (object.type === 'object') {
                if (object.subtype === 'internal#location') {
                    // Could format this nicely later, see #110
                    value = 'internal#location';
                }
                else if (object.subtype === 'null') {
                    value = 'null';
                }
                else {
                    return this.createObjectVariable(name, object, parentEvaluateName, context);
                }
            }
            else if (object.type === 'undefined') {
                value = 'undefined';
            }
            else if (object.type === 'function') {
                return Promise.resolve(this.createFunctionVariable(name, object, parentEvaluateName));
            }
            else {
                // The value is a primitive value, or something that has a description (not object, primitive, or undefined). And force to be string
                if (typeof object.value === 'undefined') {
                    value = object.description;
                }
                else if (object.type === 'number') {
                    // .value is truncated, so use .description, the full string representation
                    // Should be like '3' or 'Infinity'.
                    value = object.description;
                }
                else {
                    value = stringify ? `"${object.value}"` : object.value;
                }
            }
        }
        return Promise.resolve({
            name,
            value,
            variablesReference: 0,
            evaluateName: ChromeUtils.getEvaluateName(parentEvaluateName, name)
        });
    }
    createFunctionVariable(name, object, parentEvaluateName) {
        let value;
        const firstBraceIdx = object.description.indexOf('{');
        if (firstBraceIdx >= 0) {
            value = object.description.substring(0, firstBraceIdx) + '{ … }';
        }
        else {
            const firstArrowIdx = object.description.indexOf('=>');
            value = firstArrowIdx >= 0 ?
                object.description.substring(0, firstArrowIdx + 2) + ' …' :
                object.description;
        }
        const evaluateName = ChromeUtils.getEvaluateName(parentEvaluateName, name);
        return {
            name,
            value,
            variablesReference: this._variableHandles.create(new variables_1.PropertyContainer(object.objectId, evaluateName)),
            type: value,
            evaluateName
        };
    }
    createObjectVariable(name, object, parentEvaluateName, context) {
        let value = object.description;
        let propCountP;
        if (object.subtype === 'array' || object.subtype === 'typedarray') {
            value = Variables.getArrayPreview(object, context);
            if (object.preview && !object.preview.overflow) {
                propCountP = Promise.resolve(this.getArrayNumPropsByPreview(object));
            }
            else {
                propCountP = this.getArrayNumPropsByEval(object.objectId);
            }
        }
        else if (object.subtype === 'set' || object.subtype === 'map') {
            if (object.preview && !object.preview.overflow) {
                propCountP = Promise.resolve(this.getCollectionNumPropsByPreview(object));
            }
            else {
                propCountP = this.getCollectionNumPropsByEval(object.objectId);
            }
        }
        else {
            if (object.subtype === 'error') {
                // The Error's description contains the whole stack which is not a nice description.
                // Up to the first newline is just the error name/message.
                const firstNewlineIdx = object.description.indexOf('\n');
                if (firstNewlineIdx >= 0)
                    value = object.description.substr(0, firstNewlineIdx);
            }
            else if (object.subtype === 'promise' && object.preview) {
                const promiseStatus = object.preview.properties.filter(prop => prop.name === '[[PromiseStatus]]')[0];
                if (promiseStatus)
                    value = object.description + ' { ' + promiseStatus.value + ' }';
            }
            else if (object.subtype === 'generator' && object.preview) {
                const generatorStatus = object.preview.properties.filter(prop => prop.name === '[[GeneratorStatus]]')[0];
                if (generatorStatus)
                    value = object.description + ' { ' + generatorStatus.value + ' }';
            }
            else if (object.type === 'object' && object.preview) {
                value = Variables.getObjectPreview(object, context);
            }
            propCountP = Promise.resolve({});
        }
        const evaluateName = ChromeUtils.getEvaluateName(parentEvaluateName, name);
        const variablesReference = this._variableHandles.create(new variables_1.PropertyContainer(object.objectId, evaluateName));
        return propCountP.then(({ indexedVariables, namedVariables }) => ({
            name,
            value,
            type: value,
            variablesReference,
            indexedVariables,
            namedVariables,
            evaluateName
        }));
    }
    completions(args) {
        const text = args.text;
        const column = args.column;
        // 1-indexed column
        const prefix = text.substring(0, column - 1);
        let expression;
        const dot = prefix.lastIndexOf('.');
        if (dot >= 0) {
            expression = prefix.substr(0, dot);
        }
        if (expression) {
            logger.verbose(`Completions: Returning for expression '${expression}'`);
            const getCompletionsFn = `(function(x){var a=[];for(var o=x;o!==null&&typeof o !== 'undefined';o=o.__proto__){a.push(Object.getOwnPropertyNames(o))};return a})(${expression})`;
            let evalPromise;
            if (typeof args.frameId === 'number') {
                const frame = this._frameHandles.get(args.frameId);
                if (!frame) {
                    return Promise.reject(errors.completionsStackFrameNotValid());
                }
                const callFrameId = frame.callFrameId;
                evalPromise = this.chrome.Debugger.evaluateOnCallFrame({ callFrameId, expression: getCompletionsFn, silent: true, returnByValue: true });
            }
            else {
                evalPromise = this.globalEvaluate({ expression: getCompletionsFn, silent: true, returnByValue: true });
            }
            return evalPromise.then(response => {
                if (response.exceptionDetails) {
                    return { targets: [] };
                }
                else {
                    return { targets: this.getFlatAndUniqueCompletionItems(response.result.value) };
                }
            });
        }
        else {
            logger.verbose(`Completions: Returning global completions`);
            // If no expression was passed, we must be getting global completions at a breakpoint
            if (typeof args.frameId !== "number" || !this._frameHandles.get(args.frameId)) {
                return Promise.reject(errors.completionsStackFrameNotValid());
            }
            const callFrame = this._frameHandles.get(args.frameId);
            const scopeExpandPs = callFrame.scopeChain
                .map(scope => new variables_1.ScopeContainer(callFrame.callFrameId, undefined, scope.object.objectId).expand(this));
            return Promise.all(scopeExpandPs)
                .then((variableArrs) => {
                const targets = this.getFlatAndUniqueCompletionItems(variableArrs.map(variableArr => variableArr.map(variable => variable.name)));
                return { targets };
            });
        }
    }
    getFlatAndUniqueCompletionItems(arrays) {
        const set = new Set();
        const items = [];
        for (let i = 0; i < arrays.length; i++) {
            for (let name of arrays[i]) {
                if (!variables_1.isIndexedPropName(name) && !set.has(name)) {
                    set.add(name);
                    items.push({
                        label: name,
                        type: 'property'
                    });
                }
            }
        }
        return items;
    }
    getArrayNumPropsByEval(objectId) {
        // +2 for __proto__ and length
        const getNumPropsFn = `function() { return [this.length, Object.keys(this).length - this.length + 2]; }`;
        return this.getNumPropsByEval(objectId, getNumPropsFn);
    }
    getArrayNumPropsByPreview(object) {
        let indexedVariables = 0;
        let namedVariables = 2; // 2 for __proto__ and length
        object.preview.properties.forEach(prop => variables_1.isIndexedPropName(prop.name) ? indexedVariables++ : namedVariables++);
        return { indexedVariables, namedVariables };
    }
    getCollectionNumPropsByEval(objectId) {
        const getNumPropsFn = `function() { return [0, Object.keys(this).length + 1]; }`; // +1 for [[Entries]];
        return this.getNumPropsByEval(objectId, getNumPropsFn);
    }
    getCollectionNumPropsByPreview(object) {
        let indexedVariables = 0;
        let namedVariables = object.preview.properties.length + 1; // +1 for [[Entries]];
        return { indexedVariables, namedVariables };
    }
    getNumPropsByEval(objectId, getNumPropsFn) {
        return this.chrome.Runtime.callFunctionOn({
            objectId,
            functionDeclaration: getNumPropsFn,
            silent: true,
            returnByValue: true
        }).then(response => {
            if (response.exceptionDetails) {
                const errMsg = ChromeUtils.errorMessageFromExceptionDetails(response.exceptionDetails);
                return Promise.reject(errors.errorFromEvaluate(errMsg));
            }
            else {
                const resultProps = response.result.value;
                if (resultProps.length !== 2) {
                    return Promise.reject(errors.errorFromEvaluate("Did not get expected props, got " + JSON.stringify(resultProps)));
                }
                return { indexedVariables: resultProps[0], namedVariables: resultProps[1] };
            }
        }, error => Promise.reject(errors.errorFromEvaluate(error.message)));
    }
    shouldIgnoreScript(script) {
        return script.url.startsWith('extensions::') || script.url.startsWith('chrome-extension://');
    }
}
ChromeDebugAdapter.PLACEHOLDER_URL_PROTOCOL = 'eval://';
ChromeDebugAdapter.SCRIPTS_COMMAND = '.scripts';
ChromeDebugAdapter.THREAD_ID = 1;
ChromeDebugAdapter.SET_BREAKPOINTS_TIMEOUT = 3000;
ChromeDebugAdapter.HITCONDITION_MATCHER = /^(>|>=|=|<|<=|%)?\s*([0-9]+)$/;
exports.ChromeDebugAdapter = ChromeDebugAdapter;

//# sourceMappingURL=chromeDebugAdapter.js.map
