/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const os = require("os");
const vscode_debugadapter_1 = require("vscode-debugadapter");
const utils = require("../utils");
const logger = require("../logger");
class ChromeDebugSession extends vscode_debugadapter_1.DebugSession {
    /**
     * This needs a bit of explanation -
     * The Session is reinstantiated for each session, but consumers need to configure their instance of
     * ChromeDebugSession. Consumers should call getSession with their config options, then call
     * DebugSession.run with the result. Alternatively they could subclass ChromeDebugSession and pass
     * their options to the super constructor, but I think this is easier to follow.
     */
    static getSession(opts) {
        // class expression!
        return class extends ChromeDebugSession {
            constructor(debuggerLinesAndColumnsStartAt1, isServer) {
                super(debuggerLinesAndColumnsStartAt1, isServer, opts);
            }
        };
    }
    constructor(debuggerLinesAndColumnsStartAt1, isServer, opts) {
        super();
        this._extensionName = opts.extensionName;
        this._debugAdapter = new opts.adapter(opts, this);
        const logFilePath = opts.logFilePath;
        logger.init((msg, level) => this.onLog(msg, level), logFilePath, isServer);
        logVersionInfo();
        const safeGetErrDetails = err => {
            let errMsg;
            try {
                errMsg = err.stack ? err.stack : JSON.stringify(err);
            }
            catch (e) {
                errMsg = 'Error while handling previous error: ' + e.stack;
            }
            return errMsg;
        };
        process.on('uncaughtException', (err) => {
            logger.error(`******** Unhandled error in debug adapter: ${safeGetErrDetails(err)}`);
            throw err;
        });
        process.addListener('unhandledRejection', (err) => {
            // Node tests are watching for the ********, so fix the tests if it's changed
            //logger.error(`******** Unhandled error in debug adapter - Unhandled promise rejection: ${safeGetErrDetails(err)}`);
        });
    }
    /**
     * Overload sendEvent to log
     */
    sendEvent(event) {
        if (event.event !== 'output') {
            // Don't create an infinite loop...
            logger.verbose(`To client: ${JSON.stringify(event)}`);
        }
        super.sendEvent(event);
    }
    /**
     * Overload sendRequest to log
     */
    sendRequest(command, args, timeout, cb) {
        logger.verbose(`To client: ${JSON.stringify(command)}(${JSON.stringify(args)}), timeout: ${timeout}`);
        super.sendRequest(command, args, timeout, cb);
    }
    /**
     * Overload sendResponse to log
     */
    sendResponse(response) {
        logger.verbose(`To client: ${JSON.stringify(response)}`);
        super.sendResponse(response);
    }
    onLog(msg, level) {
        const outputCategory = level === logger.LogLevel.Error ? 'stderr' : 'console';
        if (level === logger.LogLevel.Verbose) {
            // Distinguish verbose messages with this prefix - makes the logs much more readable
            msg = `  ›${msg}`;
        }
        this.sendEvent(new vscode_debugadapter_1.OutputEvent(msg, outputCategory));
    }
    /**
     * Takes a response and a promise to the response body. If the promise is successful, assigns the response body and sends the response.
     * If the promise fails, sets the appropriate response parameters and sends the response.
     */
    sendResponseAsync(request, response, responseP) {
        responseP.then((body) => {
            response.body = body;
            this.sendResponse(response);
        }, e => {
            if (e.format) {
                this.sendErrorResponse(response, e);
                return;
            }
            const eStr = e ? e.message : 'Unknown error';
            if (eStr === 'Error: unknowncommand') {
                this.sendErrorResponse(response, 1014, `[${this._extensionName}] Unrecognized request: ${request.command}`, null, vscode_debugadapter_1.ErrorDestination.Telemetry);
                return;
            }
            if (request.command === 'evaluate') {
                // Errors from evaluate show up in the console or watches pane. Doesn't seem right
                // as it's not really a failed request. So it doesn't need the [extensionName] tag and worth special casing.
                response.message = eStr;
                response.success = false;
                this.sendResponse(response);
                return;
            }
            else {
                this.failedRequest(request.command, response, e);
            }
        });
    }
    /**
     * Overload dispatchRequest to the debug adapters' Promise-based methods instead of DebugSession's callback-based methods
     */
    dispatchRequest(request) {
        const response = new vscode_debugadapter_1.Response(request);
        try {
            logger.verbose(`From client: ${request.command}(${JSON.stringify(request.arguments)})`);
            const responseP = (request.command in this._debugAdapter) ?
                Promise.resolve(this._debugAdapter[request.command](request.arguments, request.seq)) :
                utils.errP('unknowncommand');
            this.sendResponseAsync(request, response, responseP);
        }
        catch (e) {
            this.failedRequest(request.command, response, e);
        }
    }
    failedRequest(requestType, response, error) {
        let errMsg;
        if (error.data) {
            // Error response from runtime
            errMsg = error.message + ': ' + error.data;
        }
        else {
            errMsg = error.stack || error.message;
        }
        logger.error(`Error processing "${requestType}": ${errMsg}`);
        // These errors show up in the message bar at the top (or nowhere), sometimes not obvious that they
        // come from the adapter, so add extensionName
        this.sendErrorResponse(response, 1104, '[{_extensionName}] Error processing "{_requestType}": {_stack}', { _extensionName: this._extensionName, _requestType: requestType, _stack: errMsg }, vscode_debugadapter_1.ErrorDestination.Telemetry);
    }
}
exports.ChromeDebugSession = ChromeDebugSession;
function logVersionInfo() {
    logger.log(`OS: ${os.platform()} ${os.arch()}`);
    logger.log(`Adapter node: ${process.version} ${process.arch}`);
    logger.log('vscode-chrome-debug-core: ' + require('../../../package.json').version);
}

//# sourceMappingURL=chromeDebugSession.js.map
