/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
class BaseVariableContainer {
    constructor(objectId, evaluateName) {
        this.objectId = objectId;
        this.evaluateName = evaluateName;
    }
    expand(adapter, filter, start, count) {
        return adapter.getVariablesForObjectId(this.objectId, this.evaluateName, filter, start, count);
    }
}
exports.BaseVariableContainer = BaseVariableContainer;
class PropertyContainer extends BaseVariableContainer {
    setValue(adapter, name, value) {
        return adapter.setPropertyValue(this.objectId, name, value);
    }
}
exports.PropertyContainer = PropertyContainer;
class ScopeContainer extends BaseVariableContainer {
    constructor(frameId, origScopeIndex, objectId, thisObj, returnValue) {
        super(objectId, '');
        this._thisObj = thisObj;
        this._returnValue = returnValue;
        this._frameId = frameId;
        this._origScopeIndex = origScopeIndex;
    }
    /**
     * Call super then insert the 'this' object if needed
     */
    expand(adapter, filter, start, count) {
        // No filtering in scopes right now
        return super.expand(adapter, 'all', start, count).then(variables => {
            if (this._thisObj) {
                // If this is a scope that should have the 'this', prop, insert it at the top of the list
                return this.insertRemoteObject(adapter, variables, 'this', this._thisObj);
            }
            return variables;
        }).then(variables => {
            if (this._returnValue) {
                return this.insertRemoteObject(adapter, variables, 'Return value', this._returnValue);
            }
            return variables;
        });
    }
    setValue(adapter, name, value) {
        return adapter.setVariableValue(this._frameId, this._origScopeIndex, name, value);
    }
    insertRemoteObject(adapter, variables, name, obj) {
        return adapter.remoteObjectToVariable(name, obj).then(variable => {
            variables.unshift(variable);
            return variables;
        });
    }
}
exports.ScopeContainer = ScopeContainer;
class ExceptionContainer extends PropertyContainer {
    constructor(objectId, exception) {
        super(exception.objectId, undefined);
        this._exception = exception;
    }
    /**
     * Expand the exception as if it were a Scope
     */
    static create(exception) {
        return exception.objectId ?
            new ExceptionContainer(exception.objectId, exception) :
            new ExceptionValueContainer(exception);
    }
}
exports.ExceptionContainer = ExceptionContainer;
/**
 * For when a value is thrown instead of an object
 */
class ExceptionValueContainer extends ExceptionContainer {
    constructor(exception) {
        super('EXCEPTION_ID', exception);
    }
    /**
     * Make up a fake 'Exception' property to hold the thrown value, displayed under the Exception Scope
     */
    expand(adapter, filter, start, count) {
        const excValuePropDescriptor = { name: 'Exception', value: this._exception };
        return adapter.propertyDescriptorToVariable(excValuePropDescriptor)
            .then(variable => [variable]);
    }
}
exports.ExceptionValueContainer = ExceptionValueContainer;
function isIndexedPropName(name) {
    return !isNaN(parseInt(name, 10));
}
exports.isIndexedPropName = isIndexedPropName;
const PREVIEW_PROPS_DEFAULT = 3;
const PREVIEW_PROPS_CONSOLE = 8;
const PREVIEW_PROP_LENGTH = 50;
const ELLIPSIS = '…';
function getArrayPreview(object, context) {
    let value = object.description;
    if (object.preview) {
        const numProps = context === 'repl' ? PREVIEW_PROPS_CONSOLE : PREVIEW_PROPS_DEFAULT;
        const props = object.preview.properties.slice(0, numProps);
        let propsPreview = props
            .map(prop => propertyPreviewToString(prop))
            .join(', ');
        if (object.preview.overflow || object.preview.properties.length > numProps) {
            propsPreview += ', …';
        }
        value += ` [${propsPreview}]`;
    }
    return value;
}
exports.getArrayPreview = getArrayPreview;
function getObjectPreview(object, context) {
    let value = object.description;
    if (object.preview) {
        const numProps = context === 'repl' ? PREVIEW_PROPS_CONSOLE : PREVIEW_PROPS_DEFAULT;
        const props = object.preview.properties.slice(0, numProps);
        let propsPreview = props
            .map(prop => `${prop.name}: ${propertyPreviewToString(prop)}`)
            .join(', ');
        if (object.preview.overflow || object.preview.properties.length > numProps) {
            propsPreview += ', …';
        }
        value += ` {${propsPreview}}`;
    }
    return value;
}
exports.getObjectPreview = getObjectPreview;
function propertyPreviewToString(prop) {
    const value = trimProperty(prop.value);
    return prop.type === 'string' ?
        `"${value}"` :
        value;
}
function trimProperty(value) {
    return (value.length > PREVIEW_PROP_LENGTH) ?
        value.substr(0, PREVIEW_PROP_LENGTH) + ELLIPSIS :
        value;
}

//# sourceMappingURL=variables.js.map
