/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const sourceMapFactory_1 = require("./sourceMapFactory");
class SourceMaps {
    constructor(webRoot, sourceMapPathOverrides) {
        // Maps absolute paths to generated/authored source files to their corresponding SourceMap object
        this._generatedPathToSourceMap = new Map();
        this._authoredPathToSourceMap = new Map();
        this._webRoot = webRoot;
        this._sourceMapPathOverrides = sourceMapPathOverrides;
    }
    /**
     * Returns the generated script path for an authored source path
     * @param pathToSource - The absolute path to the authored file
     */
    getGeneratedPathFromAuthoredPath(authoredPath) {
        authoredPath = authoredPath.toLowerCase();
        return this._authoredPathToSourceMap.has(authoredPath) ?
            this._authoredPathToSourceMap.get(authoredPath).generatedPath() :
            null;
    }
    mapToGenerated(authoredPath, line, column) {
        authoredPath = authoredPath.toLowerCase();
        return this._authoredPathToSourceMap.has(authoredPath) ?
            this._authoredPathToSourceMap.get(authoredPath)
                .generatedPositionFor(authoredPath, line, column) :
            null;
    }
    mapToAuthored(pathToGenerated, line, column) {
        pathToGenerated = pathToGenerated.toLowerCase();
        return this._generatedPathToSourceMap.has(pathToGenerated) ?
            this._generatedPathToSourceMap.get(pathToGenerated)
                .authoredPositionFor(line, column) :
            null;
    }
    allMappedSources(pathToGenerated) {
        pathToGenerated = pathToGenerated.toLowerCase();
        return this._generatedPathToSourceMap.has(pathToGenerated) ?
            this._generatedPathToSourceMap.get(pathToGenerated).authoredSources :
            null;
    }
    allSourcePathDetails(pathToGenerated) {
        pathToGenerated = pathToGenerated.toLowerCase();
        return this._generatedPathToSourceMap.has(pathToGenerated) ?
            this._generatedPathToSourceMap.get(pathToGenerated).allSourcePathDetails :
            null;
    }
    sourceContentFor(authoredPath) {
        authoredPath = authoredPath.toLowerCase();
        return this._authoredPathToSourceMap.has(authoredPath) ?
            this._authoredPathToSourceMap.get(authoredPath)
                .sourceContentFor(authoredPath) :
            null;
    }
    /**
     * Given a new path to a new script file, finds and loads the sourcemap for that file
     */
    processNewSourceMap(pathToGenerated, sourceMapURL) {
        return this._generatedPathToSourceMap.has(pathToGenerated.toLowerCase()) ?
            Promise.resolve() :
            sourceMapFactory_1.getMapForGeneratedPath(pathToGenerated, sourceMapURL, this._webRoot, this._sourceMapPathOverrides).then(sourceMap => {
                if (sourceMap) {
                    this._generatedPathToSourceMap.set(pathToGenerated.toLowerCase(), sourceMap);
                    sourceMap.authoredSources.forEach(authoredSource => this._authoredPathToSourceMap.set(authoredSource.toLowerCase(), sourceMap));
                }
            });
    }
}
exports.SourceMaps = SourceMaps;

//# sourceMappingURL=sourceMaps.js.map
