/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const WebSocket = require("ws");
const telemetry_1 = require("../telemetry");
const errors = require("../errors");
const utils = require("../utils");
const vscode_debugadapter_1 = require("vscode-debugadapter");
const chromeTargetDiscoveryStrategy_1 = require("./chromeTargetDiscoveryStrategy");
const noice_json_rpc_1 = require("noice-json-rpc");
/**
 * A subclass of WebSocket that logs all traffic
 */
class LoggingSocket extends WebSocket {
    constructor(address, protocols, options) {
        super(address, protocols, options);
        this.on('error', e => {
            vscode_debugadapter_1.logger.log('Websocket error: ' + e.toString());
        });
        this.on('close', () => {
            vscode_debugadapter_1.logger.log('Websocket closed');
        });
        this.on('message', msgStr => {
            let msgObj;
            try {
                msgObj = JSON.parse(msgStr);
            }
            catch (e) {
                vscode_debugadapter_1.logger.error(`Invalid JSON from target: (${e.message}): ${msgStr}`);
                return;
            }
            if (msgObj && !(msgObj.method && msgObj.method.startsWith('Network.'))) {
                // Not really the right place to examine the content of the message, but don't log annoying Network activity notifications.
                vscode_debugadapter_1.logger.verbose('← From target: ' + msgStr);
            }
        });
    }
    send(data, cb) {
        super.send.apply(this, arguments);
        const msgStr = JSON.stringify(data);
        vscode_debugadapter_1.logger.verbose('→ To target: ' + msgStr);
    }
}
/**
 * Connects to a target supporting the Chrome Debug Protocol and sends and receives messages
 */
class ChromeConnection {
    constructor(targetDiscovery, targetFilter) {
        this._targetFilter = targetFilter;
        this._targetDiscoveryStrategy = targetDiscovery || new chromeTargetDiscoveryStrategy_1.ChromeTargetDiscovery(vscode_debugadapter_1.logger, telemetry_1.telemetry);
    }
    get isAttached() { return !!this._client; }
    get api() {
        return this._client && this._client.api();
    }
    /**
     * Attach the websocket to the first available tab in the chrome instance with the given remote debugging port number.
     */
    attach(address = '127.0.0.1', port = 9222, targetUrl, timeout) {
        return this._attach(address, port, targetUrl, timeout)
            .then(() => { });
    }
    attachToWebsocketUrl(wsUrl) {
        this._socket = new LoggingSocket(wsUrl);
        this._client = new noice_json_rpc_1.Client(this._socket);
        this._client.on('error', e => vscode_debugadapter_1.logger.error('Error handling message from target: ' + e.message));
    }
    _attach(address, port, targetUrl, timeout = ChromeConnection.ATTACH_TIMEOUT) {
        return utils.retryAsync(() => this._targetDiscoveryStrategy.getTarget(address, port, this._targetFilter, targetUrl), timeout, /*intervalDelay=*/ 200)
            .catch(err => Promise.reject(errors.runtimeConnectionTimeout(timeout, err.message)))
            .then(wsUrl => {
            return this.attachToWebsocketUrl(wsUrl);
        });
    }
    run() {
        // This is a CDP version difference which will have to be handled more elegantly with others later...
        // For now, we need to send both messages and ignore a failing one.
        return Promise.all([
            this.api.Runtime.runIfWaitingForDebugger(),
            this.api.Runtime.run()
        ])
            .then(() => { }, e => { });
    }
    close() {
        this._socket.close();
    }
    onClose(handler) {
        this._socket.on('close', handler);
    }
}
ChromeConnection.ATTACH_TIMEOUT = 10000; // ms
exports.ChromeConnection = ChromeConnection;

//# sourceMappingURL=chromeConnection.js.map
