/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments)).next());
    });
};
const utils = require("../utils");
const logger = require("../logger");
class SkipFilesManager {
    constructor(_cda) {
        this._cda = _cda;
        this._skipFileStatuses = new Map();
        this._blackboxedRegexes = [];
    }
    get chrome() {
        return this._cda.chrome;
    }
    onScriptParsed(script, mappedUrl) {
        this.resolveSkipFiles(script, mappedUrl);
    }
    onAttach(launchAttachArgs) {
        let patterns = [];
        if (launchAttachArgs.skipFiles) {
            const skipFilesArgs = launchAttachArgs.skipFiles.filter(glob => {
                if (glob.startsWith('!')) {
                    logger.log(`Warning: skipFiles entries starting with '!' aren't supported and will be ignored. ("${glob}")`, /*forceLog=*/ true);
                    return false;
                }
                return true;
            });
            patterns = skipFilesArgs.map(glob => utils.pathGlobToBlackboxedRegex(glob));
        }
        if (launchAttachArgs.skipFileRegExps) {
            patterns = patterns.concat(launchAttachArgs.skipFileRegExps);
        }
        if (patterns.length) {
            this._blackboxedRegexes = patterns.map(pattern => new RegExp(pattern, 'i'));
            this.chrome.Debugger.setBlackboxPatterns({ patterns })
                .catch(() => this.warnNoSkipFiles());
        }
    }
    resolveSkipFiles(script, mappedUrl, toggling) {
        return __awaiter(this, void 0, void 0, function* () {
            const details = yield this._cda.sourceMapTransformer.allSourcePathDetails(mappedUrl);
            if (details && details.length) {
                const parentIsSkipped = this.shouldSkipSource(script.url);
                const libPositions = [];
                // Figure out skip/noskip transitions within script
                let inLibRange = parentIsSkipped;
                details.forEach((detail, i) => __awaiter(this, void 0, void 0, function* () {
                    let isSkippedFile = this.shouldSkipSource(detail.inferredPath);
                    if (typeof isSkippedFile !== 'boolean') {
                        // Inherit the parent's status
                        isSkippedFile = parentIsSkipped;
                    }
                    this._skipFileStatuses.set(detail.inferredPath, isSkippedFile);
                    if ((isSkippedFile && !inLibRange) || (!isSkippedFile && inLibRange)) {
                        libPositions.push({
                            lineNumber: detail.startPosition.line,
                            columnNumber: detail.startPosition.column
                        });
                        inLibRange = !inLibRange;
                    }
                }));
                // If there's any change from the default, set proper blackboxed ranges
                if (libPositions.length || toggling) {
                    if (parentIsSkipped) {
                        libPositions.splice(0, 0, { lineNumber: 0, columnNumber: 0 });
                    }
                    yield this.chrome.Debugger.setBlackboxedRanges({
                        scriptId: script.scriptId,
                        positions: []
                    }).catch(() => this.warnNoSkipFiles());
                    if (libPositions.length) {
                        this.chrome.Debugger.setBlackboxedRanges({
                            scriptId: script.scriptId,
                            positions: libPositions
                        }).catch(() => this.warnNoSkipFiles());
                    }
                }
            }
            else {
                // Dynamically skip or un-skip non-sourcemapped script if needed
                const status = yield this.getSkipStatus(mappedUrl);
                const skippedByPattern = this.matchesSkipFilesPatterns(mappedUrl);
                if (typeof status === 'boolean' && status !== skippedByPattern) {
                    const positions = status ? [{ lineNumber: 0, columnNumber: 0 }] : [];
                    this.chrome.Debugger.setBlackboxedRanges({
                        scriptId: script.scriptId,
                        positions
                    }).catch(() => this.warnNoSkipFiles());
                }
            }
        });
    }
    /**
     * If the source has a saved skip status, return that, whether true or false.
     * If not, check it against the patterns list.
     */
    shouldSkipSource(sourcePath) {
        const status = this.getSkipStatus(sourcePath);
        if (typeof status === 'boolean') {
            return status;
        }
        if (this.matchesSkipFilesPatterns(sourcePath)) {
            return true;
        }
        return undefined;
    }
    /**
     * Returns true if this path matches one of the static skip patterns
     */
    matchesSkipFilesPatterns(sourcePath) {
        return this._blackboxedRegexes.some(regex => {
            return regex.test(sourcePath);
        });
    }
    /**
     * Returns the current skip status for this path, which is either an authored or generated script.
     */
    getSkipStatus(sourcePath) {
        if (this._skipFileStatuses.has(sourcePath)) {
            return this._skipFileStatuses.get(sourcePath);
        }
        return undefined;
    }
    toggleSkipFileStatus(args) {
        return __awaiter(this, void 0, void 0, function* () {
            const aPath = utils.fileUrlToPath(args.path);
            logger.log(`Toggling the skip file status for: ${aPath}`);
            if (!(yield this._cda.isInCurrentStack(aPath))) {
                // Only valid for files that are in the current stack
                logger.log(`Can't toggle the skipFile status for ${aPath} - it's not in the current stack.`);
                return;
            }
            const generatedPath = yield this.sourceMapTransformer.getGeneratedPathFromAuthoredPath(aPath);
            if (!generatedPath) {
                // haven't heard of this script
                return;
            }
            const sources = yield this.sourceMapTransformer.allSources(generatedPath);
            if (generatedPath === aPath && sources.length) {
                // Ignore toggling skip status for generated scripts with sources
                logger.log(`Can't toggle skipFile status for ${aPath} - it's a script with a sourcemap`);
                return;
            }
            const newStatus = !this.shouldSkipSource(aPath);
            this._skipFileStatuses.set(aPath, newStatus);
            const targetPath = this.pathTransformer.getTargetPathFromClientPath(generatedPath);
            const script = this._scriptsByUrl.get(targetPath);
            yield this.resolveSkipFiles(script, generatedPath, sources, /*toggling=*/ true);
            if (!newStatus) {
                this.removeMatchingRegexes(script.url);
            }
            if (generatedPath === aPath) {
                if (newStatus) {
                    this._blackboxedRegexes.push(new RegExp(script.url, 'i'));
                }
            }
            this.onPaused(this._lastPauseState.event, this._lastPauseState.expecting);
        });
    }
    removeMatchingRegexes(path) {
        this._blackboxedRegexes = this._blackboxedRegexes.filter(regex => !regex.test(path));
        this.chrome.Debugger.setBlackboxPatterns({
            patterns: this._blackboxedRegexes.map(regex => regex.source)
        }).catch(() => this.warnNoSkipFiles());
    }
    warnNoSkipFiles() {
        logger.log('Warning: this runtime does not support skipFiles');
    }
}
exports.SkipFilesManager = SkipFilesManager;

//# sourceMappingURL=skipFiles.js.map
