/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
const path = require("path");
const sourceMaps_1 = require("../sourceMaps/sourceMaps");
const utils = require("../utils");
const vscode_debugadapter_1 = require("vscode-debugadapter");
const nls = require("vscode-nls");
const localize = nls.config(process.env.VSCODE_NLS_CONFIG)(__filename);
/**
 * If sourcemaps are enabled, converts from source files on the client side to runtime files on the target side
 */
class BaseSourceMapTransformer {
    constructor(sourceHandles) {
        this._preLoad = Promise.resolve();
        this._sourceHandles = sourceHandles;
    }
    get sourceMaps() {
        return this._sourceMaps;
    }
    launch(args) {
        this.init(args);
    }
    attach(args) {
        this.init(args);
    }
    init(args) {
        if (args.sourceMaps) {
            this._sourceMaps = new sourceMaps_1.SourceMaps(args.webRoot, args.sourceMapPathOverrides);
            this._requestSeqToSetBreakpointsArgs = new Map();
            this._allRuntimeScriptPaths = new Set();
            this._authoredPathsToMappedBPs = new Map();
        }
    }
    clearTargetContext() {
        this._allRuntimeScriptPaths = new Set();
    }
    /**
     * Apply sourcemapping to the setBreakpoints request path/lines.
     * Returns true if completed successfully, and setBreakpoint should continue.
     */
    setBreakpoints(args, requestSeq) {
        if (!this._sourceMaps) {
            return;
        }
        const originalBPs = JSON.parse(JSON.stringify(args.breakpoints));
        if (args.source.sourceReference) {
            // If the source contents were inlined, then args.source has no path, but we
            // stored it in the handle
            const handle = this._sourceHandles.get(args.source.sourceReference);
            if (handle.mappedPath) {
                args.source.path = handle.mappedPath;
            }
        }
        if (args.source.path) {
            const argsPath = args.source.path;
            const mappedPath = this._sourceMaps.getGeneratedPathFromAuthoredPath(argsPath);
            if (mappedPath) {
                vscode_debugadapter_1.logger.log(`SourceMaps.setBP: Mapped ${argsPath} to ${mappedPath}`);
                args.authoredPath = argsPath;
                args.source.path = mappedPath;
                // DebugProtocol doesn't send cols yet, but they need to be added from sourcemaps
                args.breakpoints.forEach(bp => {
                    const { line, column = 0 } = bp;
                    const mapped = this._sourceMaps.mapToGenerated(argsPath, line, column);
                    if (mapped) {
                        vscode_debugadapter_1.logger.log(`SourceMaps.setBP: Mapped ${argsPath}:${line + 1}:${column + 1} to ${mappedPath}:${mapped.line + 1}:${mapped.column + 1}`);
                        bp.line = mapped.line;
                        bp.column = mapped.column;
                    }
                    else {
                        vscode_debugadapter_1.logger.log(`SourceMaps.setBP: Mapped ${argsPath} but not line ${line + 1}, column 1`);
                        bp.column = column; // take 0 default if needed
                    }
                });
                this._authoredPathsToMappedBPs.set(argsPath, args.breakpoints);
                // Include BPs from other files that map to the same file. Ensure the current file's breakpoints go first
                this._sourceMaps.allMappedSources(mappedPath).forEach(sourcePath => {
                    if (sourcePath === argsPath) {
                        return;
                    }
                    const sourceBPs = this._authoredPathsToMappedBPs.get(sourcePath);
                    if (sourceBPs) {
                        // Don't modify the cached array
                        args.breakpoints = args.breakpoints.concat(sourceBPs);
                    }
                });
            }
            else if (this._allRuntimeScriptPaths.has(argsPath)) {
                // It's a generated file which is loaded
                vscode_debugadapter_1.logger.log(`SourceMaps.setBP: SourceMaps are enabled but ${argsPath} is a runtime script`);
            }
            else {
                // Source (or generated) file which is not loaded, need to wait
                vscode_debugadapter_1.logger.log(`SourceMaps.setBP: ${argsPath} can't be resolved to a loaded script. It may just not be loaded yet.`);
                return;
            }
        }
        else {
        }
        this._requestSeqToSetBreakpointsArgs.set(requestSeq, {
            originalBPs,
            authoredPath: args.authoredPath,
            generatedPath: args.source.path
        });
        return;
    }
    /**
     * Apply sourcemapping back to authored files from the response
     */
    setBreakpointsResponse(response, requestSeq) {
        if (this._sourceMaps && this._requestSeqToSetBreakpointsArgs.has(requestSeq)) {
            const args = this._requestSeqToSetBreakpointsArgs.get(requestSeq);
            if (args.authoredPath) {
                // authoredPath is set, so the file was mapped to source.
                // Remove breakpoints from files that map to the same file, and map back to source.
                response.breakpoints = response.breakpoints.filter((_, i) => i < args.originalBPs.length);
                response.breakpoints.forEach((bp, i) => {
                    const mapped = this._sourceMaps.mapToAuthored(args.generatedPath, bp.line, bp.column);
                    if (mapped) {
                        vscode_debugadapter_1.logger.log(`SourceMaps.setBP: Mapped ${args.generatedPath}:${bp.line + 1}:${bp.column + 1} to ${mapped.source}:${mapped.line + 1}`);
                        bp.line = mapped.line;
                        bp.column = mapped.column;
                    }
                    else {
                        vscode_debugadapter_1.logger.log(`SourceMaps.setBP: Can't map ${args.generatedPath}:${bp.line + 1}:${bp.column + 1}, keeping original line numbers.`);
                        bp.line = args.originalBPs[i].line;
                        bp.column = args.originalBPs[i].column;
                    }
                    this._requestSeqToSetBreakpointsArgs.delete(requestSeq);
                });
            }
        }
    }
    /**
     * Apply sourcemapping to the stacktrace response
     */
    stackTraceResponse(response) {
        if (this._sourceMaps) {
            response.stackFrames.forEach(stackFrame => {
                if (!stackFrame.source) {
                    return;
                }
                const mapped = this._sourceMaps.mapToAuthored(stackFrame.source.path, stackFrame.line, stackFrame.column);
                if (mapped && utils.existsSync(mapped.source)) {
                    // Script was mapped to a valid path
                    stackFrame.source.path = mapped.source;
                    stackFrame.source.sourceReference = undefined;
                    stackFrame.source.name = path.basename(mapped.source);
                    stackFrame.line = mapped.line;
                    stackFrame.column = mapped.column;
                    stackFrame.isSourceMapped = true;
                }
                else {
                    const inlinedSource = mapped && this._sourceMaps.sourceContentFor(mapped.source);
                    if (mapped && inlinedSource) {
                        // Clear the path and set the sourceReference - the client will ask for
                        // the source later and it will be returned from the sourcemap
                        stackFrame.source.name = path.basename(mapped.source);
                        stackFrame.source.path = mapped.source;
                        stackFrame.source.sourceReference = this.getSourceReferenceForScriptPath(mapped.source, inlinedSource);
                        stackFrame.source.origin = localize(0, null);
                        stackFrame.line = mapped.line;
                        stackFrame.column = mapped.column;
                        stackFrame.isSourceMapped = true;
                    }
                    else if (utils.existsSync(stackFrame.source.path)) {
                        // Script could not be mapped, but does exist on disk. Keep it and clear the sourceReference.
                        stackFrame.source.sourceReference = undefined;
                        stackFrame.source.origin = undefined;
                    }
                }
            });
        }
    }
    /**
     * Get the existing handle for this script, identified by runtime scriptId, or create a new one
     */
    getSourceReferenceForScriptPath(mappedPath, contents) {
        return this._sourceHandles.lookupF(container => container.mappedPath === mappedPath) ||
            this._sourceHandles.create({ contents, mappedPath });
    }
    scriptParsed(pathToGenerated, sourceMapURL) {
        if (this._sourceMaps) {
            this._allRuntimeScriptPaths.add(pathToGenerated);
            if (!sourceMapURL)
                return Promise.resolve(null);
            // Load the sourcemap for this new script and log its sources
            return this._sourceMaps.processNewSourceMap(pathToGenerated, sourceMapURL).then(() => {
                const sources = this._sourceMaps.allMappedSources(pathToGenerated);
                if (sources) {
                    vscode_debugadapter_1.logger.log(`SourceMaps.scriptParsed: ${pathToGenerated} was just loaded and has mapped sources: ${JSON.stringify(sources)}`);
                }
                return sources;
            });
        }
        else {
            return Promise.resolve(null);
        }
    }
    breakpointResolved(bp, scriptPath) {
        if (this._sourceMaps) {
            const mapped = this._sourceMaps.mapToAuthored(scriptPath, bp.line, bp.column);
            if (mapped) {
                // No need to send back the path, the bp can only move within its script
                bp.line = mapped.line;
                bp.column = mapped.column;
            }
        }
    }
    scopesResponse(pathToGenerated, scopesResponse) {
        if (this._sourceMaps) {
            scopesResponse.scopes.forEach(scope => this.mapScopeLocations(pathToGenerated, scope));
        }
    }
    mapScopeLocations(pathToGenerated, scope) {
        if (typeof scope.line !== 'number') {
            return;
        }
        const mappedStart = this._sourceMaps.mapToAuthored(pathToGenerated, scope.line, scope.column);
        if (mappedStart) {
            // Only apply changes if both mappings are found
            const mappedEnd = this._sourceMaps.mapToAuthored(pathToGenerated, scope.endLine, scope.endColumn);
            if (mappedEnd) {
                scope.line = mappedStart.line;
                scope.column = mappedStart.column;
                scope.endLine = mappedEnd.line;
                scope.endColumn = mappedEnd.column;
            }
        }
    }
    mapToGenerated(authoredPath, line, column) {
        if (!this._sourceMaps)
            return Promise.resolve(null);
        return this._preLoad.then(() => this._sourceMaps.mapToGenerated(authoredPath, line, column));
    }
    mapToAuthored(pathToGenerated, line, column) {
        if (!this._sourceMaps)
            return Promise.resolve(null);
        return this._preLoad.then(() => this._sourceMaps.mapToAuthored(pathToGenerated, line, column));
    }
    getGeneratedPathFromAuthoredPath(authoredPath) {
        if (!this._sourceMaps)
            return Promise.resolve(authoredPath);
        return this._preLoad.then(() => {
            // Find the generated path, or check whether this script is actually a runtime path - if so, return that
            return this._sourceMaps.getGeneratedPathFromAuthoredPath(authoredPath) ||
                (this._allRuntimeScriptPaths.has(authoredPath) ? authoredPath : null);
        });
    }
    allSources(pathToGenerated) {
        if (!this._sourceMaps)
            return Promise.resolve([]);
        return this._preLoad.then(() => {
            return this._sourceMaps.allMappedSources(pathToGenerated) || [];
        });
    }
    allSourcePathDetails(pathToGenerated) {
        if (!this._sourceMaps)
            return Promise.resolve([]);
        return this._preLoad.then(() => {
            return this._sourceMaps.allSourcePathDetails(pathToGenerated) || [];
        });
    }
}
exports.BaseSourceMapTransformer = BaseSourceMapTransformer;

//# sourceMappingURL=baseSourceMapTransformer.js.map
