/*---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var fs = require('fs');
var debugSession_1 = require('./debugSession');
(function (LogLevel) {
    LogLevel[LogLevel["Verbose"] = 0] = "Verbose";
    LogLevel[LogLevel["Log"] = 1] = "Log";
    LogLevel[LogLevel["Warn"] = 2] = "Warn";
    LogLevel[LogLevel["Error"] = 3] = "Error";
})(exports.LogLevel || (exports.LogLevel = {}));
var LogLevel = exports.LogLevel;
var Logger = (function () {
    function Logger() {
        this._pendingLogQ = [];
    }
    Logger.prototype.log = function (msg, level) {
        if (level === void 0) { level = LogLevel.Log; }
        msg = msg + '\n';
        this._write(msg, level);
    };
    Logger.prototype.verbose = function (msg) {
        this.log(msg, LogLevel.Verbose);
    };
    Logger.prototype.warn = function (msg) {
        this.log(msg, LogLevel.Warn);
    };
    Logger.prototype.error = function (msg) {
        this.log(msg, LogLevel.Error);
    };
    /**
     * `log` adds a newline, `write` doesn't
     */
    Logger.prototype._write = function (msg, level) {
        if (level === void 0) { level = LogLevel.Log; }
        // [null, undefined] => string
        msg = msg + '';
        if (this._pendingLogQ) {
            this._pendingLogQ.push({ msg: msg, level: level });
        }
        else {
            this._currentLogger.log(msg, level);
        }
    };
    /**
     * Set the logger's minimum level to log in the console, and whether to log to the file. Log messages are queued before this is
     * called the first time, because minLogLevel defaults to Warn.
     */
    Logger.prototype.setup = function (consoleMinLogLevel, logToFile) {
        var _this = this;
        if (this._currentLogger) {
            this._currentLogger.setup(consoleMinLogLevel, logToFile);
            // Now that we have a minimum logLevel, we can clear out the queue of pending messages
            if (this._pendingLogQ) {
                var logQ = this._pendingLogQ;
                this._pendingLogQ = null;
                logQ.forEach(function (item) { return _this._write(item.msg, item.level); });
            }
        }
    };
    Logger.prototype.init = function (logCallback, logFilePath, logToConsole) {
        // Re-init, create new global Logger
        this._pendingLogQ = this._pendingLogQ || [];
        this._currentLogger = new InternalLogger(logCallback, logFilePath, logToConsole);
        if (logFilePath) {
            var d = new Date();
            var timestamp = d.toLocaleTimeString() + ', ' + d.toLocaleDateString();
            this.verbose(timestamp);
        }
    };
    return Logger;
}());
exports.Logger = Logger;
exports.logger = new Logger();
/**
 * Manages logging, whether to console.log, file, or VS Code console.
 * Encapsulates the state specific to each logging session
 */
var InternalLogger = (function () {
    function InternalLogger(logCallback, logFilePath, isServer) {
        this._logCallback = logCallback;
        this._logFilePath = logFilePath;
        this._logToConsole = isServer;
        this._minLogLevel = LogLevel.Warn;
    }
    InternalLogger.prototype.setup = function (consoleMinLogLevel, logToFile) {
        var _this = this;
        this._minLogLevel = consoleMinLogLevel;
        // Open a log file in the specified location. Overwritten on each run.
        if (logToFile) {
            this.log("Verbose logs are written to:\n", LogLevel.Warn);
            this.log(this._logFilePath + '\n', LogLevel.Warn);
            this._logFileStream = fs.createWriteStream(this._logFilePath);
            this._logFileStream.on('error', function (e) {
                _this.sendLog("Error involving log file at path: " + _this._logFilePath + ". Error: " + e.toString(), LogLevel.Error);
            });
        }
    };
    InternalLogger.prototype.log = function (msg, level) {
        if (level >= this._minLogLevel) {
            this.sendLog(msg, level);
        }
        if (this._logToConsole) {
            var logFn = level === LogLevel.Error ? console.error :
                level === LogLevel.Warn ? console.warn :
                    console.log;
            logFn(trimLastNewline(msg));
        }
        // If an error, prepend with '[Error]'
        if (level === LogLevel.Error) {
            msg = "[" + LogLevel[level] + "] " + msg;
        }
        if (this._logFileStream) {
            this._logFileStream.write(msg);
        }
    };
    InternalLogger.prototype.sendLog = function (msg, level) {
        // Truncate long messages, they can hang VS Code
        if (msg.length > 1500) {
            var endsInNewline = !!msg.match(/(\n|\r\n)$/);
            msg = msg.substr(0, 1500) + '[...]';
            if (endsInNewline) {
                msg = msg + '\n';
            }
        }
        if (this._logCallback) {
            var event_1 = new LogOutputEvent(msg, level);
            this._logCallback(event_1);
        }
    };
    return InternalLogger;
}());
var LogOutputEvent = (function (_super) {
    __extends(LogOutputEvent, _super);
    function LogOutputEvent(msg, level) {
        var category = level === LogLevel.Error ? 'stderr' :
            level === LogLevel.Warn ? 'console' :
                'stdout';
        _super.call(this, msg, category);
    }
    return LogOutputEvent;
}(debugSession_1.OutputEvent));
exports.LogOutputEvent = LogOutputEvent;
function trimLastNewline(str) {
    return str.replace(/(\n|\r\n)$/, '');
}
exports.trimLastNewline = trimLastNewline;
//# sourceMappingURL=data:application/json;base64,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