/**
 * Copyright (c) 2012-2015, Christopher Jeffrey (MIT License)
 * Copyright (c) 2016, Daniel Imms (MIT License).
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b)
        if (b.hasOwnProperty(p))
            d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var extend = require("extend");
var net = require("net");
var path = require("path");
var terminal_1 = require("./terminal");
var pty;
try {
    pty = require(path.join('..', 'build', 'Release', 'pty.node'));
}
catch (e) {
    pty = require(path.join('..', 'build', 'Debug', 'pty.node'));
}
;
var UnixTerminal = (function (_super) {
    __extends(UnixTerminal, _super);
    function UnixTerminal(file, args, opt) {
        var _this = _super.call(this) || this;
        // Initialize arguments
        args = args || [];
        file = file || 'sh';
        opt = opt || {};
        opt.env = opt.env || process.env;
        var cols = opt.cols || terminal_1.Terminal.DEFAULT_COLS;
        var rows = opt.rows || terminal_1.Terminal.DEFAULT_ROWS;
        var uid = opt.uid != null ? opt.uid : -1;
        var gid = opt.gid != null ? opt.gid : -1;
        var env = extend({}, opt.env);
        if (opt.env === process.env) {
            _this._sanitizeEnv(env);
        }
        var cwd = opt.cwd || process.cwd();
        var name = opt.name || env.TERM || 'xterm';
        env.TERM = name;
        var parsedEnv = _this._parseEnv(env);
        var onexit = function (code, signal) {
            // XXX Sometimes a data event is emitted after exit. Wait til socket is
            // destroyed.
            if (!_this._emittedClose) {
                if (_this._boundClose)
                    return;
                _this._boundClose = true;
                _this.once('close', function () { return _this.emit('exit', code, signal); });
                return;
            }
            _this.emit('exit', code, signal);
        };
        // fork
        var term = pty.fork(file, args, parsedEnv, cwd, cols, rows, uid, gid, onexit);
        _this.socket = new PipeSocket(term.fd);
        _this.socket.setEncoding('utf8');
        _this.socket.resume();
        // setup
        _this.socket.on('error', function (err) {
            // NOTE: fs.ReadStream gets EAGAIN twice at first:
            if (err.code) {
                if (~err.code.indexOf('EAGAIN')) {
                    return;
                }
            }
            // close
            _this._close();
            // EIO on exit from fs.ReadStream:
            if (!_this._emittedClose) {
                _this._emittedClose = true;
                _this.emit('close');
            }
            // EIO, happens when someone closes our child process: the only process in
            // the terminal.
            // node < 0.6.14: errno 5
            // node >= 0.6.14: read EIO
            if (err.code) {
                if (~err.code.indexOf('errno 5') || ~err.code.indexOf('EIO')) {
                    return;
                }
            }
            // throw anything else
            if (_this.listeners('error').length < 2) {
                throw err;
            }
        });
        _this.pid = term.pid;
        _this.fd = term.fd;
        _this.pty = term.pty;
        _this.file = file;
        _this.name = name;
        _this.readable = true;
        _this.writable = true;
        _this.socket.on('close', function () {
            if (_this._emittedClose) {
                return;
            }
            _this._emittedClose = true;
            _this._close();
            _this.emit('close');
        });
        return _this;
    }
    /**
     * openpty
     */
    UnixTerminal.open = function (opt) {
        var self = Object.create(UnixTerminal.prototype);
        opt = opt || {};
        if (arguments.length > 1) {
            opt = {
                cols: arguments[1],
                rows: arguments[2]
            };
        }
        var cols = opt.cols || terminal_1.Terminal.DEFAULT_COLS;
        var rows = opt.rows || terminal_1.Terminal.DEFAULT_ROWS;
        // open
        var term = pty.open(cols, rows);
        self.master = new PipeSocket(term.master);
        self.master.setEncoding('utf8');
        self.master.resume();
        self.slave = new PipeSocket(term.slave);
        self.slave.setEncoding('utf8');
        self.slave.resume();
        self.socket = self.master;
        self.pid = null;
        self.fd = term.master;
        self.pty = term.pty;
        self.file = process.argv[0] || 'node';
        self.name = process.env.TERM || '';
        self.readable = true;
        self.writable = true;
        self.socket.on('error', function (err) {
            self._close();
            if (self.listeners('error').length < 2) {
                throw err;
            }
        });
        self.socket.on('close', function () {
            self._close();
        });
        return self;
    };
    ;
    UnixTerminal.prototype.write = function (data) {
        this.socket.write(data);
    };
    UnixTerminal.prototype.destroy = function () {
        var _this = this;
        this._close();
        // Need to close the read stream so node stops reading a dead file
        // descriptor. Then we can safely SIGHUP the shell.
        this.socket.once('close', function () {
            _this.kill('SIGHUP');
        });
        this.socket.destroy();
    };
    UnixTerminal.prototype.kill = function (signal) {
        try {
            process.kill(this.pid, signal || 'SIGHUP');
        }
        catch (e) { }
    };
    Object.defineProperty(UnixTerminal.prototype, "process", {
        /**
         * Gets the name of the process.
         */
        get: function () {
            return pty.process(this.fd, this.pty) || this.file;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * TTY
     */
    UnixTerminal.prototype.resize = function (cols, rows) {
        pty.resize(this.fd, cols, rows);
    };
    UnixTerminal.prototype._sanitizeEnv = function (env) {
        // Make sure we didn't start our server from inside tmux.
        delete env['TMUX'];
        delete env['TMUX_PANE'];
        // Make sure we didn't start our server from inside screen.
        // http://web.mit.edu/gnu/doc/html/screen_20.html
        delete env['STY'];
        delete env['WINDOW'];
        // Delete some variables that might confuse our terminal.
        delete env['WINDOWID'];
        delete env['TERMCAP'];
        delete env['COLUMNS'];
        delete env['LINES'];
    };
    return UnixTerminal;
}(terminal_1.Terminal));
exports.UnixTerminal = UnixTerminal;
/**
 * Wraps net.Socket to force the handle type "PIPE" by temporarily overwriting
 * tty_wrap.guessHandleType.
 * See: https://github.com/chjj/pty.js/issues/103
 */
var PipeSocket = (function (_super) {
    __extends(PipeSocket, _super);
    function PipeSocket(fd) {
        var _this = this;
        var tty = process.binding('tty_wrap');
        var guessHandleType = tty.guessHandleType;
        tty.guessHandleType = function () { return 'PIPE'; };
        _this = _super.call(this, { fd: fd }) || this;
        tty.guessHandleType = guessHandleType;
        return _this;
    }
    return PipeSocket;
}(net.Socket));
//# sourceMappingURL=unixTerminal.js.map 
//# sourceMappingURL=unixTerminal.js.map