/**
 * Copyright (c) 2012-2015, Christopher Jeffrey, Peter Sunde (MIT License)
 * Copyright (c) 2016, Daniel Imms (MIT License).
 */
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b)
        if (b.hasOwnProperty(p))
            d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var extend = require("extend");
var terminal_1 = require("./terminal");
var windowsPtyAgent_1 = require("./windowsPtyAgent");
var WindowsTerminal = (function (_super) {
    __extends(WindowsTerminal, _super);
    function WindowsTerminal(file, args, opt) {
        var _this = _super.call(this) || this;
        // Initialize arguments
        args = args || [];
        file = file || 'cmd.exe';
        opt = opt || {};
        opt.env = opt.env || process.env;
        var env = extend({}, opt.env);
        var cols = opt.cols || terminal_1.Terminal.DEFAULT_COLS;
        var rows = opt.rows || terminal_1.Terminal.DEFAULT_ROWS;
        var cwd = opt.cwd || process.cwd();
        var name = opt.name || env.TERM || 'Windows Shell';
        var parsedEnv = _this._parseEnv(env);
        // If the terminal is ready
        _this.isReady = false;
        // Functions that need to run after `ready` event is emitted.
        _this.deferreds = [];
        // Create new termal.
        _this.agent = new windowsPtyAgent_1.WindowsPtyAgent(file, args, parsedEnv, cwd, cols, rows, false);
        _this.socket = _this.agent.outSocket;
        // Not available until `ready` event emitted.
        _this.pid = _this.agent.pid;
        _this.fd = _this.agent.fd;
        _this.pty = _this.agent.pty;
        // The forked windows terminal is not available until `ready` event is
        // emitted.
        _this.socket.on('ready_datapipe', function () {
            // These events needs to be forwarded.
            ['connect', 'data', 'end', 'timeout', 'drain'].forEach(function (event) {
                _this.socket.on(event, function (data) {
                    // Wait until the first data event is fired then we can run deferreds.
                    if (!_this.isReady && event === 'data') {
                        // Terminal is now ready and we can avoid having to defer method
                        // calls.
                        _this.isReady = true;
                        // Execute all deferred methods
                        _this.deferreds.forEach(function (fn) {
                            // NB! In order to ensure that `this` has all its references
                            // updated any variable that need to be available in `this` before
                            // the deferred is run has to be declared above this forEach
                            // statement.
                            fn.run();
                        });
                        // Reset
                        _this.deferreds = [];
                    }
                });
            });
            // Resume socket.
            _this.socket.resume();
            // Shutdown if `error` event is emitted.
            _this.socket.on('error', function (err) {
                // Close terminal session.
                _this._close();
                // EIO, happens when someone closes our child process: the only process
                // in the terminal.
                // node < 0.6.14: errno 5
                // node >= 0.6.14: read EIO
                if (err.code) {
                    if (~err.code.indexOf('errno 5') || ~err.code.indexOf('EIO'))
                        return;
                }
                // Throw anything else.
                if (_this.listeners('error').length < 2) {
                    throw err;
                }
            });
            // Cleanup after the socket is closed.
            _this.socket.on('close', function () {
                _this.emit('exit', null);
                _this._close();
            });
        });
        _this.file = file;
        _this.name = name;
        _this.readable = true;
        _this.writable = true;
        return _this;
    }
    /**
     * openpty
     */
    WindowsTerminal.open = function (options) {
        throw new Error('open() not supported on windows, use Fork() instead.');
    };
    /**
     * Events
     */
    WindowsTerminal.prototype.write = function (data) {
        var _this = this;
        this._defer(function () {
            _this.agent.inSocket.write(data);
        });
    };
    /**
     * TTY
     */
    WindowsTerminal.prototype.resize = function (cols, rows) {
        var _this = this;
        this._defer(function () {
            _this.agent.resize(cols, rows);
        });
    };
    WindowsTerminal.prototype.destroy = function () {
        var _this = this;
        this._defer(function () {
            _this.kill();
        });
    };
    WindowsTerminal.prototype.kill = function (signal) {
        var _this = this;
        this._defer(function () {
            if (signal) {
                throw new Error('Signals not supported on windows.');
            }
            _this._close();
            _this.agent.kill();
        });
    };
    WindowsTerminal.prototype._defer = function (deferredFn) {
        var _this = this;
        // Ensure that this method is only used within Terminal class.
        if (!(this instanceof WindowsTerminal)) {
            throw new Error('Must be instanceof WindowsTerminal');
        }
        // If the terminal is ready, execute.
        if (this.isReady) {
            deferredFn.apply(this, null);
            return;
        }
        // Queue until terminal is ready.
        this.deferreds.push({
            run: function () { return deferredFn.apply(_this, null); }
        });
    };
    Object.defineProperty(WindowsTerminal.prototype, "process", {
        /**
         * Gets the name of the process.
         */
        get: function () { return this.name; },
        enumerable: true,
        configurable: true
    });
    return WindowsTerminal;
}(terminal_1.Terminal));
exports.WindowsTerminal = WindowsTerminal;
//# sourceMappingURL=windowsTerminal.js.map 
//# sourceMappingURL=windowsTerminal.js.map