/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import java.util.Arrays;
import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.exception.util.LocalizedFormats;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.BlockRealMatrix;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.QRDecomposition;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.util.FastMath;

public class QRDecompositionImpl
implements QRDecomposition {
    private double[][] qrt;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;

    public QRDecompositionImpl(RealMatrix matrix) {
        int m = matrix.getRowDimension();
        int n = matrix.getColumnDimension();
        this.qrt = matrix.transpose().getData();
        this.rDiag = new double[FastMath.min(m, n)];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
            double a2;
            double[] qrtMinor = this.qrt[minor];
            double xNormSqr = 0.0;
            for (int row = minor; row < m; ++row) {
                double c2 = qrtMinor[row];
                xNormSqr += c2 * c2;
            }
            this.rDiag[minor] = a2 = qrtMinor[minor] > 0.0 ? -FastMath.sqrt(xNormSqr) : FastMath.sqrt(xNormSqr);
            if (a2 == 0.0) continue;
            int n2 = minor;
            qrtMinor[n2] = qrtMinor[n2] - a2;
            for (int col = minor + 1; col < n; ++col) {
                int row;
                double[] qrtCol = this.qrt[col];
                double alpha = 0.0;
                for (row = minor; row < m; ++row) {
                    alpha -= qrtCol[row] * qrtMinor[row];
                }
                alpha /= a2 * qrtMinor[minor];
                for (row = minor; row < m; ++row) {
                    int n3 = row;
                    qrtCol[n3] = qrtCol[n3] - alpha * qrtMinor[row];
                }
            }
        }
    }

    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedR = MatrixUtils.createRealMatrix(m, n);
            for (int row = FastMath.min(m, n) - 1; row >= 0; --row) {
                this.cachedR.setEntry(row, row, this.rDiag[row]);
                for (int col = row + 1; col < n; ++col) {
                    this.cachedR.setEntry(row, col, this.qrt[col][row]);
                }
            }
        }
        return this.cachedR;
    }

    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int minor;
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedQT = MatrixUtils.createRealMatrix(m, m);
            for (minor = m - 1; minor >= FastMath.min(m, n); --minor) {
                this.cachedQT.setEntry(minor, minor, 1.0);
            }
            for (minor = FastMath.min(m, n) - 1; minor >= 0; --minor) {
                double[] qrtMinor = this.qrt[minor];
                this.cachedQT.setEntry(minor, minor, 1.0);
                if (qrtMinor[minor] == 0.0) continue;
                for (int col = minor; col < m; ++col) {
                    int row;
                    double alpha = 0.0;
                    for (row = minor; row < m; ++row) {
                        alpha -= this.cachedQT.getEntry(col, row) * qrtMinor[row];
                    }
                    alpha /= this.rDiag[minor] * qrtMinor[minor];
                    for (row = minor; row < m; ++row) {
                        this.cachedQT.addToEntry(col, row, -alpha * qrtMinor[row]);
                    }
                }
            }
        }
        return this.cachedQT;
    }

    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedH = MatrixUtils.createRealMatrix(m, n);
            for (int i = 0; i < m; ++i) {
                for (int j = 0; j < FastMath.min(i + 1, n); ++j) {
                    this.cachedH.setEntry(i, j, this.qrt[j][i] / -this.rDiag[j]);
                }
            }
        }
        return this.cachedH;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.qrt, this.rDiag);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qrt;
        private final double[] rDiag;

        private Solver(double[][] qrt, double[] rDiag) {
            this.qrt = qrt;
            this.rDiag = rDiag;
        }

        public boolean isNonSingular() {
            for (double diag : this.rDiag) {
                if (diag != 0.0) continue;
                return false;
            }
            return true;
        }

        public double[] solve(double[] b2) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b2.length != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b2.length, m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x = new double[n];
            double[] y = (double[])b2.clone();
            for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
                int row;
                double[] qrtMinor = this.qrt[minor];
                double dotProduct = 0.0;
                for (row = minor; row < m; ++row) {
                    dotProduct += y[row] * qrtMinor[row];
                }
                dotProduct /= this.rDiag[minor] * qrtMinor[minor];
                for (row = minor; row < m; ++row) {
                    int n2 = row;
                    y[n2] = y[n2] + dotProduct * qrtMinor[row];
                }
            }
            for (int row = this.rDiag.length - 1; row >= 0; --row) {
                int n3 = row;
                y[n3] = y[n3] / this.rDiag[row];
                double yRow = y[row];
                double[] qrtRow = this.qrt[row];
                x[row] = yRow;
                for (int i = 0; i < row; ++i) {
                    int n4 = i;
                    y[n4] = y[n4] - yRow * qrtRow[i];
                }
            }
            return x;
        }

        public RealVector solve(RealVector b2) throws IllegalArgumentException, InvalidMatrixException {
            try {
                return this.solve((ArrayRealVector)b2);
            }
            catch (ClassCastException cce) {
                return new ArrayRealVector(this.solve(b2.getData()), false);
            }
        }

        public ArrayRealVector solve(ArrayRealVector b2) throws IllegalArgumentException, InvalidMatrixException {
            return new ArrayRealVector(this.solve(b2.getDataRef()), false);
        }

        public RealMatrix solve(RealMatrix b2) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b2.getRowDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.DIMENSIONS_MISMATCH_2x2, b2.getRowDimension(), b2.getColumnDimension(), m, "n");
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b2.getColumnDimension();
            int blockSize = 52;
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(n, columns);
            double[][] y = new double[b2.getRowDimension()][52];
            double[] alpha = new double[52];
            for (int kBlock = 0; kBlock < cBlocks; ++kBlock) {
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b2.copySubMatrix(0, m - 1, kStart, kEnd - 1, y);
                for (int minor = 0; minor < FastMath.min(m, n); ++minor) {
                    int k;
                    double[] yRow;
                    double d2;
                    int row;
                    double[] qrtMinor = this.qrt[minor];
                    double factor = 1.0 / (this.rDiag[minor] * qrtMinor[minor]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    for (row = minor; row < m; ++row) {
                        d2 = qrtMinor[row];
                        yRow = y[row];
                        for (k = 0; k < kWidth; ++k) {
                            int n2 = k;
                            alpha[n2] = alpha[n2] + d2 * yRow[k];
                        }
                    }
                    int k2 = 0;
                    while (k2 < kWidth) {
                        int n3 = k2++;
                        alpha[n3] = alpha[n3] * factor;
                    }
                    for (row = minor; row < m; ++row) {
                        d2 = qrtMinor[row];
                        yRow = y[row];
                        for (k = 0; k < kWidth; ++k) {
                            int n4 = k;
                            yRow[n4] = yRow[n4] + alpha[k] * d2;
                        }
                    }
                }
                for (int j = this.rDiag.length - 1; j >= 0; --j) {
                    int jBlock = j / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[j];
                    double[] yJ = y[j];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (j - jStart) * kWidth;
                    for (int k = 0; k < kWidth; ++k) {
                        int n5 = k;
                        yJ[n5] = yJ[n5] * factor;
                        xBlock[index++] = yJ[k];
                    }
                    double[] qrtJ = this.qrt[j];
                    for (int i = 0; i < j; ++i) {
                        double rIJ = qrtJ[i];
                        double[] yI = y[i];
                        for (int k = 0; k < kWidth; ++k) {
                            int n6 = k;
                            yI[n6] = yI[n6] - yJ[k] * rIJ;
                        }
                    }
                }
            }
            return new BlockRealMatrix(n, columns, xBlocks, false);
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.rDiag.length));
        }
    }
}

